'use strict';

var obsidian = require('obsidian');
var state = require('@codemirror/state');
var view = require('@codemirror/view');
var language = require('@codemirror/language');

// these aren't really private, but nor are they really useful to document

/**
 * @private
 */
class LuxonError extends Error {}

/**
 * @private
 */
class InvalidDateTimeError extends LuxonError {
  constructor(reason) {
    super(`Invalid DateTime: ${reason.toMessage()}`);
  }
}

/**
 * @private
 */
class InvalidIntervalError extends LuxonError {
  constructor(reason) {
    super(`Invalid Interval: ${reason.toMessage()}`);
  }
}

/**
 * @private
 */
class InvalidDurationError extends LuxonError {
  constructor(reason) {
    super(`Invalid Duration: ${reason.toMessage()}`);
  }
}

/**
 * @private
 */
class ConflictingSpecificationError extends LuxonError {}

/**
 * @private
 */
class InvalidUnitError extends LuxonError {
  constructor(unit) {
    super(`Invalid unit ${unit}`);
  }
}

/**
 * @private
 */
class InvalidArgumentError extends LuxonError {}

/**
 * @private
 */
class ZoneIsAbstractError extends LuxonError {
  constructor() {
    super("Zone is an abstract class");
  }
}

/**
 * @private
 */

const n$1 = "numeric",
  s$2 = "short",
  l$2 = "long";

const DATE_SHORT = {
  year: n$1,
  month: n$1,
  day: n$1,
};

const DATE_MED = {
  year: n$1,
  month: s$2,
  day: n$1,
};

const DATE_MED_WITH_WEEKDAY = {
  year: n$1,
  month: s$2,
  day: n$1,
  weekday: s$2,
};

const DATE_FULL = {
  year: n$1,
  month: l$2,
  day: n$1,
};

const DATE_HUGE = {
  year: n$1,
  month: l$2,
  day: n$1,
  weekday: l$2,
};

const TIME_SIMPLE = {
  hour: n$1,
  minute: n$1,
};

const TIME_WITH_SECONDS = {
  hour: n$1,
  minute: n$1,
  second: n$1,
};

const TIME_WITH_SHORT_OFFSET = {
  hour: n$1,
  minute: n$1,
  second: n$1,
  timeZoneName: s$2,
};

const TIME_WITH_LONG_OFFSET = {
  hour: n$1,
  minute: n$1,
  second: n$1,
  timeZoneName: l$2,
};

const TIME_24_SIMPLE = {
  hour: n$1,
  minute: n$1,
  hourCycle: "h23",
};

const TIME_24_WITH_SECONDS = {
  hour: n$1,
  minute: n$1,
  second: n$1,
  hourCycle: "h23",
};

const TIME_24_WITH_SHORT_OFFSET = {
  hour: n$1,
  minute: n$1,
  second: n$1,
  hourCycle: "h23",
  timeZoneName: s$2,
};

const TIME_24_WITH_LONG_OFFSET = {
  hour: n$1,
  minute: n$1,
  second: n$1,
  hourCycle: "h23",
  timeZoneName: l$2,
};

const DATETIME_SHORT = {
  year: n$1,
  month: n$1,
  day: n$1,
  hour: n$1,
  minute: n$1,
};

const DATETIME_SHORT_WITH_SECONDS = {
  year: n$1,
  month: n$1,
  day: n$1,
  hour: n$1,
  minute: n$1,
  second: n$1,
};

const DATETIME_MED = {
  year: n$1,
  month: s$2,
  day: n$1,
  hour: n$1,
  minute: n$1,
};

const DATETIME_MED_WITH_SECONDS = {
  year: n$1,
  month: s$2,
  day: n$1,
  hour: n$1,
  minute: n$1,
  second: n$1,
};

const DATETIME_MED_WITH_WEEKDAY = {
  year: n$1,
  month: s$2,
  day: n$1,
  weekday: s$2,
  hour: n$1,
  minute: n$1,
};

const DATETIME_FULL = {
  year: n$1,
  month: l$2,
  day: n$1,
  hour: n$1,
  minute: n$1,
  timeZoneName: s$2,
};

const DATETIME_FULL_WITH_SECONDS = {
  year: n$1,
  month: l$2,
  day: n$1,
  hour: n$1,
  minute: n$1,
  second: n$1,
  timeZoneName: s$2,
};

const DATETIME_HUGE = {
  year: n$1,
  month: l$2,
  day: n$1,
  weekday: l$2,
  hour: n$1,
  minute: n$1,
  timeZoneName: l$2,
};

const DATETIME_HUGE_WITH_SECONDS = {
  year: n$1,
  month: l$2,
  day: n$1,
  weekday: l$2,
  hour: n$1,
  minute: n$1,
  second: n$1,
  timeZoneName: l$2,
};

/**
 * @interface
 */
class Zone {
  /**
   * The type of zone
   * @abstract
   * @type {string}
   */
  get type() {
    throw new ZoneIsAbstractError();
  }

  /**
   * The name of this zone.
   * @abstract
   * @type {string}
   */
  get name() {
    throw new ZoneIsAbstractError();
  }

  get ianaName() {
    return this.name;
  }

  /**
   * Returns whether the offset is known to be fixed for the whole year.
   * @abstract
   * @type {boolean}
   */
  get isUniversal() {
    throw new ZoneIsAbstractError();
  }

  /**
   * Returns the offset's common name (such as EST) at the specified timestamp
   * @abstract
   * @param {number} ts - Epoch milliseconds for which to get the name
   * @param {Object} opts - Options to affect the format
   * @param {string} opts.format - What style of offset to return. Accepts 'long' or 'short'.
   * @param {string} opts.locale - What locale to return the offset name in.
   * @return {string}
   */
  offsetName(ts, opts) {
    throw new ZoneIsAbstractError();
  }

  /**
   * Returns the offset's value as a string
   * @abstract
   * @param {number} ts - Epoch milliseconds for which to get the offset
   * @param {string} format - What style of offset to return.
   *                          Accepts 'narrow', 'short', or 'techie'. Returning '+6', '+06:00', or '+0600' respectively
   * @return {string}
   */
  formatOffset(ts, format) {
    throw new ZoneIsAbstractError();
  }

  /**
   * Return the offset in minutes for this zone at the specified timestamp.
   * @abstract
   * @param {number} ts - Epoch milliseconds for which to compute the offset
   * @return {number}
   */
  offset(ts) {
    throw new ZoneIsAbstractError();
  }

  /**
   * Return whether this Zone is equal to another zone
   * @abstract
   * @param {Zone} otherZone - the zone to compare
   * @return {boolean}
   */
  equals(otherZone) {
    throw new ZoneIsAbstractError();
  }

  /**
   * Return whether this Zone is valid.
   * @abstract
   * @type {boolean}
   */
  get isValid() {
    throw new ZoneIsAbstractError();
  }
}

let singleton$1 = null;

/**
 * Represents the local zone for this JavaScript environment.
 * @implements {Zone}
 */
class SystemZone extends Zone {
  /**
   * Get a singleton instance of the local zone
   * @return {SystemZone}
   */
  static get instance() {
    if (singleton$1 === null) {
      singleton$1 = new SystemZone();
    }
    return singleton$1;
  }

  /** @override **/
  get type() {
    return "system";
  }

  /** @override **/
  get name() {
    return new Intl.DateTimeFormat().resolvedOptions().timeZone;
  }

  /** @override **/
  get isUniversal() {
    return false;
  }

  /** @override **/
  offsetName(ts, { format, locale }) {
    return parseZoneInfo(ts, format, locale);
  }

  /** @override **/
  formatOffset(ts, format) {
    return formatOffset(this.offset(ts), format);
  }

  /** @override **/
  offset(ts) {
    return -new Date(ts).getTimezoneOffset();
  }

  /** @override **/
  equals(otherZone) {
    return otherZone.type === "system";
  }

  /** @override **/
  get isValid() {
    return true;
  }
}

let dtfCache = {};
function makeDTF(zone) {
  if (!dtfCache[zone]) {
    dtfCache[zone] = new Intl.DateTimeFormat("en-US", {
      hour12: false,
      timeZone: zone,
      year: "numeric",
      month: "2-digit",
      day: "2-digit",
      hour: "2-digit",
      minute: "2-digit",
      second: "2-digit",
      era: "short",
    });
  }
  return dtfCache[zone];
}

const typeToPos = {
  year: 0,
  month: 1,
  day: 2,
  era: 3,
  hour: 4,
  minute: 5,
  second: 6,
};

function hackyOffset(dtf, date) {
  const formatted = dtf.format(date).replace(/\u200E/g, ""),
    parsed = /(\d+)\/(\d+)\/(\d+) (AD|BC),? (\d+):(\d+):(\d+)/.exec(formatted),
    [, fMonth, fDay, fYear, fadOrBc, fHour, fMinute, fSecond] = parsed;
  return [fYear, fMonth, fDay, fadOrBc, fHour, fMinute, fSecond];
}

function partsOffset(dtf, date) {
  const formatted = dtf.formatToParts(date);
  const filled = [];
  for (let i = 0; i < formatted.length; i++) {
    const { type, value } = formatted[i];
    const pos = typeToPos[type];

    if (type === "era") {
      filled[pos] = value;
    } else if (!isUndefined(pos)) {
      filled[pos] = parseInt(value, 10);
    }
  }
  return filled;
}

let ianaZoneCache = {};
/**
 * A zone identified by an IANA identifier, like America/New_York
 * @implements {Zone}
 */
class IANAZone extends Zone {
  /**
   * @param {string} name - Zone name
   * @return {IANAZone}
   */
  static create(name) {
    if (!ianaZoneCache[name]) {
      ianaZoneCache[name] = new IANAZone(name);
    }
    return ianaZoneCache[name];
  }

  /**
   * Reset local caches. Should only be necessary in testing scenarios.
   * @return {void}
   */
  static resetCache() {
    ianaZoneCache = {};
    dtfCache = {};
  }

  /**
   * Returns whether the provided string is a valid specifier. This only checks the string's format, not that the specifier identifies a known zone; see isValidZone for that.
   * @param {string} s - The string to check validity on
   * @example IANAZone.isValidSpecifier("America/New_York") //=> true
   * @example IANAZone.isValidSpecifier("Sport~~blorp") //=> false
   * @deprecated This method returns false for some valid IANA names. Use isValidZone instead.
   * @return {boolean}
   */
  static isValidSpecifier(s) {
    return this.isValidZone(s);
  }

  /**
   * Returns whether the provided string identifies a real zone
   * @param {string} zone - The string to check
   * @example IANAZone.isValidZone("America/New_York") //=> true
   * @example IANAZone.isValidZone("Fantasia/Castle") //=> false
   * @example IANAZone.isValidZone("Sport~~blorp") //=> false
   * @return {boolean}
   */
  static isValidZone(zone) {
    if (!zone) {
      return false;
    }
    try {
      new Intl.DateTimeFormat("en-US", { timeZone: zone }).format();
      return true;
    } catch (e) {
      return false;
    }
  }

  constructor(name) {
    super();
    /** @private **/
    this.zoneName = name;
    /** @private **/
    this.valid = IANAZone.isValidZone(name);
  }

  /** @override **/
  get type() {
    return "iana";
  }

  /** @override **/
  get name() {
    return this.zoneName;
  }

  /** @override **/
  get isUniversal() {
    return false;
  }

  /** @override **/
  offsetName(ts, { format, locale }) {
    return parseZoneInfo(ts, format, locale, this.name);
  }

  /** @override **/
  formatOffset(ts, format) {
    return formatOffset(this.offset(ts), format);
  }

  /** @override **/
  offset(ts) {
    const date = new Date(ts);

    if (isNaN(date)) return NaN;

    const dtf = makeDTF(this.name);
    let [year, month, day, adOrBc, hour, minute, second] = dtf.formatToParts
      ? partsOffset(dtf, date)
      : hackyOffset(dtf, date);

    if (adOrBc === "BC") {
      year = -Math.abs(year) + 1;
    }

    // because we're using hour12 and https://bugs.chromium.org/p/chromium/issues/detail?id=1025564&can=2&q=%2224%3A00%22%20datetimeformat
    const adjustedHour = hour === 24 ? 0 : hour;

    const asUTC = objToLocalTS({
      year,
      month,
      day,
      hour: adjustedHour,
      minute,
      second,
      millisecond: 0,
    });

    let asTS = +date;
    const over = asTS % 1000;
    asTS -= over >= 0 ? over : 1000 + over;
    return (asUTC - asTS) / (60 * 1000);
  }

  /** @override **/
  equals(otherZone) {
    return otherZone.type === "iana" && otherZone.name === this.name;
  }

  /** @override **/
  get isValid() {
    return this.valid;
  }
}

// todo - remap caching

let intlLFCache = {};
function getCachedLF(locString, opts = {}) {
  const key = JSON.stringify([locString, opts]);
  let dtf = intlLFCache[key];
  if (!dtf) {
    dtf = new Intl.ListFormat(locString, opts);
    intlLFCache[key] = dtf;
  }
  return dtf;
}

let intlDTCache = {};
function getCachedDTF(locString, opts = {}) {
  const key = JSON.stringify([locString, opts]);
  let dtf = intlDTCache[key];
  if (!dtf) {
    dtf = new Intl.DateTimeFormat(locString, opts);
    intlDTCache[key] = dtf;
  }
  return dtf;
}

let intlNumCache = {};
function getCachedINF(locString, opts = {}) {
  const key = JSON.stringify([locString, opts]);
  let inf = intlNumCache[key];
  if (!inf) {
    inf = new Intl.NumberFormat(locString, opts);
    intlNumCache[key] = inf;
  }
  return inf;
}

let intlRelCache = {};
function getCachedRTF(locString, opts = {}) {
  const { base, ...cacheKeyOpts } = opts; // exclude `base` from the options
  const key = JSON.stringify([locString, cacheKeyOpts]);
  let inf = intlRelCache[key];
  if (!inf) {
    inf = new Intl.RelativeTimeFormat(locString, opts);
    intlRelCache[key] = inf;
  }
  return inf;
}

let sysLocaleCache = null;
function systemLocale() {
  if (sysLocaleCache) {
    return sysLocaleCache;
  } else {
    sysLocaleCache = new Intl.DateTimeFormat().resolvedOptions().locale;
    return sysLocaleCache;
  }
}

function parseLocaleString(localeStr) {
  // I really want to avoid writing a BCP 47 parser
  // see, e.g. https://github.com/wooorm/bcp-47
  // Instead, we'll do this:

  // a) if the string has no -u extensions, just leave it alone
  // b) if it does, use Intl to resolve everything
  // c) if Intl fails, try again without the -u

  // private subtags and unicode subtags have ordering requirements,
  // and we're not properly parsing this, so just strip out the
  // private ones if they exist.
  const xIndex = localeStr.indexOf("-x-");
  if (xIndex !== -1) {
    localeStr = localeStr.substring(0, xIndex);
  }

  const uIndex = localeStr.indexOf("-u-");
  if (uIndex === -1) {
    return [localeStr];
  } else {
    let options;
    let selectedStr;
    try {
      options = getCachedDTF(localeStr).resolvedOptions();
      selectedStr = localeStr;
    } catch (e) {
      const smaller = localeStr.substring(0, uIndex);
      options = getCachedDTF(smaller).resolvedOptions();
      selectedStr = smaller;
    }

    const { numberingSystem, calendar } = options;
    return [selectedStr, numberingSystem, calendar];
  }
}

function intlConfigString(localeStr, numberingSystem, outputCalendar) {
  if (outputCalendar || numberingSystem) {
    if (!localeStr.includes("-u-")) {
      localeStr += "-u";
    }

    if (outputCalendar) {
      localeStr += `-ca-${outputCalendar}`;
    }

    if (numberingSystem) {
      localeStr += `-nu-${numberingSystem}`;
    }
    return localeStr;
  } else {
    return localeStr;
  }
}

function mapMonths(f) {
  const ms = [];
  for (let i = 1; i <= 12; i++) {
    const dt = DateTime.utc(2009, i, 1);
    ms.push(f(dt));
  }
  return ms;
}

function mapWeekdays(f) {
  const ms = [];
  for (let i = 1; i <= 7; i++) {
    const dt = DateTime.utc(2016, 11, 13 + i);
    ms.push(f(dt));
  }
  return ms;
}

function listStuff(loc, length, englishFn, intlFn) {
  const mode = loc.listingMode();

  if (mode === "error") {
    return null;
  } else if (mode === "en") {
    return englishFn(length);
  } else {
    return intlFn(length);
  }
}

function supportsFastNumbers(loc) {
  if (loc.numberingSystem && loc.numberingSystem !== "latn") {
    return false;
  } else {
    return (
      loc.numberingSystem === "latn" ||
      !loc.locale ||
      loc.locale.startsWith("en") ||
      new Intl.DateTimeFormat(loc.intl).resolvedOptions().numberingSystem === "latn"
    );
  }
}

/**
 * @private
 */

class PolyNumberFormatter {
  constructor(intl, forceSimple, opts) {
    this.padTo = opts.padTo || 0;
    this.floor = opts.floor || false;

    const { padTo, floor, ...otherOpts } = opts;

    if (!forceSimple || Object.keys(otherOpts).length > 0) {
      const intlOpts = { useGrouping: false, ...opts };
      if (opts.padTo > 0) intlOpts.minimumIntegerDigits = opts.padTo;
      this.inf = getCachedINF(intl, intlOpts);
    }
  }

  format(i) {
    if (this.inf) {
      const fixed = this.floor ? Math.floor(i) : i;
      return this.inf.format(fixed);
    } else {
      // to match the browser's numberformatter defaults
      const fixed = this.floor ? Math.floor(i) : roundTo(i, 3);
      return padStart(fixed, this.padTo);
    }
  }
}

/**
 * @private
 */

class PolyDateFormatter {
  constructor(dt, intl, opts) {
    this.opts = opts;
    this.originalZone = undefined;

    let z = undefined;
    if (this.opts.timeZone) {
      // Don't apply any workarounds if a timeZone is explicitly provided in opts
      this.dt = dt;
    } else if (dt.zone.type === "fixed") {
      // UTC-8 or Etc/UTC-8 are not part of tzdata, only Etc/GMT+8 and the like.
      // That is why fixed-offset TZ is set to that unless it is:
      // 1. Representing offset 0 when UTC is used to maintain previous behavior and does not become GMT.
      // 2. Unsupported by the browser:
      //    - some do not support Etc/
      //    - < Etc/GMT-14, > Etc/GMT+12, and 30-minute or 45-minute offsets are not part of tzdata
      const gmtOffset = -1 * (dt.offset / 60);
      const offsetZ = gmtOffset >= 0 ? `Etc/GMT+${gmtOffset}` : `Etc/GMT${gmtOffset}`;
      if (dt.offset !== 0 && IANAZone.create(offsetZ).valid) {
        z = offsetZ;
        this.dt = dt;
      } else {
        // Not all fixed-offset zones like Etc/+4:30 are present in tzdata so
        // we manually apply the offset and substitute the zone as needed.
        z = "UTC";
        this.dt = dt.offset === 0 ? dt : dt.setZone("UTC").plus({ minutes: dt.offset });
        this.originalZone = dt.zone;
      }
    } else if (dt.zone.type === "system") {
      this.dt = dt;
    } else if (dt.zone.type === "iana") {
      this.dt = dt;
      z = dt.zone.name;
    } else {
      // Custom zones can have any offset / offsetName so we just manually
      // apply the offset and substitute the zone as needed.
      z = "UTC";
      this.dt = dt.setZone("UTC").plus({ minutes: dt.offset });
      this.originalZone = dt.zone;
    }

    const intlOpts = { ...this.opts };
    intlOpts.timeZone = intlOpts.timeZone || z;
    this.dtf = getCachedDTF(intl, intlOpts);
  }

  format() {
    if (this.originalZone) {
      // If we have to substitute in the actual zone name, we have to use
      // formatToParts so that the timezone can be replaced.
      return this.formatToParts()
        .map(({ value }) => value)
        .join("");
    }
    return this.dtf.format(this.dt.toJSDate());
  }

  formatToParts() {
    const parts = this.dtf.formatToParts(this.dt.toJSDate());
    if (this.originalZone) {
      return parts.map((part) => {
        if (part.type === "timeZoneName") {
          const offsetName = this.originalZone.offsetName(this.dt.ts, {
            locale: this.dt.locale,
            format: this.opts.timeZoneName,
          });
          return {
            ...part,
            value: offsetName,
          };
        } else {
          return part;
        }
      });
    }
    return parts;
  }

  resolvedOptions() {
    return this.dtf.resolvedOptions();
  }
}

/**
 * @private
 */
class PolyRelFormatter {
  constructor(intl, isEnglish, opts) {
    this.opts = { style: "long", ...opts };
    if (!isEnglish && hasRelative()) {
      this.rtf = getCachedRTF(intl, opts);
    }
  }

  format(count, unit) {
    if (this.rtf) {
      return this.rtf.format(count, unit);
    } else {
      return formatRelativeTime(unit, count, this.opts.numeric, this.opts.style !== "long");
    }
  }

  formatToParts(count, unit) {
    if (this.rtf) {
      return this.rtf.formatToParts(count, unit);
    } else {
      return [];
    }
  }
}

/**
 * @private
 */

class Locale {
  static fromOpts(opts) {
    return Locale.create(opts.locale, opts.numberingSystem, opts.outputCalendar, opts.defaultToEN);
  }

  static create(locale, numberingSystem, outputCalendar, defaultToEN = false) {
    const specifiedLocale = locale || Settings.defaultLocale;
    // the system locale is useful for human readable strings but annoying for parsing/formatting known formats
    const localeR = specifiedLocale || (defaultToEN ? "en-US" : systemLocale());
    const numberingSystemR = numberingSystem || Settings.defaultNumberingSystem;
    const outputCalendarR = outputCalendar || Settings.defaultOutputCalendar;
    return new Locale(localeR, numberingSystemR, outputCalendarR, specifiedLocale);
  }

  static resetCache() {
    sysLocaleCache = null;
    intlDTCache = {};
    intlNumCache = {};
    intlRelCache = {};
  }

  static fromObject({ locale, numberingSystem, outputCalendar } = {}) {
    return Locale.create(locale, numberingSystem, outputCalendar);
  }

  constructor(locale, numbering, outputCalendar, specifiedLocale) {
    const [parsedLocale, parsedNumberingSystem, parsedOutputCalendar] = parseLocaleString(locale);

    this.locale = parsedLocale;
    this.numberingSystem = numbering || parsedNumberingSystem || null;
    this.outputCalendar = outputCalendar || parsedOutputCalendar || null;
    this.intl = intlConfigString(this.locale, this.numberingSystem, this.outputCalendar);

    this.weekdaysCache = { format: {}, standalone: {} };
    this.monthsCache = { format: {}, standalone: {} };
    this.meridiemCache = null;
    this.eraCache = {};

    this.specifiedLocale = specifiedLocale;
    this.fastNumbersCached = null;
  }

  get fastNumbers() {
    if (this.fastNumbersCached == null) {
      this.fastNumbersCached = supportsFastNumbers(this);
    }

    return this.fastNumbersCached;
  }

  listingMode() {
    const isActuallyEn = this.isEnglish();
    const hasNoWeirdness =
      (this.numberingSystem === null || this.numberingSystem === "latn") &&
      (this.outputCalendar === null || this.outputCalendar === "gregory");
    return isActuallyEn && hasNoWeirdness ? "en" : "intl";
  }

  clone(alts) {
    if (!alts || Object.getOwnPropertyNames(alts).length === 0) {
      return this;
    } else {
      return Locale.create(
        alts.locale || this.specifiedLocale,
        alts.numberingSystem || this.numberingSystem,
        alts.outputCalendar || this.outputCalendar,
        alts.defaultToEN || false
      );
    }
  }

  redefaultToEN(alts = {}) {
    return this.clone({ ...alts, defaultToEN: true });
  }

  redefaultToSystem(alts = {}) {
    return this.clone({ ...alts, defaultToEN: false });
  }

  months(length, format = false) {
    return listStuff(this, length, months, () => {
      const intl = format ? { month: length, day: "numeric" } : { month: length },
        formatStr = format ? "format" : "standalone";
      if (!this.monthsCache[formatStr][length]) {
        this.monthsCache[formatStr][length] = mapMonths((dt) => this.extract(dt, intl, "month"));
      }
      return this.monthsCache[formatStr][length];
    });
  }

  weekdays(length, format = false) {
    return listStuff(this, length, weekdays, () => {
      const intl = format
          ? { weekday: length, year: "numeric", month: "long", day: "numeric" }
          : { weekday: length },
        formatStr = format ? "format" : "standalone";
      if (!this.weekdaysCache[formatStr][length]) {
        this.weekdaysCache[formatStr][length] = mapWeekdays((dt) =>
          this.extract(dt, intl, "weekday")
        );
      }
      return this.weekdaysCache[formatStr][length];
    });
  }

  meridiems() {
    return listStuff(
      this,
      undefined,
      () => meridiems,
      () => {
        // In theory there could be aribitrary day periods. We're gonna assume there are exactly two
        // for AM and PM. This is probably wrong, but it's makes parsing way easier.
        if (!this.meridiemCache) {
          const intl = { hour: "numeric", hourCycle: "h12" };
          this.meridiemCache = [DateTime.utc(2016, 11, 13, 9), DateTime.utc(2016, 11, 13, 19)].map(
            (dt) => this.extract(dt, intl, "dayperiod")
          );
        }

        return this.meridiemCache;
      }
    );
  }

  eras(length) {
    return listStuff(this, length, eras, () => {
      const intl = { era: length };

      // This is problematic. Different calendars are going to define eras totally differently. What I need is the minimum set of dates
      // to definitely enumerate them.
      if (!this.eraCache[length]) {
        this.eraCache[length] = [DateTime.utc(-40, 1, 1), DateTime.utc(2017, 1, 1)].map((dt) =>
          this.extract(dt, intl, "era")
        );
      }

      return this.eraCache[length];
    });
  }

  extract(dt, intlOpts, field) {
    const df = this.dtFormatter(dt, intlOpts),
      results = df.formatToParts(),
      matching = results.find((m) => m.type.toLowerCase() === field);
    return matching ? matching.value : null;
  }

  numberFormatter(opts = {}) {
    // this forcesimple option is never used (the only caller short-circuits on it, but it seems safer to leave)
    // (in contrast, the rest of the condition is used heavily)
    return new PolyNumberFormatter(this.intl, opts.forceSimple || this.fastNumbers, opts);
  }

  dtFormatter(dt, intlOpts = {}) {
    return new PolyDateFormatter(dt, this.intl, intlOpts);
  }

  relFormatter(opts = {}) {
    return new PolyRelFormatter(this.intl, this.isEnglish(), opts);
  }

  listFormatter(opts = {}) {
    return getCachedLF(this.intl, opts);
  }

  isEnglish() {
    return (
      this.locale === "en" ||
      this.locale.toLowerCase() === "en-us" ||
      new Intl.DateTimeFormat(this.intl).resolvedOptions().locale.startsWith("en-us")
    );
  }

  equals(other) {
    return (
      this.locale === other.locale &&
      this.numberingSystem === other.numberingSystem &&
      this.outputCalendar === other.outputCalendar
    );
  }
}

let singleton = null;

/**
 * A zone with a fixed offset (meaning no DST)
 * @implements {Zone}
 */
class FixedOffsetZone extends Zone {
  /**
   * Get a singleton instance of UTC
   * @return {FixedOffsetZone}
   */
  static get utcInstance() {
    if (singleton === null) {
      singleton = new FixedOffsetZone(0);
    }
    return singleton;
  }

  /**
   * Get an instance with a specified offset
   * @param {number} offset - The offset in minutes
   * @return {FixedOffsetZone}
   */
  static instance(offset) {
    return offset === 0 ? FixedOffsetZone.utcInstance : new FixedOffsetZone(offset);
  }

  /**
   * Get an instance of FixedOffsetZone from a UTC offset string, like "UTC+6"
   * @param {string} s - The offset string to parse
   * @example FixedOffsetZone.parseSpecifier("UTC+6")
   * @example FixedOffsetZone.parseSpecifier("UTC+06")
   * @example FixedOffsetZone.parseSpecifier("UTC-6:00")
   * @return {FixedOffsetZone}
   */
  static parseSpecifier(s) {
    if (s) {
      const r = s.match(/^utc(?:([+-]\d{1,2})(?::(\d{2}))?)?$/i);
      if (r) {
        return new FixedOffsetZone(signedOffset(r[1], r[2]));
      }
    }
    return null;
  }

  constructor(offset) {
    super();
    /** @private **/
    this.fixed = offset;
  }

  /** @override **/
  get type() {
    return "fixed";
  }

  /** @override **/
  get name() {
    return this.fixed === 0 ? "UTC" : `UTC${formatOffset(this.fixed, "narrow")}`;
  }

  get ianaName() {
    if (this.fixed === 0) {
      return "Etc/UTC";
    } else {
      return `Etc/GMT${formatOffset(-this.fixed, "narrow")}`;
    }
  }

  /** @override **/
  offsetName() {
    return this.name;
  }

  /** @override **/
  formatOffset(ts, format) {
    return formatOffset(this.fixed, format);
  }

  /** @override **/
  get isUniversal() {
    return true;
  }

  /** @override **/
  offset() {
    return this.fixed;
  }

  /** @override **/
  equals(otherZone) {
    return otherZone.type === "fixed" && otherZone.fixed === this.fixed;
  }

  /** @override **/
  get isValid() {
    return true;
  }
}

/**
 * A zone that failed to parse. You should never need to instantiate this.
 * @implements {Zone}
 */
class InvalidZone extends Zone {
  constructor(zoneName) {
    super();
    /**  @private */
    this.zoneName = zoneName;
  }

  /** @override **/
  get type() {
    return "invalid";
  }

  /** @override **/
  get name() {
    return this.zoneName;
  }

  /** @override **/
  get isUniversal() {
    return false;
  }

  /** @override **/
  offsetName() {
    return null;
  }

  /** @override **/
  formatOffset() {
    return "";
  }

  /** @override **/
  offset() {
    return NaN;
  }

  /** @override **/
  equals() {
    return false;
  }

  /** @override **/
  get isValid() {
    return false;
  }
}

/**
 * @private
 */

function normalizeZone(input, defaultZone) {
  if (isUndefined(input) || input === null) {
    return defaultZone;
  } else if (input instanceof Zone) {
    return input;
  } else if (isString(input)) {
    const lowered = input.toLowerCase();
    if (lowered === "default") return defaultZone;
    else if (lowered === "local" || lowered === "system") return SystemZone.instance;
    else if (lowered === "utc" || lowered === "gmt") return FixedOffsetZone.utcInstance;
    else return FixedOffsetZone.parseSpecifier(lowered) || IANAZone.create(input);
  } else if (isNumber(input)) {
    return FixedOffsetZone.instance(input);
  } else if (typeof input === "object" && "offset" in input && typeof input.offset === "function") {
    // This is dumb, but the instanceof check above doesn't seem to really work
    // so we're duck checking it
    return input;
  } else {
    return new InvalidZone(input);
  }
}

let now = () => Date.now(),
  defaultZone = "system",
  defaultLocale = null,
  defaultNumberingSystem = null,
  defaultOutputCalendar = null,
  twoDigitCutoffYear = 60,
  throwOnInvalid;

/**
 * Settings contains static getters and setters that control Luxon's overall behavior. Luxon is a simple library with few options, but the ones it does have live here.
 */
class Settings {
  /**
   * Get the callback for returning the current timestamp.
   * @type {function}
   */
  static get now() {
    return now;
  }

  /**
   * Set the callback for returning the current timestamp.
   * The function should return a number, which will be interpreted as an Epoch millisecond count
   * @type {function}
   * @example Settings.now = () => Date.now() + 3000 // pretend it is 3 seconds in the future
   * @example Settings.now = () => 0 // always pretend it's Jan 1, 1970 at midnight in UTC time
   */
  static set now(n) {
    now = n;
  }

  /**
   * Set the default time zone to create DateTimes in. Does not affect existing instances.
   * Use the value "system" to reset this value to the system's time zone.
   * @type {string}
   */
  static set defaultZone(zone) {
    defaultZone = zone;
  }

  /**
   * Get the default time zone object currently used to create DateTimes. Does not affect existing instances.
   * The default value is the system's time zone (the one set on the machine that runs this code).
   * @type {Zone}
   */
  static get defaultZone() {
    return normalizeZone(defaultZone, SystemZone.instance);
  }

  /**
   * Get the default locale to create DateTimes with. Does not affect existing instances.
   * @type {string}
   */
  static get defaultLocale() {
    return defaultLocale;
  }

  /**
   * Set the default locale to create DateTimes with. Does not affect existing instances.
   * @type {string}
   */
  static set defaultLocale(locale) {
    defaultLocale = locale;
  }

  /**
   * Get the default numbering system to create DateTimes with. Does not affect existing instances.
   * @type {string}
   */
  static get defaultNumberingSystem() {
    return defaultNumberingSystem;
  }

  /**
   * Set the default numbering system to create DateTimes with. Does not affect existing instances.
   * @type {string}
   */
  static set defaultNumberingSystem(numberingSystem) {
    defaultNumberingSystem = numberingSystem;
  }

  /**
   * Get the default output calendar to create DateTimes with. Does not affect existing instances.
   * @type {string}
   */
  static get defaultOutputCalendar() {
    return defaultOutputCalendar;
  }

  /**
   * Set the default output calendar to create DateTimes with. Does not affect existing instances.
   * @type {string}
   */
  static set defaultOutputCalendar(outputCalendar) {
    defaultOutputCalendar = outputCalendar;
  }

  /**
   * Get the cutoff year after which a string encoding a year as two digits is interpreted to occur in the current century.
   * @type {number}
   */
  static get twoDigitCutoffYear() {
    return twoDigitCutoffYear;
  }

  /**
   * Set the cutoff year after which a string encoding a year as two digits is interpreted to occur in the current century.
   * @type {number}
   * @example Settings.twoDigitCutoffYear = 0 // cut-off year is 0, so all 'yy' are interpreted as current century
   * @example Settings.twoDigitCutoffYear = 50 // '49' -> 1949; '50' -> 2050
   * @example Settings.twoDigitCutoffYear = 1950 // interpreted as 50
   * @example Settings.twoDigitCutoffYear = 2050 // ALSO interpreted as 50
   */
  static set twoDigitCutoffYear(cutoffYear) {
    twoDigitCutoffYear = cutoffYear % 100;
  }

  /**
   * Get whether Luxon will throw when it encounters invalid DateTimes, Durations, or Intervals
   * @type {boolean}
   */
  static get throwOnInvalid() {
    return throwOnInvalid;
  }

  /**
   * Set whether Luxon will throw when it encounters invalid DateTimes, Durations, or Intervals
   * @type {boolean}
   */
  static set throwOnInvalid(t) {
    throwOnInvalid = t;
  }

  /**
   * Reset Luxon's global caches. Should only be necessary in testing scenarios.
   * @return {void}
   */
  static resetCaches() {
    Locale.resetCache();
    IANAZone.resetCache();
  }
}

/*
  This is just a junk drawer, containing anything used across multiple classes.
  Because Luxon is small(ish), this should stay small and we won't worry about splitting
  it up into, say, parsingUtil.js and basicUtil.js and so on. But they are divided up by feature area.
*/

/**
 * @private
 */

// TYPES

function isUndefined(o) {
  return typeof o === "undefined";
}

function isNumber(o) {
  return typeof o === "number";
}

function isInteger(o) {
  return typeof o === "number" && o % 1 === 0;
}

function isString(o) {
  return typeof o === "string";
}

function isDate(o) {
  return Object.prototype.toString.call(o) === "[object Date]";
}

// CAPABILITIES

function hasRelative() {
  try {
    return typeof Intl !== "undefined" && !!Intl.RelativeTimeFormat;
  } catch (e) {
    return false;
  }
}

// OBJECTS AND ARRAYS

function maybeArray(thing) {
  return Array.isArray(thing) ? thing : [thing];
}

function bestBy(arr, by, compare) {
  if (arr.length === 0) {
    return undefined;
  }
  return arr.reduce((best, next) => {
    const pair = [by(next), next];
    if (!best) {
      return pair;
    } else if (compare(best[0], pair[0]) === best[0]) {
      return best;
    } else {
      return pair;
    }
  }, null)[1];
}

function pick(obj, keys) {
  return keys.reduce((a, k) => {
    a[k] = obj[k];
    return a;
  }, {});
}

function hasOwnProperty(obj, prop) {
  return Object.prototype.hasOwnProperty.call(obj, prop);
}

// NUMBERS AND STRINGS

function integerBetween(thing, bottom, top) {
  return isInteger(thing) && thing >= bottom && thing <= top;
}

// x % n but takes the sign of n instead of x
function floorMod(x, n) {
  return x - n * Math.floor(x / n);
}

function padStart(input, n = 2) {
  const isNeg = input < 0;
  let padded;
  if (isNeg) {
    padded = "-" + ("" + -input).padStart(n, "0");
  } else {
    padded = ("" + input).padStart(n, "0");
  }
  return padded;
}

function parseInteger(string) {
  if (isUndefined(string) || string === null || string === "") {
    return undefined;
  } else {
    return parseInt(string, 10);
  }
}

function parseFloating(string) {
  if (isUndefined(string) || string === null || string === "") {
    return undefined;
  } else {
    return parseFloat(string);
  }
}

function parseMillis(fraction) {
  // Return undefined (instead of 0) in these cases, where fraction is not set
  if (isUndefined(fraction) || fraction === null || fraction === "") {
    return undefined;
  } else {
    const f = parseFloat("0." + fraction) * 1000;
    return Math.floor(f);
  }
}

function roundTo(number, digits, towardZero = false) {
  const factor = 10 ** digits,
    rounder = towardZero ? Math.trunc : Math.round;
  return rounder(number * factor) / factor;
}

// DATE BASICS

function isLeapYear(year) {
  return year % 4 === 0 && (year % 100 !== 0 || year % 400 === 0);
}

function daysInYear(year) {
  return isLeapYear(year) ? 366 : 365;
}

function daysInMonth(year, month) {
  const modMonth = floorMod(month - 1, 12) + 1,
    modYear = year + (month - modMonth) / 12;

  if (modMonth === 2) {
    return isLeapYear(modYear) ? 29 : 28;
  } else {
    return [31, null, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31][modMonth - 1];
  }
}

// convert a calendar object to a local timestamp (epoch, but with the offset baked in)
function objToLocalTS(obj) {
  let d = Date.UTC(
    obj.year,
    obj.month - 1,
    obj.day,
    obj.hour,
    obj.minute,
    obj.second,
    obj.millisecond
  );

  // for legacy reasons, years between 0 and 99 are interpreted as 19XX; revert that
  if (obj.year < 100 && obj.year >= 0) {
    d = new Date(d);
    // set the month and day again, this is necessary because year 2000 is a leap year, but year 100 is not
    // so if obj.year is in 99, but obj.day makes it roll over into year 100,
    // the calculations done by Date.UTC are using year 2000 - which is incorrect
    d.setUTCFullYear(obj.year, obj.month - 1, obj.day);
  }
  return +d;
}

function weeksInWeekYear(weekYear) {
  const p1 =
      (weekYear +
        Math.floor(weekYear / 4) -
        Math.floor(weekYear / 100) +
        Math.floor(weekYear / 400)) %
      7,
    last = weekYear - 1,
    p2 = (last + Math.floor(last / 4) - Math.floor(last / 100) + Math.floor(last / 400)) % 7;
  return p1 === 4 || p2 === 3 ? 53 : 52;
}

function untruncateYear(year) {
  if (year > 99) {
    return year;
  } else return year > Settings.twoDigitCutoffYear ? 1900 + year : 2000 + year;
}

// PARSING

function parseZoneInfo(ts, offsetFormat, locale, timeZone = null) {
  const date = new Date(ts),
    intlOpts = {
      hourCycle: "h23",
      year: "numeric",
      month: "2-digit",
      day: "2-digit",
      hour: "2-digit",
      minute: "2-digit",
    };

  if (timeZone) {
    intlOpts.timeZone = timeZone;
  }

  const modified = { timeZoneName: offsetFormat, ...intlOpts };

  const parsed = new Intl.DateTimeFormat(locale, modified)
    .formatToParts(date)
    .find((m) => m.type.toLowerCase() === "timezonename");
  return parsed ? parsed.value : null;
}

// signedOffset('-5', '30') -> -330
function signedOffset(offHourStr, offMinuteStr) {
  let offHour = parseInt(offHourStr, 10);

  // don't || this because we want to preserve -0
  if (Number.isNaN(offHour)) {
    offHour = 0;
  }

  const offMin = parseInt(offMinuteStr, 10) || 0,
    offMinSigned = offHour < 0 || Object.is(offHour, -0) ? -offMin : offMin;
  return offHour * 60 + offMinSigned;
}

// COERCION

function asNumber(value) {
  const numericValue = Number(value);
  if (typeof value === "boolean" || value === "" || Number.isNaN(numericValue))
    throw new InvalidArgumentError(`Invalid unit value ${value}`);
  return numericValue;
}

function normalizeObject(obj, normalizer) {
  const normalized = {};
  for (const u in obj) {
    if (hasOwnProperty(obj, u)) {
      const v = obj[u];
      if (v === undefined || v === null) continue;
      normalized[normalizer(u)] = asNumber(v);
    }
  }
  return normalized;
}

function formatOffset(offset, format) {
  const hours = Math.trunc(Math.abs(offset / 60)),
    minutes = Math.trunc(Math.abs(offset % 60)),
    sign = offset >= 0 ? "+" : "-";

  switch (format) {
    case "short":
      return `${sign}${padStart(hours, 2)}:${padStart(minutes, 2)}`;
    case "narrow":
      return `${sign}${hours}${minutes > 0 ? `:${minutes}` : ""}`;
    case "techie":
      return `${sign}${padStart(hours, 2)}${padStart(minutes, 2)}`;
    default:
      throw new RangeError(`Value format ${format} is out of range for property format`);
  }
}

function timeObject(obj) {
  return pick(obj, ["hour", "minute", "second", "millisecond"]);
}

/**
 * @private
 */

const monthsLong = [
  "January",
  "February",
  "March",
  "April",
  "May",
  "June",
  "July",
  "August",
  "September",
  "October",
  "November",
  "December",
];

const monthsShort = [
  "Jan",
  "Feb",
  "Mar",
  "Apr",
  "May",
  "Jun",
  "Jul",
  "Aug",
  "Sep",
  "Oct",
  "Nov",
  "Dec",
];

const monthsNarrow = ["J", "F", "M", "A", "M", "J", "J", "A", "S", "O", "N", "D"];

function months(length) {
  switch (length) {
    case "narrow":
      return [...monthsNarrow];
    case "short":
      return [...monthsShort];
    case "long":
      return [...monthsLong];
    case "numeric":
      return ["1", "2", "3", "4", "5", "6", "7", "8", "9", "10", "11", "12"];
    case "2-digit":
      return ["01", "02", "03", "04", "05", "06", "07", "08", "09", "10", "11", "12"];
    default:
      return null;
  }
}

const weekdaysLong = [
  "Monday",
  "Tuesday",
  "Wednesday",
  "Thursday",
  "Friday",
  "Saturday",
  "Sunday",
];

const weekdaysShort = ["Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun"];

const weekdaysNarrow = ["M", "T", "W", "T", "F", "S", "S"];

function weekdays(length) {
  switch (length) {
    case "narrow":
      return [...weekdaysNarrow];
    case "short":
      return [...weekdaysShort];
    case "long":
      return [...weekdaysLong];
    case "numeric":
      return ["1", "2", "3", "4", "5", "6", "7"];
    default:
      return null;
  }
}

const meridiems = ["AM", "PM"];

const erasLong = ["Before Christ", "Anno Domini"];

const erasShort = ["BC", "AD"];

const erasNarrow = ["B", "A"];

function eras(length) {
  switch (length) {
    case "narrow":
      return [...erasNarrow];
    case "short":
      return [...erasShort];
    case "long":
      return [...erasLong];
    default:
      return null;
  }
}

function meridiemForDateTime(dt) {
  return meridiems[dt.hour < 12 ? 0 : 1];
}

function weekdayForDateTime(dt, length) {
  return weekdays(length)[dt.weekday - 1];
}

function monthForDateTime(dt, length) {
  return months(length)[dt.month - 1];
}

function eraForDateTime(dt, length) {
  return eras(length)[dt.year < 0 ? 0 : 1];
}

function formatRelativeTime(unit, count, numeric = "always", narrow = false) {
  const units = {
    years: ["year", "yr."],
    quarters: ["quarter", "qtr."],
    months: ["month", "mo."],
    weeks: ["week", "wk."],
    days: ["day", "day", "days"],
    hours: ["hour", "hr."],
    minutes: ["minute", "min."],
    seconds: ["second", "sec."],
  };

  const lastable = ["hours", "minutes", "seconds"].indexOf(unit) === -1;

  if (numeric === "auto" && lastable) {
    const isDay = unit === "days";
    switch (count) {
      case 1:
        return isDay ? "tomorrow" : `next ${units[unit][0]}`;
      case -1:
        return isDay ? "yesterday" : `last ${units[unit][0]}`;
      case 0:
        return isDay ? "today" : `this ${units[unit][0]}`;
    }
  }

  const isInPast = Object.is(count, -0) || count < 0,
    fmtValue = Math.abs(count),
    singular = fmtValue === 1,
    lilUnits = units[unit],
    fmtUnit = narrow
      ? singular
        ? lilUnits[1]
        : lilUnits[2] || lilUnits[1]
      : singular
      ? units[unit][0]
      : unit;
  return isInPast ? `${fmtValue} ${fmtUnit} ago` : `in ${fmtValue} ${fmtUnit}`;
}

function stringifyTokens(splits, tokenToString) {
  let s = "";
  for (const token of splits) {
    if (token.literal) {
      s += token.val;
    } else {
      s += tokenToString(token.val);
    }
  }
  return s;
}

const macroTokenToFormatOpts = {
  D: DATE_SHORT,
  DD: DATE_MED,
  DDD: DATE_FULL,
  DDDD: DATE_HUGE,
  t: TIME_SIMPLE,
  tt: TIME_WITH_SECONDS,
  ttt: TIME_WITH_SHORT_OFFSET,
  tttt: TIME_WITH_LONG_OFFSET,
  T: TIME_24_SIMPLE,
  TT: TIME_24_WITH_SECONDS,
  TTT: TIME_24_WITH_SHORT_OFFSET,
  TTTT: TIME_24_WITH_LONG_OFFSET,
  f: DATETIME_SHORT,
  ff: DATETIME_MED,
  fff: DATETIME_FULL,
  ffff: DATETIME_HUGE,
  F: DATETIME_SHORT_WITH_SECONDS,
  FF: DATETIME_MED_WITH_SECONDS,
  FFF: DATETIME_FULL_WITH_SECONDS,
  FFFF: DATETIME_HUGE_WITH_SECONDS,
};

/**
 * @private
 */

class Formatter {
  static create(locale, opts = {}) {
    return new Formatter(locale, opts);
  }

  static parseFormat(fmt) {
    // white-space is always considered a literal in user-provided formats
    // the " " token has a special meaning (see unitForToken)

    let current = null,
      currentFull = "",
      bracketed = false;
    const splits = [];
    for (let i = 0; i < fmt.length; i++) {
      const c = fmt.charAt(i);
      if (c === "'") {
        if (currentFull.length > 0) {
          splits.push({ literal: bracketed || /^\s+$/.test(currentFull), val: currentFull });
        }
        current = null;
        currentFull = "";
        bracketed = !bracketed;
      } else if (bracketed) {
        currentFull += c;
      } else if (c === current) {
        currentFull += c;
      } else {
        if (currentFull.length > 0) {
          splits.push({ literal: /^\s+$/.test(currentFull), val: currentFull });
        }
        currentFull = c;
        current = c;
      }
    }

    if (currentFull.length > 0) {
      splits.push({ literal: bracketed || /^\s+$/.test(currentFull), val: currentFull });
    }

    return splits;
  }

  static macroTokenToFormatOpts(token) {
    return macroTokenToFormatOpts[token];
  }

  constructor(locale, formatOpts) {
    this.opts = formatOpts;
    this.loc = locale;
    this.systemLoc = null;
  }

  formatWithSystemDefault(dt, opts) {
    if (this.systemLoc === null) {
      this.systemLoc = this.loc.redefaultToSystem();
    }
    const df = this.systemLoc.dtFormatter(dt, { ...this.opts, ...opts });
    return df.format();
  }

  dtFormatter(dt, opts = {}) {
    return this.loc.dtFormatter(dt, { ...this.opts, ...opts });
  }

  formatDateTime(dt, opts) {
    return this.dtFormatter(dt, opts).format();
  }

  formatDateTimeParts(dt, opts) {
    return this.dtFormatter(dt, opts).formatToParts();
  }

  formatInterval(interval, opts) {
    const df = this.dtFormatter(interval.start, opts);
    return df.dtf.formatRange(interval.start.toJSDate(), interval.end.toJSDate());
  }

  resolvedOptions(dt, opts) {
    return this.dtFormatter(dt, opts).resolvedOptions();
  }

  num(n, p = 0) {
    // we get some perf out of doing this here, annoyingly
    if (this.opts.forceSimple) {
      return padStart(n, p);
    }

    const opts = { ...this.opts };

    if (p > 0) {
      opts.padTo = p;
    }

    return this.loc.numberFormatter(opts).format(n);
  }

  formatDateTimeFromString(dt, fmt) {
    const knownEnglish = this.loc.listingMode() === "en",
      useDateTimeFormatter = this.loc.outputCalendar && this.loc.outputCalendar !== "gregory",
      string = (opts, extract) => this.loc.extract(dt, opts, extract),
      formatOffset = (opts) => {
        if (dt.isOffsetFixed && dt.offset === 0 && opts.allowZ) {
          return "Z";
        }

        return dt.isValid ? dt.zone.formatOffset(dt.ts, opts.format) : "";
      },
      meridiem = () =>
        knownEnglish
          ? meridiemForDateTime(dt)
          : string({ hour: "numeric", hourCycle: "h12" }, "dayperiod"),
      month = (length, standalone) =>
        knownEnglish
          ? monthForDateTime(dt, length)
          : string(standalone ? { month: length } : { month: length, day: "numeric" }, "month"),
      weekday = (length, standalone) =>
        knownEnglish
          ? weekdayForDateTime(dt, length)
          : string(
              standalone ? { weekday: length } : { weekday: length, month: "long", day: "numeric" },
              "weekday"
            ),
      maybeMacro = (token) => {
        const formatOpts = Formatter.macroTokenToFormatOpts(token);
        if (formatOpts) {
          return this.formatWithSystemDefault(dt, formatOpts);
        } else {
          return token;
        }
      },
      era = (length) =>
        knownEnglish ? eraForDateTime(dt, length) : string({ era: length }, "era"),
      tokenToString = (token) => {
        // Where possible: https://cldr.unicode.org/translation/date-time/date-time-symbols
        switch (token) {
          // ms
          case "S":
            return this.num(dt.millisecond);
          case "u":
          // falls through
          case "SSS":
            return this.num(dt.millisecond, 3);
          // seconds
          case "s":
            return this.num(dt.second);
          case "ss":
            return this.num(dt.second, 2);
          // fractional seconds
          case "uu":
            return this.num(Math.floor(dt.millisecond / 10), 2);
          case "uuu":
            return this.num(Math.floor(dt.millisecond / 100));
          // minutes
          case "m":
            return this.num(dt.minute);
          case "mm":
            return this.num(dt.minute, 2);
          // hours
          case "h":
            return this.num(dt.hour % 12 === 0 ? 12 : dt.hour % 12);
          case "hh":
            return this.num(dt.hour % 12 === 0 ? 12 : dt.hour % 12, 2);
          case "H":
            return this.num(dt.hour);
          case "HH":
            return this.num(dt.hour, 2);
          // offset
          case "Z":
            // like +6
            return formatOffset({ format: "narrow", allowZ: this.opts.allowZ });
          case "ZZ":
            // like +06:00
            return formatOffset({ format: "short", allowZ: this.opts.allowZ });
          case "ZZZ":
            // like +0600
            return formatOffset({ format: "techie", allowZ: this.opts.allowZ });
          case "ZZZZ":
            // like EST
            return dt.zone.offsetName(dt.ts, { format: "short", locale: this.loc.locale });
          case "ZZZZZ":
            // like Eastern Standard Time
            return dt.zone.offsetName(dt.ts, { format: "long", locale: this.loc.locale });
          // zone
          case "z":
            // like America/New_York
            return dt.zoneName;
          // meridiems
          case "a":
            return meridiem();
          // dates
          case "d":
            return useDateTimeFormatter ? string({ day: "numeric" }, "day") : this.num(dt.day);
          case "dd":
            return useDateTimeFormatter ? string({ day: "2-digit" }, "day") : this.num(dt.day, 2);
          // weekdays - standalone
          case "c":
            // like 1
            return this.num(dt.weekday);
          case "ccc":
            // like 'Tues'
            return weekday("short", true);
          case "cccc":
            // like 'Tuesday'
            return weekday("long", true);
          case "ccccc":
            // like 'T'
            return weekday("narrow", true);
          // weekdays - format
          case "E":
            // like 1
            return this.num(dt.weekday);
          case "EEE":
            // like 'Tues'
            return weekday("short", false);
          case "EEEE":
            // like 'Tuesday'
            return weekday("long", false);
          case "EEEEE":
            // like 'T'
            return weekday("narrow", false);
          // months - standalone
          case "L":
            // like 1
            return useDateTimeFormatter
              ? string({ month: "numeric", day: "numeric" }, "month")
              : this.num(dt.month);
          case "LL":
            // like 01, doesn't seem to work
            return useDateTimeFormatter
              ? string({ month: "2-digit", day: "numeric" }, "month")
              : this.num(dt.month, 2);
          case "LLL":
            // like Jan
            return month("short", true);
          case "LLLL":
            // like January
            return month("long", true);
          case "LLLLL":
            // like J
            return month("narrow", true);
          // months - format
          case "M":
            // like 1
            return useDateTimeFormatter
              ? string({ month: "numeric" }, "month")
              : this.num(dt.month);
          case "MM":
            // like 01
            return useDateTimeFormatter
              ? string({ month: "2-digit" }, "month")
              : this.num(dt.month, 2);
          case "MMM":
            // like Jan
            return month("short", false);
          case "MMMM":
            // like January
            return month("long", false);
          case "MMMMM":
            // like J
            return month("narrow", false);
          // years
          case "y":
            // like 2014
            return useDateTimeFormatter ? string({ year: "numeric" }, "year") : this.num(dt.year);
          case "yy":
            // like 14
            return useDateTimeFormatter
              ? string({ year: "2-digit" }, "year")
              : this.num(dt.year.toString().slice(-2), 2);
          case "yyyy":
            // like 0012
            return useDateTimeFormatter
              ? string({ year: "numeric" }, "year")
              : this.num(dt.year, 4);
          case "yyyyyy":
            // like 000012
            return useDateTimeFormatter
              ? string({ year: "numeric" }, "year")
              : this.num(dt.year, 6);
          // eras
          case "G":
            // like AD
            return era("short");
          case "GG":
            // like Anno Domini
            return era("long");
          case "GGGGG":
            return era("narrow");
          case "kk":
            return this.num(dt.weekYear.toString().slice(-2), 2);
          case "kkkk":
            return this.num(dt.weekYear, 4);
          case "W":
            return this.num(dt.weekNumber);
          case "WW":
            return this.num(dt.weekNumber, 2);
          case "o":
            return this.num(dt.ordinal);
          case "ooo":
            return this.num(dt.ordinal, 3);
          case "q":
            // like 1
            return this.num(dt.quarter);
          case "qq":
            // like 01
            return this.num(dt.quarter, 2);
          case "X":
            return this.num(Math.floor(dt.ts / 1000));
          case "x":
            return this.num(dt.ts);
          default:
            return maybeMacro(token);
        }
      };

    return stringifyTokens(Formatter.parseFormat(fmt), tokenToString);
  }

  formatDurationFromString(dur, fmt) {
    const tokenToField = (token) => {
        switch (token[0]) {
          case "S":
            return "millisecond";
          case "s":
            return "second";
          case "m":
            return "minute";
          case "h":
            return "hour";
          case "d":
            return "day";
          case "w":
            return "week";
          case "M":
            return "month";
          case "y":
            return "year";
          default:
            return null;
        }
      },
      tokenToString = (lildur) => (token) => {
        const mapped = tokenToField(token);
        if (mapped) {
          return this.num(lildur.get(mapped), token.length);
        } else {
          return token;
        }
      },
      tokens = Formatter.parseFormat(fmt),
      realTokens = tokens.reduce(
        (found, { literal, val }) => (literal ? found : found.concat(val)),
        []
      ),
      collapsed = dur.shiftTo(...realTokens.map(tokenToField).filter((t) => t));
    return stringifyTokens(tokens, tokenToString(collapsed));
  }
}

class Invalid {
  constructor(reason, explanation) {
    this.reason = reason;
    this.explanation = explanation;
  }

  toMessage() {
    if (this.explanation) {
      return `${this.reason}: ${this.explanation}`;
    } else {
      return this.reason;
    }
  }
}

/*
 * This file handles parsing for well-specified formats. Here's how it works:
 * Two things go into parsing: a regex to match with and an extractor to take apart the groups in the match.
 * An extractor is just a function that takes a regex match array and returns a { year: ..., month: ... } object
 * parse() does the work of executing the regex and applying the extractor. It takes multiple regex/extractor pairs to try in sequence.
 * Extractors can take a "cursor" representing the offset in the match to look at. This makes it easy to combine extractors.
 * combineExtractors() does the work of combining them, keeping track of the cursor through multiple extractions.
 * Some extractions are super dumb and simpleParse and fromStrings help DRY them.
 */

const ianaRegex = /[A-Za-z_+-]{1,256}(?::?\/[A-Za-z0-9_+-]{1,256}(?:\/[A-Za-z0-9_+-]{1,256})?)?/;

function combineRegexes(...regexes) {
  const full = regexes.reduce((f, r) => f + r.source, "");
  return RegExp(`^${full}$`);
}

function combineExtractors(...extractors) {
  return (m) =>
    extractors
      .reduce(
        ([mergedVals, mergedZone, cursor], ex) => {
          const [val, zone, next] = ex(m, cursor);
          return [{ ...mergedVals, ...val }, zone || mergedZone, next];
        },
        [{}, null, 1]
      )
      .slice(0, 2);
}

function parse(s, ...patterns) {
  if (s == null) {
    return [null, null];
  }

  for (const [regex, extractor] of patterns) {
    const m = regex.exec(s);
    if (m) {
      return extractor(m);
    }
  }
  return [null, null];
}

function simpleParse(...keys) {
  return (match, cursor) => {
    const ret = {};
    let i;

    for (i = 0; i < keys.length; i++) {
      ret[keys[i]] = parseInteger(match[cursor + i]);
    }
    return [ret, null, cursor + i];
  };
}

// ISO and SQL parsing
const offsetRegex = /(?:(Z)|([+-]\d\d)(?::?(\d\d))?)/;
const isoExtendedZone = `(?:${offsetRegex.source}?(?:\\[(${ianaRegex.source})\\])?)?`;
const isoTimeBaseRegex = /(\d\d)(?::?(\d\d)(?::?(\d\d)(?:[.,](\d{1,30}))?)?)?/;
const isoTimeRegex = RegExp(`${isoTimeBaseRegex.source}${isoExtendedZone}`);
const isoTimeExtensionRegex = RegExp(`(?:T${isoTimeRegex.source})?`);
const isoYmdRegex = /([+-]\d{6}|\d{4})(?:-?(\d\d)(?:-?(\d\d))?)?/;
const isoWeekRegex = /(\d{4})-?W(\d\d)(?:-?(\d))?/;
const isoOrdinalRegex = /(\d{4})-?(\d{3})/;
const extractISOWeekData = simpleParse("weekYear", "weekNumber", "weekDay");
const extractISOOrdinalData = simpleParse("year", "ordinal");
const sqlYmdRegex = /(\d{4})-(\d\d)-(\d\d)/; // dumbed-down version of the ISO one
const sqlTimeRegex = RegExp(
  `${isoTimeBaseRegex.source} ?(?:${offsetRegex.source}|(${ianaRegex.source}))?`
);
const sqlTimeExtensionRegex = RegExp(`(?: ${sqlTimeRegex.source})?`);

function int(match, pos, fallback) {
  const m = match[pos];
  return isUndefined(m) ? fallback : parseInteger(m);
}

function extractISOYmd(match, cursor) {
  const item = {
    year: int(match, cursor),
    month: int(match, cursor + 1, 1),
    day: int(match, cursor + 2, 1),
  };

  return [item, null, cursor + 3];
}

function extractISOTime(match, cursor) {
  const item = {
    hours: int(match, cursor, 0),
    minutes: int(match, cursor + 1, 0),
    seconds: int(match, cursor + 2, 0),
    milliseconds: parseMillis(match[cursor + 3]),
  };

  return [item, null, cursor + 4];
}

function extractISOOffset(match, cursor) {
  const local = !match[cursor] && !match[cursor + 1],
    fullOffset = signedOffset(match[cursor + 1], match[cursor + 2]),
    zone = local ? null : FixedOffsetZone.instance(fullOffset);
  return [{}, zone, cursor + 3];
}

function extractIANAZone(match, cursor) {
  const zone = match[cursor] ? IANAZone.create(match[cursor]) : null;
  return [{}, zone, cursor + 1];
}

// ISO time parsing

const isoTimeOnly = RegExp(`^T?${isoTimeBaseRegex.source}$`);

// ISO duration parsing

const isoDuration =
  /^-?P(?:(?:(-?\d{1,20}(?:\.\d{1,20})?)Y)?(?:(-?\d{1,20}(?:\.\d{1,20})?)M)?(?:(-?\d{1,20}(?:\.\d{1,20})?)W)?(?:(-?\d{1,20}(?:\.\d{1,20})?)D)?(?:T(?:(-?\d{1,20}(?:\.\d{1,20})?)H)?(?:(-?\d{1,20}(?:\.\d{1,20})?)M)?(?:(-?\d{1,20})(?:[.,](-?\d{1,20}))?S)?)?)$/;

function extractISODuration(match) {
  const [s, yearStr, monthStr, weekStr, dayStr, hourStr, minuteStr, secondStr, millisecondsStr] =
    match;

  const hasNegativePrefix = s[0] === "-";
  const negativeSeconds = secondStr && secondStr[0] === "-";

  const maybeNegate = (num, force = false) =>
    num !== undefined && (force || (num && hasNegativePrefix)) ? -num : num;

  return [
    {
      years: maybeNegate(parseFloating(yearStr)),
      months: maybeNegate(parseFloating(monthStr)),
      weeks: maybeNegate(parseFloating(weekStr)),
      days: maybeNegate(parseFloating(dayStr)),
      hours: maybeNegate(parseFloating(hourStr)),
      minutes: maybeNegate(parseFloating(minuteStr)),
      seconds: maybeNegate(parseFloating(secondStr), secondStr === "-0"),
      milliseconds: maybeNegate(parseMillis(millisecondsStr), negativeSeconds),
    },
  ];
}

// These are a little braindead. EDT *should* tell us that we're in, say, America/New_York
// and not just that we're in -240 *right now*. But since I don't think these are used that often
// I'm just going to ignore that
const obsOffsets = {
  GMT: 0,
  EDT: -4 * 60,
  EST: -5 * 60,
  CDT: -5 * 60,
  CST: -6 * 60,
  MDT: -6 * 60,
  MST: -7 * 60,
  PDT: -7 * 60,
  PST: -8 * 60,
};

function fromStrings(weekdayStr, yearStr, monthStr, dayStr, hourStr, minuteStr, secondStr) {
  const result = {
    year: yearStr.length === 2 ? untruncateYear(parseInteger(yearStr)) : parseInteger(yearStr),
    month: monthsShort.indexOf(monthStr) + 1,
    day: parseInteger(dayStr),
    hour: parseInteger(hourStr),
    minute: parseInteger(minuteStr),
  };

  if (secondStr) result.second = parseInteger(secondStr);
  if (weekdayStr) {
    result.weekday =
      weekdayStr.length > 3
        ? weekdaysLong.indexOf(weekdayStr) + 1
        : weekdaysShort.indexOf(weekdayStr) + 1;
  }

  return result;
}

// RFC 2822/5322
const rfc2822 =
  /^(?:(Mon|Tue|Wed|Thu|Fri|Sat|Sun),\s)?(\d{1,2})\s(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)\s(\d{2,4})\s(\d\d):(\d\d)(?::(\d\d))?\s(?:(UT|GMT|[ECMP][SD]T)|([Zz])|(?:([+-]\d\d)(\d\d)))$/;

function extractRFC2822(match) {
  const [
      ,
      weekdayStr,
      dayStr,
      monthStr,
      yearStr,
      hourStr,
      minuteStr,
      secondStr,
      obsOffset,
      milOffset,
      offHourStr,
      offMinuteStr,
    ] = match,
    result = fromStrings(weekdayStr, yearStr, monthStr, dayStr, hourStr, minuteStr, secondStr);

  let offset;
  if (obsOffset) {
    offset = obsOffsets[obsOffset];
  } else if (milOffset) {
    offset = 0;
  } else {
    offset = signedOffset(offHourStr, offMinuteStr);
  }

  return [result, new FixedOffsetZone(offset)];
}

function preprocessRFC2822(s) {
  // Remove comments and folding whitespace and replace multiple-spaces with a single space
  return s
    .replace(/\([^()]*\)|[\n\t]/g, " ")
    .replace(/(\s\s+)/g, " ")
    .trim();
}

// http date

const rfc1123 =
    /^(Mon|Tue|Wed|Thu|Fri|Sat|Sun), (\d\d) (Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) (\d{4}) (\d\d):(\d\d):(\d\d) GMT$/,
  rfc850 =
    /^(Monday|Tuesday|Wednesday|Thursday|Friday|Saturday|Sunday), (\d\d)-(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)-(\d\d) (\d\d):(\d\d):(\d\d) GMT$/,
  ascii =
    /^(Mon|Tue|Wed|Thu|Fri|Sat|Sun) (Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) ( \d|\d\d) (\d\d):(\d\d):(\d\d) (\d{4})$/;

function extractRFC1123Or850(match) {
  const [, weekdayStr, dayStr, monthStr, yearStr, hourStr, minuteStr, secondStr] = match,
    result = fromStrings(weekdayStr, yearStr, monthStr, dayStr, hourStr, minuteStr, secondStr);
  return [result, FixedOffsetZone.utcInstance];
}

function extractASCII(match) {
  const [, weekdayStr, monthStr, dayStr, hourStr, minuteStr, secondStr, yearStr] = match,
    result = fromStrings(weekdayStr, yearStr, monthStr, dayStr, hourStr, minuteStr, secondStr);
  return [result, FixedOffsetZone.utcInstance];
}

const isoYmdWithTimeExtensionRegex = combineRegexes(isoYmdRegex, isoTimeExtensionRegex);
const isoWeekWithTimeExtensionRegex = combineRegexes(isoWeekRegex, isoTimeExtensionRegex);
const isoOrdinalWithTimeExtensionRegex = combineRegexes(isoOrdinalRegex, isoTimeExtensionRegex);
const isoTimeCombinedRegex = combineRegexes(isoTimeRegex);

const extractISOYmdTimeAndOffset = combineExtractors(
  extractISOYmd,
  extractISOTime,
  extractISOOffset,
  extractIANAZone
);
const extractISOWeekTimeAndOffset = combineExtractors(
  extractISOWeekData,
  extractISOTime,
  extractISOOffset,
  extractIANAZone
);
const extractISOOrdinalDateAndTime = combineExtractors(
  extractISOOrdinalData,
  extractISOTime,
  extractISOOffset,
  extractIANAZone
);
const extractISOTimeAndOffset = combineExtractors(
  extractISOTime,
  extractISOOffset,
  extractIANAZone
);

/*
 * @private
 */

function parseISODate(s) {
  return parse(
    s,
    [isoYmdWithTimeExtensionRegex, extractISOYmdTimeAndOffset],
    [isoWeekWithTimeExtensionRegex, extractISOWeekTimeAndOffset],
    [isoOrdinalWithTimeExtensionRegex, extractISOOrdinalDateAndTime],
    [isoTimeCombinedRegex, extractISOTimeAndOffset]
  );
}

function parseRFC2822Date(s) {
  return parse(preprocessRFC2822(s), [rfc2822, extractRFC2822]);
}

function parseHTTPDate(s) {
  return parse(
    s,
    [rfc1123, extractRFC1123Or850],
    [rfc850, extractRFC1123Or850],
    [ascii, extractASCII]
  );
}

function parseISODuration(s) {
  return parse(s, [isoDuration, extractISODuration]);
}

const extractISOTimeOnly = combineExtractors(extractISOTime);

function parseISOTimeOnly(s) {
  return parse(s, [isoTimeOnly, extractISOTimeOnly]);
}

const sqlYmdWithTimeExtensionRegex = combineRegexes(sqlYmdRegex, sqlTimeExtensionRegex);
const sqlTimeCombinedRegex = combineRegexes(sqlTimeRegex);

const extractISOTimeOffsetAndIANAZone = combineExtractors(
  extractISOTime,
  extractISOOffset,
  extractIANAZone
);

function parseSQL(s) {
  return parse(
    s,
    [sqlYmdWithTimeExtensionRegex, extractISOYmdTimeAndOffset],
    [sqlTimeCombinedRegex, extractISOTimeOffsetAndIANAZone]
  );
}

const INVALID$2 = "Invalid Duration";

// unit conversion constants
const lowOrderMatrix = {
    weeks: {
      days: 7,
      hours: 7 * 24,
      minutes: 7 * 24 * 60,
      seconds: 7 * 24 * 60 * 60,
      milliseconds: 7 * 24 * 60 * 60 * 1000,
    },
    days: {
      hours: 24,
      minutes: 24 * 60,
      seconds: 24 * 60 * 60,
      milliseconds: 24 * 60 * 60 * 1000,
    },
    hours: { minutes: 60, seconds: 60 * 60, milliseconds: 60 * 60 * 1000 },
    minutes: { seconds: 60, milliseconds: 60 * 1000 },
    seconds: { milliseconds: 1000 },
  },
  casualMatrix = {
    years: {
      quarters: 4,
      months: 12,
      weeks: 52,
      days: 365,
      hours: 365 * 24,
      minutes: 365 * 24 * 60,
      seconds: 365 * 24 * 60 * 60,
      milliseconds: 365 * 24 * 60 * 60 * 1000,
    },
    quarters: {
      months: 3,
      weeks: 13,
      days: 91,
      hours: 91 * 24,
      minutes: 91 * 24 * 60,
      seconds: 91 * 24 * 60 * 60,
      milliseconds: 91 * 24 * 60 * 60 * 1000,
    },
    months: {
      weeks: 4,
      days: 30,
      hours: 30 * 24,
      minutes: 30 * 24 * 60,
      seconds: 30 * 24 * 60 * 60,
      milliseconds: 30 * 24 * 60 * 60 * 1000,
    },

    ...lowOrderMatrix,
  },
  daysInYearAccurate = 146097.0 / 400,
  daysInMonthAccurate = 146097.0 / 4800,
  accurateMatrix = {
    years: {
      quarters: 4,
      months: 12,
      weeks: daysInYearAccurate / 7,
      days: daysInYearAccurate,
      hours: daysInYearAccurate * 24,
      minutes: daysInYearAccurate * 24 * 60,
      seconds: daysInYearAccurate * 24 * 60 * 60,
      milliseconds: daysInYearAccurate * 24 * 60 * 60 * 1000,
    },
    quarters: {
      months: 3,
      weeks: daysInYearAccurate / 28,
      days: daysInYearAccurate / 4,
      hours: (daysInYearAccurate * 24) / 4,
      minutes: (daysInYearAccurate * 24 * 60) / 4,
      seconds: (daysInYearAccurate * 24 * 60 * 60) / 4,
      milliseconds: (daysInYearAccurate * 24 * 60 * 60 * 1000) / 4,
    },
    months: {
      weeks: daysInMonthAccurate / 7,
      days: daysInMonthAccurate,
      hours: daysInMonthAccurate * 24,
      minutes: daysInMonthAccurate * 24 * 60,
      seconds: daysInMonthAccurate * 24 * 60 * 60,
      milliseconds: daysInMonthAccurate * 24 * 60 * 60 * 1000,
    },
    ...lowOrderMatrix,
  };

// units ordered by size
const orderedUnits$1 = [
  "years",
  "quarters",
  "months",
  "weeks",
  "days",
  "hours",
  "minutes",
  "seconds",
  "milliseconds",
];

const reverseUnits = orderedUnits$1.slice(0).reverse();

// clone really means "create another instance just like this one, but with these changes"
function clone$1(dur, alts, clear = false) {
  // deep merge for vals
  const conf = {
    values: clear ? alts.values : { ...dur.values, ...(alts.values || {}) },
    loc: dur.loc.clone(alts.loc),
    conversionAccuracy: alts.conversionAccuracy || dur.conversionAccuracy,
    matrix: alts.matrix || dur.matrix,
  };
  return new Duration(conf);
}

function durationToMillis(matrix, vals) {
  let sum = vals.milliseconds ?? 0;
  for (const unit of reverseUnits.slice(1)) {
    if (vals[unit]) {
      sum += vals[unit] * matrix[unit]["milliseconds"];
    }
  }
  return sum;
}

// NB: mutates parameters
function normalizeValues(matrix, vals) {
  // the logic below assumes the overall value of the duration is positive
  // if this is not the case, factor is used to make it so
  const factor = durationToMillis(matrix, vals) < 0 ? -1 : 1;

  orderedUnits$1.reduceRight((previous, current) => {
    if (!isUndefined(vals[current])) {
      if (previous) {
        const previousVal = vals[previous] * factor;
        const conv = matrix[current][previous];

        // if (previousVal < 0):
        // lower order unit is negative (e.g. { years: 2, days: -2 })
        // normalize this by reducing the higher order unit by the appropriate amount
        // and increasing the lower order unit
        // this can never make the higher order unit negative, because this function only operates
        // on positive durations, so the amount of time represented by the lower order unit cannot
        // be larger than the higher order unit
        // else:
        // lower order unit is positive (e.g. { years: 2, days: 450 } or { years: -2, days: 450 })
        // in this case we attempt to convert as much as possible from the lower order unit into
        // the higher order one
        //
        // Math.floor takes care of both of these cases, rounding away from 0
        // if previousVal < 0 it makes the absolute value larger
        // if previousVal >= it makes the absolute value smaller
        const rollUp = Math.floor(previousVal / conv);
        vals[current] += rollUp * factor;
        vals[previous] -= rollUp * conv * factor;
      }
      return current;
    } else {
      return previous;
    }
  }, null);

  // try to convert any decimals into smaller units if possible
  // for example for { years: 2.5, days: 0, seconds: 0 } we want to get { years: 2, days: 182, hours: 12 }
  orderedUnits$1.reduce((previous, current) => {
    if (!isUndefined(vals[current])) {
      if (previous) {
        const fraction = vals[previous] % 1;
        vals[previous] -= fraction;
        vals[current] += fraction * matrix[previous][current];
      }
      return current;
    } else {
      return previous;
    }
  }, null);
}

// Remove all properties with a value of 0 from an object
function removeZeroes(vals) {
  const newVals = {};
  for (const [key, value] of Object.entries(vals)) {
    if (value !== 0) {
      newVals[key] = value;
    }
  }
  return newVals;
}

/**
 * A Duration object represents a period of time, like "2 months" or "1 day, 1 hour". Conceptually, it's just a map of units to their quantities, accompanied by some additional configuration and methods for creating, parsing, interrogating, transforming, and formatting them. They can be used on their own or in conjunction with other Luxon types; for example, you can use {@link DateTime#plus} to add a Duration object to a DateTime, producing another DateTime.
 *
 * Here is a brief overview of commonly used methods and getters in Duration:
 *
 * * **Creation** To create a Duration, use {@link Duration.fromMillis}, {@link Duration.fromObject}, or {@link Duration.fromISO}.
 * * **Unit values** See the {@link Duration#years}, {@link Duration#months}, {@link Duration#weeks}, {@link Duration#days}, {@link Duration#hours}, {@link Duration#minutes}, {@link Duration#seconds}, {@link Duration#milliseconds} accessors.
 * * **Configuration** See  {@link Duration#locale} and {@link Duration#numberingSystem} accessors.
 * * **Transformation** To create new Durations out of old ones use {@link Duration#plus}, {@link Duration#minus}, {@link Duration#normalize}, {@link Duration#set}, {@link Duration#reconfigure}, {@link Duration#shiftTo}, and {@link Duration#negate}.
 * * **Output** To convert the Duration into other representations, see {@link Duration#as}, {@link Duration#toISO}, {@link Duration#toFormat}, and {@link Duration#toJSON}
 *
 * There's are more methods documented below. In addition, for more information on subtler topics like internationalization and validity, see the external documentation.
 */
class Duration {
  /**
   * @private
   */
  constructor(config) {
    const accurate = config.conversionAccuracy === "longterm" || false;
    let matrix = accurate ? accurateMatrix : casualMatrix;

    if (config.matrix) {
      matrix = config.matrix;
    }

    /**
     * @access private
     */
    this.values = config.values;
    /**
     * @access private
     */
    this.loc = config.loc || Locale.create();
    /**
     * @access private
     */
    this.conversionAccuracy = accurate ? "longterm" : "casual";
    /**
     * @access private
     */
    this.invalid = config.invalid || null;
    /**
     * @access private
     */
    this.matrix = matrix;
    /**
     * @access private
     */
    this.isLuxonDuration = true;
  }

  /**
   * Create Duration from a number of milliseconds.
   * @param {number} count of milliseconds
   * @param {Object} opts - options for parsing
   * @param {string} [opts.locale='en-US'] - the locale to use
   * @param {string} opts.numberingSystem - the numbering system to use
   * @param {string} [opts.conversionAccuracy='casual'] - the conversion system to use
   * @return {Duration}
   */
  static fromMillis(count, opts) {
    return Duration.fromObject({ milliseconds: count }, opts);
  }

  /**
   * Create a Duration from a JavaScript object with keys like 'years' and 'hours'.
   * If this object is empty then a zero milliseconds duration is returned.
   * @param {Object} obj - the object to create the DateTime from
   * @param {number} obj.years
   * @param {number} obj.quarters
   * @param {number} obj.months
   * @param {number} obj.weeks
   * @param {number} obj.days
   * @param {number} obj.hours
   * @param {number} obj.minutes
   * @param {number} obj.seconds
   * @param {number} obj.milliseconds
   * @param {Object} [opts=[]] - options for creating this Duration
   * @param {string} [opts.locale='en-US'] - the locale to use
   * @param {string} opts.numberingSystem - the numbering system to use
   * @param {string} [opts.conversionAccuracy='casual'] - the preset conversion system to use
   * @param {string} [opts.matrix=Object] - the custom conversion system to use
   * @return {Duration}
   */
  static fromObject(obj, opts = {}) {
    if (obj == null || typeof obj !== "object") {
      throw new InvalidArgumentError(
        `Duration.fromObject: argument expected to be an object, got ${
          obj === null ? "null" : typeof obj
        }`
      );
    }

    return new Duration({
      values: normalizeObject(obj, Duration.normalizeUnit),
      loc: Locale.fromObject(opts),
      conversionAccuracy: opts.conversionAccuracy,
      matrix: opts.matrix,
    });
  }

  /**
   * Create a Duration from DurationLike.
   *
   * @param {Object | number | Duration} durationLike
   * One of:
   * - object with keys like 'years' and 'hours'.
   * - number representing milliseconds
   * - Duration instance
   * @return {Duration}
   */
  static fromDurationLike(durationLike) {
    if (isNumber(durationLike)) {
      return Duration.fromMillis(durationLike);
    } else if (Duration.isDuration(durationLike)) {
      return durationLike;
    } else if (typeof durationLike === "object") {
      return Duration.fromObject(durationLike);
    } else {
      throw new InvalidArgumentError(
        `Unknown duration argument ${durationLike} of type ${typeof durationLike}`
      );
    }
  }

  /**
   * Create a Duration from an ISO 8601 duration string.
   * @param {string} text - text to parse
   * @param {Object} opts - options for parsing
   * @param {string} [opts.locale='en-US'] - the locale to use
   * @param {string} opts.numberingSystem - the numbering system to use
   * @param {string} [opts.conversionAccuracy='casual'] - the preset conversion system to use
   * @param {string} [opts.matrix=Object] - the preset conversion system to use
   * @see https://en.wikipedia.org/wiki/ISO_8601#Durations
   * @example Duration.fromISO('P3Y6M1W4DT12H30M5S').toObject() //=> { years: 3, months: 6, weeks: 1, days: 4, hours: 12, minutes: 30, seconds: 5 }
   * @example Duration.fromISO('PT23H').toObject() //=> { hours: 23 }
   * @example Duration.fromISO('P5Y3M').toObject() //=> { years: 5, months: 3 }
   * @return {Duration}
   */
  static fromISO(text, opts) {
    const [parsed] = parseISODuration(text);
    if (parsed) {
      return Duration.fromObject(parsed, opts);
    } else {
      return Duration.invalid("unparsable", `the input "${text}" can't be parsed as ISO 8601`);
    }
  }

  /**
   * Create a Duration from an ISO 8601 time string.
   * @param {string} text - text to parse
   * @param {Object} opts - options for parsing
   * @param {string} [opts.locale='en-US'] - the locale to use
   * @param {string} opts.numberingSystem - the numbering system to use
   * @param {string} [opts.conversionAccuracy='casual'] - the preset conversion system to use
   * @param {string} [opts.matrix=Object] - the conversion system to use
   * @see https://en.wikipedia.org/wiki/ISO_8601#Times
   * @example Duration.fromISOTime('11:22:33.444').toObject() //=> { hours: 11, minutes: 22, seconds: 33, milliseconds: 444 }
   * @example Duration.fromISOTime('11:00').toObject() //=> { hours: 11, minutes: 0, seconds: 0 }
   * @example Duration.fromISOTime('T11:00').toObject() //=> { hours: 11, minutes: 0, seconds: 0 }
   * @example Duration.fromISOTime('1100').toObject() //=> { hours: 11, minutes: 0, seconds: 0 }
   * @example Duration.fromISOTime('T1100').toObject() //=> { hours: 11, minutes: 0, seconds: 0 }
   * @return {Duration}
   */
  static fromISOTime(text, opts) {
    const [parsed] = parseISOTimeOnly(text);
    if (parsed) {
      return Duration.fromObject(parsed, opts);
    } else {
      return Duration.invalid("unparsable", `the input "${text}" can't be parsed as ISO 8601`);
    }
  }

  /**
   * Create an invalid Duration.
   * @param {string} reason - simple string of why this datetime is invalid. Should not contain parameters or anything else data-dependent
   * @param {string} [explanation=null] - longer explanation, may include parameters and other useful debugging information
   * @return {Duration}
   */
  static invalid(reason, explanation = null) {
    if (!reason) {
      throw new InvalidArgumentError("need to specify a reason the Duration is invalid");
    }

    const invalid = reason instanceof Invalid ? reason : new Invalid(reason, explanation);

    if (Settings.throwOnInvalid) {
      throw new InvalidDurationError(invalid);
    } else {
      return new Duration({ invalid });
    }
  }

  /**
   * @private
   */
  static normalizeUnit(unit) {
    const normalized = {
      year: "years",
      years: "years",
      quarter: "quarters",
      quarters: "quarters",
      month: "months",
      months: "months",
      week: "weeks",
      weeks: "weeks",
      day: "days",
      days: "days",
      hour: "hours",
      hours: "hours",
      minute: "minutes",
      minutes: "minutes",
      second: "seconds",
      seconds: "seconds",
      millisecond: "milliseconds",
      milliseconds: "milliseconds",
    }[unit ? unit.toLowerCase() : unit];

    if (!normalized) throw new InvalidUnitError(unit);

    return normalized;
  }

  /**
   * Check if an object is a Duration. Works across context boundaries
   * @param {object} o
   * @return {boolean}
   */
  static isDuration(o) {
    return (o && o.isLuxonDuration) || false;
  }

  /**
   * Get  the locale of a Duration, such 'en-GB'
   * @type {string}
   */
  get locale() {
    return this.isValid ? this.loc.locale : null;
  }

  /**
   * Get the numbering system of a Duration, such 'beng'. The numbering system is used when formatting the Duration
   *
   * @type {string}
   */
  get numberingSystem() {
    return this.isValid ? this.loc.numberingSystem : null;
  }

  /**
   * Returns a string representation of this Duration formatted according to the specified format string. You may use these tokens:
   * * `S` for milliseconds
   * * `s` for seconds
   * * `m` for minutes
   * * `h` for hours
   * * `d` for days
   * * `w` for weeks
   * * `M` for months
   * * `y` for years
   * Notes:
   * * Add padding by repeating the token, e.g. "yy" pads the years to two digits, "hhhh" pads the hours out to four digits
   * * Tokens can be escaped by wrapping with single quotes.
   * * The duration will be converted to the set of units in the format string using {@link Duration#shiftTo} and the Durations's conversion accuracy setting.
   * @param {string} fmt - the format string
   * @param {Object} opts - options
   * @param {boolean} [opts.floor=true] - floor numerical values
   * @example Duration.fromObject({ years: 1, days: 6, seconds: 2 }).toFormat("y d s") //=> "1 6 2"
   * @example Duration.fromObject({ years: 1, days: 6, seconds: 2 }).toFormat("yy dd sss") //=> "01 06 002"
   * @example Duration.fromObject({ years: 1, days: 6, seconds: 2 }).toFormat("M S") //=> "12 518402000"
   * @return {string}
   */
  toFormat(fmt, opts = {}) {
    // reverse-compat since 1.2; we always round down now, never up, and we do it by default
    const fmtOpts = {
      ...opts,
      floor: opts.round !== false && opts.floor !== false,
    };
    return this.isValid
      ? Formatter.create(this.loc, fmtOpts).formatDurationFromString(this, fmt)
      : INVALID$2;
  }

  /**
   * Returns a string representation of a Duration with all units included.
   * To modify its behavior use the `listStyle` and any Intl.NumberFormat option, though `unitDisplay` is especially relevant.
   * @see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Intl/NumberFormat
   * @param opts - On option object to override the formatting. Accepts the same keys as the options parameter of the native `Int.NumberFormat` constructor, as well as `listStyle`.
   * @example
   * ```js
   * var dur = Duration.fromObject({ days: 1, hours: 5, minutes: 6 })
   * dur.toHuman() //=> '1 day, 5 hours, 6 minutes'
   * dur.toHuman({ listStyle: "long" }) //=> '1 day, 5 hours, and 6 minutes'
   * dur.toHuman({ unitDisplay: "short" }) //=> '1 day, 5 hr, 6 min'
   * ```
   */
  toHuman(opts = {}) {
    if (!this.isValid) return INVALID$2;

    const l = orderedUnits$1
      .map((unit) => {
        const val = this.values[unit];
        if (isUndefined(val)) {
          return null;
        }
        return this.loc
          .numberFormatter({ style: "unit", unitDisplay: "long", ...opts, unit: unit.slice(0, -1) })
          .format(val);
      })
      .filter((n) => n);

    return this.loc
      .listFormatter({ type: "conjunction", style: opts.listStyle || "narrow", ...opts })
      .format(l);
  }

  /**
   * Returns a JavaScript object with this Duration's values.
   * @example Duration.fromObject({ years: 1, days: 6, seconds: 2 }).toObject() //=> { years: 1, days: 6, seconds: 2 }
   * @return {Object}
   */
  toObject() {
    if (!this.isValid) return {};
    return { ...this.values };
  }

  /**
   * Returns an ISO 8601-compliant string representation of this Duration.
   * @see https://en.wikipedia.org/wiki/ISO_8601#Durations
   * @example Duration.fromObject({ years: 3, seconds: 45 }).toISO() //=> 'P3YT45S'
   * @example Duration.fromObject({ months: 4, seconds: 45 }).toISO() //=> 'P4MT45S'
   * @example Duration.fromObject({ months: 5 }).toISO() //=> 'P5M'
   * @example Duration.fromObject({ minutes: 5 }).toISO() //=> 'PT5M'
   * @example Duration.fromObject({ milliseconds: 6 }).toISO() //=> 'PT0.006S'
   * @return {string}
   */
  toISO() {
    // we could use the formatter, but this is an easier way to get the minimum string
    if (!this.isValid) return null;

    let s = "P";
    if (this.years !== 0) s += this.years + "Y";
    if (this.months !== 0 || this.quarters !== 0) s += this.months + this.quarters * 3 + "M";
    if (this.weeks !== 0) s += this.weeks + "W";
    if (this.days !== 0) s += this.days + "D";
    if (this.hours !== 0 || this.minutes !== 0 || this.seconds !== 0 || this.milliseconds !== 0)
      s += "T";
    if (this.hours !== 0) s += this.hours + "H";
    if (this.minutes !== 0) s += this.minutes + "M";
    if (this.seconds !== 0 || this.milliseconds !== 0)
      // this will handle "floating point madness" by removing extra decimal places
      // https://stackoverflow.com/questions/588004/is-floating-point-math-broken
      s += roundTo(this.seconds + this.milliseconds / 1000, 3) + "S";
    if (s === "P") s += "T0S";
    return s;
  }

  /**
   * Returns an ISO 8601-compliant string representation of this Duration, formatted as a time of day.
   * Note that this will return null if the duration is invalid, negative, or equal to or greater than 24 hours.
   * @see https://en.wikipedia.org/wiki/ISO_8601#Times
   * @param {Object} opts - options
   * @param {boolean} [opts.suppressMilliseconds=false] - exclude milliseconds from the format if they're 0
   * @param {boolean} [opts.suppressSeconds=false] - exclude seconds from the format if they're 0
   * @param {boolean} [opts.includePrefix=false] - include the `T` prefix
   * @param {string} [opts.format='extended'] - choose between the basic and extended format
   * @example Duration.fromObject({ hours: 11 }).toISOTime() //=> '11:00:00.000'
   * @example Duration.fromObject({ hours: 11 }).toISOTime({ suppressMilliseconds: true }) //=> '11:00:00'
   * @example Duration.fromObject({ hours: 11 }).toISOTime({ suppressSeconds: true }) //=> '11:00'
   * @example Duration.fromObject({ hours: 11 }).toISOTime({ includePrefix: true }) //=> 'T11:00:00.000'
   * @example Duration.fromObject({ hours: 11 }).toISOTime({ format: 'basic' }) //=> '110000.000'
   * @return {string}
   */
  toISOTime(opts = {}) {
    if (!this.isValid) return null;

    const millis = this.toMillis();
    if (millis < 0 || millis >= 86400000) return null;

    opts = {
      suppressMilliseconds: false,
      suppressSeconds: false,
      includePrefix: false,
      format: "extended",
      ...opts,
      includeOffset: false,
    };

    const dateTime = DateTime.fromMillis(millis, { zone: "UTC" });
    return dateTime.toISOTime(opts);
  }

  /**
   * Returns an ISO 8601 representation of this Duration appropriate for use in JSON.
   * @return {string}
   */
  toJSON() {
    return this.toISO();
  }

  /**
   * Returns an ISO 8601 representation of this Duration appropriate for use in debugging.
   * @return {string}
   */
  toString() {
    return this.toISO();
  }

  /**
   * Returns an milliseconds value of this Duration.
   * @return {number}
   */
  toMillis() {
    if (!this.isValid) return NaN;

    return durationToMillis(this.matrix, this.values);
  }

  /**
   * Returns an milliseconds value of this Duration. Alias of {@link toMillis}
   * @return {number}
   */
  valueOf() {
    return this.toMillis();
  }

  /**
   * Make this Duration longer by the specified amount. Return a newly-constructed Duration.
   * @param {Duration|Object|number} duration - The amount to add. Either a Luxon Duration, a number of milliseconds, the object argument to Duration.fromObject()
   * @return {Duration}
   */
  plus(duration) {
    if (!this.isValid) return this;

    const dur = Duration.fromDurationLike(duration),
      result = {};

    for (const k of orderedUnits$1) {
      if (hasOwnProperty(dur.values, k) || hasOwnProperty(this.values, k)) {
        result[k] = dur.get(k) + this.get(k);
      }
    }

    return clone$1(this, { values: result }, true);
  }

  /**
   * Make this Duration shorter by the specified amount. Return a newly-constructed Duration.
   * @param {Duration|Object|number} duration - The amount to subtract. Either a Luxon Duration, a number of milliseconds, the object argument to Duration.fromObject()
   * @return {Duration}
   */
  minus(duration) {
    if (!this.isValid) return this;

    const dur = Duration.fromDurationLike(duration);
    return this.plus(dur.negate());
  }

  /**
   * Scale this Duration by the specified amount. Return a newly-constructed Duration.
   * @param {function} fn - The function to apply to each unit. Arity is 1 or 2: the value of the unit and, optionally, the unit name. Must return a number.
   * @example Duration.fromObject({ hours: 1, minutes: 30 }).mapUnits(x => x * 2) //=> { hours: 2, minutes: 60 }
   * @example Duration.fromObject({ hours: 1, minutes: 30 }).mapUnits((x, u) => u === "hours" ? x * 2 : x) //=> { hours: 2, minutes: 30 }
   * @return {Duration}
   */
  mapUnits(fn) {
    if (!this.isValid) return this;
    const result = {};
    for (const k of Object.keys(this.values)) {
      result[k] = asNumber(fn(this.values[k], k));
    }
    return clone$1(this, { values: result }, true);
  }

  /**
   * Get the value of unit.
   * @param {string} unit - a unit such as 'minute' or 'day'
   * @example Duration.fromObject({years: 2, days: 3}).get('years') //=> 2
   * @example Duration.fromObject({years: 2, days: 3}).get('months') //=> 0
   * @example Duration.fromObject({years: 2, days: 3}).get('days') //=> 3
   * @return {number}
   */
  get(unit) {
    return this[Duration.normalizeUnit(unit)];
  }

  /**
   * "Set" the values of specified units. Return a newly-constructed Duration.
   * @param {Object} values - a mapping of units to numbers
   * @example dur.set({ years: 2017 })
   * @example dur.set({ hours: 8, minutes: 30 })
   * @return {Duration}
   */
  set(values) {
    if (!this.isValid) return this;

    const mixed = { ...this.values, ...normalizeObject(values, Duration.normalizeUnit) };
    return clone$1(this, { values: mixed });
  }

  /**
   * "Set" the locale and/or numberingSystem.  Returns a newly-constructed Duration.
   * @example dur.reconfigure({ locale: 'en-GB' })
   * @return {Duration}
   */
  reconfigure({ locale, numberingSystem, conversionAccuracy, matrix } = {}) {
    const loc = this.loc.clone({ locale, numberingSystem });
    const opts = { loc, matrix, conversionAccuracy };
    return clone$1(this, opts);
  }

  /**
   * Return the length of the duration in the specified unit.
   * @param {string} unit - a unit such as 'minutes' or 'days'
   * @example Duration.fromObject({years: 1}).as('days') //=> 365
   * @example Duration.fromObject({years: 1}).as('months') //=> 12
   * @example Duration.fromObject({hours: 60}).as('days') //=> 2.5
   * @return {number}
   */
  as(unit) {
    return this.isValid ? this.shiftTo(unit).get(unit) : NaN;
  }

  /**
   * Reduce this Duration to its canonical representation in its current units.
   * Assuming the overall value of the Duration is positive, this means:
   * - excessive values for lower-order units are converted to higher-order units (if possible, see first and second example)
   * - negative lower-order units are converted to higher order units (there must be such a higher order unit, otherwise
   *   the overall value would be negative, see second example)
   * - fractional values for higher-order units are converted to lower-order units (if possible, see fourth example)
   *
   * If the overall value is negative, the result of this method is equivalent to `this.negate().normalize().negate()`.
   * @example Duration.fromObject({ years: 2, days: 5000 }).normalize().toObject() //=> { years: 15, days: 255 }
   * @example Duration.fromObject({ days: 5000 }).normalize().toObject() //=> { days: 5000 }
   * @example Duration.fromObject({ hours: 12, minutes: -45 }).normalize().toObject() //=> { hours: 11, minutes: 15 }
   * @example Duration.fromObject({ years: 2.5, days: 0, hours: 0 }).normalize().toObject() //=> { years: 2, days: 182, hours: 12 }
   * @return {Duration}
   */
  normalize() {
    if (!this.isValid) return this;
    const vals = this.toObject();
    normalizeValues(this.matrix, vals);
    return clone$1(this, { values: vals }, true);
  }

  /**
   * Rescale units to its largest representation
   * @example Duration.fromObject({ milliseconds: 90000 }).rescale().toObject() //=> { minutes: 1, seconds: 30 }
   * @return {Duration}
   */
  rescale() {
    if (!this.isValid) return this;
    const vals = removeZeroes(this.normalize().shiftToAll().toObject());
    return clone$1(this, { values: vals }, true);
  }

  /**
   * Convert this Duration into its representation in a different set of units.
   * @example Duration.fromObject({ hours: 1, seconds: 30 }).shiftTo('minutes', 'milliseconds').toObject() //=> { minutes: 60, milliseconds: 30000 }
   * @return {Duration}
   */
  shiftTo(...units) {
    if (!this.isValid) return this;

    if (units.length === 0) {
      return this;
    }

    units = units.map((u) => Duration.normalizeUnit(u));

    const built = {},
      accumulated = {},
      vals = this.toObject();
    let lastUnit;

    for (const k of orderedUnits$1) {
      if (units.indexOf(k) >= 0) {
        lastUnit = k;

        let own = 0;

        // anything we haven't boiled down yet should get boiled to this unit
        for (const ak in accumulated) {
          own += this.matrix[ak][k] * accumulated[ak];
          accumulated[ak] = 0;
        }

        // plus anything that's already in this unit
        if (isNumber(vals[k])) {
          own += vals[k];
        }

        // only keep the integer part for now in the hopes of putting any decimal part
        // into a smaller unit later
        const i = Math.trunc(own);
        built[k] = i;
        accumulated[k] = (own * 1000 - i * 1000) / 1000;

        // otherwise, keep it in the wings to boil it later
      } else if (isNumber(vals[k])) {
        accumulated[k] = vals[k];
      }
    }

    // anything leftover becomes the decimal for the last unit
    // lastUnit must be defined since units is not empty
    for (const key in accumulated) {
      if (accumulated[key] !== 0) {
        built[lastUnit] +=
          key === lastUnit ? accumulated[key] : accumulated[key] / this.matrix[lastUnit][key];
      }
    }

    normalizeValues(this.matrix, built);
    return clone$1(this, { values: built }, true);
  }

  /**
   * Shift this Duration to all available units.
   * Same as shiftTo("years", "months", "weeks", "days", "hours", "minutes", "seconds", "milliseconds")
   * @return {Duration}
   */
  shiftToAll() {
    if (!this.isValid) return this;
    return this.shiftTo(
      "years",
      "months",
      "weeks",
      "days",
      "hours",
      "minutes",
      "seconds",
      "milliseconds"
    );
  }

  /**
   * Return the negative of this Duration.
   * @example Duration.fromObject({ hours: 1, seconds: 30 }).negate().toObject() //=> { hours: -1, seconds: -30 }
   * @return {Duration}
   */
  negate() {
    if (!this.isValid) return this;
    const negated = {};
    for (const k of Object.keys(this.values)) {
      negated[k] = this.values[k] === 0 ? 0 : -this.values[k];
    }
    return clone$1(this, { values: negated }, true);
  }

  /**
   * Get the years.
   * @type {number}
   */
  get years() {
    return this.isValid ? this.values.years || 0 : NaN;
  }

  /**
   * Get the quarters.
   * @type {number}
   */
  get quarters() {
    return this.isValid ? this.values.quarters || 0 : NaN;
  }

  /**
   * Get the months.
   * @type {number}
   */
  get months() {
    return this.isValid ? this.values.months || 0 : NaN;
  }

  /**
   * Get the weeks
   * @type {number}
   */
  get weeks() {
    return this.isValid ? this.values.weeks || 0 : NaN;
  }

  /**
   * Get the days.
   * @type {number}
   */
  get days() {
    return this.isValid ? this.values.days || 0 : NaN;
  }

  /**
   * Get the hours.
   * @type {number}
   */
  get hours() {
    return this.isValid ? this.values.hours || 0 : NaN;
  }

  /**
   * Get the minutes.
   * @type {number}
   */
  get minutes() {
    return this.isValid ? this.values.minutes || 0 : NaN;
  }

  /**
   * Get the seconds.
   * @return {number}
   */
  get seconds() {
    return this.isValid ? this.values.seconds || 0 : NaN;
  }

  /**
   * Get the milliseconds.
   * @return {number}
   */
  get milliseconds() {
    return this.isValid ? this.values.milliseconds || 0 : NaN;
  }

  /**
   * Returns whether the Duration is invalid. Invalid durations are returned by diff operations
   * on invalid DateTimes or Intervals.
   * @return {boolean}
   */
  get isValid() {
    return this.invalid === null;
  }

  /**
   * Returns an error code if this Duration became invalid, or null if the Duration is valid
   * @return {string}
   */
  get invalidReason() {
    return this.invalid ? this.invalid.reason : null;
  }

  /**
   * Returns an explanation of why this Duration became invalid, or null if the Duration is valid
   * @type {string}
   */
  get invalidExplanation() {
    return this.invalid ? this.invalid.explanation : null;
  }

  /**
   * Equality check
   * Two Durations are equal iff they have the same units and the same values for each unit.
   * @param {Duration} other
   * @return {boolean}
   */
  equals(other) {
    if (!this.isValid || !other.isValid) {
      return false;
    }

    if (!this.loc.equals(other.loc)) {
      return false;
    }

    function eq(v1, v2) {
      // Consider 0 and undefined as equal
      if (v1 === undefined || v1 === 0) return v2 === undefined || v2 === 0;
      return v1 === v2;
    }

    for (const u of orderedUnits$1) {
      if (!eq(this.values[u], other.values[u])) {
        return false;
      }
    }
    return true;
  }
}

const INVALID$1 = "Invalid Interval";

// checks if the start is equal to or before the end
function validateStartEnd(start, end) {
  if (!start || !start.isValid) {
    return Interval.invalid("missing or invalid start");
  } else if (!end || !end.isValid) {
    return Interval.invalid("missing or invalid end");
  } else if (end < start) {
    return Interval.invalid(
      "end before start",
      `The end of an interval must be after its start, but you had start=${start.toISO()} and end=${end.toISO()}`
    );
  } else {
    return null;
  }
}

/**
 * An Interval object represents a half-open interval of time, where each endpoint is a {@link DateTime}. Conceptually, it's a container for those two endpoints, accompanied by methods for creating, parsing, interrogating, comparing, transforming, and formatting them.
 *
 * Here is a brief overview of the most commonly used methods and getters in Interval:
 *
 * * **Creation** To create an Interval, use {@link Interval.fromDateTimes}, {@link Interval.after}, {@link Interval.before}, or {@link Interval.fromISO}.
 * * **Accessors** Use {@link Interval#start} and {@link Interval#end} to get the start and end.
 * * **Interrogation** To analyze the Interval, use {@link Interval#count}, {@link Interval#length}, {@link Interval#hasSame}, {@link Interval#contains}, {@link Interval#isAfter}, or {@link Interval#isBefore}.
 * * **Transformation** To create other Intervals out of this one, use {@link Interval#set}, {@link Interval#splitAt}, {@link Interval#splitBy}, {@link Interval#divideEqually}, {@link Interval.merge}, {@link Interval.xor}, {@link Interval#union}, {@link Interval#intersection}, or {@link Interval#difference}.
 * * **Comparison** To compare this Interval to another one, use {@link Interval#equals}, {@link Interval#overlaps}, {@link Interval#abutsStart}, {@link Interval#abutsEnd}, {@link Interval#engulfs}
 * * **Output** To convert the Interval into other representations, see {@link Interval#toString}, {@link Interval#toLocaleString}, {@link Interval#toISO}, {@link Interval#toISODate}, {@link Interval#toISOTime}, {@link Interval#toFormat}, and {@link Interval#toDuration}.
 */
class Interval {
  /**
   * @private
   */
  constructor(config) {
    /**
     * @access private
     */
    this.s = config.start;
    /**
     * @access private
     */
    this.e = config.end;
    /**
     * @access private
     */
    this.invalid = config.invalid || null;
    /**
     * @access private
     */
    this.isLuxonInterval = true;
  }

  /**
   * Create an invalid Interval.
   * @param {string} reason - simple string of why this Interval is invalid. Should not contain parameters or anything else data-dependent
   * @param {string} [explanation=null] - longer explanation, may include parameters and other useful debugging information
   * @return {Interval}
   */
  static invalid(reason, explanation = null) {
    if (!reason) {
      throw new InvalidArgumentError("need to specify a reason the Interval is invalid");
    }

    const invalid = reason instanceof Invalid ? reason : new Invalid(reason, explanation);

    if (Settings.throwOnInvalid) {
      throw new InvalidIntervalError(invalid);
    } else {
      return new Interval({ invalid });
    }
  }

  /**
   * Create an Interval from a start DateTime and an end DateTime. Inclusive of the start but not the end.
   * @param {DateTime|Date|Object} start
   * @param {DateTime|Date|Object} end
   * @return {Interval}
   */
  static fromDateTimes(start, end) {
    const builtStart = friendlyDateTime(start),
      builtEnd = friendlyDateTime(end);

    const validateError = validateStartEnd(builtStart, builtEnd);

    if (validateError == null) {
      return new Interval({
        start: builtStart,
        end: builtEnd,
      });
    } else {
      return validateError;
    }
  }

  /**
   * Create an Interval from a start DateTime and a Duration to extend to.
   * @param {DateTime|Date|Object} start
   * @param {Duration|Object|number} duration - the length of the Interval.
   * @return {Interval}
   */
  static after(start, duration) {
    const dur = Duration.fromDurationLike(duration),
      dt = friendlyDateTime(start);
    return Interval.fromDateTimes(dt, dt.plus(dur));
  }

  /**
   * Create an Interval from an end DateTime and a Duration to extend backwards to.
   * @param {DateTime|Date|Object} end
   * @param {Duration|Object|number} duration - the length of the Interval.
   * @return {Interval}
   */
  static before(end, duration) {
    const dur = Duration.fromDurationLike(duration),
      dt = friendlyDateTime(end);
    return Interval.fromDateTimes(dt.minus(dur), dt);
  }

  /**
   * Create an Interval from an ISO 8601 string.
   * Accepts `<start>/<end>`, `<start>/<duration>`, and `<duration>/<end>` formats.
   * @param {string} text - the ISO string to parse
   * @param {Object} [opts] - options to pass {@link DateTime#fromISO} and optionally {@link Duration#fromISO}
   * @see https://en.wikipedia.org/wiki/ISO_8601#Time_intervals
   * @return {Interval}
   */
  static fromISO(text, opts) {
    const [s, e] = (text || "").split("/", 2);
    if (s && e) {
      let start, startIsValid;
      try {
        start = DateTime.fromISO(s, opts);
        startIsValid = start.isValid;
      } catch (e) {
        startIsValid = false;
      }

      let end, endIsValid;
      try {
        end = DateTime.fromISO(e, opts);
        endIsValid = end.isValid;
      } catch (e) {
        endIsValid = false;
      }

      if (startIsValid && endIsValid) {
        return Interval.fromDateTimes(start, end);
      }

      if (startIsValid) {
        const dur = Duration.fromISO(e, opts);
        if (dur.isValid) {
          return Interval.after(start, dur);
        }
      } else if (endIsValid) {
        const dur = Duration.fromISO(s, opts);
        if (dur.isValid) {
          return Interval.before(end, dur);
        }
      }
    }
    return Interval.invalid("unparsable", `the input "${text}" can't be parsed as ISO 8601`);
  }

  /**
   * Check if an object is an Interval. Works across context boundaries
   * @param {object} o
   * @return {boolean}
   */
  static isInterval(o) {
    return (o && o.isLuxonInterval) || false;
  }

  /**
   * Returns the start of the Interval
   * @type {DateTime}
   */
  get start() {
    return this.isValid ? this.s : null;
  }

  /**
   * Returns the end of the Interval
   * @type {DateTime}
   */
  get end() {
    return this.isValid ? this.e : null;
  }

  /**
   * Returns whether this Interval's end is at least its start, meaning that the Interval isn't 'backwards'.
   * @type {boolean}
   */
  get isValid() {
    return this.invalidReason === null;
  }

  /**
   * Returns an error code if this Interval is invalid, or null if the Interval is valid
   * @type {string}
   */
  get invalidReason() {
    return this.invalid ? this.invalid.reason : null;
  }

  /**
   * Returns an explanation of why this Interval became invalid, or null if the Interval is valid
   * @type {string}
   */
  get invalidExplanation() {
    return this.invalid ? this.invalid.explanation : null;
  }

  /**
   * Returns the length of the Interval in the specified unit.
   * @param {string} unit - the unit (such as 'hours' or 'days') to return the length in.
   * @return {number}
   */
  length(unit = "milliseconds") {
    return this.isValid ? this.toDuration(...[unit]).get(unit) : NaN;
  }

  /**
   * Returns the count of minutes, hours, days, months, or years included in the Interval, even in part.
   * Unlike {@link Interval#length} this counts sections of the calendar, not periods of time, e.g. specifying 'day'
   * asks 'what dates are included in this interval?', not 'how many days long is this interval?'
   * @param {string} [unit='milliseconds'] - the unit of time to count.
   * @return {number}
   */
  count(unit = "milliseconds") {
    if (!this.isValid) return NaN;
    const start = this.start.startOf(unit),
      end = this.end.startOf(unit);
    return Math.floor(end.diff(start, unit).get(unit)) + (end.valueOf() !== this.end.valueOf());
  }

  /**
   * Returns whether this Interval's start and end are both in the same unit of time
   * @param {string} unit - the unit of time to check sameness on
   * @return {boolean}
   */
  hasSame(unit) {
    return this.isValid ? this.isEmpty() || this.e.minus(1).hasSame(this.s, unit) : false;
  }

  /**
   * Return whether this Interval has the same start and end DateTimes.
   * @return {boolean}
   */
  isEmpty() {
    return this.s.valueOf() === this.e.valueOf();
  }

  /**
   * Return whether this Interval's start is after the specified DateTime.
   * @param {DateTime} dateTime
   * @return {boolean}
   */
  isAfter(dateTime) {
    if (!this.isValid) return false;
    return this.s > dateTime;
  }

  /**
   * Return whether this Interval's end is before the specified DateTime.
   * @param {DateTime} dateTime
   * @return {boolean}
   */
  isBefore(dateTime) {
    if (!this.isValid) return false;
    return this.e <= dateTime;
  }

  /**
   * Return whether this Interval contains the specified DateTime.
   * @param {DateTime} dateTime
   * @return {boolean}
   */
  contains(dateTime) {
    if (!this.isValid) return false;
    return this.s <= dateTime && this.e > dateTime;
  }

  /**
   * "Sets" the start and/or end dates. Returns a newly-constructed Interval.
   * @param {Object} values - the values to set
   * @param {DateTime} values.start - the starting DateTime
   * @param {DateTime} values.end - the ending DateTime
   * @return {Interval}
   */
  set({ start, end } = {}) {
    if (!this.isValid) return this;
    return Interval.fromDateTimes(start || this.s, end || this.e);
  }

  /**
   * Split this Interval at each of the specified DateTimes
   * @param {...DateTime} dateTimes - the unit of time to count.
   * @return {Array}
   */
  splitAt(...dateTimes) {
    if (!this.isValid) return [];
    const sorted = dateTimes
        .map(friendlyDateTime)
        .filter((d) => this.contains(d))
        .sort(),
      results = [];
    let { s } = this,
      i = 0;

    while (s < this.e) {
      const added = sorted[i] || this.e,
        next = +added > +this.e ? this.e : added;
      results.push(Interval.fromDateTimes(s, next));
      s = next;
      i += 1;
    }

    return results;
  }

  /**
   * Split this Interval into smaller Intervals, each of the specified length.
   * Left over time is grouped into a smaller interval
   * @param {Duration|Object|number} duration - The length of each resulting interval.
   * @return {Array}
   */
  splitBy(duration) {
    const dur = Duration.fromDurationLike(duration);

    if (!this.isValid || !dur.isValid || dur.as("milliseconds") === 0) {
      return [];
    }

    let { s } = this,
      idx = 1,
      next;

    const results = [];
    while (s < this.e) {
      const added = this.start.plus(dur.mapUnits((x) => x * idx));
      next = +added > +this.e ? this.e : added;
      results.push(Interval.fromDateTimes(s, next));
      s = next;
      idx += 1;
    }

    return results;
  }

  /**
   * Split this Interval into the specified number of smaller intervals.
   * @param {number} numberOfParts - The number of Intervals to divide the Interval into.
   * @return {Array}
   */
  divideEqually(numberOfParts) {
    if (!this.isValid) return [];
    return this.splitBy(this.length() / numberOfParts).slice(0, numberOfParts);
  }

  /**
   * Return whether this Interval overlaps with the specified Interval
   * @param {Interval} other
   * @return {boolean}
   */
  overlaps(other) {
    return this.e > other.s && this.s < other.e;
  }

  /**
   * Return whether this Interval's end is adjacent to the specified Interval's start.
   * @param {Interval} other
   * @return {boolean}
   */
  abutsStart(other) {
    if (!this.isValid) return false;
    return +this.e === +other.s;
  }

  /**
   * Return whether this Interval's start is adjacent to the specified Interval's end.
   * @param {Interval} other
   * @return {boolean}
   */
  abutsEnd(other) {
    if (!this.isValid) return false;
    return +other.e === +this.s;
  }

  /**
   * Return whether this Interval engulfs the start and end of the specified Interval.
   * @param {Interval} other
   * @return {boolean}
   */
  engulfs(other) {
    if (!this.isValid) return false;
    return this.s <= other.s && this.e >= other.e;
  }

  /**
   * Return whether this Interval has the same start and end as the specified Interval.
   * @param {Interval} other
   * @return {boolean}
   */
  equals(other) {
    if (!this.isValid || !other.isValid) {
      return false;
    }

    return this.s.equals(other.s) && this.e.equals(other.e);
  }

  /**
   * Return an Interval representing the intersection of this Interval and the specified Interval.
   * Specifically, the resulting Interval has the maximum start time and the minimum end time of the two Intervals.
   * Returns null if the intersection is empty, meaning, the intervals don't intersect.
   * @param {Interval} other
   * @return {Interval}
   */
  intersection(other) {
    if (!this.isValid) return this;
    const s = this.s > other.s ? this.s : other.s,
      e = this.e < other.e ? this.e : other.e;

    if (s >= e) {
      return null;
    } else {
      return Interval.fromDateTimes(s, e);
    }
  }

  /**
   * Return an Interval representing the union of this Interval and the specified Interval.
   * Specifically, the resulting Interval has the minimum start time and the maximum end time of the two Intervals.
   * @param {Interval} other
   * @return {Interval}
   */
  union(other) {
    if (!this.isValid) return this;
    const s = this.s < other.s ? this.s : other.s,
      e = this.e > other.e ? this.e : other.e;
    return Interval.fromDateTimes(s, e);
  }

  /**
   * Merge an array of Intervals into a equivalent minimal set of Intervals.
   * Combines overlapping and adjacent Intervals.
   * @param {Array} intervals
   * @return {Array}
   */
  static merge(intervals) {
    const [found, final] = intervals
      .sort((a, b) => a.s - b.s)
      .reduce(
        ([sofar, current], item) => {
          if (!current) {
            return [sofar, item];
          } else if (current.overlaps(item) || current.abutsStart(item)) {
            return [sofar, current.union(item)];
          } else {
            return [sofar.concat([current]), item];
          }
        },
        [[], null]
      );
    if (final) {
      found.push(final);
    }
    return found;
  }

  /**
   * Return an array of Intervals representing the spans of time that only appear in one of the specified Intervals.
   * @param {Array} intervals
   * @return {Array}
   */
  static xor(intervals) {
    let start = null,
      currentCount = 0;
    const results = [],
      ends = intervals.map((i) => [
        { time: i.s, type: "s" },
        { time: i.e, type: "e" },
      ]),
      flattened = Array.prototype.concat(...ends),
      arr = flattened.sort((a, b) => a.time - b.time);

    for (const i of arr) {
      currentCount += i.type === "s" ? 1 : -1;

      if (currentCount === 1) {
        start = i.time;
      } else {
        if (start && +start !== +i.time) {
          results.push(Interval.fromDateTimes(start, i.time));
        }

        start = null;
      }
    }

    return Interval.merge(results);
  }

  /**
   * Return an Interval representing the span of time in this Interval that doesn't overlap with any of the specified Intervals.
   * @param {...Interval} intervals
   * @return {Array}
   */
  difference(...intervals) {
    return Interval.xor([this].concat(intervals))
      .map((i) => this.intersection(i))
      .filter((i) => i && !i.isEmpty());
  }

  /**
   * Returns a string representation of this Interval appropriate for debugging.
   * @return {string}
   */
  toString() {
    if (!this.isValid) return INVALID$1;
    return `[${this.s.toISO()} – ${this.e.toISO()})`;
  }

  /**
   * Returns a localized string representing this Interval. Accepts the same options as the
   * Intl.DateTimeFormat constructor and any presets defined by Luxon, such as
   * {@link DateTime.DATE_FULL} or {@link DateTime.TIME_SIMPLE}. The exact behavior of this method
   * is browser-specific, but in general it will return an appropriate representation of the
   * Interval in the assigned locale. Defaults to the system's locale if no locale has been
   * specified.
   * @see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/DateTimeFormat
   * @param {Object} [formatOpts=DateTime.DATE_SHORT] - Either a DateTime preset or
   * Intl.DateTimeFormat constructor options.
   * @param {Object} opts - Options to override the configuration of the start DateTime.
   * @example Interval.fromISO('2022-11-07T09:00Z/2022-11-08T09:00Z').toLocaleString(); //=> 11/7/2022 – 11/8/2022
   * @example Interval.fromISO('2022-11-07T09:00Z/2022-11-08T09:00Z').toLocaleString(DateTime.DATE_FULL); //=> November 7 – 8, 2022
   * @example Interval.fromISO('2022-11-07T09:00Z/2022-11-08T09:00Z').toLocaleString(DateTime.DATE_FULL, { locale: 'fr-FR' }); //=> 7–8 novembre 2022
   * @example Interval.fromISO('2022-11-07T17:00Z/2022-11-07T19:00Z').toLocaleString(DateTime.TIME_SIMPLE); //=> 6:00 – 8:00 PM
   * @example Interval.fromISO('2022-11-07T17:00Z/2022-11-07T19:00Z').toLocaleString({ weekday: 'short', month: 'short', day: '2-digit', hour: '2-digit', minute: '2-digit' }); //=> Mon, Nov 07, 6:00 – 8:00 p
   * @return {string}
   */
  toLocaleString(formatOpts = DATE_SHORT, opts = {}) {
    return this.isValid
      ? Formatter.create(this.s.loc.clone(opts), formatOpts).formatInterval(this)
      : INVALID$1;
  }

  /**
   * Returns an ISO 8601-compliant string representation of this Interval.
   * @see https://en.wikipedia.org/wiki/ISO_8601#Time_intervals
   * @param {Object} opts - The same options as {@link DateTime#toISO}
   * @return {string}
   */
  toISO(opts) {
    if (!this.isValid) return INVALID$1;
    return `${this.s.toISO(opts)}/${this.e.toISO(opts)}`;
  }

  /**
   * Returns an ISO 8601-compliant string representation of date of this Interval.
   * The time components are ignored.
   * @see https://en.wikipedia.org/wiki/ISO_8601#Time_intervals
   * @return {string}
   */
  toISODate() {
    if (!this.isValid) return INVALID$1;
    return `${this.s.toISODate()}/${this.e.toISODate()}`;
  }

  /**
   * Returns an ISO 8601-compliant string representation of time of this Interval.
   * The date components are ignored.
   * @see https://en.wikipedia.org/wiki/ISO_8601#Time_intervals
   * @param {Object} opts - The same options as {@link DateTime#toISO}
   * @return {string}
   */
  toISOTime(opts) {
    if (!this.isValid) return INVALID$1;
    return `${this.s.toISOTime(opts)}/${this.e.toISOTime(opts)}`;
  }

  /**
   * Returns a string representation of this Interval formatted according to the specified format
   * string. **You may not want this.** See {@link Interval#toLocaleString} for a more flexible
   * formatting tool.
   * @param {string} dateFormat - The format string. This string formats the start and end time.
   * See {@link DateTime#toFormat} for details.
   * @param {Object} opts - Options.
   * @param {string} [opts.separator =  ' – '] - A separator to place between the start and end
   * representations.
   * @return {string}
   */
  toFormat(dateFormat, { separator = " – " } = {}) {
    if (!this.isValid) return INVALID$1;
    return `${this.s.toFormat(dateFormat)}${separator}${this.e.toFormat(dateFormat)}`;
  }

  /**
   * Return a Duration representing the time spanned by this interval.
   * @param {string|string[]} [unit=['milliseconds']] - the unit or units (such as 'hours' or 'days') to include in the duration.
   * @param {Object} opts - options that affect the creation of the Duration
   * @param {string} [opts.conversionAccuracy='casual'] - the conversion system to use
   * @example Interval.fromDateTimes(dt1, dt2).toDuration().toObject() //=> { milliseconds: 88489257 }
   * @example Interval.fromDateTimes(dt1, dt2).toDuration('days').toObject() //=> { days: 1.0241812152777778 }
   * @example Interval.fromDateTimes(dt1, dt2).toDuration(['hours', 'minutes']).toObject() //=> { hours: 24, minutes: 34.82095 }
   * @example Interval.fromDateTimes(dt1, dt2).toDuration(['hours', 'minutes', 'seconds']).toObject() //=> { hours: 24, minutes: 34, seconds: 49.257 }
   * @example Interval.fromDateTimes(dt1, dt2).toDuration('seconds').toObject() //=> { seconds: 88489.257 }
   * @return {Duration}
   */
  toDuration(unit, opts) {
    if (!this.isValid) {
      return Duration.invalid(this.invalidReason);
    }
    return this.e.diff(this.s, unit, opts);
  }

  /**
   * Run mapFn on the interval start and end, returning a new Interval from the resulting DateTimes
   * @param {function} mapFn
   * @return {Interval}
   * @example Interval.fromDateTimes(dt1, dt2).mapEndpoints(endpoint => endpoint.toUTC())
   * @example Interval.fromDateTimes(dt1, dt2).mapEndpoints(endpoint => endpoint.plus({ hours: 2 }))
   */
  mapEndpoints(mapFn) {
    return Interval.fromDateTimes(mapFn(this.s), mapFn(this.e));
  }
}

/**
 * The Info class contains static methods for retrieving general time and date related data. For example, it has methods for finding out if a time zone has a DST, for listing the months in any supported locale, and for discovering which of Luxon features are available in the current environment.
 */
class Info {
  /**
   * Return whether the specified zone contains a DST.
   * @param {string|Zone} [zone='local'] - Zone to check. Defaults to the environment's local zone.
   * @return {boolean}
   */
  static hasDST(zone = Settings.defaultZone) {
    const proto = DateTime.now().setZone(zone).set({ month: 12 });

    return !zone.isUniversal && proto.offset !== proto.set({ month: 6 }).offset;
  }

  /**
   * Return whether the specified zone is a valid IANA specifier.
   * @param {string} zone - Zone to check
   * @return {boolean}
   */
  static isValidIANAZone(zone) {
    return IANAZone.isValidZone(zone);
  }

  /**
   * Converts the input into a {@link Zone} instance.
   *
   * * If `input` is already a Zone instance, it is returned unchanged.
   * * If `input` is a string containing a valid time zone name, a Zone instance
   *   with that name is returned.
   * * If `input` is a string that doesn't refer to a known time zone, a Zone
   *   instance with {@link Zone#isValid} == false is returned.
   * * If `input is a number, a Zone instance with the specified fixed offset
   *   in minutes is returned.
   * * If `input` is `null` or `undefined`, the default zone is returned.
   * @param {string|Zone|number} [input] - the value to be converted
   * @return {Zone}
   */
  static normalizeZone(input) {
    return normalizeZone(input, Settings.defaultZone);
  }

  /**
   * Return an array of standalone month names.
   * @see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/DateTimeFormat
   * @param {string} [length='long'] - the length of the month representation, such as "numeric", "2-digit", "narrow", "short", "long"
   * @param {Object} opts - options
   * @param {string} [opts.locale] - the locale code
   * @param {string} [opts.numberingSystem=null] - the numbering system
   * @param {string} [opts.locObj=null] - an existing locale object to use
   * @param {string} [opts.outputCalendar='gregory'] - the calendar
   * @example Info.months()[0] //=> 'January'
   * @example Info.months('short')[0] //=> 'Jan'
   * @example Info.months('numeric')[0] //=> '1'
   * @example Info.months('short', { locale: 'fr-CA' } )[0] //=> 'janv.'
   * @example Info.months('numeric', { locale: 'ar' })[0] //=> '١'
   * @example Info.months('long', { outputCalendar: 'islamic' })[0] //=> 'Rabiʻ I'
   * @return {Array}
   */
  static months(
    length = "long",
    { locale = null, numberingSystem = null, locObj = null, outputCalendar = "gregory" } = {}
  ) {
    return (locObj || Locale.create(locale, numberingSystem, outputCalendar)).months(length);
  }

  /**
   * Return an array of format month names.
   * Format months differ from standalone months in that they're meant to appear next to the day of the month. In some languages, that
   * changes the string.
   * See {@link Info#months}
   * @param {string} [length='long'] - the length of the month representation, such as "numeric", "2-digit", "narrow", "short", "long"
   * @param {Object} opts - options
   * @param {string} [opts.locale] - the locale code
   * @param {string} [opts.numberingSystem=null] - the numbering system
   * @param {string} [opts.locObj=null] - an existing locale object to use
   * @param {string} [opts.outputCalendar='gregory'] - the calendar
   * @return {Array}
   */
  static monthsFormat(
    length = "long",
    { locale = null, numberingSystem = null, locObj = null, outputCalendar = "gregory" } = {}
  ) {
    return (locObj || Locale.create(locale, numberingSystem, outputCalendar)).months(length, true);
  }

  /**
   * Return an array of standalone week names.
   * @see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/DateTimeFormat
   * @param {string} [length='long'] - the length of the weekday representation, such as "narrow", "short", "long".
   * @param {Object} opts - options
   * @param {string} [opts.locale] - the locale code
   * @param {string} [opts.numberingSystem=null] - the numbering system
   * @param {string} [opts.locObj=null] - an existing locale object to use
   * @example Info.weekdays()[0] //=> 'Monday'
   * @example Info.weekdays('short')[0] //=> 'Mon'
   * @example Info.weekdays('short', { locale: 'fr-CA' })[0] //=> 'lun.'
   * @example Info.weekdays('short', { locale: 'ar' })[0] //=> 'الاثنين'
   * @return {Array}
   */
  static weekdays(length = "long", { locale = null, numberingSystem = null, locObj = null } = {}) {
    return (locObj || Locale.create(locale, numberingSystem, null)).weekdays(length);
  }

  /**
   * Return an array of format week names.
   * Format weekdays differ from standalone weekdays in that they're meant to appear next to more date information. In some languages, that
   * changes the string.
   * See {@link Info#weekdays}
   * @param {string} [length='long'] - the length of the month representation, such as "narrow", "short", "long".
   * @param {Object} opts - options
   * @param {string} [opts.locale=null] - the locale code
   * @param {string} [opts.numberingSystem=null] - the numbering system
   * @param {string} [opts.locObj=null] - an existing locale object to use
   * @return {Array}
   */
  static weekdaysFormat(
    length = "long",
    { locale = null, numberingSystem = null, locObj = null } = {}
  ) {
    return (locObj || Locale.create(locale, numberingSystem, null)).weekdays(length, true);
  }

  /**
   * Return an array of meridiems.
   * @param {Object} opts - options
   * @param {string} [opts.locale] - the locale code
   * @example Info.meridiems() //=> [ 'AM', 'PM' ]
   * @example Info.meridiems({ locale: 'my' }) //=> [ 'နံနက်', 'ညနေ' ]
   * @return {Array}
   */
  static meridiems({ locale = null } = {}) {
    return Locale.create(locale).meridiems();
  }

  /**
   * Return an array of eras, such as ['BC', 'AD']. The locale can be specified, but the calendar system is always Gregorian.
   * @param {string} [length='short'] - the length of the era representation, such as "short" or "long".
   * @param {Object} opts - options
   * @param {string} [opts.locale] - the locale code
   * @example Info.eras() //=> [ 'BC', 'AD' ]
   * @example Info.eras('long') //=> [ 'Before Christ', 'Anno Domini' ]
   * @example Info.eras('long', { locale: 'fr' }) //=> [ 'avant Jésus-Christ', 'après Jésus-Christ' ]
   * @return {Array}
   */
  static eras(length = "short", { locale = null } = {}) {
    return Locale.create(locale, null, "gregory").eras(length);
  }

  /**
   * Return the set of available features in this environment.
   * Some features of Luxon are not available in all environments. For example, on older browsers, relative time formatting support is not available. Use this function to figure out if that's the case.
   * Keys:
   * * `relative`: whether this environment supports relative time formatting
   * @example Info.features() //=> { relative: false }
   * @return {Object}
   */
  static features() {
    return { relative: hasRelative() };
  }
}

function dayDiff(earlier, later) {
  const utcDayStart = (dt) => dt.toUTC(0, { keepLocalTime: true }).startOf("day").valueOf(),
    ms = utcDayStart(later) - utcDayStart(earlier);
  return Math.floor(Duration.fromMillis(ms).as("days"));
}

function highOrderDiffs(cursor, later, units) {
  const differs = [
    ["years", (a, b) => b.year - a.year],
    ["quarters", (a, b) => b.quarter - a.quarter + (b.year - a.year) * 4],
    ["months", (a, b) => b.month - a.month + (b.year - a.year) * 12],
    [
      "weeks",
      (a, b) => {
        const days = dayDiff(a, b);
        return (days - (days % 7)) / 7;
      },
    ],
    ["days", dayDiff],
  ];

  const results = {};
  const earlier = cursor;
  let lowestOrder, highWater;

  /* This loop tries to diff using larger units first.
     If we overshoot, we backtrack and try the next smaller unit.
     "cursor" starts out at the earlier timestamp and moves closer and closer to "later"
     as we use smaller and smaller units.
     highWater keeps track of where we would be if we added one more of the smallest unit,
     this is used later to potentially convert any difference smaller than the smallest higher order unit
     into a fraction of that smallest higher order unit
  */
  for (const [unit, differ] of differs) {
    if (units.indexOf(unit) >= 0) {
      lowestOrder = unit;

      results[unit] = differ(cursor, later);
      highWater = earlier.plus(results);

      if (highWater > later) {
        // we overshot the end point, backtrack cursor by 1
        results[unit]--;
        cursor = earlier.plus(results);

        // if we are still overshooting now, we need to backtrack again
        // this happens in certain situations when diffing times in different zones,
        // because this calculation ignores time zones
        if (cursor > later) {
          // keep the "overshot by 1" around as highWater
          highWater = cursor;
          // backtrack cursor by 1
          results[unit]--;
          cursor = earlier.plus(results);
        }
      } else {
        cursor = highWater;
      }
    }
  }

  return [cursor, results, highWater, lowestOrder];
}

function diff (earlier, later, units, opts) {
  let [cursor, results, highWater, lowestOrder] = highOrderDiffs(earlier, later, units);

  const remainingMillis = later - cursor;

  const lowerOrderUnits = units.filter(
    (u) => ["hours", "minutes", "seconds", "milliseconds"].indexOf(u) >= 0
  );

  if (lowerOrderUnits.length === 0) {
    if (highWater < later) {
      highWater = cursor.plus({ [lowestOrder]: 1 });
    }

    if (highWater !== cursor) {
      results[lowestOrder] = (results[lowestOrder] || 0) + remainingMillis / (highWater - cursor);
    }
  }

  const duration = Duration.fromObject(results, opts);

  if (lowerOrderUnits.length > 0) {
    return Duration.fromMillis(remainingMillis, opts)
      .shiftTo(...lowerOrderUnits)
      .plus(duration);
  } else {
    return duration;
  }
}

const numberingSystems = {
  arab: "[\u0660-\u0669]",
  arabext: "[\u06F0-\u06F9]",
  bali: "[\u1B50-\u1B59]",
  beng: "[\u09E6-\u09EF]",
  deva: "[\u0966-\u096F]",
  fullwide: "[\uFF10-\uFF19]",
  gujr: "[\u0AE6-\u0AEF]",
  hanidec: "[〇|一|二|三|四|五|六|七|八|九]",
  khmr: "[\u17E0-\u17E9]",
  knda: "[\u0CE6-\u0CEF]",
  laoo: "[\u0ED0-\u0ED9]",
  limb: "[\u1946-\u194F]",
  mlym: "[\u0D66-\u0D6F]",
  mong: "[\u1810-\u1819]",
  mymr: "[\u1040-\u1049]",
  orya: "[\u0B66-\u0B6F]",
  tamldec: "[\u0BE6-\u0BEF]",
  telu: "[\u0C66-\u0C6F]",
  thai: "[\u0E50-\u0E59]",
  tibt: "[\u0F20-\u0F29]",
  latn: "\\d",
};

const numberingSystemsUTF16 = {
  arab: [1632, 1641],
  arabext: [1776, 1785],
  bali: [6992, 7001],
  beng: [2534, 2543],
  deva: [2406, 2415],
  fullwide: [65296, 65303],
  gujr: [2790, 2799],
  khmr: [6112, 6121],
  knda: [3302, 3311],
  laoo: [3792, 3801],
  limb: [6470, 6479],
  mlym: [3430, 3439],
  mong: [6160, 6169],
  mymr: [4160, 4169],
  orya: [2918, 2927],
  tamldec: [3046, 3055],
  telu: [3174, 3183],
  thai: [3664, 3673],
  tibt: [3872, 3881],
};

const hanidecChars = numberingSystems.hanidec.replace(/[\[|\]]/g, "").split("");

function parseDigits(str) {
  let value = parseInt(str, 10);
  if (isNaN(value)) {
    value = "";
    for (let i = 0; i < str.length; i++) {
      const code = str.charCodeAt(i);

      if (str[i].search(numberingSystems.hanidec) !== -1) {
        value += hanidecChars.indexOf(str[i]);
      } else {
        for (const key in numberingSystemsUTF16) {
          const [min, max] = numberingSystemsUTF16[key];
          if (code >= min && code <= max) {
            value += code - min;
          }
        }
      }
    }
    return parseInt(value, 10);
  } else {
    return value;
  }
}

function digitRegex({ numberingSystem }, append = "") {
  return new RegExp(`${numberingSystems[numberingSystem || "latn"]}${append}`);
}

const MISSING_FTP = "missing Intl.DateTimeFormat.formatToParts support";

function intUnit(regex, post = (i) => i) {
  return { regex, deser: ([s]) => post(parseDigits(s)) };
}

const NBSP = String.fromCharCode(160);
const spaceOrNBSP = `[ ${NBSP}]`;
const spaceOrNBSPRegExp = new RegExp(spaceOrNBSP, "g");

function fixListRegex(s) {
  // make dots optional and also make them literal
  // make space and non breakable space characters interchangeable
  return s.replace(/\./g, "\\.?").replace(spaceOrNBSPRegExp, spaceOrNBSP);
}

function stripInsensitivities(s) {
  return s
    .replace(/\./g, "") // ignore dots that were made optional
    .replace(spaceOrNBSPRegExp, " ") // interchange space and nbsp
    .toLowerCase();
}

function oneOf(strings, startIndex) {
  if (strings === null) {
    return null;
  } else {
    return {
      regex: RegExp(strings.map(fixListRegex).join("|")),
      deser: ([s]) =>
        strings.findIndex((i) => stripInsensitivities(s) === stripInsensitivities(i)) + startIndex,
    };
  }
}

function offset(regex, groups) {
  return { regex, deser: ([, h, m]) => signedOffset(h, m), groups };
}

function simple(regex) {
  return { regex, deser: ([s]) => s };
}

function escapeToken(value) {
  return value.replace(/[\-\[\]{}()*+?.,\\\^$|#\s]/g, "\\$&");
}

/**
 * @param token
 * @param {Locale} loc
 */
function unitForToken(token, loc) {
  const one = digitRegex(loc),
    two = digitRegex(loc, "{2}"),
    three = digitRegex(loc, "{3}"),
    four = digitRegex(loc, "{4}"),
    six = digitRegex(loc, "{6}"),
    oneOrTwo = digitRegex(loc, "{1,2}"),
    oneToThree = digitRegex(loc, "{1,3}"),
    oneToSix = digitRegex(loc, "{1,6}"),
    oneToNine = digitRegex(loc, "{1,9}"),
    twoToFour = digitRegex(loc, "{2,4}"),
    fourToSix = digitRegex(loc, "{4,6}"),
    literal = (t) => ({ regex: RegExp(escapeToken(t.val)), deser: ([s]) => s, literal: true }),
    unitate = (t) => {
      if (token.literal) {
        return literal(t);
      }
      switch (t.val) {
        // era
        case "G":
          return oneOf(loc.eras("short"), 0);
        case "GG":
          return oneOf(loc.eras("long"), 0);
        // years
        case "y":
          return intUnit(oneToSix);
        case "yy":
          return intUnit(twoToFour, untruncateYear);
        case "yyyy":
          return intUnit(four);
        case "yyyyy":
          return intUnit(fourToSix);
        case "yyyyyy":
          return intUnit(six);
        // months
        case "M":
          return intUnit(oneOrTwo);
        case "MM":
          return intUnit(two);
        case "MMM":
          return oneOf(loc.months("short", true), 1);
        case "MMMM":
          return oneOf(loc.months("long", true), 1);
        case "L":
          return intUnit(oneOrTwo);
        case "LL":
          return intUnit(two);
        case "LLL":
          return oneOf(loc.months("short", false), 1);
        case "LLLL":
          return oneOf(loc.months("long", false), 1);
        // dates
        case "d":
          return intUnit(oneOrTwo);
        case "dd":
          return intUnit(two);
        // ordinals
        case "o":
          return intUnit(oneToThree);
        case "ooo":
          return intUnit(three);
        // time
        case "HH":
          return intUnit(two);
        case "H":
          return intUnit(oneOrTwo);
        case "hh":
          return intUnit(two);
        case "h":
          return intUnit(oneOrTwo);
        case "mm":
          return intUnit(two);
        case "m":
          return intUnit(oneOrTwo);
        case "q":
          return intUnit(oneOrTwo);
        case "qq":
          return intUnit(two);
        case "s":
          return intUnit(oneOrTwo);
        case "ss":
          return intUnit(two);
        case "S":
          return intUnit(oneToThree);
        case "SSS":
          return intUnit(three);
        case "u":
          return simple(oneToNine);
        case "uu":
          return simple(oneOrTwo);
        case "uuu":
          return intUnit(one);
        // meridiem
        case "a":
          return oneOf(loc.meridiems(), 0);
        // weekYear (k)
        case "kkkk":
          return intUnit(four);
        case "kk":
          return intUnit(twoToFour, untruncateYear);
        // weekNumber (W)
        case "W":
          return intUnit(oneOrTwo);
        case "WW":
          return intUnit(two);
        // weekdays
        case "E":
        case "c":
          return intUnit(one);
        case "EEE":
          return oneOf(loc.weekdays("short", false), 1);
        case "EEEE":
          return oneOf(loc.weekdays("long", false), 1);
        case "ccc":
          return oneOf(loc.weekdays("short", true), 1);
        case "cccc":
          return oneOf(loc.weekdays("long", true), 1);
        // offset/zone
        case "Z":
        case "ZZ":
          return offset(new RegExp(`([+-]${oneOrTwo.source})(?::(${two.source}))?`), 2);
        case "ZZZ":
          return offset(new RegExp(`([+-]${oneOrTwo.source})(${two.source})?`), 2);
        // we don't support ZZZZ (PST) or ZZZZZ (Pacific Standard Time) in parsing
        // because we don't have any way to figure out what they are
        case "z":
          return simple(/[a-z_+-/]{1,256}?/i);
        // this special-case "token" represents a place where a macro-token expanded into a white-space literal
        // in this case we accept any non-newline white-space
        case " ":
          return simple(/[^\S\n\r]/);
        default:
          return literal(t);
      }
    };

  const unit = unitate(token) || {
    invalidReason: MISSING_FTP,
  };

  unit.token = token;

  return unit;
}

const partTypeStyleToTokenVal = {
  year: {
    "2-digit": "yy",
    numeric: "yyyyy",
  },
  month: {
    numeric: "M",
    "2-digit": "MM",
    short: "MMM",
    long: "MMMM",
  },
  day: {
    numeric: "d",
    "2-digit": "dd",
  },
  weekday: {
    short: "EEE",
    long: "EEEE",
  },
  dayperiod: "a",
  dayPeriod: "a",
  hour12: {
    numeric: "h",
    "2-digit": "hh",
  },
  hour24: {
    numeric: "H",
    "2-digit": "HH",
  },
  minute: {
    numeric: "m",
    "2-digit": "mm",
  },
  second: {
    numeric: "s",
    "2-digit": "ss",
  },
  timeZoneName: {
    long: "ZZZZZ",
    short: "ZZZ",
  },
};

function tokenForPart(part, formatOpts, resolvedOpts) {
  const { type, value } = part;

  if (type === "literal") {
    const isSpace = /^\s+$/.test(value);
    return {
      literal: !isSpace,
      val: isSpace ? " " : value,
    };
  }

  const style = formatOpts[type];

  // The user might have explicitly specified hour12 or hourCycle
  // if so, respect their decision
  // if not, refer back to the resolvedOpts, which are based on the locale
  let actualType = type;
  if (type === "hour") {
    if (formatOpts.hour12 != null) {
      actualType = formatOpts.hour12 ? "hour12" : "hour24";
    } else if (formatOpts.hourCycle != null) {
      if (formatOpts.hourCycle === "h11" || formatOpts.hourCycle === "h12") {
        actualType = "hour12";
      } else {
        actualType = "hour24";
      }
    } else {
      // tokens only differentiate between 24 hours or not,
      // so we do not need to check hourCycle here, which is less supported anyways
      actualType = resolvedOpts.hour12 ? "hour12" : "hour24";
    }
  }
  let val = partTypeStyleToTokenVal[actualType];
  if (typeof val === "object") {
    val = val[style];
  }

  if (val) {
    return {
      literal: false,
      val,
    };
  }

  return undefined;
}

function buildRegex(units) {
  const re = units.map((u) => u.regex).reduce((f, r) => `${f}(${r.source})`, "");
  return [`^${re}$`, units];
}

function match(input, regex, handlers) {
  const matches = input.match(regex);

  if (matches) {
    const all = {};
    let matchIndex = 1;
    for (const i in handlers) {
      if (hasOwnProperty(handlers, i)) {
        const h = handlers[i],
          groups = h.groups ? h.groups + 1 : 1;
        if (!h.literal && h.token) {
          all[h.token.val[0]] = h.deser(matches.slice(matchIndex, matchIndex + groups));
        }
        matchIndex += groups;
      }
    }
    return [matches, all];
  } else {
    return [matches, {}];
  }
}

function dateTimeFromMatches(matches) {
  const toField = (token) => {
    switch (token) {
      case "S":
        return "millisecond";
      case "s":
        return "second";
      case "m":
        return "minute";
      case "h":
      case "H":
        return "hour";
      case "d":
        return "day";
      case "o":
        return "ordinal";
      case "L":
      case "M":
        return "month";
      case "y":
        return "year";
      case "E":
      case "c":
        return "weekday";
      case "W":
        return "weekNumber";
      case "k":
        return "weekYear";
      case "q":
        return "quarter";
      default:
        return null;
    }
  };

  let zone = null;
  let specificOffset;
  if (!isUndefined(matches.z)) {
    zone = IANAZone.create(matches.z);
  }

  if (!isUndefined(matches.Z)) {
    if (!zone) {
      zone = new FixedOffsetZone(matches.Z);
    }
    specificOffset = matches.Z;
  }

  if (!isUndefined(matches.q)) {
    matches.M = (matches.q - 1) * 3 + 1;
  }

  if (!isUndefined(matches.h)) {
    if (matches.h < 12 && matches.a === 1) {
      matches.h += 12;
    } else if (matches.h === 12 && matches.a === 0) {
      matches.h = 0;
    }
  }

  if (matches.G === 0 && matches.y) {
    matches.y = -matches.y;
  }

  if (!isUndefined(matches.u)) {
    matches.S = parseMillis(matches.u);
  }

  const vals = Object.keys(matches).reduce((r, k) => {
    const f = toField(k);
    if (f) {
      r[f] = matches[k];
    }

    return r;
  }, {});

  return [vals, zone, specificOffset];
}

let dummyDateTimeCache = null;

function getDummyDateTime() {
  if (!dummyDateTimeCache) {
    dummyDateTimeCache = DateTime.fromMillis(1555555555555);
  }

  return dummyDateTimeCache;
}

function maybeExpandMacroToken(token, locale) {
  if (token.literal) {
    return token;
  }

  const formatOpts = Formatter.macroTokenToFormatOpts(token.val);
  const tokens = formatOptsToTokens(formatOpts, locale);

  if (tokens == null || tokens.includes(undefined)) {
    return token;
  }

  return tokens;
}

function expandMacroTokens(tokens, locale) {
  return Array.prototype.concat(...tokens.map((t) => maybeExpandMacroToken(t, locale)));
}

/**
 * @private
 */

function explainFromTokens(locale, input, format) {
  const tokens = expandMacroTokens(Formatter.parseFormat(format), locale),
    units = tokens.map((t) => unitForToken(t, locale)),
    disqualifyingUnit = units.find((t) => t.invalidReason);

  if (disqualifyingUnit) {
    return { input, tokens, invalidReason: disqualifyingUnit.invalidReason };
  } else {
    const [regexString, handlers] = buildRegex(units),
      regex = RegExp(regexString, "i"),
      [rawMatches, matches] = match(input, regex, handlers),
      [result, zone, specificOffset] = matches
        ? dateTimeFromMatches(matches)
        : [null, null, undefined];
    if (hasOwnProperty(matches, "a") && hasOwnProperty(matches, "H")) {
      throw new ConflictingSpecificationError(
        "Can't include meridiem when specifying 24-hour format"
      );
    }
    return { input, tokens, regex, rawMatches, matches, result, zone, specificOffset };
  }
}

function parseFromTokens(locale, input, format) {
  const { result, zone, specificOffset, invalidReason } = explainFromTokens(locale, input, format);
  return [result, zone, specificOffset, invalidReason];
}

function formatOptsToTokens(formatOpts, locale) {
  if (!formatOpts) {
    return null;
  }

  const formatter = Formatter.create(locale, formatOpts);
  const df = formatter.dtFormatter(getDummyDateTime());
  const parts = df.formatToParts();
  const resolvedOpts = df.resolvedOptions();
  return parts.map((p) => tokenForPart(p, formatOpts, resolvedOpts));
}

const nonLeapLadder = [0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334],
  leapLadder = [0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335];

function unitOutOfRange(unit, value) {
  return new Invalid(
    "unit out of range",
    `you specified ${value} (of type ${typeof value}) as a ${unit}, which is invalid`
  );
}

function dayOfWeek(year, month, day) {
  const d = new Date(Date.UTC(year, month - 1, day));

  if (year < 100 && year >= 0) {
    d.setUTCFullYear(d.getUTCFullYear() - 1900);
  }

  const js = d.getUTCDay();

  return js === 0 ? 7 : js;
}

function computeOrdinal(year, month, day) {
  return day + (isLeapYear(year) ? leapLadder : nonLeapLadder)[month - 1];
}

function uncomputeOrdinal(year, ordinal) {
  const table = isLeapYear(year) ? leapLadder : nonLeapLadder,
    month0 = table.findIndex((i) => i < ordinal),
    day = ordinal - table[month0];
  return { month: month0 + 1, day };
}

/**
 * @private
 */

function gregorianToWeek(gregObj) {
  const { year, month, day } = gregObj,
    ordinal = computeOrdinal(year, month, day),
    weekday = dayOfWeek(year, month, day);

  let weekNumber = Math.floor((ordinal - weekday + 10) / 7),
    weekYear;

  if (weekNumber < 1) {
    weekYear = year - 1;
    weekNumber = weeksInWeekYear(weekYear);
  } else if (weekNumber > weeksInWeekYear(year)) {
    weekYear = year + 1;
    weekNumber = 1;
  } else {
    weekYear = year;
  }

  return { weekYear, weekNumber, weekday, ...timeObject(gregObj) };
}

function weekToGregorian(weekData) {
  const { weekYear, weekNumber, weekday } = weekData,
    weekdayOfJan4 = dayOfWeek(weekYear, 1, 4),
    yearInDays = daysInYear(weekYear);

  let ordinal = weekNumber * 7 + weekday - weekdayOfJan4 - 3,
    year;

  if (ordinal < 1) {
    year = weekYear - 1;
    ordinal += daysInYear(year);
  } else if (ordinal > yearInDays) {
    year = weekYear + 1;
    ordinal -= daysInYear(weekYear);
  } else {
    year = weekYear;
  }

  const { month, day } = uncomputeOrdinal(year, ordinal);
  return { year, month, day, ...timeObject(weekData) };
}

function gregorianToOrdinal(gregData) {
  const { year, month, day } = gregData;
  const ordinal = computeOrdinal(year, month, day);
  return { year, ordinal, ...timeObject(gregData) };
}

function ordinalToGregorian(ordinalData) {
  const { year, ordinal } = ordinalData;
  const { month, day } = uncomputeOrdinal(year, ordinal);
  return { year, month, day, ...timeObject(ordinalData) };
}

function hasInvalidWeekData(obj) {
  const validYear = isInteger(obj.weekYear),
    validWeek = integerBetween(obj.weekNumber, 1, weeksInWeekYear(obj.weekYear)),
    validWeekday = integerBetween(obj.weekday, 1, 7);

  if (!validYear) {
    return unitOutOfRange("weekYear", obj.weekYear);
  } else if (!validWeek) {
    return unitOutOfRange("week", obj.week);
  } else if (!validWeekday) {
    return unitOutOfRange("weekday", obj.weekday);
  } else return false;
}

function hasInvalidOrdinalData(obj) {
  const validYear = isInteger(obj.year),
    validOrdinal = integerBetween(obj.ordinal, 1, daysInYear(obj.year));

  if (!validYear) {
    return unitOutOfRange("year", obj.year);
  } else if (!validOrdinal) {
    return unitOutOfRange("ordinal", obj.ordinal);
  } else return false;
}

function hasInvalidGregorianData(obj) {
  const validYear = isInteger(obj.year),
    validMonth = integerBetween(obj.month, 1, 12),
    validDay = integerBetween(obj.day, 1, daysInMonth(obj.year, obj.month));

  if (!validYear) {
    return unitOutOfRange("year", obj.year);
  } else if (!validMonth) {
    return unitOutOfRange("month", obj.month);
  } else if (!validDay) {
    return unitOutOfRange("day", obj.day);
  } else return false;
}

function hasInvalidTimeData(obj) {
  const { hour, minute, second, millisecond } = obj;
  const validHour =
      integerBetween(hour, 0, 23) ||
      (hour === 24 && minute === 0 && second === 0 && millisecond === 0),
    validMinute = integerBetween(minute, 0, 59),
    validSecond = integerBetween(second, 0, 59),
    validMillisecond = integerBetween(millisecond, 0, 999);

  if (!validHour) {
    return unitOutOfRange("hour", hour);
  } else if (!validMinute) {
    return unitOutOfRange("minute", minute);
  } else if (!validSecond) {
    return unitOutOfRange("second", second);
  } else if (!validMillisecond) {
    return unitOutOfRange("millisecond", millisecond);
  } else return false;
}

const INVALID = "Invalid DateTime";
const MAX_DATE = 8.64e15;

function unsupportedZone(zone) {
  return new Invalid("unsupported zone", `the zone "${zone.name}" is not supported`);
}

// we cache week data on the DT object and this intermediates the cache
function possiblyCachedWeekData(dt) {
  if (dt.weekData === null) {
    dt.weekData = gregorianToWeek(dt.c);
  }
  return dt.weekData;
}

// clone really means, "make a new object with these modifications". all "setters" really use this
// to create a new object while only changing some of the properties
function clone(inst, alts) {
  const current = {
    ts: inst.ts,
    zone: inst.zone,
    c: inst.c,
    o: inst.o,
    loc: inst.loc,
    invalid: inst.invalid,
  };
  return new DateTime({ ...current, ...alts, old: current });
}

// find the right offset a given local time. The o input is our guess, which determines which
// offset we'll pick in ambiguous cases (e.g. there are two 3 AMs b/c Fallback DST)
function fixOffset(localTS, o, tz) {
  // Our UTC time is just a guess because our offset is just a guess
  let utcGuess = localTS - o * 60 * 1000;

  // Test whether the zone matches the offset for this ts
  const o2 = tz.offset(utcGuess);

  // If so, offset didn't change and we're done
  if (o === o2) {
    return [utcGuess, o];
  }

  // If not, change the ts by the difference in the offset
  utcGuess -= (o2 - o) * 60 * 1000;

  // If that gives us the local time we want, we're done
  const o3 = tz.offset(utcGuess);
  if (o2 === o3) {
    return [utcGuess, o2];
  }

  // If it's different, we're in a hole time. The offset has changed, but the we don't adjust the time
  return [localTS - Math.min(o2, o3) * 60 * 1000, Math.max(o2, o3)];
}

// convert an epoch timestamp into a calendar object with the given offset
function tsToObj(ts, offset) {
  ts += offset * 60 * 1000;

  const d = new Date(ts);

  return {
    year: d.getUTCFullYear(),
    month: d.getUTCMonth() + 1,
    day: d.getUTCDate(),
    hour: d.getUTCHours(),
    minute: d.getUTCMinutes(),
    second: d.getUTCSeconds(),
    millisecond: d.getUTCMilliseconds(),
  };
}

// convert a calendar object to a epoch timestamp
function objToTS(obj, offset, zone) {
  return fixOffset(objToLocalTS(obj), offset, zone);
}

// create a new DT instance by adding a duration, adjusting for DSTs
function adjustTime(inst, dur) {
  const oPre = inst.o,
    year = inst.c.year + Math.trunc(dur.years),
    month = inst.c.month + Math.trunc(dur.months) + Math.trunc(dur.quarters) * 3,
    c = {
      ...inst.c,
      year,
      month,
      day:
        Math.min(inst.c.day, daysInMonth(year, month)) +
        Math.trunc(dur.days) +
        Math.trunc(dur.weeks) * 7,
    },
    millisToAdd = Duration.fromObject({
      years: dur.years - Math.trunc(dur.years),
      quarters: dur.quarters - Math.trunc(dur.quarters),
      months: dur.months - Math.trunc(dur.months),
      weeks: dur.weeks - Math.trunc(dur.weeks),
      days: dur.days - Math.trunc(dur.days),
      hours: dur.hours,
      minutes: dur.minutes,
      seconds: dur.seconds,
      milliseconds: dur.milliseconds,
    }).as("milliseconds"),
    localTS = objToLocalTS(c);

  let [ts, o] = fixOffset(localTS, oPre, inst.zone);

  if (millisToAdd !== 0) {
    ts += millisToAdd;
    // that could have changed the offset by going over a DST, but we want to keep the ts the same
    o = inst.zone.offset(ts);
  }

  return { ts, o };
}

// helper useful in turning the results of parsing into real dates
// by handling the zone options
function parseDataToDateTime(parsed, parsedZone, opts, format, text, specificOffset) {
  const { setZone, zone } = opts;
  if ((parsed && Object.keys(parsed).length !== 0) || parsedZone) {
    const interpretationZone = parsedZone || zone,
      inst = DateTime.fromObject(parsed, {
        ...opts,
        zone: interpretationZone,
        specificOffset,
      });
    return setZone ? inst : inst.setZone(zone);
  } else {
    return DateTime.invalid(
      new Invalid("unparsable", `the input "${text}" can't be parsed as ${format}`)
    );
  }
}

// if you want to output a technical format (e.g. RFC 2822), this helper
// helps handle the details
function toTechFormat(dt, format, allowZ = true) {
  return dt.isValid
    ? Formatter.create(Locale.create("en-US"), {
        allowZ,
        forceSimple: true,
      }).formatDateTimeFromString(dt, format)
    : null;
}

function toISODate(o, extended) {
  const longFormat = o.c.year > 9999 || o.c.year < 0;
  let c = "";
  if (longFormat && o.c.year >= 0) c += "+";
  c += padStart(o.c.year, longFormat ? 6 : 4);

  if (extended) {
    c += "-";
    c += padStart(o.c.month);
    c += "-";
    c += padStart(o.c.day);
  } else {
    c += padStart(o.c.month);
    c += padStart(o.c.day);
  }
  return c;
}

function toISOTime(
  o,
  extended,
  suppressSeconds,
  suppressMilliseconds,
  includeOffset,
  extendedZone
) {
  let c = padStart(o.c.hour);
  if (extended) {
    c += ":";
    c += padStart(o.c.minute);
    if (o.c.millisecond !== 0 || o.c.second !== 0 || !suppressSeconds) {
      c += ":";
    }
  } else {
    c += padStart(o.c.minute);
  }

  if (o.c.millisecond !== 0 || o.c.second !== 0 || !suppressSeconds) {
    c += padStart(o.c.second);

    if (o.c.millisecond !== 0 || !suppressMilliseconds) {
      c += ".";
      c += padStart(o.c.millisecond, 3);
    }
  }

  if (includeOffset) {
    if (o.isOffsetFixed && o.offset === 0 && !extendedZone) {
      c += "Z";
    } else if (o.o < 0) {
      c += "-";
      c += padStart(Math.trunc(-o.o / 60));
      c += ":";
      c += padStart(Math.trunc(-o.o % 60));
    } else {
      c += "+";
      c += padStart(Math.trunc(o.o / 60));
      c += ":";
      c += padStart(Math.trunc(o.o % 60));
    }
  }

  if (extendedZone) {
    c += "[" + o.zone.ianaName + "]";
  }
  return c;
}

// defaults for unspecified units in the supported calendars
const defaultUnitValues = {
    month: 1,
    day: 1,
    hour: 0,
    minute: 0,
    second: 0,
    millisecond: 0,
  },
  defaultWeekUnitValues = {
    weekNumber: 1,
    weekday: 1,
    hour: 0,
    minute: 0,
    second: 0,
    millisecond: 0,
  },
  defaultOrdinalUnitValues = {
    ordinal: 1,
    hour: 0,
    minute: 0,
    second: 0,
    millisecond: 0,
  };

// Units in the supported calendars, sorted by bigness
const orderedUnits = ["year", "month", "day", "hour", "minute", "second", "millisecond"],
  orderedWeekUnits = [
    "weekYear",
    "weekNumber",
    "weekday",
    "hour",
    "minute",
    "second",
    "millisecond",
  ],
  orderedOrdinalUnits = ["year", "ordinal", "hour", "minute", "second", "millisecond"];

// standardize case and plurality in units
function normalizeUnit(unit) {
  const normalized = {
    year: "year",
    years: "year",
    month: "month",
    months: "month",
    day: "day",
    days: "day",
    hour: "hour",
    hours: "hour",
    minute: "minute",
    minutes: "minute",
    quarter: "quarter",
    quarters: "quarter",
    second: "second",
    seconds: "second",
    millisecond: "millisecond",
    milliseconds: "millisecond",
    weekday: "weekday",
    weekdays: "weekday",
    weeknumber: "weekNumber",
    weeksnumber: "weekNumber",
    weeknumbers: "weekNumber",
    weekyear: "weekYear",
    weekyears: "weekYear",
    ordinal: "ordinal",
  }[unit.toLowerCase()];

  if (!normalized) throw new InvalidUnitError(unit);

  return normalized;
}

// this is a dumbed down version of fromObject() that runs about 60% faster
// but doesn't do any validation, makes a bunch of assumptions about what units
// are present, and so on.
function quickDT(obj, opts) {
  const zone = normalizeZone(opts.zone, Settings.defaultZone),
    loc = Locale.fromObject(opts),
    tsNow = Settings.now();

  let ts, o;

  // assume we have the higher-order units
  if (!isUndefined(obj.year)) {
    for (const u of orderedUnits) {
      if (isUndefined(obj[u])) {
        obj[u] = defaultUnitValues[u];
      }
    }

    const invalid = hasInvalidGregorianData(obj) || hasInvalidTimeData(obj);
    if (invalid) {
      return DateTime.invalid(invalid);
    }

    const offsetProvis = zone.offset(tsNow);
    [ts, o] = objToTS(obj, offsetProvis, zone);
  } else {
    ts = tsNow;
  }

  return new DateTime({ ts, zone, loc, o });
}

function diffRelative(start, end, opts) {
  const round = isUndefined(opts.round) ? true : opts.round,
    format = (c, unit) => {
      c = roundTo(c, round || opts.calendary ? 0 : 2, true);
      const formatter = end.loc.clone(opts).relFormatter(opts);
      return formatter.format(c, unit);
    },
    differ = (unit) => {
      if (opts.calendary) {
        if (!end.hasSame(start, unit)) {
          return end.startOf(unit).diff(start.startOf(unit), unit).get(unit);
        } else return 0;
      } else {
        return end.diff(start, unit).get(unit);
      }
    };

  if (opts.unit) {
    return format(differ(opts.unit), opts.unit);
  }

  for (const unit of opts.units) {
    const count = differ(unit);
    if (Math.abs(count) >= 1) {
      return format(count, unit);
    }
  }
  return format(start > end ? -0 : 0, opts.units[opts.units.length - 1]);
}

function lastOpts(argList) {
  let opts = {},
    args;
  if (argList.length > 0 && typeof argList[argList.length - 1] === "object") {
    opts = argList[argList.length - 1];
    args = Array.from(argList).slice(0, argList.length - 1);
  } else {
    args = Array.from(argList);
  }
  return [opts, args];
}

/**
 * A DateTime is an immutable data structure representing a specific date and time and accompanying methods. It contains class and instance methods for creating, parsing, interrogating, transforming, and formatting them.
 *
 * A DateTime comprises of:
 * * A timestamp. Each DateTime instance refers to a specific millisecond of the Unix epoch.
 * * A time zone. Each instance is considered in the context of a specific zone (by default the local system's zone).
 * * Configuration properties that effect how output strings are formatted, such as `locale`, `numberingSystem`, and `outputCalendar`.
 *
 * Here is a brief overview of the most commonly used functionality it provides:
 *
 * * **Creation**: To create a DateTime from its components, use one of its factory class methods: {@link DateTime.local}, {@link DateTime.utc}, and (most flexibly) {@link DateTime.fromObject}. To create one from a standard string format, use {@link DateTime.fromISO}, {@link DateTime.fromHTTP}, and {@link DateTime.fromRFC2822}. To create one from a custom string format, use {@link DateTime.fromFormat}. To create one from a native JS date, use {@link DateTime.fromJSDate}.
 * * **Gregorian calendar and time**: To examine the Gregorian properties of a DateTime individually (i.e as opposed to collectively through {@link DateTime#toObject}), use the {@link DateTime#year}, {@link DateTime#month},
 * {@link DateTime#day}, {@link DateTime#hour}, {@link DateTime#minute}, {@link DateTime#second}, {@link DateTime#millisecond} accessors.
 * * **Week calendar**: For ISO week calendar attributes, see the {@link DateTime#weekYear}, {@link DateTime#weekNumber}, and {@link DateTime#weekday} accessors.
 * * **Configuration** See the {@link DateTime#locale} and {@link DateTime#numberingSystem} accessors.
 * * **Transformation**: To transform the DateTime into other DateTimes, use {@link DateTime#set}, {@link DateTime#reconfigure}, {@link DateTime#setZone}, {@link DateTime#setLocale}, {@link DateTime.plus}, {@link DateTime#minus}, {@link DateTime#endOf}, {@link DateTime#startOf}, {@link DateTime#toUTC}, and {@link DateTime#toLocal}.
 * * **Output**: To convert the DateTime to other representations, use the {@link DateTime#toRelative}, {@link DateTime#toRelativeCalendar}, {@link DateTime#toJSON}, {@link DateTime#toISO}, {@link DateTime#toHTTP}, {@link DateTime#toObject}, {@link DateTime#toRFC2822}, {@link DateTime#toString}, {@link DateTime#toLocaleString}, {@link DateTime#toFormat}, {@link DateTime#toMillis} and {@link DateTime#toJSDate}.
 *
 * There's plenty others documented below. In addition, for more information on subtler topics like internationalization, time zones, alternative calendars, validity, and so on, see the external documentation.
 */
class DateTime {
  /**
   * @access private
   */
  constructor(config) {
    const zone = config.zone || Settings.defaultZone;

    let invalid =
      config.invalid ||
      (Number.isNaN(config.ts) ? new Invalid("invalid input") : null) ||
      (!zone.isValid ? unsupportedZone(zone) : null);
    /**
     * @access private
     */
    this.ts = isUndefined(config.ts) ? Settings.now() : config.ts;

    let c = null,
      o = null;
    if (!invalid) {
      const unchanged = config.old && config.old.ts === this.ts && config.old.zone.equals(zone);

      if (unchanged) {
        [c, o] = [config.old.c, config.old.o];
      } else {
        const ot = zone.offset(this.ts);
        c = tsToObj(this.ts, ot);
        invalid = Number.isNaN(c.year) ? new Invalid("invalid input") : null;
        c = invalid ? null : c;
        o = invalid ? null : ot;
      }
    }

    /**
     * @access private
     */
    this._zone = zone;
    /**
     * @access private
     */
    this.loc = config.loc || Locale.create();
    /**
     * @access private
     */
    this.invalid = invalid;
    /**
     * @access private
     */
    this.weekData = null;
    /**
     * @access private
     */
    this.c = c;
    /**
     * @access private
     */
    this.o = o;
    /**
     * @access private
     */
    this.isLuxonDateTime = true;
  }

  // CONSTRUCT

  /**
   * Create a DateTime for the current instant, in the system's time zone.
   *
   * Use Settings to override these default values if needed.
   * @example DateTime.now().toISO() //~> now in the ISO format
   * @return {DateTime}
   */
  static now() {
    return new DateTime({});
  }

  /**
   * Create a local DateTime
   * @param {number} [year] - The calendar year. If omitted (as in, call `local()` with no arguments), the current time will be used
   * @param {number} [month=1] - The month, 1-indexed
   * @param {number} [day=1] - The day of the month, 1-indexed
   * @param {number} [hour=0] - The hour of the day, in 24-hour time
   * @param {number} [minute=0] - The minute of the hour, meaning a number between 0 and 59
   * @param {number} [second=0] - The second of the minute, meaning a number between 0 and 59
   * @param {number} [millisecond=0] - The millisecond of the second, meaning a number between 0 and 999
   * @example DateTime.local()                                  //~> now
   * @example DateTime.local({ zone: "America/New_York" })      //~> now, in US east coast time
   * @example DateTime.local(2017)                              //~> 2017-01-01T00:00:00
   * @example DateTime.local(2017, 3)                           //~> 2017-03-01T00:00:00
   * @example DateTime.local(2017, 3, 12, { locale: "fr" })     //~> 2017-03-12T00:00:00, with a French locale
   * @example DateTime.local(2017, 3, 12, 5)                    //~> 2017-03-12T05:00:00
   * @example DateTime.local(2017, 3, 12, 5, { zone: "utc" })   //~> 2017-03-12T05:00:00, in UTC
   * @example DateTime.local(2017, 3, 12, 5, 45)                //~> 2017-03-12T05:45:00
   * @example DateTime.local(2017, 3, 12, 5, 45, 10)            //~> 2017-03-12T05:45:10
   * @example DateTime.local(2017, 3, 12, 5, 45, 10, 765)       //~> 2017-03-12T05:45:10.765
   * @return {DateTime}
   */
  static local() {
    const [opts, args] = lastOpts(arguments),
      [year, month, day, hour, minute, second, millisecond] = args;
    return quickDT({ year, month, day, hour, minute, second, millisecond }, opts);
  }

  /**
   * Create a DateTime in UTC
   * @param {number} [year] - The calendar year. If omitted (as in, call `utc()` with no arguments), the current time will be used
   * @param {number} [month=1] - The month, 1-indexed
   * @param {number} [day=1] - The day of the month
   * @param {number} [hour=0] - The hour of the day, in 24-hour time
   * @param {number} [minute=0] - The minute of the hour, meaning a number between 0 and 59
   * @param {number} [second=0] - The second of the minute, meaning a number between 0 and 59
   * @param {number} [millisecond=0] - The millisecond of the second, meaning a number between 0 and 999
   * @param {Object} options - configuration options for the DateTime
   * @param {string} [options.locale] - a locale to set on the resulting DateTime instance
   * @param {string} [options.outputCalendar] - the output calendar to set on the resulting DateTime instance
   * @param {string} [options.numberingSystem] - the numbering system to set on the resulting DateTime instance
   * @example DateTime.utc()                                              //~> now
   * @example DateTime.utc(2017)                                          //~> 2017-01-01T00:00:00Z
   * @example DateTime.utc(2017, 3)                                       //~> 2017-03-01T00:00:00Z
   * @example DateTime.utc(2017, 3, 12)                                   //~> 2017-03-12T00:00:00Z
   * @example DateTime.utc(2017, 3, 12, 5)                                //~> 2017-03-12T05:00:00Z
   * @example DateTime.utc(2017, 3, 12, 5, 45)                            //~> 2017-03-12T05:45:00Z
   * @example DateTime.utc(2017, 3, 12, 5, 45, { locale: "fr" })          //~> 2017-03-12T05:45:00Z with a French locale
   * @example DateTime.utc(2017, 3, 12, 5, 45, 10)                        //~> 2017-03-12T05:45:10Z
   * @example DateTime.utc(2017, 3, 12, 5, 45, 10, 765, { locale: "fr" }) //~> 2017-03-12T05:45:10.765Z with a French locale
   * @return {DateTime}
   */
  static utc() {
    const [opts, args] = lastOpts(arguments),
      [year, month, day, hour, minute, second, millisecond] = args;

    opts.zone = FixedOffsetZone.utcInstance;
    return quickDT({ year, month, day, hour, minute, second, millisecond }, opts);
  }

  /**
   * Create a DateTime from a JavaScript Date object. Uses the default zone.
   * @param {Date} date - a JavaScript Date object
   * @param {Object} options - configuration options for the DateTime
   * @param {string|Zone} [options.zone='local'] - the zone to place the DateTime into
   * @return {DateTime}
   */
  static fromJSDate(date, options = {}) {
    const ts = isDate(date) ? date.valueOf() : NaN;
    if (Number.isNaN(ts)) {
      return DateTime.invalid("invalid input");
    }

    const zoneToUse = normalizeZone(options.zone, Settings.defaultZone);
    if (!zoneToUse.isValid) {
      return DateTime.invalid(unsupportedZone(zoneToUse));
    }

    return new DateTime({
      ts: ts,
      zone: zoneToUse,
      loc: Locale.fromObject(options),
    });
  }

  /**
   * Create a DateTime from a number of milliseconds since the epoch (meaning since 1 January 1970 00:00:00 UTC). Uses the default zone.
   * @param {number} milliseconds - a number of milliseconds since 1970 UTC
   * @param {Object} options - configuration options for the DateTime
   * @param {string|Zone} [options.zone='local'] - the zone to place the DateTime into
   * @param {string} [options.locale] - a locale to set on the resulting DateTime instance
   * @param {string} options.outputCalendar - the output calendar to set on the resulting DateTime instance
   * @param {string} options.numberingSystem - the numbering system to set on the resulting DateTime instance
   * @return {DateTime}
   */
  static fromMillis(milliseconds, options = {}) {
    if (!isNumber(milliseconds)) {
      throw new InvalidArgumentError(
        `fromMillis requires a numerical input, but received a ${typeof milliseconds} with value ${milliseconds}`
      );
    } else if (milliseconds < -MAX_DATE || milliseconds > MAX_DATE) {
      // this isn't perfect because because we can still end up out of range because of additional shifting, but it's a start
      return DateTime.invalid("Timestamp out of range");
    } else {
      return new DateTime({
        ts: milliseconds,
        zone: normalizeZone(options.zone, Settings.defaultZone),
        loc: Locale.fromObject(options),
      });
    }
  }

  /**
   * Create a DateTime from a number of seconds since the epoch (meaning since 1 January 1970 00:00:00 UTC). Uses the default zone.
   * @param {number} seconds - a number of seconds since 1970 UTC
   * @param {Object} options - configuration options for the DateTime
   * @param {string|Zone} [options.zone='local'] - the zone to place the DateTime into
   * @param {string} [options.locale] - a locale to set on the resulting DateTime instance
   * @param {string} options.outputCalendar - the output calendar to set on the resulting DateTime instance
   * @param {string} options.numberingSystem - the numbering system to set on the resulting DateTime instance
   * @return {DateTime}
   */
  static fromSeconds(seconds, options = {}) {
    if (!isNumber(seconds)) {
      throw new InvalidArgumentError("fromSeconds requires a numerical input");
    } else {
      return new DateTime({
        ts: seconds * 1000,
        zone: normalizeZone(options.zone, Settings.defaultZone),
        loc: Locale.fromObject(options),
      });
    }
  }

  /**
   * Create a DateTime from a JavaScript object with keys like 'year' and 'hour' with reasonable defaults.
   * @param {Object} obj - the object to create the DateTime from
   * @param {number} obj.year - a year, such as 1987
   * @param {number} obj.month - a month, 1-12
   * @param {number} obj.day - a day of the month, 1-31, depending on the month
   * @param {number} obj.ordinal - day of the year, 1-365 or 366
   * @param {number} obj.weekYear - an ISO week year
   * @param {number} obj.weekNumber - an ISO week number, between 1 and 52 or 53, depending on the year
   * @param {number} obj.weekday - an ISO weekday, 1-7, where 1 is Monday and 7 is Sunday
   * @param {number} obj.hour - hour of the day, 0-23
   * @param {number} obj.minute - minute of the hour, 0-59
   * @param {number} obj.second - second of the minute, 0-59
   * @param {number} obj.millisecond - millisecond of the second, 0-999
   * @param {Object} opts - options for creating this DateTime
   * @param {string|Zone} [opts.zone='local'] - interpret the numbers in the context of a particular zone. Can take any value taken as the first argument to setZone()
   * @param {string} [opts.locale='system's locale'] - a locale to set on the resulting DateTime instance
   * @param {string} opts.outputCalendar - the output calendar to set on the resulting DateTime instance
   * @param {string} opts.numberingSystem - the numbering system to set on the resulting DateTime instance
   * @example DateTime.fromObject({ year: 1982, month: 5, day: 25}).toISODate() //=> '1982-05-25'
   * @example DateTime.fromObject({ year: 1982 }).toISODate() //=> '1982-01-01'
   * @example DateTime.fromObject({ hour: 10, minute: 26, second: 6 }) //~> today at 10:26:06
   * @example DateTime.fromObject({ hour: 10, minute: 26, second: 6 }, { zone: 'utc' }),
   * @example DateTime.fromObject({ hour: 10, minute: 26, second: 6 }, { zone: 'local' })
   * @example DateTime.fromObject({ hour: 10, minute: 26, second: 6 }, { zone: 'America/New_York' })
   * @example DateTime.fromObject({ weekYear: 2016, weekNumber: 2, weekday: 3 }).toISODate() //=> '2016-01-13'
   * @return {DateTime}
   */
  static fromObject(obj, opts = {}) {
    obj = obj || {};
    const zoneToUse = normalizeZone(opts.zone, Settings.defaultZone);
    if (!zoneToUse.isValid) {
      return DateTime.invalid(unsupportedZone(zoneToUse));
    }

    const tsNow = Settings.now(),
      offsetProvis = !isUndefined(opts.specificOffset)
        ? opts.specificOffset
        : zoneToUse.offset(tsNow),
      normalized = normalizeObject(obj, normalizeUnit),
      containsOrdinal = !isUndefined(normalized.ordinal),
      containsGregorYear = !isUndefined(normalized.year),
      containsGregorMD = !isUndefined(normalized.month) || !isUndefined(normalized.day),
      containsGregor = containsGregorYear || containsGregorMD,
      definiteWeekDef = normalized.weekYear || normalized.weekNumber,
      loc = Locale.fromObject(opts);

    // cases:
    // just a weekday -> this week's instance of that weekday, no worries
    // (gregorian data or ordinal) + (weekYear or weekNumber) -> error
    // (gregorian month or day) + ordinal -> error
    // otherwise just use weeks or ordinals or gregorian, depending on what's specified

    if ((containsGregor || containsOrdinal) && definiteWeekDef) {
      throw new ConflictingSpecificationError(
        "Can't mix weekYear/weekNumber units with year/month/day or ordinals"
      );
    }

    if (containsGregorMD && containsOrdinal) {
      throw new ConflictingSpecificationError("Can't mix ordinal dates with month/day");
    }

    const useWeekData = definiteWeekDef || (normalized.weekday && !containsGregor);

    // configure ourselves to deal with gregorian dates or week stuff
    let units,
      defaultValues,
      objNow = tsToObj(tsNow, offsetProvis);
    if (useWeekData) {
      units = orderedWeekUnits;
      defaultValues = defaultWeekUnitValues;
      objNow = gregorianToWeek(objNow);
    } else if (containsOrdinal) {
      units = orderedOrdinalUnits;
      defaultValues = defaultOrdinalUnitValues;
      objNow = gregorianToOrdinal(objNow);
    } else {
      units = orderedUnits;
      defaultValues = defaultUnitValues;
    }

    // set default values for missing stuff
    let foundFirst = false;
    for (const u of units) {
      const v = normalized[u];
      if (!isUndefined(v)) {
        foundFirst = true;
      } else if (foundFirst) {
        normalized[u] = defaultValues[u];
      } else {
        normalized[u] = objNow[u];
      }
    }

    // make sure the values we have are in range
    const higherOrderInvalid = useWeekData
        ? hasInvalidWeekData(normalized)
        : containsOrdinal
        ? hasInvalidOrdinalData(normalized)
        : hasInvalidGregorianData(normalized),
      invalid = higherOrderInvalid || hasInvalidTimeData(normalized);

    if (invalid) {
      return DateTime.invalid(invalid);
    }

    // compute the actual time
    const gregorian = useWeekData
        ? weekToGregorian(normalized)
        : containsOrdinal
        ? ordinalToGregorian(normalized)
        : normalized,
      [tsFinal, offsetFinal] = objToTS(gregorian, offsetProvis, zoneToUse),
      inst = new DateTime({
        ts: tsFinal,
        zone: zoneToUse,
        o: offsetFinal,
        loc,
      });

    // gregorian data + weekday serves only to validate
    if (normalized.weekday && containsGregor && obj.weekday !== inst.weekday) {
      return DateTime.invalid(
        "mismatched weekday",
        `you can't specify both a weekday of ${normalized.weekday} and a date of ${inst.toISO()}`
      );
    }

    return inst;
  }

  /**
   * Create a DateTime from an ISO 8601 string
   * @param {string} text - the ISO string
   * @param {Object} opts - options to affect the creation
   * @param {string|Zone} [opts.zone='local'] - use this zone if no offset is specified in the input string itself. Will also convert the time to this zone
   * @param {boolean} [opts.setZone=false] - override the zone with a fixed-offset zone specified in the string itself, if it specifies one
   * @param {string} [opts.locale='system's locale'] - a locale to set on the resulting DateTime instance
   * @param {string} [opts.outputCalendar] - the output calendar to set on the resulting DateTime instance
   * @param {string} [opts.numberingSystem] - the numbering system to set on the resulting DateTime instance
   * @example DateTime.fromISO('2016-05-25T09:08:34.123')
   * @example DateTime.fromISO('2016-05-25T09:08:34.123+06:00')
   * @example DateTime.fromISO('2016-05-25T09:08:34.123+06:00', {setZone: true})
   * @example DateTime.fromISO('2016-05-25T09:08:34.123', {zone: 'utc'})
   * @example DateTime.fromISO('2016-W05-4')
   * @return {DateTime}
   */
  static fromISO(text, opts = {}) {
    const [vals, parsedZone] = parseISODate(text);
    return parseDataToDateTime(vals, parsedZone, opts, "ISO 8601", text);
  }

  /**
   * Create a DateTime from an RFC 2822 string
   * @param {string} text - the RFC 2822 string
   * @param {Object} opts - options to affect the creation
   * @param {string|Zone} [opts.zone='local'] - convert the time to this zone. Since the offset is always specified in the string itself, this has no effect on the interpretation of string, merely the zone the resulting DateTime is expressed in.
   * @param {boolean} [opts.setZone=false] - override the zone with a fixed-offset zone specified in the string itself, if it specifies one
   * @param {string} [opts.locale='system's locale'] - a locale to set on the resulting DateTime instance
   * @param {string} opts.outputCalendar - the output calendar to set on the resulting DateTime instance
   * @param {string} opts.numberingSystem - the numbering system to set on the resulting DateTime instance
   * @example DateTime.fromRFC2822('25 Nov 2016 13:23:12 GMT')
   * @example DateTime.fromRFC2822('Fri, 25 Nov 2016 13:23:12 +0600')
   * @example DateTime.fromRFC2822('25 Nov 2016 13:23 Z')
   * @return {DateTime}
   */
  static fromRFC2822(text, opts = {}) {
    const [vals, parsedZone] = parseRFC2822Date(text);
    return parseDataToDateTime(vals, parsedZone, opts, "RFC 2822", text);
  }

  /**
   * Create a DateTime from an HTTP header date
   * @see https://www.w3.org/Protocols/rfc2616/rfc2616-sec3.html#sec3.3.1
   * @param {string} text - the HTTP header date
   * @param {Object} opts - options to affect the creation
   * @param {string|Zone} [opts.zone='local'] - convert the time to this zone. Since HTTP dates are always in UTC, this has no effect on the interpretation of string, merely the zone the resulting DateTime is expressed in.
   * @param {boolean} [opts.setZone=false] - override the zone with the fixed-offset zone specified in the string. For HTTP dates, this is always UTC, so this option is equivalent to setting the `zone` option to 'utc', but this option is included for consistency with similar methods.
   * @param {string} [opts.locale='system's locale'] - a locale to set on the resulting DateTime instance
   * @param {string} opts.outputCalendar - the output calendar to set on the resulting DateTime instance
   * @param {string} opts.numberingSystem - the numbering system to set on the resulting DateTime instance
   * @example DateTime.fromHTTP('Sun, 06 Nov 1994 08:49:37 GMT')
   * @example DateTime.fromHTTP('Sunday, 06-Nov-94 08:49:37 GMT')
   * @example DateTime.fromHTTP('Sun Nov  6 08:49:37 1994')
   * @return {DateTime}
   */
  static fromHTTP(text, opts = {}) {
    const [vals, parsedZone] = parseHTTPDate(text);
    return parseDataToDateTime(vals, parsedZone, opts, "HTTP", opts);
  }

  /**
   * Create a DateTime from an input string and format string.
   * Defaults to en-US if no locale has been specified, regardless of the system's locale. For a table of tokens and their interpretations, see [here](https://moment.github.io/luxon/#/parsing?id=table-of-tokens).
   * @param {string} text - the string to parse
   * @param {string} fmt - the format the string is expected to be in (see the link below for the formats)
   * @param {Object} opts - options to affect the creation
   * @param {string|Zone} [opts.zone='local'] - use this zone if no offset is specified in the input string itself. Will also convert the DateTime to this zone
   * @param {boolean} [opts.setZone=false] - override the zone with a zone specified in the string itself, if it specifies one
   * @param {string} [opts.locale='en-US'] - a locale string to use when parsing. Will also set the DateTime to this locale
   * @param {string} opts.numberingSystem - the numbering system to use when parsing. Will also set the resulting DateTime to this numbering system
   * @param {string} opts.outputCalendar - the output calendar to set on the resulting DateTime instance
   * @return {DateTime}
   */
  static fromFormat(text, fmt, opts = {}) {
    if (isUndefined(text) || isUndefined(fmt)) {
      throw new InvalidArgumentError("fromFormat requires an input string and a format");
    }

    const { locale = null, numberingSystem = null } = opts,
      localeToUse = Locale.fromOpts({
        locale,
        numberingSystem,
        defaultToEN: true,
      }),
      [vals, parsedZone, specificOffset, invalid] = parseFromTokens(localeToUse, text, fmt);
    if (invalid) {
      return DateTime.invalid(invalid);
    } else {
      return parseDataToDateTime(vals, parsedZone, opts, `format ${fmt}`, text, specificOffset);
    }
  }

  /**
   * @deprecated use fromFormat instead
   */
  static fromString(text, fmt, opts = {}) {
    return DateTime.fromFormat(text, fmt, opts);
  }

  /**
   * Create a DateTime from a SQL date, time, or datetime
   * Defaults to en-US if no locale has been specified, regardless of the system's locale
   * @param {string} text - the string to parse
   * @param {Object} opts - options to affect the creation
   * @param {string|Zone} [opts.zone='local'] - use this zone if no offset is specified in the input string itself. Will also convert the DateTime to this zone
   * @param {boolean} [opts.setZone=false] - override the zone with a zone specified in the string itself, if it specifies one
   * @param {string} [opts.locale='en-US'] - a locale string to use when parsing. Will also set the DateTime to this locale
   * @param {string} opts.numberingSystem - the numbering system to use when parsing. Will also set the resulting DateTime to this numbering system
   * @param {string} opts.outputCalendar - the output calendar to set on the resulting DateTime instance
   * @example DateTime.fromSQL('2017-05-15')
   * @example DateTime.fromSQL('2017-05-15 09:12:34')
   * @example DateTime.fromSQL('2017-05-15 09:12:34.342')
   * @example DateTime.fromSQL('2017-05-15 09:12:34.342+06:00')
   * @example DateTime.fromSQL('2017-05-15 09:12:34.342 America/Los_Angeles')
   * @example DateTime.fromSQL('2017-05-15 09:12:34.342 America/Los_Angeles', { setZone: true })
   * @example DateTime.fromSQL('2017-05-15 09:12:34.342', { zone: 'America/Los_Angeles' })
   * @example DateTime.fromSQL('09:12:34.342')
   * @return {DateTime}
   */
  static fromSQL(text, opts = {}) {
    const [vals, parsedZone] = parseSQL(text);
    return parseDataToDateTime(vals, parsedZone, opts, "SQL", text);
  }

  /**
   * Create an invalid DateTime.
   * @param {string} reason - simple string of why this DateTime is invalid. Should not contain parameters or anything else data-dependent.
   * @param {string} [explanation=null] - longer explanation, may include parameters and other useful debugging information
   * @return {DateTime}
   */
  static invalid(reason, explanation = null) {
    if (!reason) {
      throw new InvalidArgumentError("need to specify a reason the DateTime is invalid");
    }

    const invalid = reason instanceof Invalid ? reason : new Invalid(reason, explanation);

    if (Settings.throwOnInvalid) {
      throw new InvalidDateTimeError(invalid);
    } else {
      return new DateTime({ invalid });
    }
  }

  /**
   * Check if an object is an instance of DateTime. Works across context boundaries
   * @param {object} o
   * @return {boolean}
   */
  static isDateTime(o) {
    return (o && o.isLuxonDateTime) || false;
  }

  /**
   * Produce the format string for a set of options
   * @param formatOpts
   * @param localeOpts
   * @returns {string}
   */
  static parseFormatForOpts(formatOpts, localeOpts = {}) {
    const tokenList = formatOptsToTokens(formatOpts, Locale.fromObject(localeOpts));
    return !tokenList ? null : tokenList.map((t) => (t ? t.val : null)).join("");
  }

  /**
   * Produce the the fully expanded format token for the locale
   * Does NOT quote characters, so quoted tokens will not round trip correctly
   * @param fmt
   * @param localeOpts
   * @returns {string}
   */
  static expandFormat(fmt, localeOpts = {}) {
    const expanded = expandMacroTokens(Formatter.parseFormat(fmt), Locale.fromObject(localeOpts));
    return expanded.map((t) => t.val).join("");
  }

  // INFO

  /**
   * Get the value of unit.
   * @param {string} unit - a unit such as 'minute' or 'day'
   * @example DateTime.local(2017, 7, 4).get('month'); //=> 7
   * @example DateTime.local(2017, 7, 4).get('day'); //=> 4
   * @return {number}
   */
  get(unit) {
    return this[unit];
  }

  /**
   * Returns whether the DateTime is valid. Invalid DateTimes occur when:
   * * The DateTime was created from invalid calendar information, such as the 13th month or February 30
   * * The DateTime was created by an operation on another invalid date
   * @type {boolean}
   */
  get isValid() {
    return this.invalid === null;
  }

  /**
   * Returns an error code if this DateTime is invalid, or null if the DateTime is valid
   * @type {string}
   */
  get invalidReason() {
    return this.invalid ? this.invalid.reason : null;
  }

  /**
   * Returns an explanation of why this DateTime became invalid, or null if the DateTime is valid
   * @type {string}
   */
  get invalidExplanation() {
    return this.invalid ? this.invalid.explanation : null;
  }

  /**
   * Get the locale of a DateTime, such 'en-GB'. The locale is used when formatting the DateTime
   *
   * @type {string}
   */
  get locale() {
    return this.isValid ? this.loc.locale : null;
  }

  /**
   * Get the numbering system of a DateTime, such 'beng'. The numbering system is used when formatting the DateTime
   *
   * @type {string}
   */
  get numberingSystem() {
    return this.isValid ? this.loc.numberingSystem : null;
  }

  /**
   * Get the output calendar of a DateTime, such 'islamic'. The output calendar is used when formatting the DateTime
   *
   * @type {string}
   */
  get outputCalendar() {
    return this.isValid ? this.loc.outputCalendar : null;
  }

  /**
   * Get the time zone associated with this DateTime.
   * @type {Zone}
   */
  get zone() {
    return this._zone;
  }

  /**
   * Get the name of the time zone.
   * @type {string}
   */
  get zoneName() {
    return this.isValid ? this.zone.name : null;
  }

  /**
   * Get the year
   * @example DateTime.local(2017, 5, 25).year //=> 2017
   * @type {number}
   */
  get year() {
    return this.isValid ? this.c.year : NaN;
  }

  /**
   * Get the quarter
   * @example DateTime.local(2017, 5, 25).quarter //=> 2
   * @type {number}
   */
  get quarter() {
    return this.isValid ? Math.ceil(this.c.month / 3) : NaN;
  }

  /**
   * Get the month (1-12).
   * @example DateTime.local(2017, 5, 25).month //=> 5
   * @type {number}
   */
  get month() {
    return this.isValid ? this.c.month : NaN;
  }

  /**
   * Get the day of the month (1-30ish).
   * @example DateTime.local(2017, 5, 25).day //=> 25
   * @type {number}
   */
  get day() {
    return this.isValid ? this.c.day : NaN;
  }

  /**
   * Get the hour of the day (0-23).
   * @example DateTime.local(2017, 5, 25, 9).hour //=> 9
   * @type {number}
   */
  get hour() {
    return this.isValid ? this.c.hour : NaN;
  }

  /**
   * Get the minute of the hour (0-59).
   * @example DateTime.local(2017, 5, 25, 9, 30).minute //=> 30
   * @type {number}
   */
  get minute() {
    return this.isValid ? this.c.minute : NaN;
  }

  /**
   * Get the second of the minute (0-59).
   * @example DateTime.local(2017, 5, 25, 9, 30, 52).second //=> 52
   * @type {number}
   */
  get second() {
    return this.isValid ? this.c.second : NaN;
  }

  /**
   * Get the millisecond of the second (0-999).
   * @example DateTime.local(2017, 5, 25, 9, 30, 52, 654).millisecond //=> 654
   * @type {number}
   */
  get millisecond() {
    return this.isValid ? this.c.millisecond : NaN;
  }

  /**
   * Get the week year
   * @see https://en.wikipedia.org/wiki/ISO_week_date
   * @example DateTime.local(2014, 12, 31).weekYear //=> 2015
   * @type {number}
   */
  get weekYear() {
    return this.isValid ? possiblyCachedWeekData(this).weekYear : NaN;
  }

  /**
   * Get the week number of the week year (1-52ish).
   * @see https://en.wikipedia.org/wiki/ISO_week_date
   * @example DateTime.local(2017, 5, 25).weekNumber //=> 21
   * @type {number}
   */
  get weekNumber() {
    return this.isValid ? possiblyCachedWeekData(this).weekNumber : NaN;
  }

  /**
   * Get the day of the week.
   * 1 is Monday and 7 is Sunday
   * @see https://en.wikipedia.org/wiki/ISO_week_date
   * @example DateTime.local(2014, 11, 31).weekday //=> 4
   * @type {number}
   */
  get weekday() {
    return this.isValid ? possiblyCachedWeekData(this).weekday : NaN;
  }

  /**
   * Get the ordinal (meaning the day of the year)
   * @example DateTime.local(2017, 5, 25).ordinal //=> 145
   * @type {number|DateTime}
   */
  get ordinal() {
    return this.isValid ? gregorianToOrdinal(this.c).ordinal : NaN;
  }

  /**
   * Get the human readable short month name, such as 'Oct'.
   * Defaults to the system's locale if no locale has been specified
   * @example DateTime.local(2017, 10, 30).monthShort //=> Oct
   * @type {string}
   */
  get monthShort() {
    return this.isValid ? Info.months("short", { locObj: this.loc })[this.month - 1] : null;
  }

  /**
   * Get the human readable long month name, such as 'October'.
   * Defaults to the system's locale if no locale has been specified
   * @example DateTime.local(2017, 10, 30).monthLong //=> October
   * @type {string}
   */
  get monthLong() {
    return this.isValid ? Info.months("long", { locObj: this.loc })[this.month - 1] : null;
  }

  /**
   * Get the human readable short weekday, such as 'Mon'.
   * Defaults to the system's locale if no locale has been specified
   * @example DateTime.local(2017, 10, 30).weekdayShort //=> Mon
   * @type {string}
   */
  get weekdayShort() {
    return this.isValid ? Info.weekdays("short", { locObj: this.loc })[this.weekday - 1] : null;
  }

  /**
   * Get the human readable long weekday, such as 'Monday'.
   * Defaults to the system's locale if no locale has been specified
   * @example DateTime.local(2017, 10, 30).weekdayLong //=> Monday
   * @type {string}
   */
  get weekdayLong() {
    return this.isValid ? Info.weekdays("long", { locObj: this.loc })[this.weekday - 1] : null;
  }

  /**
   * Get the UTC offset of this DateTime in minutes
   * @example DateTime.now().offset //=> -240
   * @example DateTime.utc().offset //=> 0
   * @type {number}
   */
  get offset() {
    return this.isValid ? +this.o : NaN;
  }

  /**
   * Get the short human name for the zone's current offset, for example "EST" or "EDT".
   * Defaults to the system's locale if no locale has been specified
   * @type {string}
   */
  get offsetNameShort() {
    if (this.isValid) {
      return this.zone.offsetName(this.ts, {
        format: "short",
        locale: this.locale,
      });
    } else {
      return null;
    }
  }

  /**
   * Get the long human name for the zone's current offset, for example "Eastern Standard Time" or "Eastern Daylight Time".
   * Defaults to the system's locale if no locale has been specified
   * @type {string}
   */
  get offsetNameLong() {
    if (this.isValid) {
      return this.zone.offsetName(this.ts, {
        format: "long",
        locale: this.locale,
      });
    } else {
      return null;
    }
  }

  /**
   * Get whether this zone's offset ever changes, as in a DST.
   * @type {boolean}
   */
  get isOffsetFixed() {
    return this.isValid ? this.zone.isUniversal : null;
  }

  /**
   * Get whether the DateTime is in a DST.
   * @type {boolean}
   */
  get isInDST() {
    if (this.isOffsetFixed) {
      return false;
    } else {
      return (
        this.offset > this.set({ month: 1, day: 1 }).offset ||
        this.offset > this.set({ month: 5 }).offset
      );
    }
  }

  /**
   * Get those DateTimes which have the same local time as this DateTime, but a different offset from UTC
   * in this DateTime's zone. During DST changes local time can be ambiguous, for example
   * `2023-10-29T02:30:00` in `Europe/Berlin` can have offset `+01:00` or `+02:00`.
   * This method will return both possible DateTimes if this DateTime's local time is ambiguous.
   * @returns {DateTime[]}
   */
  getPossibleOffsets() {
    if (!this.isValid || this.isOffsetFixed) {
      return [this];
    }
    const dayMs = 86400000;
    const minuteMs = 60000;
    const localTS = objToLocalTS(this.c);
    const oEarlier = this.zone.offset(localTS - dayMs);
    const oLater = this.zone.offset(localTS + dayMs);

    const o1 = this.zone.offset(localTS - oEarlier * minuteMs);
    const o2 = this.zone.offset(localTS - oLater * minuteMs);
    if (o1 === o2) {
      return [this];
    }
    const ts1 = localTS - o1 * minuteMs;
    const ts2 = localTS - o2 * minuteMs;
    const c1 = tsToObj(ts1, o1);
    const c2 = tsToObj(ts2, o2);
    if (
      c1.hour === c2.hour &&
      c1.minute === c2.minute &&
      c1.second === c2.second &&
      c1.millisecond === c2.millisecond
    ) {
      return [clone(this, { ts: ts1 }), clone(this, { ts: ts2 })];
    }
    return [this];
  }

  /**
   * Returns true if this DateTime is in a leap year, false otherwise
   * @example DateTime.local(2016).isInLeapYear //=> true
   * @example DateTime.local(2013).isInLeapYear //=> false
   * @type {boolean}
   */
  get isInLeapYear() {
    return isLeapYear(this.year);
  }

  /**
   * Returns the number of days in this DateTime's month
   * @example DateTime.local(2016, 2).daysInMonth //=> 29
   * @example DateTime.local(2016, 3).daysInMonth //=> 31
   * @type {number}
   */
  get daysInMonth() {
    return daysInMonth(this.year, this.month);
  }

  /**
   * Returns the number of days in this DateTime's year
   * @example DateTime.local(2016).daysInYear //=> 366
   * @example DateTime.local(2013).daysInYear //=> 365
   * @type {number}
   */
  get daysInYear() {
    return this.isValid ? daysInYear(this.year) : NaN;
  }

  /**
   * Returns the number of weeks in this DateTime's year
   * @see https://en.wikipedia.org/wiki/ISO_week_date
   * @example DateTime.local(2004).weeksInWeekYear //=> 53
   * @example DateTime.local(2013).weeksInWeekYear //=> 52
   * @type {number}
   */
  get weeksInWeekYear() {
    return this.isValid ? weeksInWeekYear(this.weekYear) : NaN;
  }

  /**
   * Returns the resolved Intl options for this DateTime.
   * This is useful in understanding the behavior of formatting methods
   * @param {Object} opts - the same options as toLocaleString
   * @return {Object}
   */
  resolvedLocaleOptions(opts = {}) {
    const { locale, numberingSystem, calendar } = Formatter.create(
      this.loc.clone(opts),
      opts
    ).resolvedOptions(this);
    return { locale, numberingSystem, outputCalendar: calendar };
  }

  // TRANSFORM

  /**
   * "Set" the DateTime's zone to UTC. Returns a newly-constructed DateTime.
   *
   * Equivalent to {@link DateTime#setZone}('utc')
   * @param {number} [offset=0] - optionally, an offset from UTC in minutes
   * @param {Object} [opts={}] - options to pass to `setZone()`
   * @return {DateTime}
   */
  toUTC(offset = 0, opts = {}) {
    return this.setZone(FixedOffsetZone.instance(offset), opts);
  }

  /**
   * "Set" the DateTime's zone to the host's local zone. Returns a newly-constructed DateTime.
   *
   * Equivalent to `setZone('local')`
   * @return {DateTime}
   */
  toLocal() {
    return this.setZone(Settings.defaultZone);
  }

  /**
   * "Set" the DateTime's zone to specified zone. Returns a newly-constructed DateTime.
   *
   * By default, the setter keeps the underlying time the same (as in, the same timestamp), but the new instance will report different local times and consider DSTs when making computations, as with {@link DateTime#plus}. You may wish to use {@link DateTime#toLocal} and {@link DateTime#toUTC} which provide simple convenience wrappers for commonly used zones.
   * @param {string|Zone} [zone='local'] - a zone identifier. As a string, that can be any IANA zone supported by the host environment, or a fixed-offset name of the form 'UTC+3', or the strings 'local' or 'utc'. You may also supply an instance of a {@link DateTime#Zone} class.
   * @param {Object} opts - options
   * @param {boolean} [opts.keepLocalTime=false] - If true, adjust the underlying time so that the local time stays the same, but in the target zone. You should rarely need this.
   * @return {DateTime}
   */
  setZone(zone, { keepLocalTime = false, keepCalendarTime = false } = {}) {
    zone = normalizeZone(zone, Settings.defaultZone);
    if (zone.equals(this.zone)) {
      return this;
    } else if (!zone.isValid) {
      return DateTime.invalid(unsupportedZone(zone));
    } else {
      let newTS = this.ts;
      if (keepLocalTime || keepCalendarTime) {
        const offsetGuess = zone.offset(this.ts);
        const asObj = this.toObject();
        [newTS] = objToTS(asObj, offsetGuess, zone);
      }
      return clone(this, { ts: newTS, zone });
    }
  }

  /**
   * "Set" the locale, numberingSystem, or outputCalendar. Returns a newly-constructed DateTime.
   * @param {Object} properties - the properties to set
   * @example DateTime.local(2017, 5, 25).reconfigure({ locale: 'en-GB' })
   * @return {DateTime}
   */
  reconfigure({ locale, numberingSystem, outputCalendar } = {}) {
    const loc = this.loc.clone({ locale, numberingSystem, outputCalendar });
    return clone(this, { loc });
  }

  /**
   * "Set" the locale. Returns a newly-constructed DateTime.
   * Just a convenient alias for reconfigure({ locale })
   * @example DateTime.local(2017, 5, 25).setLocale('en-GB')
   * @return {DateTime}
   */
  setLocale(locale) {
    return this.reconfigure({ locale });
  }

  /**
   * "Set" the values of specified units. Returns a newly-constructed DateTime.
   * You can only set units with this method; for "setting" metadata, see {@link DateTime#reconfigure} and {@link DateTime#setZone}.
   * @param {Object} values - a mapping of units to numbers
   * @example dt.set({ year: 2017 })
   * @example dt.set({ hour: 8, minute: 30 })
   * @example dt.set({ weekday: 5 })
   * @example dt.set({ year: 2005, ordinal: 234 })
   * @return {DateTime}
   */
  set(values) {
    if (!this.isValid) return this;

    const normalized = normalizeObject(values, normalizeUnit),
      settingWeekStuff =
        !isUndefined(normalized.weekYear) ||
        !isUndefined(normalized.weekNumber) ||
        !isUndefined(normalized.weekday),
      containsOrdinal = !isUndefined(normalized.ordinal),
      containsGregorYear = !isUndefined(normalized.year),
      containsGregorMD = !isUndefined(normalized.month) || !isUndefined(normalized.day),
      containsGregor = containsGregorYear || containsGregorMD,
      definiteWeekDef = normalized.weekYear || normalized.weekNumber;

    if ((containsGregor || containsOrdinal) && definiteWeekDef) {
      throw new ConflictingSpecificationError(
        "Can't mix weekYear/weekNumber units with year/month/day or ordinals"
      );
    }

    if (containsGregorMD && containsOrdinal) {
      throw new ConflictingSpecificationError("Can't mix ordinal dates with month/day");
    }

    let mixed;
    if (settingWeekStuff) {
      mixed = weekToGregorian({ ...gregorianToWeek(this.c), ...normalized });
    } else if (!isUndefined(normalized.ordinal)) {
      mixed = ordinalToGregorian({ ...gregorianToOrdinal(this.c), ...normalized });
    } else {
      mixed = { ...this.toObject(), ...normalized };

      // if we didn't set the day but we ended up on an overflow date,
      // use the last day of the right month
      if (isUndefined(normalized.day)) {
        mixed.day = Math.min(daysInMonth(mixed.year, mixed.month), mixed.day);
      }
    }

    const [ts, o] = objToTS(mixed, this.o, this.zone);
    return clone(this, { ts, o });
  }

  /**
   * Add a period of time to this DateTime and return the resulting DateTime
   *
   * Adding hours, minutes, seconds, or milliseconds increases the timestamp by the right number of milliseconds. Adding days, months, or years shifts the calendar, accounting for DSTs and leap years along the way. Thus, `dt.plus({ hours: 24 })` may result in a different time than `dt.plus({ days: 1 })` if there's a DST shift in between.
   * @param {Duration|Object|number} duration - The amount to add. Either a Luxon Duration, a number of milliseconds, the object argument to Duration.fromObject()
   * @example DateTime.now().plus(123) //~> in 123 milliseconds
   * @example DateTime.now().plus({ minutes: 15 }) //~> in 15 minutes
   * @example DateTime.now().plus({ days: 1 }) //~> this time tomorrow
   * @example DateTime.now().plus({ days: -1 }) //~> this time yesterday
   * @example DateTime.now().plus({ hours: 3, minutes: 13 }) //~> in 3 hr, 13 min
   * @example DateTime.now().plus(Duration.fromObject({ hours: 3, minutes: 13 })) //~> in 3 hr, 13 min
   * @return {DateTime}
   */
  plus(duration) {
    if (!this.isValid) return this;
    const dur = Duration.fromDurationLike(duration);
    return clone(this, adjustTime(this, dur));
  }

  /**
   * Subtract a period of time to this DateTime and return the resulting DateTime
   * See {@link DateTime#plus}
   * @param {Duration|Object|number} duration - The amount to subtract. Either a Luxon Duration, a number of milliseconds, the object argument to Duration.fromObject()
   @return {DateTime}
   */
  minus(duration) {
    if (!this.isValid) return this;
    const dur = Duration.fromDurationLike(duration).negate();
    return clone(this, adjustTime(this, dur));
  }

  /**
   * "Set" this DateTime to the beginning of a unit of time.
   * @param {string} unit - The unit to go to the beginning of. Can be 'year', 'quarter', 'month', 'week', 'day', 'hour', 'minute', 'second', or 'millisecond'.
   * @example DateTime.local(2014, 3, 3).startOf('month').toISODate(); //=> '2014-03-01'
   * @example DateTime.local(2014, 3, 3).startOf('year').toISODate(); //=> '2014-01-01'
   * @example DateTime.local(2014, 3, 3).startOf('week').toISODate(); //=> '2014-03-03', weeks always start on Mondays
   * @example DateTime.local(2014, 3, 3, 5, 30).startOf('day').toISOTime(); //=> '00:00.000-05:00'
   * @example DateTime.local(2014, 3, 3, 5, 30).startOf('hour').toISOTime(); //=> '05:00:00.000-05:00'
   * @return {DateTime}
   */
  startOf(unit) {
    if (!this.isValid) return this;
    const o = {},
      normalizedUnit = Duration.normalizeUnit(unit);
    switch (normalizedUnit) {
      case "years":
        o.month = 1;
      // falls through
      case "quarters":
      case "months":
        o.day = 1;
      // falls through
      case "weeks":
      case "days":
        o.hour = 0;
      // falls through
      case "hours":
        o.minute = 0;
      // falls through
      case "minutes":
        o.second = 0;
      // falls through
      case "seconds":
        o.millisecond = 0;
        break;
      // no default, invalid units throw in normalizeUnit()
    }

    if (normalizedUnit === "weeks") {
      o.weekday = 1;
    }

    if (normalizedUnit === "quarters") {
      const q = Math.ceil(this.month / 3);
      o.month = (q - 1) * 3 + 1;
    }

    return this.set(o);
  }

  /**
   * "Set" this DateTime to the end (meaning the last millisecond) of a unit of time
   * @param {string} unit - The unit to go to the end of. Can be 'year', 'quarter', 'month', 'week', 'day', 'hour', 'minute', 'second', or 'millisecond'.
   * @example DateTime.local(2014, 3, 3).endOf('month').toISO(); //=> '2014-03-31T23:59:59.999-05:00'
   * @example DateTime.local(2014, 3, 3).endOf('year').toISO(); //=> '2014-12-31T23:59:59.999-05:00'
   * @example DateTime.local(2014, 3, 3).endOf('week').toISO(); // => '2014-03-09T23:59:59.999-05:00', weeks start on Mondays
   * @example DateTime.local(2014, 3, 3, 5, 30).endOf('day').toISO(); //=> '2014-03-03T23:59:59.999-05:00'
   * @example DateTime.local(2014, 3, 3, 5, 30).endOf('hour').toISO(); //=> '2014-03-03T05:59:59.999-05:00'
   * @return {DateTime}
   */
  endOf(unit) {
    return this.isValid
      ? this.plus({ [unit]: 1 })
          .startOf(unit)
          .minus(1)
      : this;
  }

  // OUTPUT

  /**
   * Returns a string representation of this DateTime formatted according to the specified format string.
   * **You may not want this.** See {@link DateTime#toLocaleString} for a more flexible formatting tool. For a table of tokens and their interpretations, see [here](https://moment.github.io/luxon/#/formatting?id=table-of-tokens).
   * Defaults to en-US if no locale has been specified, regardless of the system's locale.
   * @param {string} fmt - the format string
   * @param {Object} opts - opts to override the configuration options on this DateTime
   * @example DateTime.now().toFormat('yyyy LLL dd') //=> '2017 Apr 22'
   * @example DateTime.now().setLocale('fr').toFormat('yyyy LLL dd') //=> '2017 avr. 22'
   * @example DateTime.now().toFormat('yyyy LLL dd', { locale: "fr" }) //=> '2017 avr. 22'
   * @example DateTime.now().toFormat("HH 'hours and' mm 'minutes'") //=> '20 hours and 55 minutes'
   * @return {string}
   */
  toFormat(fmt, opts = {}) {
    return this.isValid
      ? Formatter.create(this.loc.redefaultToEN(opts)).formatDateTimeFromString(this, fmt)
      : INVALID;
  }

  /**
   * Returns a localized string representing this date. Accepts the same options as the Intl.DateTimeFormat constructor and any presets defined by Luxon, such as `DateTime.DATE_FULL` or `DateTime.TIME_SIMPLE`.
   * The exact behavior of this method is browser-specific, but in general it will return an appropriate representation
   * of the DateTime in the assigned locale.
   * Defaults to the system's locale if no locale has been specified
   * @see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/DateTimeFormat
   * @param formatOpts {Object} - Intl.DateTimeFormat constructor options and configuration options
   * @param {Object} opts - opts to override the configuration options on this DateTime
   * @example DateTime.now().toLocaleString(); //=> 4/20/2017
   * @example DateTime.now().setLocale('en-gb').toLocaleString(); //=> '20/04/2017'
   * @example DateTime.now().toLocaleString(DateTime.DATE_FULL); //=> 'April 20, 2017'
   * @example DateTime.now().toLocaleString(DateTime.DATE_FULL, { locale: 'fr' }); //=> '28 août 2022'
   * @example DateTime.now().toLocaleString(DateTime.TIME_SIMPLE); //=> '11:32 AM'
   * @example DateTime.now().toLocaleString(DateTime.DATETIME_SHORT); //=> '4/20/2017, 11:32 AM'
   * @example DateTime.now().toLocaleString({ weekday: 'long', month: 'long', day: '2-digit' }); //=> 'Thursday, April 20'
   * @example DateTime.now().toLocaleString({ weekday: 'short', month: 'short', day: '2-digit', hour: '2-digit', minute: '2-digit' }); //=> 'Thu, Apr 20, 11:27 AM'
   * @example DateTime.now().toLocaleString({ hour: '2-digit', minute: '2-digit', hourCycle: 'h23' }); //=> '11:32'
   * @return {string}
   */
  toLocaleString(formatOpts = DATE_SHORT, opts = {}) {
    return this.isValid
      ? Formatter.create(this.loc.clone(opts), formatOpts).formatDateTime(this)
      : INVALID;
  }

  /**
   * Returns an array of format "parts", meaning individual tokens along with metadata. This is allows callers to post-process individual sections of the formatted output.
   * Defaults to the system's locale if no locale has been specified
   * @see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/DateTimeFormat/formatToParts
   * @param opts {Object} - Intl.DateTimeFormat constructor options, same as `toLocaleString`.
   * @example DateTime.now().toLocaleParts(); //=> [
   *                                   //=>   { type: 'day', value: '25' },
   *                                   //=>   { type: 'literal', value: '/' },
   *                                   //=>   { type: 'month', value: '05' },
   *                                   //=>   { type: 'literal', value: '/' },
   *                                   //=>   { type: 'year', value: '1982' }
   *                                   //=> ]
   */
  toLocaleParts(opts = {}) {
    return this.isValid
      ? Formatter.create(this.loc.clone(opts), opts).formatDateTimeParts(this)
      : [];
  }

  /**
   * Returns an ISO 8601-compliant string representation of this DateTime
   * @param {Object} opts - options
   * @param {boolean} [opts.suppressMilliseconds=false] - exclude milliseconds from the format if they're 0
   * @param {boolean} [opts.suppressSeconds=false] - exclude seconds from the format if they're 0
   * @param {boolean} [opts.includeOffset=true] - include the offset, such as 'Z' or '-04:00'
   * @param {boolean} [opts.extendedZone=false] - add the time zone format extension
   * @param {string} [opts.format='extended'] - choose between the basic and extended format
   * @example DateTime.utc(1983, 5, 25).toISO() //=> '1982-05-25T00:00:00.000Z'
   * @example DateTime.now().toISO() //=> '2017-04-22T20:47:05.335-04:00'
   * @example DateTime.now().toISO({ includeOffset: false }) //=> '2017-04-22T20:47:05.335'
   * @example DateTime.now().toISO({ format: 'basic' }) //=> '20170422T204705.335-0400'
   * @return {string}
   */
  toISO({
    format = "extended",
    suppressSeconds = false,
    suppressMilliseconds = false,
    includeOffset = true,
    extendedZone = false,
  } = {}) {
    if (!this.isValid) {
      return null;
    }

    const ext = format === "extended";

    let c = toISODate(this, ext);
    c += "T";
    c += toISOTime(this, ext, suppressSeconds, suppressMilliseconds, includeOffset, extendedZone);
    return c;
  }

  /**
   * Returns an ISO 8601-compliant string representation of this DateTime's date component
   * @param {Object} opts - options
   * @param {string} [opts.format='extended'] - choose between the basic and extended format
   * @example DateTime.utc(1982, 5, 25).toISODate() //=> '1982-05-25'
   * @example DateTime.utc(1982, 5, 25).toISODate({ format: 'basic' }) //=> '19820525'
   * @return {string}
   */
  toISODate({ format = "extended" } = {}) {
    if (!this.isValid) {
      return null;
    }

    return toISODate(this, format === "extended");
  }

  /**
   * Returns an ISO 8601-compliant string representation of this DateTime's week date
   * @example DateTime.utc(1982, 5, 25).toISOWeekDate() //=> '1982-W21-2'
   * @return {string}
   */
  toISOWeekDate() {
    return toTechFormat(this, "kkkk-'W'WW-c");
  }

  /**
   * Returns an ISO 8601-compliant string representation of this DateTime's time component
   * @param {Object} opts - options
   * @param {boolean} [opts.suppressMilliseconds=false] - exclude milliseconds from the format if they're 0
   * @param {boolean} [opts.suppressSeconds=false] - exclude seconds from the format if they're 0
   * @param {boolean} [opts.includeOffset=true] - include the offset, such as 'Z' or '-04:00'
   * @param {boolean} [opts.extendedZone=true] - add the time zone format extension
   * @param {boolean} [opts.includePrefix=false] - include the `T` prefix
   * @param {string} [opts.format='extended'] - choose between the basic and extended format
   * @example DateTime.utc().set({ hour: 7, minute: 34 }).toISOTime() //=> '07:34:19.361Z'
   * @example DateTime.utc().set({ hour: 7, minute: 34, seconds: 0, milliseconds: 0 }).toISOTime({ suppressSeconds: true }) //=> '07:34Z'
   * @example DateTime.utc().set({ hour: 7, minute: 34 }).toISOTime({ format: 'basic' }) //=> '073419.361Z'
   * @example DateTime.utc().set({ hour: 7, minute: 34 }).toISOTime({ includePrefix: true }) //=> 'T07:34:19.361Z'
   * @return {string}
   */
  toISOTime({
    suppressMilliseconds = false,
    suppressSeconds = false,
    includeOffset = true,
    includePrefix = false,
    extendedZone = false,
    format = "extended",
  } = {}) {
    if (!this.isValid) {
      return null;
    }

    let c = includePrefix ? "T" : "";
    return (
      c +
      toISOTime(
        this,
        format === "extended",
        suppressSeconds,
        suppressMilliseconds,
        includeOffset,
        extendedZone
      )
    );
  }

  /**
   * Returns an RFC 2822-compatible string representation of this DateTime
   * @example DateTime.utc(2014, 7, 13).toRFC2822() //=> 'Sun, 13 Jul 2014 00:00:00 +0000'
   * @example DateTime.local(2014, 7, 13).toRFC2822() //=> 'Sun, 13 Jul 2014 00:00:00 -0400'
   * @return {string}
   */
  toRFC2822() {
    return toTechFormat(this, "EEE, dd LLL yyyy HH:mm:ss ZZZ", false);
  }

  /**
   * Returns a string representation of this DateTime appropriate for use in HTTP headers. The output is always expressed in GMT.
   * Specifically, the string conforms to RFC 1123.
   * @see https://www.w3.org/Protocols/rfc2616/rfc2616-sec3.html#sec3.3.1
   * @example DateTime.utc(2014, 7, 13).toHTTP() //=> 'Sun, 13 Jul 2014 00:00:00 GMT'
   * @example DateTime.utc(2014, 7, 13, 19).toHTTP() //=> 'Sun, 13 Jul 2014 19:00:00 GMT'
   * @return {string}
   */
  toHTTP() {
    return toTechFormat(this.toUTC(), "EEE, dd LLL yyyy HH:mm:ss 'GMT'");
  }

  /**
   * Returns a string representation of this DateTime appropriate for use in SQL Date
   * @example DateTime.utc(2014, 7, 13).toSQLDate() //=> '2014-07-13'
   * @return {string}
   */
  toSQLDate() {
    if (!this.isValid) {
      return null;
    }
    return toISODate(this, true);
  }

  /**
   * Returns a string representation of this DateTime appropriate for use in SQL Time
   * @param {Object} opts - options
   * @param {boolean} [opts.includeZone=false] - include the zone, such as 'America/New_York'. Overrides includeOffset.
   * @param {boolean} [opts.includeOffset=true] - include the offset, such as 'Z' or '-04:00'
   * @param {boolean} [opts.includeOffsetSpace=true] - include the space between the time and the offset, such as '05:15:16.345 -04:00'
   * @example DateTime.utc().toSQL() //=> '05:15:16.345'
   * @example DateTime.now().toSQL() //=> '05:15:16.345 -04:00'
   * @example DateTime.now().toSQL({ includeOffset: false }) //=> '05:15:16.345'
   * @example DateTime.now().toSQL({ includeZone: false }) //=> '05:15:16.345 America/New_York'
   * @return {string}
   */
  toSQLTime({ includeOffset = true, includeZone = false, includeOffsetSpace = true } = {}) {
    let fmt = "HH:mm:ss.SSS";

    if (includeZone || includeOffset) {
      if (includeOffsetSpace) {
        fmt += " ";
      }
      if (includeZone) {
        fmt += "z";
      } else if (includeOffset) {
        fmt += "ZZ";
      }
    }

    return toTechFormat(this, fmt, true);
  }

  /**
   * Returns a string representation of this DateTime appropriate for use in SQL DateTime
   * @param {Object} opts - options
   * @param {boolean} [opts.includeZone=false] - include the zone, such as 'America/New_York'. Overrides includeOffset.
   * @param {boolean} [opts.includeOffset=true] - include the offset, such as 'Z' or '-04:00'
   * @param {boolean} [opts.includeOffsetSpace=true] - include the space between the time and the offset, such as '05:15:16.345 -04:00'
   * @example DateTime.utc(2014, 7, 13).toSQL() //=> '2014-07-13 00:00:00.000 Z'
   * @example DateTime.local(2014, 7, 13).toSQL() //=> '2014-07-13 00:00:00.000 -04:00'
   * @example DateTime.local(2014, 7, 13).toSQL({ includeOffset: false }) //=> '2014-07-13 00:00:00.000'
   * @example DateTime.local(2014, 7, 13).toSQL({ includeZone: true }) //=> '2014-07-13 00:00:00.000 America/New_York'
   * @return {string}
   */
  toSQL(opts = {}) {
    if (!this.isValid) {
      return null;
    }

    return `${this.toSQLDate()} ${this.toSQLTime(opts)}`;
  }

  /**
   * Returns a string representation of this DateTime appropriate for debugging
   * @return {string}
   */
  toString() {
    return this.isValid ? this.toISO() : INVALID;
  }

  /**
   * Returns the epoch milliseconds of this DateTime. Alias of {@link DateTime#toMillis}
   * @return {number}
   */
  valueOf() {
    return this.toMillis();
  }

  /**
   * Returns the epoch milliseconds of this DateTime.
   * @return {number}
   */
  toMillis() {
    return this.isValid ? this.ts : NaN;
  }

  /**
   * Returns the epoch seconds of this DateTime.
   * @return {number}
   */
  toSeconds() {
    return this.isValid ? this.ts / 1000 : NaN;
  }

  /**
   * Returns the epoch seconds (as a whole number) of this DateTime.
   * @return {number}
   */
  toUnixInteger() {
    return this.isValid ? Math.floor(this.ts / 1000) : NaN;
  }

  /**
   * Returns an ISO 8601 representation of this DateTime appropriate for use in JSON.
   * @return {string}
   */
  toJSON() {
    return this.toISO();
  }

  /**
   * Returns a BSON serializable equivalent to this DateTime.
   * @return {Date}
   */
  toBSON() {
    return this.toJSDate();
  }

  /**
   * Returns a JavaScript object with this DateTime's year, month, day, and so on.
   * @param opts - options for generating the object
   * @param {boolean} [opts.includeConfig=false] - include configuration attributes in the output
   * @example DateTime.now().toObject() //=> { year: 2017, month: 4, day: 22, hour: 20, minute: 49, second: 42, millisecond: 268 }
   * @return {Object}
   */
  toObject(opts = {}) {
    if (!this.isValid) return {};

    const base = { ...this.c };

    if (opts.includeConfig) {
      base.outputCalendar = this.outputCalendar;
      base.numberingSystem = this.loc.numberingSystem;
      base.locale = this.loc.locale;
    }
    return base;
  }

  /**
   * Returns a JavaScript Date equivalent to this DateTime.
   * @return {Date}
   */
  toJSDate() {
    return new Date(this.isValid ? this.ts : NaN);
  }

  // COMPARE

  /**
   * Return the difference between two DateTimes as a Duration.
   * @param {DateTime} otherDateTime - the DateTime to compare this one to
   * @param {string|string[]} [unit=['milliseconds']] - the unit or array of units (such as 'hours' or 'days') to include in the duration.
   * @param {Object} opts - options that affect the creation of the Duration
   * @param {string} [opts.conversionAccuracy='casual'] - the conversion system to use
   * @example
   * var i1 = DateTime.fromISO('1982-05-25T09:45'),
   *     i2 = DateTime.fromISO('1983-10-14T10:30');
   * i2.diff(i1).toObject() //=> { milliseconds: 43807500000 }
   * i2.diff(i1, 'hours').toObject() //=> { hours: 12168.75 }
   * i2.diff(i1, ['months', 'days']).toObject() //=> { months: 16, days: 19.03125 }
   * i2.diff(i1, ['months', 'days', 'hours']).toObject() //=> { months: 16, days: 19, hours: 0.75 }
   * @return {Duration}
   */
  diff(otherDateTime, unit = "milliseconds", opts = {}) {
    if (!this.isValid || !otherDateTime.isValid) {
      return Duration.invalid("created by diffing an invalid DateTime");
    }

    const durOpts = { locale: this.locale, numberingSystem: this.numberingSystem, ...opts };

    const units = maybeArray(unit).map(Duration.normalizeUnit),
      otherIsLater = otherDateTime.valueOf() > this.valueOf(),
      earlier = otherIsLater ? this : otherDateTime,
      later = otherIsLater ? otherDateTime : this,
      diffed = diff(earlier, later, units, durOpts);

    return otherIsLater ? diffed.negate() : diffed;
  }

  /**
   * Return the difference between this DateTime and right now.
   * See {@link DateTime#diff}
   * @param {string|string[]} [unit=['milliseconds']] - the unit or units units (such as 'hours' or 'days') to include in the duration
   * @param {Object} opts - options that affect the creation of the Duration
   * @param {string} [opts.conversionAccuracy='casual'] - the conversion system to use
   * @return {Duration}
   */
  diffNow(unit = "milliseconds", opts = {}) {
    return this.diff(DateTime.now(), unit, opts);
  }

  /**
   * Return an Interval spanning between this DateTime and another DateTime
   * @param {DateTime} otherDateTime - the other end point of the Interval
   * @return {Interval}
   */
  until(otherDateTime) {
    return this.isValid ? Interval.fromDateTimes(this, otherDateTime) : this;
  }

  /**
   * Return whether this DateTime is in the same unit of time as another DateTime.
   * Higher-order units must also be identical for this function to return `true`.
   * Note that time zones are **ignored** in this comparison, which compares the **local** calendar time. Use {@link DateTime#setZone} to convert one of the dates if needed.
   * @param {DateTime} otherDateTime - the other DateTime
   * @param {string} unit - the unit of time to check sameness on
   * @example DateTime.now().hasSame(otherDT, 'day'); //~> true if otherDT is in the same current calendar day
   * @return {boolean}
   */
  hasSame(otherDateTime, unit) {
    if (!this.isValid) return false;

    const inputMs = otherDateTime.valueOf();
    const adjustedToZone = this.setZone(otherDateTime.zone, { keepLocalTime: true });
    return adjustedToZone.startOf(unit) <= inputMs && inputMs <= adjustedToZone.endOf(unit);
  }

  /**
   * Equality check
   * Two DateTimes are equal if and only if they represent the same millisecond, have the same zone and location, and are both valid.
   * To compare just the millisecond values, use `+dt1 === +dt2`.
   * @param {DateTime} other - the other DateTime
   * @return {boolean}
   */
  equals(other) {
    return (
      this.isValid &&
      other.isValid &&
      this.valueOf() === other.valueOf() &&
      this.zone.equals(other.zone) &&
      this.loc.equals(other.loc)
    );
  }

  /**
   * Returns a string representation of a this time relative to now, such as "in two days". Can only internationalize if your
   * platform supports Intl.RelativeTimeFormat. Rounds down by default.
   * @param {Object} options - options that affect the output
   * @param {DateTime} [options.base=DateTime.now()] - the DateTime to use as the basis to which this time is compared. Defaults to now.
   * @param {string} [options.style="long"] - the style of units, must be "long", "short", or "narrow"
   * @param {string|string[]} options.unit - use a specific unit or array of units; if omitted, or an array, the method will pick the best unit. Use an array or one of "years", "quarters", "months", "weeks", "days", "hours", "minutes", or "seconds"
   * @param {boolean} [options.round=true] - whether to round the numbers in the output.
   * @param {number} [options.padding=0] - padding in milliseconds. This allows you to round up the result if it fits inside the threshold. Don't use in combination with {round: false} because the decimal output will include the padding.
   * @param {string} options.locale - override the locale of this DateTime
   * @param {string} options.numberingSystem - override the numberingSystem of this DateTime. The Intl system may choose not to honor this
   * @example DateTime.now().plus({ days: 1 }).toRelative() //=> "in 1 day"
   * @example DateTime.now().setLocale("es").toRelative({ days: 1 }) //=> "dentro de 1 día"
   * @example DateTime.now().plus({ days: 1 }).toRelative({ locale: "fr" }) //=> "dans 23 heures"
   * @example DateTime.now().minus({ days: 2 }).toRelative() //=> "2 days ago"
   * @example DateTime.now().minus({ days: 2 }).toRelative({ unit: "hours" }) //=> "48 hours ago"
   * @example DateTime.now().minus({ hours: 36 }).toRelative({ round: false }) //=> "1.5 days ago"
   */
  toRelative(options = {}) {
    if (!this.isValid) return null;
    const base = options.base || DateTime.fromObject({}, { zone: this.zone }),
      padding = options.padding ? (this < base ? -options.padding : options.padding) : 0;
    let units = ["years", "months", "days", "hours", "minutes", "seconds"];
    let unit = options.unit;
    if (Array.isArray(options.unit)) {
      units = options.unit;
      unit = undefined;
    }
    return diffRelative(base, this.plus(padding), {
      ...options,
      numeric: "always",
      units,
      unit,
    });
  }

  /**
   * Returns a string representation of this date relative to today, such as "yesterday" or "next month".
   * Only internationalizes on platforms that supports Intl.RelativeTimeFormat.
   * @param {Object} options - options that affect the output
   * @param {DateTime} [options.base=DateTime.now()] - the DateTime to use as the basis to which this time is compared. Defaults to now.
   * @param {string} options.locale - override the locale of this DateTime
   * @param {string} options.unit - use a specific unit; if omitted, the method will pick the unit. Use one of "years", "quarters", "months", "weeks", or "days"
   * @param {string} options.numberingSystem - override the numberingSystem of this DateTime. The Intl system may choose not to honor this
   * @example DateTime.now().plus({ days: 1 }).toRelativeCalendar() //=> "tomorrow"
   * @example DateTime.now().setLocale("es").plus({ days: 1 }).toRelative() //=> ""mañana"
   * @example DateTime.now().plus({ days: 1 }).toRelativeCalendar({ locale: "fr" }) //=> "demain"
   * @example DateTime.now().minus({ days: 2 }).toRelativeCalendar() //=> "2 days ago"
   */
  toRelativeCalendar(options = {}) {
    if (!this.isValid) return null;

    return diffRelative(options.base || DateTime.fromObject({}, { zone: this.zone }), this, {
      ...options,
      numeric: "auto",
      units: ["years", "months", "days"],
      calendary: true,
    });
  }

  /**
   * Return the min of several date times
   * @param {...DateTime} dateTimes - the DateTimes from which to choose the minimum
   * @return {DateTime} the min DateTime, or undefined if called with no argument
   */
  static min(...dateTimes) {
    if (!dateTimes.every(DateTime.isDateTime)) {
      throw new InvalidArgumentError("min requires all arguments be DateTimes");
    }
    return bestBy(dateTimes, (i) => i.valueOf(), Math.min);
  }

  /**
   * Return the max of several date times
   * @param {...DateTime} dateTimes - the DateTimes from which to choose the maximum
   * @return {DateTime} the max DateTime, or undefined if called with no argument
   */
  static max(...dateTimes) {
    if (!dateTimes.every(DateTime.isDateTime)) {
      throw new InvalidArgumentError("max requires all arguments be DateTimes");
    }
    return bestBy(dateTimes, (i) => i.valueOf(), Math.max);
  }

  // MISC

  /**
   * Explain how a string would be parsed by fromFormat()
   * @param {string} text - the string to parse
   * @param {string} fmt - the format the string is expected to be in (see description)
   * @param {Object} options - options taken by fromFormat()
   * @return {Object}
   */
  static fromFormatExplain(text, fmt, options = {}) {
    const { locale = null, numberingSystem = null } = options,
      localeToUse = Locale.fromOpts({
        locale,
        numberingSystem,
        defaultToEN: true,
      });
    return explainFromTokens(localeToUse, text, fmt);
  }

  /**
   * @deprecated use fromFormatExplain instead
   */
  static fromStringExplain(text, fmt, options = {}) {
    return DateTime.fromFormatExplain(text, fmt, options);
  }

  // FORMAT PRESETS

  /**
   * {@link DateTime#toLocaleString} format like 10/14/1983
   * @type {Object}
   */
  static get DATE_SHORT() {
    return DATE_SHORT;
  }

  /**
   * {@link DateTime#toLocaleString} format like 'Oct 14, 1983'
   * @type {Object}
   */
  static get DATE_MED() {
    return DATE_MED;
  }

  /**
   * {@link DateTime#toLocaleString} format like 'Fri, Oct 14, 1983'
   * @type {Object}
   */
  static get DATE_MED_WITH_WEEKDAY() {
    return DATE_MED_WITH_WEEKDAY;
  }

  /**
   * {@link DateTime#toLocaleString} format like 'October 14, 1983'
   * @type {Object}
   */
  static get DATE_FULL() {
    return DATE_FULL;
  }

  /**
   * {@link DateTime#toLocaleString} format like 'Tuesday, October 14, 1983'
   * @type {Object}
   */
  static get DATE_HUGE() {
    return DATE_HUGE;
  }

  /**
   * {@link DateTime#toLocaleString} format like '09:30 AM'. Only 12-hour if the locale is.
   * @type {Object}
   */
  static get TIME_SIMPLE() {
    return TIME_SIMPLE;
  }

  /**
   * {@link DateTime#toLocaleString} format like '09:30:23 AM'. Only 12-hour if the locale is.
   * @type {Object}
   */
  static get TIME_WITH_SECONDS() {
    return TIME_WITH_SECONDS;
  }

  /**
   * {@link DateTime#toLocaleString} format like '09:30:23 AM EDT'. Only 12-hour if the locale is.
   * @type {Object}
   */
  static get TIME_WITH_SHORT_OFFSET() {
    return TIME_WITH_SHORT_OFFSET;
  }

  /**
   * {@link DateTime#toLocaleString} format like '09:30:23 AM Eastern Daylight Time'. Only 12-hour if the locale is.
   * @type {Object}
   */
  static get TIME_WITH_LONG_OFFSET() {
    return TIME_WITH_LONG_OFFSET;
  }

  /**
   * {@link DateTime#toLocaleString} format like '09:30', always 24-hour.
   * @type {Object}
   */
  static get TIME_24_SIMPLE() {
    return TIME_24_SIMPLE;
  }

  /**
   * {@link DateTime#toLocaleString} format like '09:30:23', always 24-hour.
   * @type {Object}
   */
  static get TIME_24_WITH_SECONDS() {
    return TIME_24_WITH_SECONDS;
  }

  /**
   * {@link DateTime#toLocaleString} format like '09:30:23 EDT', always 24-hour.
   * @type {Object}
   */
  static get TIME_24_WITH_SHORT_OFFSET() {
    return TIME_24_WITH_SHORT_OFFSET;
  }

  /**
   * {@link DateTime#toLocaleString} format like '09:30:23 Eastern Daylight Time', always 24-hour.
   * @type {Object}
   */
  static get TIME_24_WITH_LONG_OFFSET() {
    return TIME_24_WITH_LONG_OFFSET;
  }

  /**
   * {@link DateTime#toLocaleString} format like '10/14/1983, 9:30 AM'. Only 12-hour if the locale is.
   * @type {Object}
   */
  static get DATETIME_SHORT() {
    return DATETIME_SHORT;
  }

  /**
   * {@link DateTime#toLocaleString} format like '10/14/1983, 9:30:33 AM'. Only 12-hour if the locale is.
   * @type {Object}
   */
  static get DATETIME_SHORT_WITH_SECONDS() {
    return DATETIME_SHORT_WITH_SECONDS;
  }

  /**
   * {@link DateTime#toLocaleString} format like 'Oct 14, 1983, 9:30 AM'. Only 12-hour if the locale is.
   * @type {Object}
   */
  static get DATETIME_MED() {
    return DATETIME_MED;
  }

  /**
   * {@link DateTime#toLocaleString} format like 'Oct 14, 1983, 9:30:33 AM'. Only 12-hour if the locale is.
   * @type {Object}
   */
  static get DATETIME_MED_WITH_SECONDS() {
    return DATETIME_MED_WITH_SECONDS;
  }

  /**
   * {@link DateTime#toLocaleString} format like 'Fri, 14 Oct 1983, 9:30 AM'. Only 12-hour if the locale is.
   * @type {Object}
   */
  static get DATETIME_MED_WITH_WEEKDAY() {
    return DATETIME_MED_WITH_WEEKDAY;
  }

  /**
   * {@link DateTime#toLocaleString} format like 'October 14, 1983, 9:30 AM EDT'. Only 12-hour if the locale is.
   * @type {Object}
   */
  static get DATETIME_FULL() {
    return DATETIME_FULL;
  }

  /**
   * {@link DateTime#toLocaleString} format like 'October 14, 1983, 9:30:33 AM EDT'. Only 12-hour if the locale is.
   * @type {Object}
   */
  static get DATETIME_FULL_WITH_SECONDS() {
    return DATETIME_FULL_WITH_SECONDS;
  }

  /**
   * {@link DateTime#toLocaleString} format like 'Friday, October 14, 1983, 9:30 AM Eastern Daylight Time'. Only 12-hour if the locale is.
   * @type {Object}
   */
  static get DATETIME_HUGE() {
    return DATETIME_HUGE;
  }

  /**
   * {@link DateTime#toLocaleString} format like 'Friday, October 14, 1983, 9:30:33 AM Eastern Daylight Time'. Only 12-hour if the locale is.
   * @type {Object}
   */
  static get DATETIME_HUGE_WITH_SECONDS() {
    return DATETIME_HUGE_WITH_SECONDS;
  }
}

/**
 * @private
 */
function friendlyDateTime(dateTimeish) {
  if (DateTime.isDateTime(dateTimeish)) {
    return dateTimeish;
  } else if (dateTimeish && dateTimeish.valueOf && isNumber(dateTimeish.valueOf())) {
    return DateTime.fromJSDate(dateTimeish);
  } else if (dateTimeish && typeof dateTimeish === "object") {
    return DateTime.fromObject(dateTimeish);
  } else {
    throw new InvalidArgumentError(
      `Unknown datetime argument: ${dateTimeish}, of type ${typeof dateTimeish}`
    );
  }
}

const VERSION = "3.4.3";

var Luxon = /*#__PURE__*/Object.freeze({
  __proto__: null,
  VERSION: VERSION,
  DateTime: DateTime,
  Duration: Duration,
  Interval: Interval,
  Info: Info,
  Zone: Zone,
  FixedOffsetZone: FixedOffsetZone,
  IANAZone: IANAZone,
  InvalidZone: InvalidZone,
  SystemZone: SystemZone,
  Settings: Settings
});

////////////////////
// Query Settings //
////////////////////
const DEFAULT_QUERY_SETTINGS = {
    renderNullAs: "\\-",
    taskCompletionTracking: false,
    taskCompletionUseEmojiShorthand: false,
    taskCompletionText: "completion",
    taskCompletionDateFormat: "yyyy-MM-dd",
    recursiveSubTaskCompletion: false,
    warnOnEmptyResult: true,
    refreshEnabled: true,
    refreshInterval: 2500,
    defaultDateFormat: "MMMM dd, yyyy",
    defaultDateTimeFormat: "h:mm a - MMMM dd, yyyy",
    maxRecursiveRenderDepth: 4,
    tableIdColumnName: "File",
    tableGroupColumnName: "Group",
    showResultCount: true,
};
const DEFAULT_EXPORT_SETTINGS = {
    allowHtml: true,
};
/** Default settings for dataview on install. */
const DEFAULT_SETTINGS = {
    ...DEFAULT_QUERY_SETTINGS,
    ...DEFAULT_EXPORT_SETTINGS,
    ...{
        inlineQueryPrefix: "=",
        inlineJsQueryPrefix: "$=",
        inlineQueriesInCodeblocks: true,
        enableInlineDataview: true,
        enableDataviewJs: false,
        enableInlineDataviewJs: false,
        prettyRenderInlineFields: true,
        prettyRenderInlineFieldsInLivePreview: true,
        dataviewJsKeyword: "dataviewjs",
    },
};

/** Functional return type for error handling. */
class Success {
    value;
    successful;
    constructor(value) {
        this.value = value;
        this.successful = true;
    }
    map(f) {
        return new Success(f(this.value));
    }
    flatMap(f) {
        return f(this.value);
    }
    mapErr(f) {
        return this;
    }
    bimap(succ, _fail) {
        return this.map(succ);
    }
    orElse(_value) {
        return this.value;
    }
    cast() {
        return this;
    }
    orElseThrow(_message) {
        return this.value;
    }
}
/** Functional return type for error handling. */
class Failure {
    error;
    successful;
    constructor(error) {
        this.error = error;
        this.successful = false;
    }
    map(_f) {
        return this;
    }
    flatMap(_f) {
        return this;
    }
    mapErr(f) {
        return new Failure(f(this.error));
    }
    bimap(_succ, fail) {
        return this.mapErr(fail);
    }
    orElse(value) {
        return value;
    }
    cast() {
        return this;
    }
    orElseThrow(message) {
        if (message)
            throw new Error(message(this.error));
        else
            throw new Error("" + this.error);
    }
}
/** Monadic 'Result' type which encapsulates whether a procedure succeeded or failed, as well as it's return value. */
var Result;
(function (Result) {
    /** Construct a new success result wrapping the given value. */
    function success(value) {
        return new Success(value);
    }
    Result.success = success;
    /** Construct a new failure value wrapping the given error. */
    function failure(error) {
        return new Failure(error);
    }
    Result.failure = failure;
    /** Join two results with a bi-function and return a new result. */
    function flatMap2(first, second, f) {
        if (first.successful) {
            if (second.successful)
                return f(first.value, second.value);
            else
                return failure(second.error);
        }
        else {
            return failure(first.error);
        }
    }
    Result.flatMap2 = flatMap2;
    /** Join two results with a bi-function and return a new result. */
    function map2(first, second, f) {
        return flatMap2(first, second, (a, b) => success(f(a, b)));
    }
    Result.map2 = map2;
})(Result || (Result = {}));

var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

function getDefaultExportFromCjs (x) {
	return x && x.__esModule && Object.prototype.hasOwnProperty.call(x, 'default') ? x['default'] : x;
}

var parsimmon_umd_min = {exports: {}};

parsimmon_umd_min.exports;

(function (module, exports) {
	!function(n,t){module.exports=t();}("undefined"!=typeof self?self:commonjsGlobal,function(){return function(n){var t={};function r(e){if(t[e])return t[e].exports;var u=t[e]={i:e,l:!1,exports:{}};return n[e].call(u.exports,u,u.exports,r),u.l=!0,u.exports}return r.m=n,r.c=t,r.d=function(n,t,e){r.o(n,t)||Object.defineProperty(n,t,{configurable:!1,enumerable:!0,get:e});},r.r=function(n){Object.defineProperty(n,"__esModule",{value:!0});},r.n=function(n){var t=n&&n.__esModule?function(){return n.default}:function(){return n};return r.d(t,"a",t),t},r.o=function(n,t){return Object.prototype.hasOwnProperty.call(n,t)},r.p="",r(r.s=0)}([function(n,t,r){function e(n){if(!(this instanceof e))return new e(n);this._=n;}var u=e.prototype;function o(n,t){for(var r=0;r<n;r++)t(r);}function i(n,t,r){return function(n,t){o(t.length,function(r){n(t[r],r,t);});}(function(r,e,u){t=n(t,r,e,u);},r),t}function a(n,t){return i(function(t,r,e,u){return t.concat([n(r,e,u)])},[],t)}function f(n,t){var r={v:0,buf:t};return o(n,function(){var n;r={v:r.v<<1|(n=r.buf,n[0]>>7),buf:function(n){var t=i(function(n,t,r,e){return n.concat(r===e.length-1?Buffer.from([t,0]).readUInt16BE(0):e.readUInt16BE(r))},[],n);return Buffer.from(a(function(n){return (n<<1&65535)>>8},t))}(r.buf)};}),r}function c(){return "undefined"!=typeof Buffer}function s(){if(!c())throw new Error("Buffer global does not exist; please use webpack if you need to parse Buffers in the browser.")}function l(n){s();var t=i(function(n,t){return n+t},0,n);if(t%8!=0)throw new Error("The bits ["+n.join(", ")+"] add up to "+t+" which is not an even number of bytes; the total should be divisible by 8");var r,u=t/8,o=(r=function(n){return n>48},i(function(n,t){return n||(r(t)?t:n)},null,n));if(o)throw new Error(o+" bit range requested exceeds 48 bit (6 byte) Number max.");return new e(function(t,r){var e=u+r;return e>t.length?x(r,u.toString()+" bytes"):b(e,i(function(n,t){var r=f(t,n.buf);return {coll:n.coll.concat(r.v),buf:r.buf}},{coll:[],buf:t.slice(r,e)},n).coll)})}function h(n,t){return new e(function(r,e){return s(),e+t>r.length?x(e,t+" bytes for "+n):b(e+t,r.slice(e,e+t))})}function p(n,t){if("number"!=typeof(r=t)||Math.floor(r)!==r||t<0||t>6)throw new Error(n+" requires integer length in range [0, 6].");var r;}function d(n){return p("uintBE",n),h("uintBE("+n+")",n).map(function(t){return t.readUIntBE(0,n)})}function v(n){return p("uintLE",n),h("uintLE("+n+")",n).map(function(t){return t.readUIntLE(0,n)})}function g(n){return p("intBE",n),h("intBE("+n+")",n).map(function(t){return t.readIntBE(0,n)})}function m(n){return p("intLE",n),h("intLE("+n+")",n).map(function(t){return t.readIntLE(0,n)})}function y(n){return n instanceof e}function E(n){return "[object Array]"==={}.toString.call(n)}function w(n){return c()&&Buffer.isBuffer(n)}function b(n,t){return {status:!0,index:n,value:t,furthest:-1,expected:[]}}function x(n,t){return E(t)||(t=[t]),{status:!1,index:-1,value:null,furthest:n,expected:t}}function B(n,t){if(!t)return n;if(n.furthest>t.furthest)return n;var r=n.furthest===t.furthest?function(n,t){if(function(){if(void 0!==e._supportsSet)return e._supportsSet;var n="undefined"!=typeof Set;return e._supportsSet=n,n}()&&Array.from){for(var r=new Set(n),u=0;u<t.length;u++)r.add(t[u]);var o=Array.from(r);return o.sort(),o}for(var i={},a=0;a<n.length;a++)i[n[a]]=!0;for(var f=0;f<t.length;f++)i[t[f]]=!0;var c=[];for(var s in i)({}).hasOwnProperty.call(i,s)&&c.push(s);return c.sort(),c}(n.expected,t.expected):t.expected;return {status:n.status,index:n.index,value:n.value,furthest:t.furthest,expected:r}}var j={};function S(n,t){if(w(n))return {offset:t,line:-1,column:-1};n in j||(j[n]={});for(var r=j[n],e=0,u=0,o=0,i=t;i>=0;){if(i in r){e=r[i].line,0===o&&(o=r[i].lineStart);break}("\n"===n.charAt(i)||"\r"===n.charAt(i)&&"\n"!==n.charAt(i+1))&&(u++,0===o&&(o=i+1)),i--;}var a=e+u,f=t-o;return r[t]={line:a,lineStart:o},{offset:t,line:a+1,column:f+1}}function _(n){if(!y(n))throw new Error("not a parser: "+n)}function L(n,t){return "string"==typeof n?n.charAt(t):n[t]}function O(n){if("number"!=typeof n)throw new Error("not a number: "+n)}function k(n){if("function"!=typeof n)throw new Error("not a function: "+n)}function P(n){if("string"!=typeof n)throw new Error("not a string: "+n)}var q=2,A=3,I=8,F=5*I,M=4*I,z="  ";function R(n,t){return new Array(t+1).join(n)}function U(n,t,r){var e=t-n.length;return e<=0?n:R(r,e)+n}function W(n,t,r,e){return {from:n-t>0?n-t:0,to:n+r>e?e:n+r}}function D(n,t){var r,e,u,o,f,c=t.index,s=c.offset,l=1;if(s===n.length)return "Got the end of the input";if(w(n)){var h=s-s%I,p=s-h,d=W(h,F,M+I,n.length),v=a(function(n){return a(function(n){return U(n.toString(16),2,"0")},n)},function(n,t){var r=n.length,e=[],u=0;if(r<=t)return [n.slice()];for(var o=0;o<r;o++)e[u]||e.push([]),e[u].push(n[o]),(o+1)%t==0&&u++;return e}(n.slice(d.from,d.to).toJSON().data,I));o=function(n){return 0===n.from&&1===n.to?{from:n.from,to:n.to}:{from:n.from/I,to:Math.floor(n.to/I)}}(d),e=h/I,r=3*p,p>=4&&(r+=1),l=2,u=a(function(n){return n.length<=4?n.join(" "):n.slice(0,4).join(" ")+"  "+n.slice(4).join(" ")},v),(f=(8*(o.to>0?o.to-1:o.to)).toString(16).length)<2&&(f=2);}else {var g=n.split(/\r\n|[\n\r\u2028\u2029]/);r=c.column-1,e=c.line-1,o=W(e,q,A,g.length),u=g.slice(o.from,o.to),f=o.to.toString().length;}var m=e-o.from;return w(n)&&(f=(8*(o.to>0?o.to-1:o.to)).toString(16).length)<2&&(f=2),i(function(t,e,u){var i,a=u===m,c=a?"> ":z;return i=w(n)?U((8*(o.from+u)).toString(16),f,"0"):U((o.from+u+1).toString(),f," "),[].concat(t,[c+i+" | "+e],a?[z+R(" ",f)+" | "+U("",r," ")+R("^",l)]:[])},[],u).join("\n")}function N(n,t){return ["\n","-- PARSING FAILED "+R("-",50),"\n\n",D(n,t),"\n\n",(r=t.expected,1===r.length?"Expected:\n\n"+r[0]:"Expected one of the following: \n\n"+r.join(", ")),"\n"].join("");var r;}function G(n){return void 0!==n.flags?n.flags:[n.global?"g":"",n.ignoreCase?"i":"",n.multiline?"m":"",n.unicode?"u":"",n.sticky?"y":""].join("")}function C(){for(var n=[].slice.call(arguments),t=n.length,r=0;r<t;r+=1)_(n[r]);return e(function(r,e){for(var u,o=new Array(t),i=0;i<t;i+=1){if(!(u=B(n[i]._(r,e),u)).status)return u;o[i]=u.value,e=u.index;}return B(b(e,o),u)})}function J(){var n=[].slice.call(arguments);if(0===n.length)throw new Error("seqMap needs at least one argument");var t=n.pop();return k(t),C.apply(null,n).map(function(n){return t.apply(null,n)})}function T(){var n=[].slice.call(arguments),t=n.length;if(0===t)return Y("zero alternates");for(var r=0;r<t;r+=1)_(n[r]);return e(function(t,r){for(var e,u=0;u<n.length;u+=1)if((e=B(n[u]._(t,r),e)).status)return e;return e})}function V(n,t){return H(n,t).or(X([]))}function H(n,t){return _(n),_(t),J(n,t.then(n).many(),function(n,t){return [n].concat(t)})}function K(n){P(n);var t="'"+n+"'";return e(function(r,e){var u=e+n.length,o=r.slice(e,u);return o===n?b(u,o):x(e,t)})}function Q(n,t){!function(n){if(!(n instanceof RegExp))throw new Error("not a regexp: "+n);for(var t=G(n),r=0;r<t.length;r++){var e=t.charAt(r);if("i"!==e&&"m"!==e&&"u"!==e&&"s"!==e)throw new Error('unsupported regexp flag "'+e+'": '+n)}}(n),arguments.length>=2?O(t):t=0;var r=function(n){return RegExp("^(?:"+n.source+")",G(n))}(n),u=""+n;return e(function(n,e){var o=r.exec(n.slice(e));if(o){if(0<=t&&t<=o.length){var i=o[0],a=o[t];return b(e+i.length,a)}return x(e,"valid match group (0 to "+o.length+") in "+u)}return x(e,u)})}function X(n){return e(function(t,r){return b(r,n)})}function Y(n){return e(function(t,r){return x(r,n)})}function Z(n){if(y(n))return e(function(t,r){var e=n._(t,r);return e.index=r,e.value="",e});if("string"==typeof n)return Z(K(n));if(n instanceof RegExp)return Z(Q(n));throw new Error("not a string, regexp, or parser: "+n)}function $(n){return _(n),e(function(t,r){var e=n._(t,r),u=t.slice(r,e.index);return e.status?x(r,'not "'+u+'"'):b(r,null)})}function nn(n){return k(n),e(function(t,r){var e=L(t,r);return r<t.length&&n(e)?b(r+1,e):x(r,"a character/byte matching "+n)})}function tn(n,t){arguments.length<2&&(t=n,n=void 0);var r=e(function(n,e){return r._=t()._,r._(n,e)});return n?r.desc(n):r}function rn(){return Y("fantasy-land/empty")}u.parse=function(n){if("string"!=typeof n&&!w(n))throw new Error(".parse must be called with a string or Buffer as its argument");var t,r=this.skip(an)._(n,0);return t=r.status?{status:!0,value:r.value}:{status:!1,index:S(n,r.furthest),expected:r.expected},delete j[n],t},u.tryParse=function(n){var t=this.parse(n);if(t.status)return t.value;var r=N(n,t),e=new Error(r);throw e.type="ParsimmonError",e.result=t,e},u.assert=function(n,t){return this.chain(function(r){return n(r)?X(r):Y(t)})},u.or=function(n){return T(this,n)},u.trim=function(n){return this.wrap(n,n)},u.wrap=function(n,t){return J(n,this,t,function(n,t){return t})},u.thru=function(n){return n(this)},u.then=function(n){return _(n),C(this,n).map(function(n){return n[1]})},u.many=function(){var n=this;return e(function(t,r){for(var e=[],u=void 0;;){if(!(u=B(n._(t,r),u)).status)return B(b(r,e),u);if(r===u.index)throw new Error("infinite loop detected in .many() parser --- calling .many() on a parser which can accept zero characters is usually the cause");r=u.index,e.push(u.value);}})},u.tieWith=function(n){return P(n),this.map(function(t){if(function(n){if(!E(n))throw new Error("not an array: "+n)}(t),t.length){P(t[0]);for(var r=t[0],e=1;e<t.length;e++)P(t[e]),r+=n+t[e];return r}return ""})},u.tie=function(){return this.tieWith("")},u.times=function(n,t){var r=this;return arguments.length<2&&(t=n),O(n),O(t),e(function(e,u){for(var o=[],i=void 0,a=void 0,f=0;f<n;f+=1){if(a=B(i=r._(e,u),a),!i.status)return a;u=i.index,o.push(i.value);}for(;f<t&&(a=B(i=r._(e,u),a),i.status);f+=1)u=i.index,o.push(i.value);return B(b(u,o),a)})},u.result=function(n){return this.map(function(){return n})},u.atMost=function(n){return this.times(0,n)},u.atLeast=function(n){return J(this.times(n),this.many(),function(n,t){return n.concat(t)})},u.map=function(n){k(n);var t=this;return e(function(r,e){var u=t._(r,e);return u.status?B(b(u.index,n(u.value)),u):u})},u.contramap=function(n){k(n);var t=this;return e(function(r,e){var u=t.parse(n(r.slice(e)));return u.status?b(e+r.length,u.value):u})},u.promap=function(n,t){return k(n),k(t),this.contramap(n).map(t)},u.skip=function(n){return C(this,n).map(function(n){return n[0]})},u.mark=function(){return J(en,this,en,function(n,t,r){return {start:n,value:t,end:r}})},u.node=function(n){return J(en,this,en,function(t,r,e){return {name:n,value:r,start:t,end:e}})},u.sepBy=function(n){return V(this,n)},u.sepBy1=function(n){return H(this,n)},u.lookahead=function(n){return this.skip(Z(n))},u.notFollowedBy=function(n){return this.skip($(n))},u.desc=function(n){E(n)||(n=[n]);var t=this;return e(function(r,e){var u=t._(r,e);return u.status||(u.expected=n),u})},u.fallback=function(n){return this.or(X(n))},u.ap=function(n){return J(n,this,function(n,t){return n(t)})},u.chain=function(n){var t=this;return e(function(r,e){var u=t._(r,e);return u.status?B(n(u.value)._(r,u.index),u):u})},u.concat=u.or,u.empty=rn,u.of=X,u["fantasy-land/ap"]=u.ap,u["fantasy-land/chain"]=u.chain,u["fantasy-land/concat"]=u.concat,u["fantasy-land/empty"]=u.empty,u["fantasy-land/of"]=u.of,u["fantasy-land/map"]=u.map;var en=e(function(n,t){return b(t,S(n,t))}),un=e(function(n,t){return t>=n.length?x(t,"any character/byte"):b(t+1,L(n,t))}),on=e(function(n,t){return b(n.length,n.slice(t))}),an=e(function(n,t){return t<n.length?x(t,"EOF"):b(t,null)}),fn=Q(/[0-9]/).desc("a digit"),cn=Q(/[0-9]*/).desc("optional digits"),sn=Q(/[a-z]/i).desc("a letter"),ln=Q(/[a-z]*/i).desc("optional letters"),hn=Q(/\s*/).desc("optional whitespace"),pn=Q(/\s+/).desc("whitespace"),dn=K("\r"),vn=K("\n"),gn=K("\r\n"),mn=T(gn,vn,dn).desc("newline"),yn=T(mn,an);e.all=on,e.alt=T,e.any=un,e.cr=dn,e.createLanguage=function(n){var t={};for(var r in n)({}).hasOwnProperty.call(n,r)&&function(r){t[r]=tn(function(){return n[r](t)});}(r);return t},e.crlf=gn,e.custom=function(n){return e(n(b,x))},e.digit=fn,e.digits=cn,e.empty=rn,e.end=yn,e.eof=an,e.fail=Y,e.formatError=N,e.index=en,e.isParser=y,e.lazy=tn,e.letter=sn,e.letters=ln,e.lf=vn,e.lookahead=Z,e.makeFailure=x,e.makeSuccess=b,e.newline=mn,e.noneOf=function(n){return nn(function(t){return n.indexOf(t)<0}).desc("none of '"+n+"'")},e.notFollowedBy=$,e.of=X,e.oneOf=function(n){for(var t=n.split(""),r=0;r<t.length;r++)t[r]="'"+t[r]+"'";return nn(function(t){return n.indexOf(t)>=0}).desc(t)},e.optWhitespace=hn,e.Parser=e,e.range=function(n,t){return nn(function(r){return n<=r&&r<=t}).desc(n+"-"+t)},e.regex=Q,e.regexp=Q,e.sepBy=V,e.sepBy1=H,e.seq=C,e.seqMap=J,e.seqObj=function(){for(var n,t={},r=0,u=(n=arguments,Array.prototype.slice.call(n)),o=u.length,i=0;i<o;i+=1){var a=u[i];if(!y(a)){if(E(a)&&2===a.length&&"string"==typeof a[0]&&y(a[1])){var f=a[0];if(Object.prototype.hasOwnProperty.call(t,f))throw new Error("seqObj: duplicate key "+f);t[f]=!0,r++;continue}throw new Error("seqObj arguments must be parsers or [string, parser] array pairs.")}}if(0===r)throw new Error("seqObj expects at least one named parser, found zero");return e(function(n,t){for(var r,e={},i=0;i<o;i+=1){var a,f;if(E(u[i])?(a=u[i][0],f=u[i][1]):(a=null,f=u[i]),!(r=B(f._(n,t),r)).status)return r;a&&(e[a]=r.value),t=r.index;}return B(b(t,e),r)})},e.string=K,e.succeed=X,e.takeWhile=function(n){return k(n),e(function(t,r){for(var e=r;e<t.length&&n(L(t,e));)e++;return b(e,t.slice(r,e))})},e.test=nn,e.whitespace=pn,e["fantasy-land/empty"]=rn,e["fantasy-land/of"]=X,e.Binary={bitSeq:l,bitSeqObj:function(n){s();var t={},r=0,e=a(function(n){if(E(n)){var e=n;if(2!==e.length)throw new Error("["+e.join(", ")+"] should be length 2, got length "+e.length);if(P(e[0]),O(e[1]),Object.prototype.hasOwnProperty.call(t,e[0]))throw new Error("duplicate key in bitSeqObj: "+e[0]);return t[e[0]]=!0,r++,e}return O(n),[null,n]},n);if(r<1)throw new Error("bitSeqObj expects at least one named pair, got ["+n.join(", ")+"]");var u=a(function(n){return n[0]},e);return l(a(function(n){return n[1]},e)).map(function(n){return i(function(n,t){return null!==t[0]&&(n[t[0]]=t[1]),n},{},a(function(t,r){return [t,n[r]]},u))})},byte:function(n){if(s(),O(n),n>255)throw new Error("Value specified to byte constructor ("+n+"=0x"+n.toString(16)+") is larger in value than a single byte.");var t=(n>15?"0x":"0x0")+n.toString(16);return e(function(r,e){var u=L(r,e);return u===n?b(e+1,u):x(e,t)})},buffer:function(n){return h("buffer",n).map(function(n){return Buffer.from(n)})},encodedString:function(n,t){return h("string",t).map(function(t){return t.toString(n)})},uintBE:d,uint8BE:d(1),uint16BE:d(2),uint32BE:d(4),uintLE:v,uint8LE:v(1),uint16LE:v(2),uint32LE:v(4),intBE:g,int8BE:g(1),int16BE:g(2),int32BE:g(4),intLE:m,int8LE:m(1),int16LE:m(2),int32LE:m(4),floatBE:h("floatBE",4).map(function(n){return n.readFloatBE(0)}),floatLE:h("floatLE",4).map(function(n){return n.readFloatLE(0)}),doubleBE:h("doubleBE",8).map(function(n){return n.readDoubleBE(0)}),doubleLE:h("doubleLE",8).map(function(n){return n.readDoubleLE(0)})},n.exports=e;}])}); 
} (parsimmon_umd_min, parsimmon_umd_min.exports));

var parsimmon_umd_minExports = parsimmon_umd_min.exports;

var emojiRegex = () => {
	// https://mths.be/emoji
	return /[#*0-9]\uFE0F?\u20E3|[\xA9\xAE\u203C\u2049\u2122\u2139\u2194-\u2199\u21A9\u21AA\u231A\u231B\u2328\u23CF\u23ED-\u23EF\u23F1\u23F2\u23F8-\u23FA\u24C2\u25AA\u25AB\u25B6\u25C0\u25FB\u25FC\u25FE\u2600-\u2604\u260E\u2611\u2614\u2615\u2618\u2620\u2622\u2623\u2626\u262A\u262E\u262F\u2638-\u263A\u2640\u2642\u2648-\u2653\u265F\u2660\u2663\u2665\u2666\u2668\u267B\u267E\u267F\u2692\u2694-\u2697\u2699\u269B\u269C\u26A0\u26A7\u26AA\u26B0\u26B1\u26BD\u26BE\u26C4\u26C8\u26CF\u26D1\u26D3\u26E9\u26F0-\u26F5\u26F7\u26F8\u26FA\u2702\u2708\u2709\u270F\u2712\u2714\u2716\u271D\u2721\u2733\u2734\u2744\u2747\u2757\u2763\u27A1\u2934\u2935\u2B05-\u2B07\u2B1B\u2B1C\u2B55\u3030\u303D\u3297\u3299]\uFE0F?|[\u261D\u270C\u270D](?:\uFE0F|\uD83C[\uDFFB-\uDFFF])?|[\u270A\u270B](?:\uD83C[\uDFFB-\uDFFF])?|[\u23E9-\u23EC\u23F0\u23F3\u25FD\u2693\u26A1\u26AB\u26C5\u26CE\u26D4\u26EA\u26FD\u2705\u2728\u274C\u274E\u2753-\u2755\u2795-\u2797\u27B0\u27BF\u2B50]|\u26F9(?:\uFE0F|\uD83C[\uDFFB-\uDFFF])?(?:\u200D[\u2640\u2642]\uFE0F?)?|\u2764\uFE0F?(?:\u200D(?:\uD83D\uDD25|\uD83E\uDE79))?|\uD83C(?:[\uDC04\uDD70\uDD71\uDD7E\uDD7F\uDE02\uDE37\uDF21\uDF24-\uDF2C\uDF36\uDF7D\uDF96\uDF97\uDF99-\uDF9B\uDF9E\uDF9F\uDFCD\uDFCE\uDFD4-\uDFDF\uDFF5\uDFF7]\uFE0F?|[\uDF85\uDFC2\uDFC7](?:\uD83C[\uDFFB-\uDFFF])?|[\uDFC3\uDFC4\uDFCA](?:\uD83C[\uDFFB-\uDFFF])?(?:\u200D[\u2640\u2642]\uFE0F?)?|[\uDFCB\uDFCC](?:\uFE0F|\uD83C[\uDFFB-\uDFFF])?(?:\u200D[\u2640\u2642]\uFE0F?)?|[\uDCCF\uDD8E\uDD91-\uDD9A\uDE01\uDE1A\uDE2F\uDE32-\uDE36\uDE38-\uDE3A\uDE50\uDE51\uDF00-\uDF20\uDF2D-\uDF35\uDF37-\uDF7C\uDF7E-\uDF84\uDF86-\uDF93\uDFA0-\uDFC1\uDFC5\uDFC6\uDFC8\uDFC9\uDFCF-\uDFD3\uDFE0-\uDFF0\uDFF8-\uDFFF]|\uDDE6\uD83C[\uDDE8-\uDDEC\uDDEE\uDDF1\uDDF2\uDDF4\uDDF6-\uDDFA\uDDFC\uDDFD\uDDFF]|\uDDE7\uD83C[\uDDE6\uDDE7\uDDE9-\uDDEF\uDDF1-\uDDF4\uDDF6-\uDDF9\uDDFB\uDDFC\uDDFE\uDDFF]|\uDDE8\uD83C[\uDDE6\uDDE8\uDDE9\uDDEB-\uDDEE\uDDF0-\uDDF5\uDDF7\uDDFA-\uDDFF]|\uDDE9\uD83C[\uDDEA\uDDEC\uDDEF\uDDF0\uDDF2\uDDF4\uDDFF]|\uDDEA\uD83C[\uDDE6\uDDE8\uDDEA\uDDEC\uDDED\uDDF7-\uDDFA]|\uDDEB\uD83C[\uDDEE-\uDDF0\uDDF2\uDDF4\uDDF7]|\uDDEC\uD83C[\uDDE6\uDDE7\uDDE9-\uDDEE\uDDF1-\uDDF3\uDDF5-\uDDFA\uDDFC\uDDFE]|\uDDED\uD83C[\uDDF0\uDDF2\uDDF3\uDDF7\uDDF9\uDDFA]|\uDDEE\uD83C[\uDDE8-\uDDEA\uDDF1-\uDDF4\uDDF6-\uDDF9]|\uDDEF\uD83C[\uDDEA\uDDF2\uDDF4\uDDF5]|\uDDF0\uD83C[\uDDEA\uDDEC-\uDDEE\uDDF2\uDDF3\uDDF5\uDDF7\uDDFC\uDDFE\uDDFF]|\uDDF1\uD83C[\uDDE6-\uDDE8\uDDEE\uDDF0\uDDF7-\uDDFB\uDDFE]|\uDDF2\uD83C[\uDDE6\uDDE8-\uDDED\uDDF0-\uDDFF]|\uDDF3\uD83C[\uDDE6\uDDE8\uDDEA-\uDDEC\uDDEE\uDDF1\uDDF4\uDDF5\uDDF7\uDDFA\uDDFF]|\uDDF4\uD83C\uDDF2|\uDDF5\uD83C[\uDDE6\uDDEA-\uDDED\uDDF0-\uDDF3\uDDF7-\uDDF9\uDDFC\uDDFE]|\uDDF6\uD83C\uDDE6|\uDDF7\uD83C[\uDDEA\uDDF4\uDDF8\uDDFA\uDDFC]|\uDDF8\uD83C[\uDDE6-\uDDEA\uDDEC-\uDDF4\uDDF7-\uDDF9\uDDFB\uDDFD-\uDDFF]|\uDDF9\uD83C[\uDDE6\uDDE8\uDDE9\uDDEB-\uDDED\uDDEF-\uDDF4\uDDF7\uDDF9\uDDFB\uDDFC\uDDFF]|\uDDFA\uD83C[\uDDE6\uDDEC\uDDF2\uDDF3\uDDF8\uDDFE\uDDFF]|\uDDFB\uD83C[\uDDE6\uDDE8\uDDEA\uDDEC\uDDEE\uDDF3\uDDFA]|\uDDFC\uD83C[\uDDEB\uDDF8]|\uDDFD\uD83C\uDDF0|\uDDFE\uD83C[\uDDEA\uDDF9]|\uDDFF\uD83C[\uDDE6\uDDF2\uDDFC]|\uDFF3\uFE0F?(?:\u200D(?:\u26A7\uFE0F?|\uD83C\uDF08))?|\uDFF4(?:\u200D\u2620\uFE0F?|\uDB40\uDC67\uDB40\uDC62\uDB40(?:\uDC65\uDB40\uDC6E\uDB40\uDC67|\uDC73\uDB40\uDC63\uDB40\uDC74|\uDC77\uDB40\uDC6C\uDB40\uDC73)\uDB40\uDC7F)?)|\uD83D(?:[\uDC08\uDC26](?:\u200D\u2B1B)?|[\uDC3F\uDCFD\uDD49\uDD4A\uDD6F\uDD70\uDD73\uDD76-\uDD79\uDD87\uDD8A-\uDD8D\uDDA5\uDDA8\uDDB1\uDDB2\uDDBC\uDDC2-\uDDC4\uDDD1-\uDDD3\uDDDC-\uDDDE\uDDE1\uDDE3\uDDE8\uDDEF\uDDF3\uDDFA\uDECB\uDECD-\uDECF\uDEE0-\uDEE5\uDEE9\uDEF0\uDEF3]\uFE0F?|[\uDC42\uDC43\uDC46-\uDC50\uDC66\uDC67\uDC6B-\uDC6D\uDC72\uDC74-\uDC76\uDC78\uDC7C\uDC83\uDC85\uDC8F\uDC91\uDCAA\uDD7A\uDD95\uDD96\uDE4C\uDE4F\uDEC0\uDECC](?:\uD83C[\uDFFB-\uDFFF])?|[\uDC6E\uDC70\uDC71\uDC73\uDC77\uDC81\uDC82\uDC86\uDC87\uDE45-\uDE47\uDE4B\uDE4D\uDE4E\uDEA3\uDEB4-\uDEB6](?:\uD83C[\uDFFB-\uDFFF])?(?:\u200D[\u2640\u2642]\uFE0F?)?|[\uDD74\uDD90](?:\uFE0F|\uD83C[\uDFFB-\uDFFF])?|[\uDC00-\uDC07\uDC09-\uDC14\uDC16-\uDC25\uDC27-\uDC3A\uDC3C-\uDC3E\uDC40\uDC44\uDC45\uDC51-\uDC65\uDC6A\uDC79-\uDC7B\uDC7D-\uDC80\uDC84\uDC88-\uDC8E\uDC90\uDC92-\uDCA9\uDCAB-\uDCFC\uDCFF-\uDD3D\uDD4B-\uDD4E\uDD50-\uDD67\uDDA4\uDDFB-\uDE2D\uDE2F-\uDE34\uDE37-\uDE44\uDE48-\uDE4A\uDE80-\uDEA2\uDEA4-\uDEB3\uDEB7-\uDEBF\uDEC1-\uDEC5\uDED0-\uDED2\uDED5-\uDED7\uDEDC-\uDEDF\uDEEB\uDEEC\uDEF4-\uDEFC\uDFE0-\uDFEB\uDFF0]|\uDC15(?:\u200D\uD83E\uDDBA)?|\uDC3B(?:\u200D\u2744\uFE0F?)?|\uDC41\uFE0F?(?:\u200D\uD83D\uDDE8\uFE0F?)?|\uDC68(?:\u200D(?:[\u2695\u2696\u2708]\uFE0F?|\u2764\uFE0F?\u200D\uD83D(?:\uDC8B\u200D\uD83D)?\uDC68|\uD83C[\uDF3E\uDF73\uDF7C\uDF93\uDFA4\uDFA8\uDFEB\uDFED]|\uD83D(?:[\uDC68\uDC69]\u200D\uD83D(?:\uDC66(?:\u200D\uD83D\uDC66)?|\uDC67(?:\u200D\uD83D[\uDC66\uDC67])?)|[\uDCBB\uDCBC\uDD27\uDD2C\uDE80\uDE92]|\uDC66(?:\u200D\uD83D\uDC66)?|\uDC67(?:\u200D\uD83D[\uDC66\uDC67])?)|\uD83E[\uDDAF-\uDDB3\uDDBC\uDDBD])|\uD83C(?:\uDFFB(?:\u200D(?:[\u2695\u2696\u2708]\uFE0F?|\u2764\uFE0F?\u200D\uD83D(?:\uDC8B\u200D\uD83D)?\uDC68\uD83C[\uDFFB-\uDFFF]|\uD83C[\uDF3E\uDF73\uDF7C\uDF93\uDFA4\uDFA8\uDFEB\uDFED]|\uD83D[\uDCBB\uDCBC\uDD27\uDD2C\uDE80\uDE92]|\uD83E(?:[\uDDAF-\uDDB3\uDDBC\uDDBD]|\uDD1D\u200D\uD83D\uDC68\uD83C[\uDFFC-\uDFFF])))?|\uDFFC(?:\u200D(?:[\u2695\u2696\u2708]\uFE0F?|\u2764\uFE0F?\u200D\uD83D(?:\uDC8B\u200D\uD83D)?\uDC68\uD83C[\uDFFB-\uDFFF]|\uD83C[\uDF3E\uDF73\uDF7C\uDF93\uDFA4\uDFA8\uDFEB\uDFED]|\uD83D[\uDCBB\uDCBC\uDD27\uDD2C\uDE80\uDE92]|\uD83E(?:[\uDDAF-\uDDB3\uDDBC\uDDBD]|\uDD1D\u200D\uD83D\uDC68\uD83C[\uDFFB\uDFFD-\uDFFF])))?|\uDFFD(?:\u200D(?:[\u2695\u2696\u2708]\uFE0F?|\u2764\uFE0F?\u200D\uD83D(?:\uDC8B\u200D\uD83D)?\uDC68\uD83C[\uDFFB-\uDFFF]|\uD83C[\uDF3E\uDF73\uDF7C\uDF93\uDFA4\uDFA8\uDFEB\uDFED]|\uD83D[\uDCBB\uDCBC\uDD27\uDD2C\uDE80\uDE92]|\uD83E(?:[\uDDAF-\uDDB3\uDDBC\uDDBD]|\uDD1D\u200D\uD83D\uDC68\uD83C[\uDFFB\uDFFC\uDFFE\uDFFF])))?|\uDFFE(?:\u200D(?:[\u2695\u2696\u2708]\uFE0F?|\u2764\uFE0F?\u200D\uD83D(?:\uDC8B\u200D\uD83D)?\uDC68\uD83C[\uDFFB-\uDFFF]|\uD83C[\uDF3E\uDF73\uDF7C\uDF93\uDFA4\uDFA8\uDFEB\uDFED]|\uD83D[\uDCBB\uDCBC\uDD27\uDD2C\uDE80\uDE92]|\uD83E(?:[\uDDAF-\uDDB3\uDDBC\uDDBD]|\uDD1D\u200D\uD83D\uDC68\uD83C[\uDFFB-\uDFFD\uDFFF])))?|\uDFFF(?:\u200D(?:[\u2695\u2696\u2708]\uFE0F?|\u2764\uFE0F?\u200D\uD83D(?:\uDC8B\u200D\uD83D)?\uDC68\uD83C[\uDFFB-\uDFFF]|\uD83C[\uDF3E\uDF73\uDF7C\uDF93\uDFA4\uDFA8\uDFEB\uDFED]|\uD83D[\uDCBB\uDCBC\uDD27\uDD2C\uDE80\uDE92]|\uD83E(?:[\uDDAF-\uDDB3\uDDBC\uDDBD]|\uDD1D\u200D\uD83D\uDC68\uD83C[\uDFFB-\uDFFE])))?))?|\uDC69(?:\u200D(?:[\u2695\u2696\u2708]\uFE0F?|\u2764\uFE0F?\u200D\uD83D(?:\uDC8B\u200D\uD83D)?[\uDC68\uDC69]|\uD83C[\uDF3E\uDF73\uDF7C\uDF93\uDFA4\uDFA8\uDFEB\uDFED]|\uD83D(?:[\uDCBB\uDCBC\uDD27\uDD2C\uDE80\uDE92]|\uDC66(?:\u200D\uD83D\uDC66)?|\uDC67(?:\u200D\uD83D[\uDC66\uDC67])?|\uDC69\u200D\uD83D(?:\uDC66(?:\u200D\uD83D\uDC66)?|\uDC67(?:\u200D\uD83D[\uDC66\uDC67])?))|\uD83E[\uDDAF-\uDDB3\uDDBC\uDDBD])|\uD83C(?:\uDFFB(?:\u200D(?:[\u2695\u2696\u2708]\uFE0F?|\u2764\uFE0F?\u200D\uD83D(?:[\uDC68\uDC69]|\uDC8B\u200D\uD83D[\uDC68\uDC69])\uD83C[\uDFFB-\uDFFF]|\uD83C[\uDF3E\uDF73\uDF7C\uDF93\uDFA4\uDFA8\uDFEB\uDFED]|\uD83D[\uDCBB\uDCBC\uDD27\uDD2C\uDE80\uDE92]|\uD83E(?:[\uDDAF-\uDDB3\uDDBC\uDDBD]|\uDD1D\u200D\uD83D[\uDC68\uDC69]\uD83C[\uDFFC-\uDFFF])))?|\uDFFC(?:\u200D(?:[\u2695\u2696\u2708]\uFE0F?|\u2764\uFE0F?\u200D\uD83D(?:[\uDC68\uDC69]|\uDC8B\u200D\uD83D[\uDC68\uDC69])\uD83C[\uDFFB-\uDFFF]|\uD83C[\uDF3E\uDF73\uDF7C\uDF93\uDFA4\uDFA8\uDFEB\uDFED]|\uD83D[\uDCBB\uDCBC\uDD27\uDD2C\uDE80\uDE92]|\uD83E(?:[\uDDAF-\uDDB3\uDDBC\uDDBD]|\uDD1D\u200D\uD83D[\uDC68\uDC69]\uD83C[\uDFFB\uDFFD-\uDFFF])))?|\uDFFD(?:\u200D(?:[\u2695\u2696\u2708]\uFE0F?|\u2764\uFE0F?\u200D\uD83D(?:[\uDC68\uDC69]|\uDC8B\u200D\uD83D[\uDC68\uDC69])\uD83C[\uDFFB-\uDFFF]|\uD83C[\uDF3E\uDF73\uDF7C\uDF93\uDFA4\uDFA8\uDFEB\uDFED]|\uD83D[\uDCBB\uDCBC\uDD27\uDD2C\uDE80\uDE92]|\uD83E(?:[\uDDAF-\uDDB3\uDDBC\uDDBD]|\uDD1D\u200D\uD83D[\uDC68\uDC69]\uD83C[\uDFFB\uDFFC\uDFFE\uDFFF])))?|\uDFFE(?:\u200D(?:[\u2695\u2696\u2708]\uFE0F?|\u2764\uFE0F?\u200D\uD83D(?:[\uDC68\uDC69]|\uDC8B\u200D\uD83D[\uDC68\uDC69])\uD83C[\uDFFB-\uDFFF]|\uD83C[\uDF3E\uDF73\uDF7C\uDF93\uDFA4\uDFA8\uDFEB\uDFED]|\uD83D[\uDCBB\uDCBC\uDD27\uDD2C\uDE80\uDE92]|\uD83E(?:[\uDDAF-\uDDB3\uDDBC\uDDBD]|\uDD1D\u200D\uD83D[\uDC68\uDC69]\uD83C[\uDFFB-\uDFFD\uDFFF])))?|\uDFFF(?:\u200D(?:[\u2695\u2696\u2708]\uFE0F?|\u2764\uFE0F?\u200D\uD83D(?:[\uDC68\uDC69]|\uDC8B\u200D\uD83D[\uDC68\uDC69])\uD83C[\uDFFB-\uDFFF]|\uD83C[\uDF3E\uDF73\uDF7C\uDF93\uDFA4\uDFA8\uDFEB\uDFED]|\uD83D[\uDCBB\uDCBC\uDD27\uDD2C\uDE80\uDE92]|\uD83E(?:[\uDDAF-\uDDB3\uDDBC\uDDBD]|\uDD1D\u200D\uD83D[\uDC68\uDC69]\uD83C[\uDFFB-\uDFFE])))?))?|\uDC6F(?:\u200D[\u2640\u2642]\uFE0F?)?|\uDD75(?:\uFE0F|\uD83C[\uDFFB-\uDFFF])?(?:\u200D[\u2640\u2642]\uFE0F?)?|\uDE2E(?:\u200D\uD83D\uDCA8)?|\uDE35(?:\u200D\uD83D\uDCAB)?|\uDE36(?:\u200D\uD83C\uDF2B\uFE0F?)?)|\uD83E(?:[\uDD0C\uDD0F\uDD18-\uDD1F\uDD30-\uDD34\uDD36\uDD77\uDDB5\uDDB6\uDDBB\uDDD2\uDDD3\uDDD5\uDEC3-\uDEC5\uDEF0\uDEF2-\uDEF8](?:\uD83C[\uDFFB-\uDFFF])?|[\uDD26\uDD35\uDD37-\uDD39\uDD3D\uDD3E\uDDB8\uDDB9\uDDCD-\uDDCF\uDDD4\uDDD6-\uDDDD](?:\uD83C[\uDFFB-\uDFFF])?(?:\u200D[\u2640\u2642]\uFE0F?)?|[\uDDDE\uDDDF](?:\u200D[\u2640\u2642]\uFE0F?)?|[\uDD0D\uDD0E\uDD10-\uDD17\uDD20-\uDD25\uDD27-\uDD2F\uDD3A\uDD3F-\uDD45\uDD47-\uDD76\uDD78-\uDDB4\uDDB7\uDDBA\uDDBC-\uDDCC\uDDD0\uDDE0-\uDDFF\uDE70-\uDE7C\uDE80-\uDE88\uDE90-\uDEBD\uDEBF-\uDEC2\uDECE-\uDEDB\uDEE0-\uDEE8]|\uDD3C(?:\u200D[\u2640\u2642]\uFE0F?|\uD83C[\uDFFB-\uDFFF])?|\uDDD1(?:\u200D(?:[\u2695\u2696\u2708]\uFE0F?|\uD83C[\uDF3E\uDF73\uDF7C\uDF84\uDF93\uDFA4\uDFA8\uDFEB\uDFED]|\uD83D[\uDCBB\uDCBC\uDD27\uDD2C\uDE80\uDE92]|\uD83E(?:[\uDDAF-\uDDB3\uDDBC\uDDBD]|\uDD1D\u200D\uD83E\uDDD1))|\uD83C(?:\uDFFB(?:\u200D(?:[\u2695\u2696\u2708]\uFE0F?|\u2764\uFE0F?\u200D(?:\uD83D\uDC8B\u200D)?\uD83E\uDDD1\uD83C[\uDFFC-\uDFFF]|\uD83C[\uDF3E\uDF73\uDF7C\uDF84\uDF93\uDFA4\uDFA8\uDFEB\uDFED]|\uD83D[\uDCBB\uDCBC\uDD27\uDD2C\uDE80\uDE92]|\uD83E(?:[\uDDAF-\uDDB3\uDDBC\uDDBD]|\uDD1D\u200D\uD83E\uDDD1\uD83C[\uDFFB-\uDFFF])))?|\uDFFC(?:\u200D(?:[\u2695\u2696\u2708]\uFE0F?|\u2764\uFE0F?\u200D(?:\uD83D\uDC8B\u200D)?\uD83E\uDDD1\uD83C[\uDFFB\uDFFD-\uDFFF]|\uD83C[\uDF3E\uDF73\uDF7C\uDF84\uDF93\uDFA4\uDFA8\uDFEB\uDFED]|\uD83D[\uDCBB\uDCBC\uDD27\uDD2C\uDE80\uDE92]|\uD83E(?:[\uDDAF-\uDDB3\uDDBC\uDDBD]|\uDD1D\u200D\uD83E\uDDD1\uD83C[\uDFFB-\uDFFF])))?|\uDFFD(?:\u200D(?:[\u2695\u2696\u2708]\uFE0F?|\u2764\uFE0F?\u200D(?:\uD83D\uDC8B\u200D)?\uD83E\uDDD1\uD83C[\uDFFB\uDFFC\uDFFE\uDFFF]|\uD83C[\uDF3E\uDF73\uDF7C\uDF84\uDF93\uDFA4\uDFA8\uDFEB\uDFED]|\uD83D[\uDCBB\uDCBC\uDD27\uDD2C\uDE80\uDE92]|\uD83E(?:[\uDDAF-\uDDB3\uDDBC\uDDBD]|\uDD1D\u200D\uD83E\uDDD1\uD83C[\uDFFB-\uDFFF])))?|\uDFFE(?:\u200D(?:[\u2695\u2696\u2708]\uFE0F?|\u2764\uFE0F?\u200D(?:\uD83D\uDC8B\u200D)?\uD83E\uDDD1\uD83C[\uDFFB-\uDFFD\uDFFF]|\uD83C[\uDF3E\uDF73\uDF7C\uDF84\uDF93\uDFA4\uDFA8\uDFEB\uDFED]|\uD83D[\uDCBB\uDCBC\uDD27\uDD2C\uDE80\uDE92]|\uD83E(?:[\uDDAF-\uDDB3\uDDBC\uDDBD]|\uDD1D\u200D\uD83E\uDDD1\uD83C[\uDFFB-\uDFFF])))?|\uDFFF(?:\u200D(?:[\u2695\u2696\u2708]\uFE0F?|\u2764\uFE0F?\u200D(?:\uD83D\uDC8B\u200D)?\uD83E\uDDD1\uD83C[\uDFFB-\uDFFE]|\uD83C[\uDF3E\uDF73\uDF7C\uDF84\uDF93\uDFA4\uDFA8\uDFEB\uDFED]|\uD83D[\uDCBB\uDCBC\uDD27\uDD2C\uDE80\uDE92]|\uD83E(?:[\uDDAF-\uDDB3\uDDBC\uDDBD]|\uDD1D\u200D\uD83E\uDDD1\uD83C[\uDFFB-\uDFFF])))?))?|\uDEF1(?:\uD83C(?:\uDFFB(?:\u200D\uD83E\uDEF2\uD83C[\uDFFC-\uDFFF])?|\uDFFC(?:\u200D\uD83E\uDEF2\uD83C[\uDFFB\uDFFD-\uDFFF])?|\uDFFD(?:\u200D\uD83E\uDEF2\uD83C[\uDFFB\uDFFC\uDFFE\uDFFF])?|\uDFFE(?:\u200D\uD83E\uDEF2\uD83C[\uDFFB-\uDFFD\uDFFF])?|\uDFFF(?:\u200D\uD83E\uDEF2\uD83C[\uDFFB-\uDFFE])?))?)/g;
};

var removeMarkdown = function(md, options) {
  options = options || {};
  options.listUnicodeChar = options.hasOwnProperty('listUnicodeChar') ? options.listUnicodeChar : false;
  options.stripListLeaders = options.hasOwnProperty('stripListLeaders') ? options.stripListLeaders : true;
  options.gfm = options.hasOwnProperty('gfm') ? options.gfm : true;
  options.useImgAltText = options.hasOwnProperty('useImgAltText') ? options.useImgAltText : true;
  options.abbr = options.hasOwnProperty('abbr') ? options.abbr : false;
  options.replaceLinksWithURL = options.hasOwnProperty('replaceLinksWithURL') ? options.replaceLinksWithURL : false;
  options.htmlTagsToSkip = options.hasOwnProperty('htmlTagsToSkip') ? options.htmlTagsToSkip : [];
  options.throwError = options.hasOwnProperty('throwError') ? options.throwError : false;

  var output = md || '';

  // Remove horizontal rules (stripListHeaders conflict with this rule, which is why it has been moved to the top)
  output = output.replace(/^(-\s*?|\*\s*?|_\s*?){3,}\s*/gm, '');

  try {
    if (options.stripListLeaders) {
      if (options.listUnicodeChar)
        output = output.replace(/^([\s\t]*)([\*\-\+]|\d+\.)\s+/gm, options.listUnicodeChar + ' $1');
      else
        output = output.replace(/^([\s\t]*)([\*\-\+]|\d+\.)\s+/gm, '$1');
    }
    if (options.gfm) {
      output = output
      // Header
        .replace(/\n={2,}/g, '\n')
        // Fenced codeblocks
        .replace(/~{3}.*\n/g, '')
        // Strikethrough
        .replace(/~~/g, '')
        // Fenced codeblocks
        .replace(/`{3}.*\n/g, '');
    }
    if (options.abbr) {
      // Remove abbreviations
      output = output.replace(/\*\[.*\]:.*\n/, '');
    }
    output = output
    // Remove HTML tags
      .replace(/<[^>]*>/g, '');

    var htmlReplaceRegex = new RegExp('<[^>]*>', 'g');
    if (options.htmlTagsToSkip.length > 0) {
      // Using negative lookahead. Eg. (?!sup|sub) will not match 'sup' and 'sub' tags.
      var joinedHtmlTagsToSkip = '(?!' + options.htmlTagsToSkip.join("|") + ')';

      // Adding the lookahead literal with the default regex for html. Eg./<(?!sup|sub)[^>]*>/ig
      htmlReplaceRegex = new RegExp(
          '<' +
          joinedHtmlTagsToSkip +
          '[^>]*>', 
          'ig'
      );
    }

    output = output
      // Remove HTML tags
      .replace(htmlReplaceRegex, '')
      // Remove setext-style headers
      .replace(/^[=\-]{2,}\s*$/g, '')
      // Remove footnotes?
      .replace(/\[\^.+?\](\: .*?$)?/g, '')
      .replace(/\s{0,2}\[.*?\]: .*?$/g, '')
      // Remove images
      .replace(/\!\[(.*?)\][\[\(].*?[\]\)]/g, options.useImgAltText ? '$1' : '')
      // Remove inline links
      .replace(/\[([^\]]*?)\][\[\(].*?[\]\)]/g, options.replaceLinksWithURL ? '$2' : '$1')
      // Remove blockquotes
      .replace(/^(\n)?\s{0,3}>\s?/gm, '$1')
      // .replace(/(^|\n)\s{0,3}>\s?/g, '\n\n')
      // Remove reference-style links?
      .replace(/^\s{1,2}\[(.*?)\]: (\S+)( ".*?")?\s*$/g, '')
      // Remove atx-style headers
      .replace(/^(\n)?\s{0,}#{1,6}\s*( (.+))? +#+$|^(\n)?\s{0,}#{1,6}\s*( (.+))?$/gm, '$1$3$4$6')
      // Remove * emphasis
      .replace(/([\*]+)(\S)(.*?\S)??\1/g, '$2$3')
      // Remove _ emphasis. Unlike *, _ emphasis gets rendered only if 
      //   1. Either there is a whitespace character before opening _ and after closing _.
      //   2. Or _ is at the start/end of the string.
      .replace(/(^|\W)([_]+)(\S)(.*?\S)??\2($|\W)/g, '$1$3$4$5')
      // Remove code blocks
      .replace(/(`{3,})(.*?)\1/gm, '$2')
      // Remove inline code
      .replace(/`(.+?)`/g, '$1')
      // // Replace two or more newlines with exactly two? Not entirely sure this belongs here...
      // .replace(/\n{2,}/g, '\n\n')
      // // Remove newlines in a paragraph
      // .replace(/(\S+)\n\s*(\S+)/g, '$1 $2')
      // Replace strike through
      .replace(/~(.*?)~/g, '$1');
  } catch(e) {
    if (options.throwError) throw e;

    console.error("remove-markdown encountered error: %s", e);
    return md;
  }
  return output;
};

var removeMd = /*@__PURE__*/getDefaultExportFromCjs(removeMarkdown);

/** Normalize a duration to all of the proper units. */
function normalizeDuration(dur) {
    if (dur === undefined || dur === null)
        return dur;
    return dur.shiftToAll().normalize();
}
/** Strip the time components of a date time object. */
function stripTime(dt) {
    if (dt === null || dt === undefined)
        return dt;
    return DateTime.fromObject({
        year: dt.year,
        month: dt.month,
        day: dt.day,
    });
}
/** Get the folder containing the given path (i.e., like computing 'path/..'). */
function getParentFolder(path) {
    return path.split("/").slice(0, -1).join("/");
}
/** Get the "title" for a file, by stripping other parts of the path as well as the extension. */
function getFileTitle(path) {
    if (path.includes("/"))
        path = path.substring(path.lastIndexOf("/") + 1);
    if (path.endsWith(".md"))
        path = path.substring(0, path.length - 3);
    return path;
}
/** Get the extension of a file from the file path. */
function getExtension(path) {
    if (!path.includes("."))
        return "";
    return path.substring(path.lastIndexOf(".") + 1);
}
/** Parse all subtags out of the given tag. I.e., #hello/i/am would yield [#hello/i/am, #hello/i, #hello]. */
function extractSubtags(tag) {
    let result = [tag];
    while (tag.includes("/")) {
        tag = tag.substring(0, tag.lastIndexOf("/"));
        result.push(tag);
    }
    return result;
}
/** Try calling the given function; on failure, return the error message.  */
function tryOrPropagate(func) {
    try {
        return func();
    }
    catch (error) {
        return Result.failure("" + error + "\n\n" + error.stack);
    }
}
/** Try asynchronously calling the given function; on failure, return the error message. */
async function asyncTryOrPropagate(func) {
    try {
        return await func();
    }
    catch (error) {
        return Result.failure("" + error + "\n\n" + error.stack);
    }
}
/**
 * Escape regex characters in a string.
 * See https://developer.mozilla.org/en-US/docs/Web/JavaScript/Guide/Regular_Expressions.
 */
function escapeRegex(str) {
    return str.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");
}
/** A parsimmon parser which canonicalizes variable names while properly respecting emoji. */
const VAR_NAME_CANONICALIZER = parsimmon_umd_minExports.alt(parsimmon_umd_minExports.regex(new RegExp(emojiRegex(), "")), parsimmon_umd_minExports.regex(/[0-9\p{Letter}_-]+/u).map(str => str.toLocaleLowerCase()), parsimmon_umd_minExports.whitespace.map(_ => "-"), parsimmon_umd_minExports.any.map(_ => ""))
    .many()
    .map(result => result.join(""));
/** Convert an arbitrary variable name into something JS/query friendly. */
function canonicalizeVarName(name) {
    return VAR_NAME_CANONICALIZER.tryParse(name);
}
const HEADER_CANONICALIZER = parsimmon_umd_minExports.alt(parsimmon_umd_minExports.regex(new RegExp(emojiRegex(), "")), parsimmon_umd_minExports.regex(/[0-9\p{Letter}_-]+/u), parsimmon_umd_minExports.whitespace.map(_ => " "), parsimmon_umd_minExports.any.map(_ => " "))
    .many()
    .map(result => {
    return result.join("").split(/\s+/).join(" ").trim();
});
/**
 * Normalizes the text in a header to be something that is actually linkable to. This mimics
 * how Obsidian does it's normalization, collapsing repeated spaces and stripping out control characters.
 */
function normalizeHeaderForLink(header) {
    return HEADER_CANONICALIZER.tryParse(header);
}
/** Render a DateTime in a minimal format to save space. */
function renderMinimalDate(time, settings, locale) {
    // If there is no relevant time specified, fall back to just rendering the date.
    if (time.second == 0 && time.minute == 0 && time.hour == 0) {
        return time.toLocal().toFormat(settings.defaultDateFormat, { locale });
    }
    return time.toLocal().toFormat(settings.defaultDateTimeFormat, { locale });
}
/** Render a duration in a minimal format to save space. */
function renderMinimalDuration(dur) {
    dur = normalizeDuration(dur);
    // toHuman outputs zero quantities e.g. "0 seconds"
    dur = Duration.fromObject(Object.fromEntries(Object.entries(dur.toObject()).filter(([, quantity]) => quantity != 0)));
    return dur.toHuman();
}
/** Determine if two sets are equal in contents. */
function setsEqual(first, second) {
    if (first.size != second.size)
        return false;
    for (let elem of first)
        if (!second.has(elem))
            return false;
    return true;
}
/** Normalize a markdown string. Removes all markdown tags and obsidian links. */
function normalizeMarkdown(str) {
    // [[test]] -> test
    let interim = str.replace(/\[\[([^\|]*?)\]\]/g, "$1");
    // [[test|test]] -> test
    interim = interim.replace(/\[\[.*?\|(.*?)\]\]/, "$1");
    // remove markdown tags
    interim = removeMd(interim);
    return interim;
}

var Values;
(function (Values) {
    /** Convert an arbitrary value into a reasonable, Markdown-friendly string if possible. */
    function toString(field, setting = DEFAULT_QUERY_SETTINGS, recursive = false) {
        let wrapped = wrapValue(field);
        if (!wrapped)
            return setting.renderNullAs;
        switch (wrapped.type) {
            case "null":
                return setting.renderNullAs;
            case "string":
                return wrapped.value;
            case "number":
            case "boolean":
                return "" + wrapped.value;
            case "html":
                return wrapped.value.outerHTML;
            case "widget":
                return wrapped.value.markdown();
            case "link":
                return wrapped.value.markdown();
            case "function":
                return "<function>";
            case "array":
                let result = "";
                if (recursive)
                    result += "[";
                result += wrapped.value.map(f => toString(f, setting, true)).join(", ");
                if (recursive)
                    result += "]";
                return result;
            case "object":
                return ("{ " +
                    Object.entries(wrapped.value)
                        .map(e => e[0] + ": " + toString(e[1], setting, true))
                        .join(", ") +
                    " }");
            case "date":
                if (wrapped.value.second == 0 && wrapped.value.hour == 0 && wrapped.value.minute == 0) {
                    return wrapped.value.toFormat(setting.defaultDateFormat);
                }
                return wrapped.value.toFormat(setting.defaultDateTimeFormat);
            case "duration":
                return renderMinimalDuration(wrapped.value);
        }
    }
    Values.toString = toString;
    /** Wrap a literal value so you can switch on it easily. */
    function wrapValue(val) {
        if (isNull(val))
            return { type: "null", value: val };
        else if (isNumber(val))
            return { type: "number", value: val };
        else if (isString(val))
            return { type: "string", value: val };
        else if (isBoolean(val))
            return { type: "boolean", value: val };
        else if (isDuration(val))
            return { type: "duration", value: val };
        else if (isDate(val))
            return { type: "date", value: val };
        else if (isWidget(val))
            return { type: "widget", value: val };
        else if (isArray(val))
            return { type: "array", value: val };
        else if (isLink(val))
            return { type: "link", value: val };
        else if (isFunction(val))
            return { type: "function", value: val };
        else if (isHtml(val))
            return { type: "html", value: val };
        else if (isObject(val))
            return { type: "object", value: val };
        else
            return undefined;
    }
    Values.wrapValue = wrapValue;
    /** Recursively map complex objects at the leaves. */
    function mapLeaves(val, func) {
        if (isObject(val)) {
            let result = {};
            for (let [key, value] of Object.entries(val))
                result[key] = mapLeaves(value, func);
            return result;
        }
        else if (isArray(val)) {
            let result = [];
            for (let value of val)
                result.push(mapLeaves(value, func));
            return result;
        }
        else {
            return func(val);
        }
    }
    Values.mapLeaves = mapLeaves;
    /** Compare two arbitrary JavaScript values. Produces a total ordering over ANY possible dataview value. */
    function compareValue(val1, val2, linkNormalizer) {
        // Handle undefined/nulls first.
        if (val1 === undefined)
            val1 = null;
        if (val2 === undefined)
            val2 = null;
        if (val1 === null && val2 === null)
            return 0;
        else if (val1 === null)
            return -1;
        else if (val2 === null)
            return 1;
        // A non-null value now which we can wrap & compare on.
        let wrap1 = wrapValue(val1);
        let wrap2 = wrapValue(val2);
        if (wrap1 === undefined && wrap2 === undefined)
            return 0;
        else if (wrap1 === undefined)
            return -1;
        else if (wrap2 === undefined)
            return 1;
        // Short-circuit on different types or on reference equality.
        if (wrap1.type != wrap2.type)
            return wrap1.type.localeCompare(wrap2.type);
        if (wrap1.value === wrap2.value)
            return 0;
        switch (wrap1.type) {
            case "string":
                return wrap1.value.localeCompare(wrap2.value);
            case "number":
                if (wrap1.value < wrap2.value)
                    return -1;
                else if (wrap1.value == wrap2.value)
                    return 0;
                return 1;
            case "null":
                return 0;
            case "boolean":
                if (wrap1.value == wrap2.value)
                    return 0;
                else
                    return wrap1.value ? 1 : -1;
            case "link":
                let link1 = wrap1.value;
                let link2 = wrap2.value;
                let normalize = linkNormalizer ?? ((x) => x);
                // We can't compare by file name or display, since that would break link equality. Compare by path.
                let pathCompare = normalize(link1.path).localeCompare(normalize(link2.path));
                if (pathCompare != 0)
                    return pathCompare;
                // Then compare by type.
                let typeCompare = link1.type.localeCompare(link2.type);
                if (typeCompare != 0)
                    return typeCompare;
                // Then compare by subpath existence.
                if (link1.subpath && !link2.subpath)
                    return 1;
                if (!link1.subpath && link2.subpath)
                    return -1;
                if (!link1.subpath && !link2.subpath)
                    return 0;
                // Since both have a subpath, compare by subpath.
                return (link1.subpath ?? "").localeCompare(link2.subpath ?? "");
            case "date":
                return wrap1.value < wrap2.value
                    ? -1
                    : wrap1.value.equals(wrap2.value)
                        ? 0
                        : 1;
            case "duration":
                return wrap1.value < wrap2.value
                    ? -1
                    : wrap1.value.equals(wrap2.value)
                        ? 0
                        : 1;
            case "array":
                let f1 = wrap1.value;
                let f2 = wrap2.value;
                for (let index = 0; index < Math.min(f1.length, f2.length); index++) {
                    let comp = compareValue(f1[index], f2[index]);
                    if (comp != 0)
                        return comp;
                }
                return f1.length - f2.length;
            case "object":
                let o1 = wrap1.value;
                let o2 = wrap2.value;
                let k1 = Array.from(Object.keys(o1));
                let k2 = Array.from(Object.keys(o2));
                k1.sort();
                k2.sort();
                let keyCompare = compareValue(k1, k2);
                if (keyCompare != 0)
                    return keyCompare;
                for (let key of k1) {
                    let comp = compareValue(o1[key], o2[key]);
                    if (comp != 0)
                        return comp;
                }
                return 0;
            case "widget":
            case "html":
            case "function":
                return 0;
        }
    }
    Values.compareValue = compareValue;
    /** Find the corresponding Dataview type for an arbitrary value. */
    function typeOf(val) {
        return wrapValue(val)?.type;
    }
    Values.typeOf = typeOf;
    /** Determine if the given value is "truthy" (i.e., is non-null and has data in it). */
    function isTruthy(field) {
        let wrapped = wrapValue(field);
        if (!wrapped)
            return false;
        switch (wrapped.type) {
            case "number":
                return wrapped.value != 0;
            case "string":
                return wrapped.value.length > 0;
            case "boolean":
                return wrapped.value;
            case "link":
                return !!wrapped.value.path;
            case "date":
                return wrapped.value.toMillis() != 0;
            case "duration":
                return wrapped.value.as("seconds") != 0;
            case "object":
                return Object.keys(wrapped.value).length > 0;
            case "array":
                return wrapped.value.length > 0;
            case "null":
                return false;
            case "html":
            case "widget":
            case "function":
                return true;
        }
    }
    Values.isTruthy = isTruthy;
    /** Deep copy a field. */
    function deepCopy(field) {
        if (field === null || field === undefined)
            return field;
        if (Values.isArray(field)) {
            return [].concat(field.map(v => deepCopy(v)));
        }
        else if (Values.isObject(field)) {
            let result = {};
            for (let [key, value] of Object.entries(field))
                result[key] = deepCopy(value);
            return result;
        }
        else {
            return field;
        }
    }
    Values.deepCopy = deepCopy;
    function isString(val) {
        return typeof val == "string";
    }
    Values.isString = isString;
    function isNumber(val) {
        return typeof val == "number";
    }
    Values.isNumber = isNumber;
    function isDate(val) {
        return val instanceof DateTime;
    }
    Values.isDate = isDate;
    function isDuration(val) {
        return val instanceof Duration;
    }
    Values.isDuration = isDuration;
    function isNull(val) {
        return val === null || val === undefined;
    }
    Values.isNull = isNull;
    function isArray(val) {
        return Array.isArray(val);
    }
    Values.isArray = isArray;
    function isBoolean(val) {
        return typeof val === "boolean";
    }
    Values.isBoolean = isBoolean;
    function isLink(val) {
        return val instanceof Link;
    }
    Values.isLink = isLink;
    function isWidget(val) {
        return val instanceof Widget;
    }
    Values.isWidget = isWidget;
    function isHtml(val) {
        if (typeof HTMLElement !== "undefined") {
            return val instanceof HTMLElement;
        }
        else {
            return false;
        }
    }
    Values.isHtml = isHtml;
    /** Checks if the given value is an object (and not any other dataview-recognized object-like type). */
    function isObject(val) {
        return (typeof val == "object" &&
            !isHtml(val) &&
            !isWidget(val) &&
            !isArray(val) &&
            !isDuration(val) &&
            !isDate(val) &&
            !isLink(val) &&
            val !== undefined &&
            !isNull(val));
    }
    Values.isObject = isObject;
    function isFunction(val) {
        return typeof val == "function";
    }
    Values.isFunction = isFunction;
})(Values || (Values = {}));
///////////////
// Groupings //
///////////////
var Groupings;
(function (Groupings) {
    /** Determines if the given group entry is a standalone value, or a grouping of sub-entries. */
    function isElementGroup(entry) {
        return Values.isObject(entry) && Object.keys(entry).length == 2 && "key" in entry && "rows" in entry;
    }
    Groupings.isElementGroup = isElementGroup;
    /** Determines if the given array is a grouping array. */
    function isGrouping(entry) {
        for (let element of entry)
            if (!isElementGroup(element))
                return false;
        return true;
    }
    Groupings.isGrouping = isGrouping;
    /** Count the total number of elements in a recursive grouping. */
    function count(elements) {
        if (isGrouping(elements)) {
            let result = 0;
            for (let subgroup of elements)
                result += count(subgroup.rows);
            return result;
        }
        else {
            return elements.length;
        }
    }
    Groupings.count = count;
})(Groupings || (Groupings = {}));
//////////
// LINK //
//////////
/** The Obsidian 'link', used for uniquely describing a file, header, or block. */
class Link {
    /** The file path this link points to. */
    path;
    /** The display name associated with the link. */
    display;
    /** The block ID or header this link points to within a file, if relevant. */
    subpath;
    /** Is this link an embedded link (!)? */
    embed;
    /** The type of this link, which determines what 'subpath' refers to, if anything. */
    type;
    /** Create a link to a specific file. */
    static file(path, embed = false, display) {
        return new Link({
            path,
            embed,
            display,
            subpath: undefined,
            type: "file",
        });
    }
    static infer(linkpath, embed = false, display) {
        if (linkpath.includes("#^")) {
            let split = linkpath.split("#^");
            return Link.block(split[0], split[1], embed, display);
        }
        else if (linkpath.includes("#")) {
            let split = linkpath.split("#");
            return Link.header(split[0], split[1], embed, display);
        }
        else
            return Link.file(linkpath, embed, display);
    }
    /** Create a link to a specific file and header in that file. */
    static header(path, header, embed, display) {
        // Headers need to be normalized to alpha-numeric & with extra spacing removed.
        return new Link({
            path,
            embed,
            display,
            subpath: normalizeHeaderForLink(header),
            type: "header",
        });
    }
    /** Create a link to a specific file and block in that file. */
    static block(path, blockId, embed, display) {
        return new Link({
            path,
            embed,
            display,
            subpath: blockId,
            type: "block",
        });
    }
    static fromObject(object) {
        return new Link(object);
    }
    constructor(fields) {
        Object.assign(this, fields);
    }
    /** Checks for link equality (i.e., that the links are pointing to the same exact location). */
    equals(other) {
        if (other == undefined || other == null)
            return false;
        return this.path == other.path && this.type == other.type && this.subpath == other.subpath;
    }
    /** Convert this link to it's markdown representation. */
    toString() {
        return this.markdown();
    }
    /** Convert this link to a raw object which is serialization-friendly. */
    toObject() {
        return { path: this.path, type: this.type, subpath: this.subpath, display: this.display, embed: this.embed };
    }
    /** Update this link with a new path. */
    //@ts-ignore; error appeared after updating Obsidian to 0.15.4; it also updated other packages but didn't say which
    withPath(path) {
        return new Link(Object.assign({}, this, { path }));
    }
    /** Return a new link which points to the same location but with a new display value. */
    withDisplay(display) {
        return new Link(Object.assign({}, this, { display }));
    }
    /** Convert a file link into a link to a specific header. */
    withHeader(header) {
        return Link.header(this.path, header, this.embed, this.display);
    }
    /** Convert any link into a link to its file. */
    toFile() {
        return Link.file(this.path, this.embed, this.display);
    }
    /** Convert this link into an embedded link. */
    toEmbed() {
        if (this.embed) {
            return this;
        }
        else {
            let link = new Link(this);
            link.embed = true;
            return link;
        }
    }
    /** Convert this link into a non-embedded link. */
    fromEmbed() {
        if (!this.embed) {
            return this;
        }
        else {
            let link = new Link(this);
            link.embed = false;
            return link;
        }
    }
    /** Convert this link to markdown so it can be rendered. */
    markdown() {
        let result = (this.embed ? "!" : "") + "[[" + this.obsidianLink();
        if (this.display) {
            result += "|" + this.display;
        }
        else {
            result += "|" + getFileTitle(this.path);
            if (this.type == "header" || this.type == "block")
                result += " > " + this.subpath;
        }
        result += "]]";
        return result;
    }
    /** Convert the inner part of the link to something that Obsidian can open / understand. */
    obsidianLink() {
        const escaped = this.path.replaceAll("|", "\\|");
        if (this.type == "header")
            return escaped + "#" + this.subpath?.replaceAll("|", "\\|");
        if (this.type == "block")
            return escaped + "#^" + this.subpath?.replaceAll("|", "\\|");
        else
            return escaped;
    }
    /** The stripped name of the file this link points to. */
    fileName() {
        return getFileTitle(this.path).replace(".md", "");
    }
}
/////////////////
// WIDGET BASE //
/////////////////
/**
 * A trivial base class which just defines the '$widget' identifier type. Subtypes of
 * widget are responsible for adding whatever metadata is relevant. If you want your widget
 * to have rendering functionality (which you probably do), you should extend `RenderWidget`.
 */
class Widget {
    $widget;
    constructor($widget) {
        this.$widget = $widget;
    }
}
/** A trivial widget which renders a (key, value) pair, and allows accessing the key and value. */
class ListPairWidget extends Widget {
    key;
    value;
    constructor(key, value) {
        super("dataview:list-pair");
        this.key = key;
        this.value = value;
    }
    markdown() {
        return `${Values.toString(this.key)}: ${Values.toString(this.value)}`;
    }
}
/** A simple widget which renders an external link. */
class ExternalLinkWidget extends Widget {
    url;
    display;
    constructor(url, display) {
        super("dataview:external-link");
        this.url = url;
        this.display = display;
    }
    markdown() {
        return `[${this.display ?? this.url}](${this.url})`;
    }
}
var Widgets;
(function (Widgets) {
    /** Create a list pair widget matching the given key and value. */
    function listPair(key, value) {
        return new ListPairWidget(key, value);
    }
    Widgets.listPair = listPair;
    /** Create an external link widget which renders an external Obsidian link. */
    function externalLink(url, display) {
        return new ExternalLinkWidget(url, display);
    }
    Widgets.externalLink = externalLink;
    /** Checks if the given widget is a list pair widget. */
    function isListPair(widget) {
        return widget.$widget === "dataview:list-pair";
    }
    Widgets.isListPair = isListPair;
    function isExternalLink(widget) {
        return widget.$widget === "dataview:external-link";
    }
    Widgets.isExternalLink = isExternalLink;
    /** Determines if the given widget is any kind of built-in widget with special rendering handling. */
    function isBuiltin(widget) {
        return isListPair(widget) || isExternalLink(widget);
    }
    Widgets.isBuiltin = isBuiltin;
})(Widgets || (Widgets = {}));

/** Implementation of DataArray, minus the dynamic variable access, which is implemented via proxy. */
class DataArrayImpl {
    values;
    settings;
    defaultComparator;
    static ARRAY_FUNCTIONS = new Set([
        "where",
        "filter",
        "map",
        "flatMap",
        "mutate",
        "slice",
        "concat",
        "indexOf",
        "limit",
        "find",
        "findIndex",
        "includes",
        "join",
        "sort",
        "sortInPlace",
        "groupBy",
        "groupIn",
        "distinct",
        "every",
        "some",
        "none",
        "first",
        "last",
        "to",
        "into",
        "lwrap",
        "expand",
        "forEach",
        "length",
        "values",
        "array",
        "defaultComparator",
        "toString",
        "settings",
        "sum",
        "avg",
        "min",
        "max",
    ]);
    static ARRAY_PROXY = {
        get: function (target, prop, receiver) {
            if (typeof prop === "symbol")
                return target[prop];
            else if (typeof prop === "number")
                return target.values[prop];
            else if (prop === "constructor")
                return target.values.constructor;
            else if (!isNaN(parseInt(prop)))
                return target.values[parseInt(prop)];
            else if (DataArrayImpl.ARRAY_FUNCTIONS.has(prop.toString()))
                return target[prop.toString()];
            return target.to(prop);
        },
    };
    static wrap(arr, settings, defaultComparator = Values.compareValue) {
        return new Proxy(new DataArrayImpl(arr, settings, defaultComparator), DataArrayImpl.ARRAY_PROXY);
    }
    length;
    constructor(values, settings, defaultComparator = Values.compareValue) {
        this.values = values;
        this.settings = settings;
        this.defaultComparator = defaultComparator;
        this.length = values.length;
    }
    lwrap(values) {
        return DataArrayImpl.wrap(values, this.settings, this.defaultComparator);
    }
    where(predicate) {
        return this.lwrap(this.values.filter(predicate));
    }
    filter(predicate) {
        return this.where(predicate);
    }
    map(f) {
        return this.lwrap(this.values.map(f));
    }
    flatMap(f) {
        let result = [];
        for (let index = 0; index < this.length; index++) {
            let value = f(this.values[index], index, this.values);
            if (!value || value.length == 0)
                continue;
            for (let r of value)
                result.push(r);
        }
        return this.lwrap(result);
    }
    mutate(f) {
        for (let index = 0; index < this.values.length; index++) {
            f(this.values[index], index, this.values);
        }
        return this;
    }
    limit(count) {
        return this.lwrap(this.values.slice(0, count));
    }
    slice(start, end) {
        return this.lwrap(this.values.slice(start, end));
    }
    concat(other) {
        return this.lwrap(this.values.concat(other.values));
    }
    /** Return the first index of the given (optionally starting the search) */
    indexOf(element, fromIndex) {
        return this.findIndex(e => this.defaultComparator(e, element) == 0, fromIndex);
    }
    /** Return the first element that satisfies the given predicate. */
    find(pred) {
        let index = this.findIndex(pred);
        if (index == -1)
            return undefined;
        else
            return this.values[index];
    }
    findIndex(pred, fromIndex) {
        for (let index = fromIndex ?? 0; index < this.length; index++) {
            if (pred(this.values[index], index, this.values))
                return index;
        }
        return -1;
    }
    includes(element) {
        return this.indexOf(element, 0) != -1;
    }
    join(sep) {
        return this.map(s => Values.toString(s, this.settings))
            .array()
            .join(sep ?? ", ");
    }
    sort(key, direction, comparator) {
        if (this.values.length == 0)
            return this;
        let realComparator = comparator ?? this.defaultComparator;
        let realKey = key ?? ((l) => l);
        // Associate each entry with it's index for the key function, and then do a normal sort.
        let copy = [].concat(this.array()).map((elem, index) => {
            return { index: index, value: elem };
        });
        copy.sort((a, b) => {
            let aKey = realKey(a.value, a.index, this.values);
            let bKey = realKey(b.value, b.index, this.values);
            return direction === "desc" ? -realComparator(aKey, bKey) : realComparator(aKey, bKey);
        });
        return this.lwrap(copy.map(e => e.value));
    }
    sortInPlace(key, direction, comparator) {
        if (this.values.length == 0)
            return this;
        let realComparator = comparator ?? this.defaultComparator;
        let realKey = key ?? ((l) => l);
        this.values.sort((a, b) => {
            let aKey = realKey(a);
            let bKey = realKey(b);
            return direction == "desc" ? -realComparator(aKey, bKey) : realComparator(aKey, bKey);
        });
        return this;
    }
    groupBy(key, comparator) {
        if (this.values.length == 0)
            return this.lwrap([]);
        // JavaScript sucks and we can't make hash maps over arbitrary types (only strings/ints), so
        // we do a poor man algorithm where we SORT, followed by grouping.
        let intermediate = this.sort(key, "asc", comparator);
        comparator = comparator ?? this.defaultComparator;
        let result = [];
        let currentRow = [intermediate[0]];
        let current = key(intermediate[0], 0, intermediate.values);
        for (let index = 1; index < intermediate.length; index++) {
            let newKey = key(intermediate[index], index, intermediate.values);
            if (comparator(current, newKey) != 0) {
                result.push({ key: current, rows: this.lwrap(currentRow) });
                current = newKey;
                currentRow = [intermediate[index]];
            }
            else {
                currentRow.push(intermediate[index]);
            }
        }
        result.push({ key: current, rows: this.lwrap(currentRow) });
        return this.lwrap(result);
    }
    groupIn(key, comparator) {
        if (Groupings.isGrouping(this.values)) {
            return this.map(v => {
                return {
                    key: v.key,
                    rows: DataArray.wrap(v.rows, this.settings).groupIn(key, comparator),
                };
            });
        }
        else {
            return this.groupBy(key, comparator);
        }
    }
    distinct(key, comparator) {
        if (this.values.length == 0)
            return this;
        let realKey = key ?? (x => x);
        // For similar reasons to groupBy, do a sort and take the first element of each block.
        let intermediate = this.map((x, index) => {
            return { key: realKey(x, index, this.values), value: x };
        }).sort(x => x.key, "asc", comparator);
        comparator = comparator ?? this.defaultComparator;
        let result = [intermediate[0].value];
        for (let index = 1; index < intermediate.length; index++) {
            if (comparator(intermediate[index - 1].key, intermediate[index].key) != 0) {
                result.push(intermediate[index].value);
            }
        }
        return this.lwrap(result);
    }
    every(f) {
        return this.values.every(f);
    }
    some(f) {
        return this.values.some(f);
    }
    none(f) {
        return this.values.every((v, i, a) => !f(v, i, a));
    }
    first() {
        return this.values.length > 0 ? this.values[0] : undefined;
    }
    last() {
        return this.values.length > 0 ? this.values[this.values.length - 1] : undefined;
    }
    to(key) {
        let result = [];
        for (let child of this.values) {
            let value = child[key];
            if (value === undefined || value === null)
                continue;
            if (Array.isArray(value) || DataArray.isDataArray(value))
                value.forEach(v => result.push(v));
            else
                result.push(value);
        }
        return this.lwrap(result);
    }
    into(key) {
        let result = [];
        for (let child of this.values) {
            let value = child[key];
            if (value === undefined || value === null)
                continue;
            result.push(value);
        }
        return this.lwrap(result);
    }
    expand(key) {
        let result = [];
        let queue = [].concat(this.values);
        while (queue.length > 0) {
            let next = queue.pop();
            let value = next[key];
            if (value === undefined || value === null)
                continue;
            if (Array.isArray(value))
                value.forEach(v => queue.push(v));
            else if (value instanceof DataArrayImpl)
                value.forEach(v => queue.push(v));
            else
                queue.push(value);
            result.push(next);
        }
        return this.lwrap(result);
    }
    forEach(f) {
        for (let index = 0; index < this.values.length; index++) {
            f(this.values[index], index, this.values);
        }
    }
    sum() {
        return this.values.reduce((a, b) => a + b, 0);
    }
    avg() {
        return this.sum() / this.values.length;
    }
    min() {
        return Math.min(...this.values);
    }
    max() {
        return Math.max(...this.values);
    }
    array() {
        return [].concat(this.values);
    }
    [Symbol.iterator]() {
        return this.values[Symbol.iterator]();
    }
    toString() {
        return "[" + this.values.join(", ") + "]";
    }
}
/** Provides utility functions for generating data arrays. */
var DataArray;
(function (DataArray) {
    /** Create a new Dataview data array. */
    function wrap(raw, settings) {
        if (isDataArray(raw))
            return raw;
        return DataArrayImpl.wrap(raw, settings);
    }
    DataArray.wrap = wrap;
    /** Create a new DataArray from an iterable object. */
    function from(raw, settings) {
        if (isDataArray(raw))
            return raw;
        let data = [];
        for (let elem of raw)
            data.push(elem);
        return DataArrayImpl.wrap(data, settings);
    }
    DataArray.from = from;
    /** Return true if the given object is a data array. */
    function isDataArray(obj) {
        return obj instanceof DataArrayImpl;
    }
    DataArray.isDataArray = isDataArray;
})(DataArray || (DataArray = {}));
// A scary looking polyfill, sure, but it fixes up data array/array interop for us.
const oldArrayIsArray = Array.isArray;
Array.isArray = (arg) => {
    return oldArrayIsArray(arg) || DataArray.isDataArray(arg);
};

/** Test-environment-friendly function which fetches the current system locale. */
function currentLocale() {
    if (typeof window === "undefined")
        return "en-US";
    return window.navigator.language;
}

/** Render simple fields compactly, removing wrapping content like paragraph and span. */
async function renderCompactMarkdown(app, markdown, container, sourcePath, component, isInlineFieldLivePreview = false) {
    // check if the call is from the CM6 view plugin defined in src/ui/views/inline-field-live-preview.ts
    if (isInlineFieldLivePreview) {
        await renderCompactMarkdownForInlineFieldLivePreview(app, markdown, container, sourcePath, component);
    }
    else {
        let subcontainer = container.createSpan();
        await obsidian.MarkdownRenderer.render(app, markdown, subcontainer, sourcePath, component);
        let paragraph = subcontainer.querySelector(":scope > p");
        if (subcontainer.children.length == 1 && paragraph) {
            while (paragraph.firstChild) {
                subcontainer.appendChild(paragraph.firstChild);
            }
            subcontainer.removeChild(paragraph);
        }
    }
}
async function renderCompactMarkdownForInlineFieldLivePreview(app, markdown, container, sourcePath, component) {
    const tmpContainer = createSpan();
    await obsidian.MarkdownRenderer.render(app, markdown, tmpContainer, sourcePath, component);
    let paragraph = tmpContainer.querySelector(":scope > p");
    if (tmpContainer.childNodes.length == 1 && paragraph) {
        container.appendChild(paragraph.childNodes.item(paragraph.childNodes.length - 1));
    }
    else {
        container.replaceChildren(...tmpContainer.childNodes);
    }
    tmpContainer.remove();
}
/** Render a pre block with an error in it; returns the element to allow for dynamic updating. */
function renderErrorPre(container, error) {
    let pre = container.createEl("pre", { cls: ["dataview", "dataview-error"] });
    pre.appendText(error);
    return pre;
}
/** Render a static codeblock. */
function renderCodeBlock(container, source, language) {
    let code = container.createEl("code", { cls: ["dataview"] });
    if (language)
        code.classList.add("language-" + language);
    code.appendText(source);
    return code;
}
/** Prettily render a value into a container with the given settings. */
async function renderValue(app, field, container, originFile, component, settings, expandList = false, context = "root", depth = 0, isInlineFieldLivePreview = false) {
    // Prevent infinite recursion.
    if (depth > settings.maxRecursiveRenderDepth) {
        container.appendText("...");
        return;
    }
    if (Values.isNull(field)) {
        await renderCompactMarkdown(app, settings.renderNullAs, container, originFile, component, isInlineFieldLivePreview);
    }
    else if (Values.isDate(field)) {
        container.appendText(renderMinimalDate(field, settings, currentLocale()));
    }
    else if (Values.isDuration(field)) {
        container.appendText(renderMinimalDuration(field));
    }
    else if (Values.isString(field) || Values.isBoolean(field) || Values.isNumber(field)) {
        await renderCompactMarkdown(app, "" + field, container, originFile, component, isInlineFieldLivePreview);
    }
    else if (Values.isLink(field)) {
        await renderCompactMarkdown(app, field.markdown(), container, originFile, component, isInlineFieldLivePreview);
    }
    else if (Values.isHtml(field)) {
        container.appendChild(field);
    }
    else if (Values.isWidget(field)) {
        if (Widgets.isListPair(field)) {
            await renderValue(app, field.key, container, originFile, component, settings, expandList, context, depth, isInlineFieldLivePreview);
            container.appendText(": ");
            await renderValue(app, field.value, container, originFile, component, settings, expandList, context, depth, isInlineFieldLivePreview);
        }
        else if (Widgets.isExternalLink(field)) {
            let elem = document.createElement("a");
            elem.textContent = field.display ?? field.url;
            elem.rel = "noopener";
            elem.target = "_blank";
            elem.classList.add("external-link");
            elem.href = field.url;
            container.appendChild(elem);
        }
        else {
            container.appendText(`<unknown widget '${field.$widget}>`);
        }
    }
    else if (Values.isFunction(field)) {
        container.appendText("<function>");
    }
    else if (Values.isArray(field) || DataArray.isDataArray(field)) {
        if (expandList) {
            let list = container.createEl("ul", {
                cls: [
                    "dataview",
                    "dataview-ul",
                    context == "list" ? "dataview-result-list-ul" : "dataview-result-list-root-ul",
                ],
            });
            for (let child of field) {
                let li = list.createEl("li", { cls: "dataview-result-list-li" });
                await renderValue(app, child, li, originFile, component, settings, expandList, "list", depth + 1, isInlineFieldLivePreview);
            }
        }
        else {
            if (field.length == 0) {
                container.appendText("<empty list>");
                return;
            }
            let span = container.createEl("span", { cls: ["dataview", "dataview-result-list-span"] });
            let first = true;
            for (let val of field) {
                if (first)
                    first = false;
                else
                    span.appendText(", ");
                await renderValue(app, val, span, originFile, component, settings, expandList, "list", depth + 1, isInlineFieldLivePreview);
            }
        }
    }
    else if (Values.isObject(field)) {
        // Don't render classes in case they have recursive references; spoopy.
        if (field?.constructor?.name && field?.constructor?.name != "Object") {
            container.appendText(`<${field.constructor.name}>`);
            return;
        }
        if (expandList) {
            let list = container.createEl("ul", { cls: ["dataview", "dataview-ul", "dataview-result-object-ul"] });
            for (let [key, value] of Object.entries(field)) {
                let li = list.createEl("li", { cls: ["dataview", "dataview-li", "dataview-result-object-li"] });
                li.appendText(key + ": ");
                await renderValue(app, value, li, originFile, component, settings, expandList, "list", depth + 1, isInlineFieldLivePreview);
            }
        }
        else {
            if (Object.keys(field).length == 0) {
                container.appendText("<empty object>");
                return;
            }
            let span = container.createEl("span", { cls: ["dataview", "dataview-result-object-span"] });
            let first = true;
            for (let [key, value] of Object.entries(field)) {
                if (first)
                    first = false;
                else
                    span.appendText(", ");
                span.appendText(key + ": ");
                await renderValue(app, value, span, originFile, component, settings, expandList, "list", depth + 1, isInlineFieldLivePreview);
            }
        }
    }
    else {
        container.appendText("Unrecognized: " + JSON.stringify(field));
    }
}

var papaparse_min = {exports: {}};

/* @license
Papa Parse
v5.4.1
https://github.com/mholt/PapaParse
License: MIT
*/
papaparse_min.exports;

(function (module, exports) {
	!function(e,t){module.exports=t();}(commonjsGlobal,function s(){var f="undefined"!=typeof self?self:"undefined"!=typeof window?window:void 0!==f?f:{};var n=!f.document&&!!f.postMessage,o=f.IS_PAPA_WORKER||!1,a={},u=0,b={parse:function(e,t){var r=(t=t||{}).dynamicTyping||!1;J(r)&&(t.dynamicTypingFunction=r,r={});if(t.dynamicTyping=r,t.transform=!!J(t.transform)&&t.transform,t.worker&&b.WORKERS_SUPPORTED){var i=function(){if(!b.WORKERS_SUPPORTED)return !1;var e=(r=f.URL||f.webkitURL||null,i=s.toString(),b.BLOB_URL||(b.BLOB_URL=r.createObjectURL(new Blob(["var global = (function() { if (typeof self !== 'undefined') { return self; } if (typeof window !== 'undefined') { return window; } if (typeof global !== 'undefined') { return global; } return {}; })(); global.IS_PAPA_WORKER=true; ","(",i,")();"],{type:"text/javascript"})))),t=new f.Worker(e);var r,i;return t.onmessage=_,t.id=u++,a[t.id]=t}();return i.userStep=t.step,i.userChunk=t.chunk,i.userComplete=t.complete,i.userError=t.error,t.step=J(t.step),t.chunk=J(t.chunk),t.complete=J(t.complete),t.error=J(t.error),delete t.worker,void i.postMessage({input:e,config:t,workerId:i.id})}var n=null;b.NODE_STREAM_INPUT,"string"==typeof e?(e=function(e){if(65279===e.charCodeAt(0))return e.slice(1);return e}(e),n=t.download?new l(t):new p(t)):!0===e.readable&&J(e.read)&&J(e.on)?n=new g(t):(f.File&&e instanceof File||e instanceof Object)&&(n=new c(t));return n.stream(e)},unparse:function(e,t){var n=!1,_=!0,m=",",y="\r\n",s='"',a=s+s,r=!1,i=null,o=!1;!function(){if("object"!=typeof t)return;"string"!=typeof t.delimiter||b.BAD_DELIMITERS.filter(function(e){return -1!==t.delimiter.indexOf(e)}).length||(m=t.delimiter);("boolean"==typeof t.quotes||"function"==typeof t.quotes||Array.isArray(t.quotes))&&(n=t.quotes);"boolean"!=typeof t.skipEmptyLines&&"string"!=typeof t.skipEmptyLines||(r=t.skipEmptyLines);"string"==typeof t.newline&&(y=t.newline);"string"==typeof t.quoteChar&&(s=t.quoteChar);"boolean"==typeof t.header&&(_=t.header);if(Array.isArray(t.columns)){if(0===t.columns.length)throw new Error("Option columns is empty");i=t.columns;}void 0!==t.escapeChar&&(a=t.escapeChar+s);("boolean"==typeof t.escapeFormulae||t.escapeFormulae instanceof RegExp)&&(o=t.escapeFormulae instanceof RegExp?t.escapeFormulae:/^[=+\-@\t\r].*$/);}();var u=new RegExp(Q(s),"g");"string"==typeof e&&(e=JSON.parse(e));if(Array.isArray(e)){if(!e.length||Array.isArray(e[0]))return h(null,e,r);if("object"==typeof e[0])return h(i||Object.keys(e[0]),e,r)}else if("object"==typeof e)return "string"==typeof e.data&&(e.data=JSON.parse(e.data)),Array.isArray(e.data)&&(e.fields||(e.fields=e.meta&&e.meta.fields||i),e.fields||(e.fields=Array.isArray(e.data[0])?e.fields:"object"==typeof e.data[0]?Object.keys(e.data[0]):[]),Array.isArray(e.data[0])||"object"==typeof e.data[0]||(e.data=[e.data])),h(e.fields||[],e.data||[],r);throw new Error("Unable to serialize unrecognized input");function h(e,t,r){var i="";"string"==typeof e&&(e=JSON.parse(e)),"string"==typeof t&&(t=JSON.parse(t));var n=Array.isArray(e)&&0<e.length,s=!Array.isArray(t[0]);if(n&&_){for(var a=0;a<e.length;a++)0<a&&(i+=m),i+=v(e[a],a);0<t.length&&(i+=y);}for(var o=0;o<t.length;o++){var u=n?e.length:t[o].length,h=!1,f=n?0===Object.keys(t[o]).length:0===t[o].length;if(r&&!n&&(h="greedy"===r?""===t[o].join("").trim():1===t[o].length&&0===t[o][0].length),"greedy"===r&&n){for(var d=[],l=0;l<u;l++){var c=s?e[l]:l;d.push(t[o][c]);}h=""===d.join("").trim();}if(!h){for(var p=0;p<u;p++){0<p&&!f&&(i+=m);var g=n&&s?e[p]:p;i+=v(t[o][g],p);}o<t.length-1&&(!r||0<u&&!f)&&(i+=y);}}return i}function v(e,t){if(null==e)return "";if(e.constructor===Date)return JSON.stringify(e).slice(1,25);var r=!1;o&&"string"==typeof e&&o.test(e)&&(e="'"+e,r=!0);var i=e.toString().replace(u,a);return (r=r||!0===n||"function"==typeof n&&n(e,t)||Array.isArray(n)&&n[t]||function(e,t){for(var r=0;r<t.length;r++)if(-1<e.indexOf(t[r]))return !0;return !1}(i,b.BAD_DELIMITERS)||-1<i.indexOf(m)||" "===i.charAt(0)||" "===i.charAt(i.length-1))?s+i+s:i}}};if(b.RECORD_SEP=String.fromCharCode(30),b.UNIT_SEP=String.fromCharCode(31),b.BYTE_ORDER_MARK="\ufeff",b.BAD_DELIMITERS=["\r","\n",'"',b.BYTE_ORDER_MARK],b.WORKERS_SUPPORTED=!n&&!!f.Worker,b.NODE_STREAM_INPUT=1,b.LocalChunkSize=10485760,b.RemoteChunkSize=5242880,b.DefaultDelimiter=",",b.Parser=E,b.ParserHandle=r,b.NetworkStreamer=l,b.FileStreamer=c,b.StringStreamer=p,b.ReadableStreamStreamer=g,f.jQuery){var d=f.jQuery;d.fn.parse=function(o){var r=o.config||{},u=[];return this.each(function(e){if(!("INPUT"===d(this).prop("tagName").toUpperCase()&&"file"===d(this).attr("type").toLowerCase()&&f.FileReader)||!this.files||0===this.files.length)return !0;for(var t=0;t<this.files.length;t++)u.push({file:this.files[t],inputElem:this,instanceConfig:d.extend({},r)});}),e(),this;function e(){if(0!==u.length){var e,t,r,i,n=u[0];if(J(o.before)){var s=o.before(n.file,n.inputElem);if("object"==typeof s){if("abort"===s.action)return e="AbortError",t=n.file,r=n.inputElem,i=s.reason,void(J(o.error)&&o.error({name:e},t,r,i));if("skip"===s.action)return void h();"object"==typeof s.config&&(n.instanceConfig=d.extend(n.instanceConfig,s.config));}else if("skip"===s)return void h()}var a=n.instanceConfig.complete;n.instanceConfig.complete=function(e){J(a)&&a(e,n.file,n.inputElem),h();},b.parse(n.file,n.instanceConfig);}else J(o.complete)&&o.complete();}function h(){u.splice(0,1),e();}};}function h(e){this._handle=null,this._finished=!1,this._completed=!1,this._halted=!1,this._input=null,this._baseIndex=0,this._partialLine="",this._rowCount=0,this._start=0,this._nextChunk=null,this.isFirstChunk=!0,this._completeResults={data:[],errors:[],meta:{}},function(e){var t=w(e);t.chunkSize=parseInt(t.chunkSize),e.step||e.chunk||(t.chunkSize=null);this._handle=new r(t),(this._handle.streamer=this)._config=t;}.call(this,e),this.parseChunk=function(e,t){if(this.isFirstChunk&&J(this._config.beforeFirstChunk)){var r=this._config.beforeFirstChunk(e);void 0!==r&&(e=r);}this.isFirstChunk=!1,this._halted=!1;var i=this._partialLine+e;this._partialLine="";var n=this._handle.parse(i,this._baseIndex,!this._finished);if(!this._handle.paused()&&!this._handle.aborted()){var s=n.meta.cursor;this._finished||(this._partialLine=i.substring(s-this._baseIndex),this._baseIndex=s),n&&n.data&&(this._rowCount+=n.data.length);var a=this._finished||this._config.preview&&this._rowCount>=this._config.preview;if(o)f.postMessage({results:n,workerId:b.WORKER_ID,finished:a});else if(J(this._config.chunk)&&!t){if(this._config.chunk(n,this._handle),this._handle.paused()||this._handle.aborted())return void(this._halted=!0);n=void 0,this._completeResults=void 0;}return this._config.step||this._config.chunk||(this._completeResults.data=this._completeResults.data.concat(n.data),this._completeResults.errors=this._completeResults.errors.concat(n.errors),this._completeResults.meta=n.meta),this._completed||!a||!J(this._config.complete)||n&&n.meta.aborted||(this._config.complete(this._completeResults,this._input),this._completed=!0),a||n&&n.meta.paused||this._nextChunk(),n}this._halted=!0;},this._sendError=function(e){J(this._config.error)?this._config.error(e):o&&this._config.error&&f.postMessage({workerId:b.WORKER_ID,error:e,finished:!1});};}function l(e){var i;(e=e||{}).chunkSize||(e.chunkSize=b.RemoteChunkSize),h.call(this,e),this._nextChunk=n?function(){this._readChunk(),this._chunkLoaded();}:function(){this._readChunk();},this.stream=function(e){this._input=e,this._nextChunk();},this._readChunk=function(){if(this._finished)this._chunkLoaded();else {if(i=new XMLHttpRequest,this._config.withCredentials&&(i.withCredentials=this._config.withCredentials),n||(i.onload=v(this._chunkLoaded,this),i.onerror=v(this._chunkError,this)),i.open(this._config.downloadRequestBody?"POST":"GET",this._input,!n),this._config.downloadRequestHeaders){var e=this._config.downloadRequestHeaders;for(var t in e)i.setRequestHeader(t,e[t]);}if(this._config.chunkSize){var r=this._start+this._config.chunkSize-1;i.setRequestHeader("Range","bytes="+this._start+"-"+r);}try{i.send(this._config.downloadRequestBody);}catch(e){this._chunkError(e.message);}n&&0===i.status&&this._chunkError();}},this._chunkLoaded=function(){4===i.readyState&&(i.status<200||400<=i.status?this._chunkError():(this._start+=this._config.chunkSize?this._config.chunkSize:i.responseText.length,this._finished=!this._config.chunkSize||this._start>=function(e){var t=e.getResponseHeader("Content-Range");if(null===t)return -1;return parseInt(t.substring(t.lastIndexOf("/")+1))}(i),this.parseChunk(i.responseText)));},this._chunkError=function(e){var t=i.statusText||e;this._sendError(new Error(t));};}function c(e){var i,n;(e=e||{}).chunkSize||(e.chunkSize=b.LocalChunkSize),h.call(this,e);var s="undefined"!=typeof FileReader;this.stream=function(e){this._input=e,n=e.slice||e.webkitSlice||e.mozSlice,s?((i=new FileReader).onload=v(this._chunkLoaded,this),i.onerror=v(this._chunkError,this)):i=new FileReaderSync,this._nextChunk();},this._nextChunk=function(){this._finished||this._config.preview&&!(this._rowCount<this._config.preview)||this._readChunk();},this._readChunk=function(){var e=this._input;if(this._config.chunkSize){var t=Math.min(this._start+this._config.chunkSize,this._input.size);e=n.call(e,this._start,t);}var r=i.readAsText(e,this._config.encoding);s||this._chunkLoaded({target:{result:r}});},this._chunkLoaded=function(e){this._start+=this._config.chunkSize,this._finished=!this._config.chunkSize||this._start>=this._input.size,this.parseChunk(e.target.result);},this._chunkError=function(){this._sendError(i.error);};}function p(e){var r;h.call(this,e=e||{}),this.stream=function(e){return r=e,this._nextChunk()},this._nextChunk=function(){if(!this._finished){var e,t=this._config.chunkSize;return t?(e=r.substring(0,t),r=r.substring(t)):(e=r,r=""),this._finished=!r,this.parseChunk(e)}};}function g(e){h.call(this,e=e||{});var t=[],r=!0,i=!1;this.pause=function(){h.prototype.pause.apply(this,arguments),this._input.pause();},this.resume=function(){h.prototype.resume.apply(this,arguments),this._input.resume();},this.stream=function(e){this._input=e,this._input.on("data",this._streamData),this._input.on("end",this._streamEnd),this._input.on("error",this._streamError);},this._checkIsFinished=function(){i&&1===t.length&&(this._finished=!0);},this._nextChunk=function(){this._checkIsFinished(),t.length?this.parseChunk(t.shift()):r=!0;},this._streamData=v(function(e){try{t.push("string"==typeof e?e:e.toString(this._config.encoding)),r&&(r=!1,this._checkIsFinished(),this.parseChunk(t.shift()));}catch(e){this._streamError(e);}},this),this._streamError=v(function(e){this._streamCleanUp(),this._sendError(e);},this),this._streamEnd=v(function(){this._streamCleanUp(),i=!0,this._streamData("");},this),this._streamCleanUp=v(function(){this._input.removeListener("data",this._streamData),this._input.removeListener("end",this._streamEnd),this._input.removeListener("error",this._streamError);},this);}function r(m){var a,o,u,i=Math.pow(2,53),n=-i,s=/^\s*-?(\d+\.?|\.\d+|\d+\.\d+)([eE][-+]?\d+)?\s*$/,h=/^((\d{4}-[01]\d-[0-3]\dT[0-2]\d:[0-5]\d:[0-5]\d\.\d+([+-][0-2]\d:[0-5]\d|Z))|(\d{4}-[01]\d-[0-3]\dT[0-2]\d:[0-5]\d:[0-5]\d([+-][0-2]\d:[0-5]\d|Z))|(\d{4}-[01]\d-[0-3]\dT[0-2]\d:[0-5]\d([+-][0-2]\d:[0-5]\d|Z)))$/,t=this,r=0,f=0,d=!1,e=!1,l=[],c={data:[],errors:[],meta:{}};if(J(m.step)){var p=m.step;m.step=function(e){if(c=e,_())g();else {if(g(),0===c.data.length)return;r+=e.data.length,m.preview&&r>m.preview?o.abort():(c.data=c.data[0],p(c,t));}};}function y(e){return "greedy"===m.skipEmptyLines?""===e.join("").trim():1===e.length&&0===e[0].length}function g(){return c&&u&&(k("Delimiter","UndetectableDelimiter","Unable to auto-detect delimiting character; defaulted to '"+b.DefaultDelimiter+"'"),u=!1),m.skipEmptyLines&&(c.data=c.data.filter(function(e){return !y(e)})),_()&&function(){if(!c)return;function e(e,t){J(m.transformHeader)&&(e=m.transformHeader(e,t)),l.push(e);}if(Array.isArray(c.data[0])){for(var t=0;_()&&t<c.data.length;t++)c.data[t].forEach(e);c.data.splice(0,1);}else c.data.forEach(e);}(),function(){if(!c||!m.header&&!m.dynamicTyping&&!m.transform)return c;function e(e,t){var r,i=m.header?{}:[];for(r=0;r<e.length;r++){var n=r,s=e[r];m.header&&(n=r>=l.length?"__parsed_extra":l[r]),m.transform&&(s=m.transform(s,n)),s=v(n,s),"__parsed_extra"===n?(i[n]=i[n]||[],i[n].push(s)):i[n]=s;}return m.header&&(r>l.length?k("FieldMismatch","TooManyFields","Too many fields: expected "+l.length+" fields but parsed "+r,f+t):r<l.length&&k("FieldMismatch","TooFewFields","Too few fields: expected "+l.length+" fields but parsed "+r,f+t)),i}var t=1;!c.data.length||Array.isArray(c.data[0])?(c.data=c.data.map(e),t=c.data.length):c.data=e(c.data,0);m.header&&c.meta&&(c.meta.fields=l);return f+=t,c}()}function _(){return m.header&&0===l.length}function v(e,t){return r=e,m.dynamicTypingFunction&&void 0===m.dynamicTyping[r]&&(m.dynamicTyping[r]=m.dynamicTypingFunction(r)),!0===(m.dynamicTyping[r]||m.dynamicTyping)?"true"===t||"TRUE"===t||"false"!==t&&"FALSE"!==t&&(function(e){if(s.test(e)){var t=parseFloat(e);if(n<t&&t<i)return !0}return !1}(t)?parseFloat(t):h.test(t)?new Date(t):""===t?null:t):t;var r;}function k(e,t,r,i){var n={type:e,code:t,message:r};void 0!==i&&(n.row=i),c.errors.push(n);}this.parse=function(e,t,r){var i=m.quoteChar||'"';if(m.newline||(m.newline=function(e,t){e=e.substring(0,1048576);var r=new RegExp(Q(t)+"([^]*?)"+Q(t),"gm"),i=(e=e.replace(r,"")).split("\r"),n=e.split("\n"),s=1<n.length&&n[0].length<i[0].length;if(1===i.length||s)return "\n";for(var a=0,o=0;o<i.length;o++)"\n"===i[o][0]&&a++;return a>=i.length/2?"\r\n":"\r"}(e,i)),u=!1,m.delimiter)J(m.delimiter)&&(m.delimiter=m.delimiter(e),c.meta.delimiter=m.delimiter);else {var n=function(e,t,r,i,n){var s,a,o,u;n=n||[",","\t","|",";",b.RECORD_SEP,b.UNIT_SEP];for(var h=0;h<n.length;h++){var f=n[h],d=0,l=0,c=0;o=void 0;for(var p=new E({comments:i,delimiter:f,newline:t,preview:10}).parse(e),g=0;g<p.data.length;g++)if(r&&y(p.data[g]))c++;else {var _=p.data[g].length;l+=_,void 0!==o?0<_&&(d+=Math.abs(_-o),o=_):o=_;}0<p.data.length&&(l/=p.data.length-c),(void 0===a||d<=a)&&(void 0===u||u<l)&&1.99<l&&(a=d,s=f,u=l);}return {successful:!!(m.delimiter=s),bestDelimiter:s}}(e,m.newline,m.skipEmptyLines,m.comments,m.delimitersToGuess);n.successful?m.delimiter=n.bestDelimiter:(u=!0,m.delimiter=b.DefaultDelimiter),c.meta.delimiter=m.delimiter;}var s=w(m);return m.preview&&m.header&&s.preview++,a=e,o=new E(s),c=o.parse(a,t,r),g(),d?{meta:{paused:!0}}:c||{meta:{paused:!1}}},this.paused=function(){return d},this.pause=function(){d=!0,o.abort(),a=J(m.chunk)?"":a.substring(o.getCharIndex());},this.resume=function(){t.streamer._halted?(d=!1,t.streamer.parseChunk(a,!0)):setTimeout(t.resume,3);},this.aborted=function(){return e},this.abort=function(){e=!0,o.abort(),c.meta.aborted=!0,J(m.complete)&&m.complete(c),a="";};}function Q(e){return e.replace(/[.*+?^${}()|[\]\\]/g,"\\$&")}function E(j){var z,M=(j=j||{}).delimiter,P=j.newline,U=j.comments,q=j.step,N=j.preview,B=j.fastMode,K=z=void 0===j.quoteChar||null===j.quoteChar?'"':j.quoteChar;if(void 0!==j.escapeChar&&(K=j.escapeChar),("string"!=typeof M||-1<b.BAD_DELIMITERS.indexOf(M))&&(M=","),U===M)throw new Error("Comment character same as delimiter");!0===U?U="#":("string"!=typeof U||-1<b.BAD_DELIMITERS.indexOf(U))&&(U=!1),"\n"!==P&&"\r"!==P&&"\r\n"!==P&&(P="\n");var W=0,H=!1;this.parse=function(i,t,r){if("string"!=typeof i)throw new Error("Input must be a string");var n=i.length,e=M.length,s=P.length,a=U.length,o=J(q),u=[],h=[],f=[],d=W=0;if(!i)return L();if(j.header&&!t){var l=i.split(P)[0].split(M),c=[],p={},g=!1;for(var _ in l){var m=l[_];J(j.transformHeader)&&(m=j.transformHeader(m,_));var y=m,v=p[m]||0;for(0<v&&(g=!0,y=m+"_"+v),p[m]=v+1;c.includes(y);)y=y+"_"+v;c.push(y);}if(g){var k=i.split(P);k[0]=c.join(M),i=k.join(P);}}if(B||!1!==B&&-1===i.indexOf(z)){for(var b=i.split(P),E=0;E<b.length;E++){if(f=b[E],W+=f.length,E!==b.length-1)W+=P.length;else if(r)return L();if(!U||f.substring(0,a)!==U){if(o){if(u=[],I(f.split(M)),F(),H)return L()}else I(f.split(M));if(N&&N<=E)return u=u.slice(0,N),L(!0)}}return L()}for(var w=i.indexOf(M,W),R=i.indexOf(P,W),C=new RegExp(Q(K)+Q(z),"g"),S=i.indexOf(z,W);;)if(i[W]!==z)if(U&&0===f.length&&i.substring(W,W+a)===U){if(-1===R)return L();W=R+s,R=i.indexOf(P,W),w=i.indexOf(M,W);}else if(-1!==w&&(w<R||-1===R))f.push(i.substring(W,w)),W=w+e,w=i.indexOf(M,W);else {if(-1===R)break;if(f.push(i.substring(W,R)),D(R+s),o&&(F(),H))return L();if(N&&u.length>=N)return L(!0)}else for(S=W,W++;;){if(-1===(S=i.indexOf(z,S+1)))return r||h.push({type:"Quotes",code:"MissingQuotes",message:"Quoted field unterminated",row:u.length,index:W}),T();if(S===n-1)return T(i.substring(W,S).replace(C,z));if(z!==K||i[S+1]!==K){if(z===K||0===S||i[S-1]!==K){-1!==w&&w<S+1&&(w=i.indexOf(M,S+1)),-1!==R&&R<S+1&&(R=i.indexOf(P,S+1));var O=A(-1===R?w:Math.min(w,R));if(i.substr(S+1+O,e)===M){f.push(i.substring(W,S).replace(C,z)),i[W=S+1+O+e]!==z&&(S=i.indexOf(z,W)),w=i.indexOf(M,W),R=i.indexOf(P,W);break}var x=A(R);if(i.substring(S+1+x,S+1+x+s)===P){if(f.push(i.substring(W,S).replace(C,z)),D(S+1+x+s),w=i.indexOf(M,W),S=i.indexOf(z,W),o&&(F(),H))return L();if(N&&u.length>=N)return L(!0);break}h.push({type:"Quotes",code:"InvalidQuotes",message:"Trailing quote on quoted field is malformed",row:u.length,index:W}),S++;}}else S++;}return T();function I(e){u.push(e),d=W;}function A(e){var t=0;if(-1!==e){var r=i.substring(S+1,e);r&&""===r.trim()&&(t=r.length);}return t}function T(e){return r||(void 0===e&&(e=i.substring(W)),f.push(e),W=n,I(f),o&&F()),L()}function D(e){W=e,I(f),f=[],R=i.indexOf(P,W);}function L(e){return {data:u,errors:h,meta:{delimiter:M,linebreak:P,aborted:H,truncated:!!e,cursor:d+(t||0)}}}function F(){q(L()),u=[],h=[];}},this.abort=function(){H=!0;},this.getCharIndex=function(){return W};}function _(e){var t=e.data,r=a[t.workerId],i=!1;if(t.error)r.userError(t.error,t.file);else if(t.results&&t.results.data){var n={abort:function(){i=!0,m(t.workerId,{data:[],errors:[],meta:{aborted:!0}});},pause:y,resume:y};if(J(r.userStep)){for(var s=0;s<t.results.data.length&&(r.userStep({data:t.results.data[s],errors:t.results.errors,meta:t.results.meta},n),!i);s++);delete t.results;}else J(r.userChunk)&&(r.userChunk(t.results,n,t.file),delete t.results);}t.finished&&!i&&m(t.workerId,t.results);}function m(e,t){var r=a[e];J(r.userComplete)&&r.userComplete(t),r.terminate(),delete a[e];}function y(){throw new Error("Not implemented.")}function w(e){if("object"!=typeof e||null===e)return e;var t=Array.isArray(e)?[]:{};for(var r in e)t[r]=w(e[r]);return t}function v(e,t){return function(){e.apply(t,arguments);}}function J(e){return "function"==typeof e}return o&&(f.onmessage=function(e){var t=e.data;void 0===b.WORKER_ID&&t&&(b.WORKER_ID=t.workerId);if("string"==typeof t.input)f.postMessage({workerId:b.WORKER_ID,results:b.parse(t.input,t.config),finished:!0});else if(f.File&&t.input instanceof File||t.input instanceof Object){var r=b.parse(t.input,t.config);r&&f.postMessage({workerId:b.WORKER_ID,results:r,finished:!0});}}),(l.prototype=Object.create(h.prototype)).constructor=l,(c.prototype=Object.create(h.prototype)).constructor=c,(p.prototype=Object.create(p.prototype)).constructor=p,(g.prototype=Object.create(h.prototype)).constructor=g,b}); 
} (papaparse_min, papaparse_min.exports));

var papaparse_minExports = papaparse_min.exports;

/** Utility methods for creating & comparing fields. */
var Fields;
(function (Fields) {
    function variable(name) {
        return { type: "variable", name };
    }
    Fields.variable = variable;
    function literal(value) {
        return { type: "literal", value };
    }
    Fields.literal = literal;
    function binaryOp(left, op, right) {
        return { type: "binaryop", left, op, right };
    }
    Fields.binaryOp = binaryOp;
    function index(obj, index) {
        return { type: "index", object: obj, index };
    }
    Fields.index = index;
    /** Converts a string in dot-notation-format into a variable which indexes. */
    function indexVariable(name) {
        let parts = name.split(".");
        let result = Fields.variable(parts[0]);
        for (let index = 1; index < parts.length; index++) {
            result = Fields.index(result, Fields.literal(parts[index]));
        }
        return result;
    }
    Fields.indexVariable = indexVariable;
    function lambda(args, value) {
        return { type: "lambda", arguments: args, value };
    }
    Fields.lambda = lambda;
    function func(func, args) {
        return { type: "function", func, arguments: args };
    }
    Fields.func = func;
    function list(values) {
        return { type: "list", values };
    }
    Fields.list = list;
    function object(values) {
        return { type: "object", values };
    }
    Fields.object = object;
    function negate(child) {
        return { type: "negated", child };
    }
    Fields.negate = negate;
    function isCompareOp(op) {
        return op == "<=" || op == "<" || op == ">" || op == ">=" || op == "!=" || op == "=";
    }
    Fields.isCompareOp = isCompareOp;
    Fields.NULL = Fields.literal(null);
})(Fields || (Fields = {}));

/** AST implementation for queries over data sources. */
/** Utility functions for creating and manipulating sources. */
var Sources;
(function (Sources) {
    /** Create a source which searches from a tag. */
    function tag(tag) {
        return { type: "tag", tag };
    }
    Sources.tag = tag;
    /** Create a source which fetches from a CSV file. */
    function csv(path) {
        return { type: "csv", path };
    }
    Sources.csv = csv;
    /** Create a source which searches for files under a folder prefix. */
    function folder(prefix) {
        return { type: "folder", folder: prefix };
    }
    Sources.folder = folder;
    /** Create a source which searches for files which link to/from a given file. */
    function link(file, incoming) {
        return { type: "link", file, direction: incoming ? "incoming" : "outgoing" };
    }
    Sources.link = link;
    /** Create a source which joins two sources by a logical operator (and/or). */
    function binaryOp(left, op, right) {
        return { type: "binaryop", left, op, right };
    }
    Sources.binaryOp = binaryOp;
    /** Create a source which takes the intersection of two sources. */
    function and(left, right) {
        return { type: "binaryop", left, op: "&", right };
    }
    Sources.and = and;
    /** Create a source which takes the union of two sources. */
    function or(left, right) {
        return { type: "binaryop", left, op: "|", right };
    }
    Sources.or = or;
    /** Create a source which negates the underlying source. */
    function negate(child) {
        return { type: "negate", child };
    }
    Sources.negate = negate;
    function empty() {
        return { type: "empty" };
    }
    Sources.empty = empty;
})(Sources || (Sources = {}));

/** Emoji regex without any additional flags. */
const EMOJI_REGEX = new RegExp(emojiRegex(), "");
/** Provides a lookup table for unit durations of the given type. */
const DURATION_TYPES = {
    year: Duration.fromObject({ years: 1 }),
    years: Duration.fromObject({ years: 1 }),
    yr: Duration.fromObject({ years: 1 }),
    yrs: Duration.fromObject({ years: 1 }),
    month: Duration.fromObject({ months: 1 }),
    months: Duration.fromObject({ months: 1 }),
    mo: Duration.fromObject({ months: 1 }),
    mos: Duration.fromObject({ months: 1 }),
    week: Duration.fromObject({ weeks: 1 }),
    weeks: Duration.fromObject({ weeks: 1 }),
    wk: Duration.fromObject({ weeks: 1 }),
    wks: Duration.fromObject({ weeks: 1 }),
    w: Duration.fromObject({ weeks: 1 }),
    day: Duration.fromObject({ days: 1 }),
    days: Duration.fromObject({ days: 1 }),
    d: Duration.fromObject({ days: 1 }),
    hour: Duration.fromObject({ hours: 1 }),
    hours: Duration.fromObject({ hours: 1 }),
    hr: Duration.fromObject({ hours: 1 }),
    hrs: Duration.fromObject({ hours: 1 }),
    h: Duration.fromObject({ hours: 1 }),
    minute: Duration.fromObject({ minutes: 1 }),
    minutes: Duration.fromObject({ minutes: 1 }),
    min: Duration.fromObject({ minutes: 1 }),
    mins: Duration.fromObject({ minutes: 1 }),
    m: Duration.fromObject({ minutes: 1 }),
    second: Duration.fromObject({ seconds: 1 }),
    seconds: Duration.fromObject({ seconds: 1 }),
    sec: Duration.fromObject({ seconds: 1 }),
    secs: Duration.fromObject({ seconds: 1 }),
    s: Duration.fromObject({ seconds: 1 }),
};
/** Shorthand for common dates (relative to right now). */
const DATE_SHORTHANDS = {
    now: () => DateTime.local(),
    today: () => DateTime.local().startOf("day"),
    yesterday: () => DateTime.local()
        .startOf("day")
        .minus(Duration.fromObject({ days: 1 })),
    tomorrow: () => DateTime.local()
        .startOf("day")
        .plus(Duration.fromObject({ days: 1 })),
    sow: () => DateTime.local().startOf("week"),
    "start-of-week": () => DateTime.local().startOf("week"),
    eow: () => DateTime.local().endOf("week"),
    "end-of-week": () => DateTime.local().endOf("week"),
    soy: () => DateTime.local().startOf("year"),
    "start-of-year": () => DateTime.local().startOf("year"),
    eoy: () => DateTime.local().endOf("year"),
    "end-of-year": () => DateTime.local().endOf("year"),
    som: () => DateTime.local().startOf("month"),
    "start-of-month": () => DateTime.local().startOf("month"),
    eom: () => DateTime.local().endOf("month"),
    "end-of-month": () => DateTime.local().endOf("month"),
};
/**
 * Keywords which cannot be used as variables directly. Use `row.<thing>` if it is a variable you have defined and want
 * to access.
 */
const KEYWORDS = ["FROM", "WHERE", "LIMIT", "GROUP", "FLATTEN"];
///////////////
// Utilities //
///////////////
/** Split on unescaped pipes in an inner link. */
function splitOnUnescapedPipe(link) {
    let pipe = -1;
    while ((pipe = link.indexOf("|", pipe + 1)) >= 0) {
        if (pipe > 0 && link[pipe - 1] == "\\")
            continue;
        return [link.substring(0, pipe).replace(/\\\|/g, "|"), link.substring(pipe + 1)];
    }
    return [link.replace(/\\\|/g, "|"), undefined];
}
/** Attempt to parse the inside of a link to pull out display name, subpath, etc. */
function parseInnerLink(rawlink) {
    let [link, display] = splitOnUnescapedPipe(rawlink);
    return Link.infer(link, false, display);
}
/** Create a left-associative binary parser which parses the given sub-element and separator. Handles whitespace. */
function createBinaryParser(child, sep, combine) {
    return parsimmon_umd_minExports.seqMap(child, parsimmon_umd_minExports.seq(parsimmon_umd_minExports.optWhitespace, sep, parsimmon_umd_minExports.optWhitespace, child).many(), (first, rest) => {
        if (rest.length == 0)
            return first;
        let node = combine(first, rest[0][1], rest[0][3]);
        for (let index = 1; index < rest.length; index++) {
            node = combine(node, rest[index][1], rest[index][3]);
        }
        return node;
    });
}
function chainOpt(base, ...funcs) {
    return parsimmon_umd_minExports.custom((success, failure) => {
        return (input, i) => {
            let result = base._(input, i);
            if (!result.status)
                return result;
            for (let func of funcs) {
                let next = func(result.value)._(input, result.index);
                if (!next.status)
                    return result;
                result = next;
            }
            return result;
        };
    });
}
const EXPRESSION = parsimmon_umd_minExports.createLanguage({
    // A floating point number; the decimal point is optional.
    number: q => parsimmon_umd_minExports.regexp(/-?[0-9]+(\.[0-9]+)?/)
        .map(str => Number.parseFloat(str))
        .desc("number"),
    // A quote-surrounded string which supports escape characters ('\').
    string: q => parsimmon_umd_minExports.string('"')
        .then(parsimmon_umd_minExports.alt(q.escapeCharacter, parsimmon_umd_minExports.noneOf('"\\'))
        .atLeast(0)
        .map(chars => chars.join("")))
        .skip(parsimmon_umd_minExports.string('"'))
        .desc("string"),
    escapeCharacter: _ => parsimmon_umd_minExports.string("\\")
        .then(parsimmon_umd_minExports.any)
        .map(escaped => {
        // If we are escaping a backslash or a quote, pass in on in escaped form
        if (escaped === '"')
            return '"';
        if (escaped === "\\")
            return "\\";
        else
            return "\\" + escaped;
    }),
    // A boolean true/false value.
    bool: _ => parsimmon_umd_minExports.regexp(/true|false|True|False/)
        .map(str => str.toLowerCase() == "true")
        .desc("boolean ('true' or 'false')"),
    // A tag of the form '#stuff/hello-there'.
    tag: _ => parsimmon_umd_minExports.seqMap(parsimmon_umd_minExports.string("#"), parsimmon_umd_minExports.alt(parsimmon_umd_minExports.regexp(/[^\u2000-\u206F\u2E00-\u2E7F'!"#$%&()*+,.:;<=>?@^`{|}~\[\]\\\s]/).desc("text")).many(), (start, rest) => start + rest.join("")).desc("tag ('#hello/stuff')"),
    // A variable identifier, which is alphanumeric and must start with a letter or... emoji.
    identifier: _ => parsimmon_umd_minExports.seqMap(parsimmon_umd_minExports.alt(parsimmon_umd_minExports.regexp(/\p{Letter}/u), parsimmon_umd_minExports.regexp(EMOJI_REGEX).desc("text")), parsimmon_umd_minExports.alt(parsimmon_umd_minExports.regexp(/[0-9\p{Letter}_-]/u), parsimmon_umd_minExports.regexp(EMOJI_REGEX).desc("text")).many(), (first, rest) => first + rest.join("")).desc("variable identifier"),
    // An Obsidian link of the form [[<link>]].
    link: _ => parsimmon_umd_minExports.regexp(/\[\[([^\[\]]*?)\]\]/u, 1)
        .map(linkInner => parseInnerLink(linkInner))
        .desc("file link"),
    // An embeddable link which can start with '!'. This overlaps with the normal negation operator, so it is only
    // provided for metadata parsing.
    embedLink: q => parsimmon_umd_minExports.seqMap(parsimmon_umd_minExports.string("!").atMost(1), q.link, (p, l) => {
        if (p.length > 0)
            l.embed = true;
        return l;
    }).desc("file link"),
    // Binary plus or minus operator.
    binaryPlusMinus: _ => parsimmon_umd_minExports.regexp(/\+|-/)
        .map(str => str)
        .desc("'+' or '-'"),
    // Binary times or divide operator.
    binaryMulDiv: _ => parsimmon_umd_minExports.regexp(/\*|\/|%/)
        .map(str => str)
        .desc("'*' or '/' or '%'"),
    // Binary comparison operator.
    binaryCompareOp: _ => parsimmon_umd_minExports.regexp(/>=|<=|!=|>|<|=/)
        .map(str => str)
        .desc("'>=' or '<=' or '!=' or '=' or '>' or '<'"),
    // Binary boolean combination operator.
    binaryBooleanOp: _ => parsimmon_umd_minExports.regexp(/and|or|&|\|/i)
        .map(str => {
        if (str.toLowerCase() == "and")
            return "&";
        else if (str.toLowerCase() == "or")
            return "|";
        else
            return str;
    })
        .desc("'and' or 'or'"),
    // A date which can be YYYY-MM[-DDTHH:mm:ss].
    rootDate: _ => parsimmon_umd_minExports.seqMap(parsimmon_umd_minExports.regexp(/\d{4}/), parsimmon_umd_minExports.string("-"), parsimmon_umd_minExports.regexp(/\d{2}/), (year, _, month) => {
        return DateTime.fromObject({ year: Number.parseInt(year), month: Number.parseInt(month) });
    }).desc("date in format YYYY-MM[-DDTHH-MM-SS.MS]"),
    dateShorthand: _ => parsimmon_umd_minExports.alt(...Object.keys(DATE_SHORTHANDS)
        .sort((a, b) => b.length - a.length)
        .map(parsimmon_umd_minExports.string)),
    date: q => chainOpt(q.rootDate, (ym) => parsimmon_umd_minExports.seqMap(parsimmon_umd_minExports.string("-"), parsimmon_umd_minExports.regexp(/\d{2}/), (_, day) => ym.set({ day: Number.parseInt(day) })), (ymd) => parsimmon_umd_minExports.seqMap(parsimmon_umd_minExports.string("T"), parsimmon_umd_minExports.regexp(/\d{2}/), (_, hour) => ymd.set({ hour: Number.parseInt(hour) })), (ymdh) => parsimmon_umd_minExports.seqMap(parsimmon_umd_minExports.string(":"), parsimmon_umd_minExports.regexp(/\d{2}/), (_, minute) => ymdh.set({ minute: Number.parseInt(minute) })), (ymdhm) => parsimmon_umd_minExports.seqMap(parsimmon_umd_minExports.string(":"), parsimmon_umd_minExports.regexp(/\d{2}/), (_, second) => ymdhm.set({ second: Number.parseInt(second) })), (ymdhms) => parsimmon_umd_minExports.alt(parsimmon_umd_minExports.seqMap(parsimmon_umd_minExports.string("."), parsimmon_umd_minExports.regexp(/\d{3}/), (_, millisecond) => ymdhms.set({ millisecond: Number.parseInt(millisecond) })), parsimmon_umd_minExports.succeed(ymdhms) // pass
    ), (dt) => parsimmon_umd_minExports.alt(parsimmon_umd_minExports.seqMap(parsimmon_umd_minExports.string("+").or(parsimmon_umd_minExports.string("-")), parsimmon_umd_minExports.regexp(/\d{1,2}(:\d{2})?/), (pm, hr) => dt.setZone("UTC" + pm + hr, { keepLocalTime: true })), parsimmon_umd_minExports.seqMap(parsimmon_umd_minExports.string("Z"), () => dt.setZone("utc", { keepLocalTime: true })), parsimmon_umd_minExports.seqMap(parsimmon_umd_minExports.string("["), parsimmon_umd_minExports.regexp(/[0-9A-Za-z+-\/]+/u), parsimmon_umd_minExports.string("]"), (_a, zone, _b) => dt.setZone(zone, { keepLocalTime: true }))))
        .assert((dt) => dt.isValid, "valid date")
        .desc("date in format YYYY-MM[-DDTHH-MM-SS.MS]"),
    // A date, plus various shorthand times of day it could be.
    datePlus: q => parsimmon_umd_minExports.alt(q.dateShorthand.map(d => DATE_SHORTHANDS[d]()), q.date).desc("date in format YYYY-MM[-DDTHH-MM-SS.MS] or in shorthand"),
    // A duration of time.
    durationType: _ => parsimmon_umd_minExports.alt(...Object.keys(DURATION_TYPES)
        .sort((a, b) => b.length - a.length)
        .map(parsimmon_umd_minExports.string)),
    duration: q => parsimmon_umd_minExports.seqMap(q.number, parsimmon_umd_minExports.optWhitespace, q.durationType, (count, _, t) => DURATION_TYPES[t].mapUnits(x => x * count))
        .sepBy1(parsimmon_umd_minExports.string(",").trim(parsimmon_umd_minExports.optWhitespace).or(parsimmon_umd_minExports.optWhitespace))
        .map(durations => durations.reduce((p, c) => p.plus(c)))
        .desc("duration like 4hr2min"),
    // A raw null value.
    rawNull: _ => parsimmon_umd_minExports.string("null"),
    // Source parsing.
    tagSource: q => q.tag.map(tag => Sources.tag(tag)),
    csvSource: q => parsimmon_umd_minExports.seqMap(parsimmon_umd_minExports.string("csv(").skip(parsimmon_umd_minExports.optWhitespace), q.string, parsimmon_umd_minExports.string(")"), (_1, path, _2) => Sources.csv(path)),
    linkIncomingSource: q => q.link.map(link => Sources.link(link.path, true)),
    linkOutgoingSource: q => parsimmon_umd_minExports.seqMap(parsimmon_umd_minExports.string("outgoing(").skip(parsimmon_umd_minExports.optWhitespace), q.link, parsimmon_umd_minExports.string(")"), (_1, link, _2) => Sources.link(link.path, false)),
    folderSource: q => q.string.map(str => Sources.folder(str)),
    parensSource: q => parsimmon_umd_minExports.seqMap(parsimmon_umd_minExports.string("("), parsimmon_umd_minExports.optWhitespace, q.source, parsimmon_umd_minExports.optWhitespace, parsimmon_umd_minExports.string(")"), (_1, _2, field, _3, _4) => field),
    negateSource: q => parsimmon_umd_minExports.seqMap(parsimmon_umd_minExports.alt(parsimmon_umd_minExports.string("-"), parsimmon_umd_minExports.string("!")), q.atomSource, (_, source) => Sources.negate(source)),
    atomSource: q => parsimmon_umd_minExports.alt(q.parensSource, q.negateSource, q.linkOutgoingSource, q.linkIncomingSource, q.folderSource, q.tagSource, q.csvSource),
    binaryOpSource: q => createBinaryParser(q.atomSource, q.binaryBooleanOp.map(s => s), Sources.binaryOp),
    source: q => q.binaryOpSource,
    // Field parsing.
    variableField: q => q.identifier
        .chain(r => {
        if (KEYWORDS.includes(r.toUpperCase())) {
            return parsimmon_umd_minExports.fail("Variable fields cannot be a keyword (" + KEYWORDS.join(" or ") + ")");
        }
        else {
            return parsimmon_umd_minExports.succeed(Fields.variable(r));
        }
    })
        .desc("variable"),
    numberField: q => q.number.map(val => Fields.literal(val)).desc("number"),
    stringField: q => q.string.map(val => Fields.literal(val)).desc("string"),
    boolField: q => q.bool.map(val => Fields.literal(val)).desc("boolean"),
    dateField: q => parsimmon_umd_minExports.seqMap(parsimmon_umd_minExports.string("date("), parsimmon_umd_minExports.optWhitespace, q.datePlus, parsimmon_umd_minExports.optWhitespace, parsimmon_umd_minExports.string(")"), (prefix, _1, date, _2, postfix) => Fields.literal(date)).desc("date"),
    durationField: q => parsimmon_umd_minExports.seqMap(parsimmon_umd_minExports.string("dur("), parsimmon_umd_minExports.optWhitespace, q.duration, parsimmon_umd_minExports.optWhitespace, parsimmon_umd_minExports.string(")"), (prefix, _1, dur, _2, postfix) => Fields.literal(dur)).desc("duration"),
    nullField: q => q.rawNull.map(_ => Fields.NULL),
    linkField: q => q.link.map(f => Fields.literal(f)),
    listField: q => q.field
        .sepBy(parsimmon_umd_minExports.string(",").trim(parsimmon_umd_minExports.optWhitespace))
        .wrap(parsimmon_umd_minExports.string("[").skip(parsimmon_umd_minExports.optWhitespace), parsimmon_umd_minExports.optWhitespace.then(parsimmon_umd_minExports.string("]")))
        .map(l => Fields.list(l))
        .desc("list ('[1, 2, 3]')"),
    objectField: q => parsimmon_umd_minExports.seqMap(q.identifier.or(q.string), parsimmon_umd_minExports.string(":").trim(parsimmon_umd_minExports.optWhitespace), q.field, (name, _sep, value) => {
        return { name, value };
    })
        .sepBy(parsimmon_umd_minExports.string(",").trim(parsimmon_umd_minExports.optWhitespace))
        .wrap(parsimmon_umd_minExports.string("{").skip(parsimmon_umd_minExports.optWhitespace), parsimmon_umd_minExports.optWhitespace.then(parsimmon_umd_minExports.string("}")))
        .map(vals => {
        let res = {};
        for (let entry of vals)
            res[entry.name] = entry.value;
        return Fields.object(res);
    })
        .desc("object ('{ a: 1, b: 2 }')"),
    atomInlineField: q => parsimmon_umd_minExports.alt(q.date, q.duration.map(d => normalizeDuration(d)), q.string, q.tag, q.embedLink, q.bool, q.number, q.rawNull),
    inlineFieldList: q => q.atomInlineField.sepBy(parsimmon_umd_minExports.string(",").trim(parsimmon_umd_minExports.optWhitespace).lookahead(q.atomInlineField)),
    inlineField: q => parsimmon_umd_minExports.alt(parsimmon_umd_minExports.seqMap(q.atomInlineField, parsimmon_umd_minExports.string(",").trim(parsimmon_umd_minExports.optWhitespace), q.inlineFieldList, (f, _s, l) => [f].concat(l)), q.atomInlineField),
    atomField: q => parsimmon_umd_minExports.alt(
    // Place embed links above negated fields as they are the special parser case '![[thing]]' and are generally unambiguous.
    q.embedLink.map(l => Fields.literal(l)), q.negatedField, q.linkField, q.listField, q.objectField, q.lambdaField, q.parensField, q.boolField, q.numberField, q.stringField, q.dateField, q.durationField, q.nullField, q.variableField),
    indexField: q => parsimmon_umd_minExports.seqMap(q.atomField, parsimmon_umd_minExports.alt(q.dotPostfix, q.indexPostfix, q.functionPostfix).many(), (obj, postfixes) => {
        let result = obj;
        for (let post of postfixes) {
            switch (post.type) {
                case "dot":
                    result = Fields.index(result, Fields.literal(post.field));
                    break;
                case "index":
                    result = Fields.index(result, post.field);
                    break;
                case "function":
                    result = Fields.func(result, post.fields);
                    break;
            }
        }
        return result;
    }),
    negatedField: q => parsimmon_umd_minExports.seqMap(parsimmon_umd_minExports.string("!"), q.indexField, (_, field) => Fields.negate(field)).desc("negated field"),
    parensField: q => parsimmon_umd_minExports.seqMap(parsimmon_umd_minExports.string("("), parsimmon_umd_minExports.optWhitespace, q.field, parsimmon_umd_minExports.optWhitespace, parsimmon_umd_minExports.string(")"), (_1, _2, field, _3, _4) => field),
    lambdaField: q => parsimmon_umd_minExports.seqMap(q.identifier
        .sepBy(parsimmon_umd_minExports.string(",").trim(parsimmon_umd_minExports.optWhitespace))
        .wrap(parsimmon_umd_minExports.string("(").trim(parsimmon_umd_minExports.optWhitespace), parsimmon_umd_minExports.string(")").trim(parsimmon_umd_minExports.optWhitespace)), parsimmon_umd_minExports.string("=>").trim(parsimmon_umd_minExports.optWhitespace), q.field, (ident, _ignore, value) => {
        return { type: "lambda", arguments: ident, value };
    }),
    dotPostfix: q => parsimmon_umd_minExports.seqMap(parsimmon_umd_minExports.string("."), q.identifier, (_, field) => {
        return { type: "dot", field: field };
    }),
    indexPostfix: q => parsimmon_umd_minExports.seqMap(parsimmon_umd_minExports.string("["), parsimmon_umd_minExports.optWhitespace, q.field, parsimmon_umd_minExports.optWhitespace, parsimmon_umd_minExports.string("]"), (_, _2, field, _3, _4) => {
        return { type: "index", field };
    }),
    functionPostfix: q => parsimmon_umd_minExports.seqMap(parsimmon_umd_minExports.string("("), parsimmon_umd_minExports.optWhitespace, q.field.sepBy(parsimmon_umd_minExports.string(",").trim(parsimmon_umd_minExports.optWhitespace)), parsimmon_umd_minExports.optWhitespace, parsimmon_umd_minExports.string(")"), (_, _1, fields, _2, _3) => {
        return { type: "function", fields };
    }),
    // The precedence hierarchy of operators - multiply/divide, add/subtract, compare, and then boolean operations.
    binaryMulDivField: q => createBinaryParser(q.indexField, q.binaryMulDiv, Fields.binaryOp),
    binaryPlusMinusField: q => createBinaryParser(q.binaryMulDivField, q.binaryPlusMinus, Fields.binaryOp),
    binaryCompareField: q => createBinaryParser(q.binaryPlusMinusField, q.binaryCompareOp, Fields.binaryOp),
    binaryBooleanField: q => createBinaryParser(q.binaryCompareField, q.binaryBooleanOp, Fields.binaryOp),
    binaryOpField: q => q.binaryBooleanField,
    field: q => q.binaryOpField,
});
/**
 * Attempt to parse a field from the given text, returning a string error if the
 * parse failed.
 */
function parseField(text) {
    try {
        return Result.success(EXPRESSION.field.tryParse(text));
    }
    catch (error) {
        return Result.failure("" + error);
    }
}

/** Parse inline fields and other embedded metadata in a line. */
/** The wrapper characters that can be used to define an inline field. */
const INLINE_FIELD_WRAPPERS = Object.freeze({
    "[": "]",
    "(": ")",
});
/**
 * Find a matching closing bracket that occurs at or after `start`, respecting nesting and escapes. If found,
 * returns the value contained within and the string index after the end of the value.
 */
function findClosing(line, start, open, close) {
    let nesting = 0;
    let escaped = false;
    for (let index = start; index < line.length; index++) {
        let char = line.charAt(index);
        // Allows for double escapes like '\\' to be rendered normally.
        if (char == "\\") {
            escaped = !escaped;
            continue;
        }
        // If escaped, ignore the next character for computing nesting, regardless of what it is.
        if (escaped) {
            escaped = false;
            continue;
        }
        if (char == open)
            nesting++;
        else if (char == close)
            nesting--;
        // Only occurs if we are on a close character and there is no more nesting.
        if (nesting < 0)
            return { value: line.substring(start, index).trim(), endIndex: index + 1 };
        escaped = false;
    }
    return undefined;
}
/** Find the '::' separator in an inline field. */
function findSeparator(line, start) {
    let sep = line.indexOf("::", start);
    if (sep < 0)
        return undefined;
    return { key: line.substring(start, sep).trim(), valueIndex: sep + 2 };
}
/** Try to completely parse an inline field starting at the given position. Assumes `start` is on a wrapping character. */
function findSpecificInlineField(line, start) {
    let open = line.charAt(start);
    let key = findSeparator(line, start + 1);
    if (key === undefined)
        return undefined;
    // Fail the match if we find any separator characters (not allowed in keys).
    for (let sep of Object.keys(INLINE_FIELD_WRAPPERS).concat(Object.values(INLINE_FIELD_WRAPPERS))) {
        if (key.key.includes(sep))
            return undefined;
    }
    let value = findClosing(line, key.valueIndex, open, INLINE_FIELD_WRAPPERS[open]);
    if (value === undefined)
        return undefined;
    return {
        key: key.key,
        value: value.value,
        start: start,
        startValue: key.valueIndex,
        end: value.endIndex,
        wrapping: open,
    };
}
/** Parse a textual inline field value into something we can work with. */
function parseInlineValue(value) {
    // Empty inline values (i.e., no text) should map to null to match long-term Dataview semantics.
    // Null is also a more universal type to deal with than strings, since all functions accept nulls.
    if (value.trim() == "")
        return null;
    // The stripped literal field parser understands all of the non-array/non-object fields and can parse them for us.
    // Inline field objects are not currently supported; inline array objects have to be handled by the parser
    // separately.
    let inline = EXPRESSION.inlineField.parse(value);
    if (inline.status)
        return inline.value;
    else
        return value;
}
/** Extracts inline fields of the form '[key:: value]' from a line of text. This is done in a relatively
 * "robust" way to avoid failing due to bad nesting or other interfering Markdown symbols:
 *
 * - Look for any wrappers ('[' and '(') in the line, trying to parse whatever comes after it as an inline key::.
 * - If successful, scan until you find a matching end bracket, and parse whatever remains as an inline value.
 */
function extractInlineFields(line, includeTaskFields = false) {
    let fields = [];
    for (let wrapper of Object.keys(INLINE_FIELD_WRAPPERS)) {
        let foundIndex = line.indexOf(wrapper);
        while (foundIndex >= 0) {
            let parsedField = findSpecificInlineField(line, foundIndex);
            if (!parsedField) {
                foundIndex = line.indexOf(wrapper, foundIndex + 1);
                continue;
            }
            fields.push(parsedField);
            foundIndex = line.indexOf(wrapper, parsedField.end);
        }
    }
    if (includeTaskFields)
        fields = fields.concat(extractSpecialTaskFields(line));
    fields.sort((a, b) => a.start - b.start);
    let filteredFields = [];
    for (let i = 0; i < fields.length; i++) {
        if (i == 0 || filteredFields[filteredFields.length - 1].end < fields[i].start) {
            filteredFields.push(fields[i]);
        }
    }
    return filteredFields;
}
/** Validates that a raw field name has a valid form. */
const FULL_LINE_KEY_PART = parsimmon_umd_minExports.alt(parsimmon_umd_minExports.regexp(new RegExp(emojiRegex(), "u")), parsimmon_umd_minExports.regexp(/[0-9\p{Letter}\w\s_/-]+/u))
    .many()
    .map(parts => parts.join(""));
parsimmon_umd_minExports.regexp(/[^0-9\w\p{Letter}]*/u)
    .then(FULL_LINE_KEY_PART)
    .skip(parsimmon_umd_minExports.regexp(/[_\*~`]*/u));
const CREATED_DATE_REGEX = /\u{2795}\s*(\d{4}-\d{2}-\d{2})/u;
const DUE_DATE_REGEX = /(?:\u{1F4C5}|\u{1F4C6}|\u{1F5D3}\u{FE0F}?)\s*(\d{4}-\d{2}-\d{2})/u;
const DONE_DATE_REGEX = /\u{2705}\s*(\d{4}-\d{2}-\d{2})/u;
const SCHEDULED_DATE_REGEX = /[\u{23F3}\u{231B}]\s*(\d{4}-\d{2}-\d{2})/u;
const START_DATE_REGEX = /\u{1F6EB}\s*(\d{4}-\d{2}-\d{2})/u;
const EMOJI_REGEXES = [
    { regex: CREATED_DATE_REGEX, key: "created" },
    { regex: START_DATE_REGEX, key: "start" },
    { regex: SCHEDULED_DATE_REGEX, key: "scheduled" },
    { regex: DUE_DATE_REGEX, key: "due" },
    { regex: DONE_DATE_REGEX, key: "completion" },
];
/** Parse special completed/due/done task fields which are marked via emoji. */
function extractSpecialTaskFields(line) {
    let results = [];
    for (let { regex, key } of EMOJI_REGEXES) {
        const match = regex.exec(line);
        if (!match)
            continue;
        results.push({
            key,
            value: match[1],
            start: match.index,
            startValue: match.index + 1,
            end: match.index + match[0].length,
            wrapping: "emoji-shorthand",
        });
    }
    return results;
}
/** Sets or replaces the value of an inline field; if the value is 'undefined', deletes the key. */
function setInlineField(source, key, value) {
    let existing = extractInlineFields(source);
    let existingKeys = existing.filter(f => f.key == key);
    // Don't do anything if there are duplicate keys OR the key already doesn't exist.
    if (existingKeys.length > 2 || (existingKeys.length == 0 && !value))
        return source;
    let existingKey = existingKeys[0];
    let annotation = value ? `[${key}:: ${value}]` : "";
    if (existingKey) {
        let prefix = source.substring(0, existingKey.start);
        let suffix = source.substring(existingKey.end);
        if (annotation)
            return `${prefix}${annotation}${suffix}`;
        else
            return `${prefix}${suffix.trimStart()}`;
    }
    else if (annotation) {
        return `${source.trimEnd()} ${annotation}`;
    }
    return source;
}
function setEmojiShorthandCompletionField(source, value) {
    const existing = extractInlineFields(source, true);
    const existingKeys = existing.filter(f => f.key === "completion" && f.wrapping === "emoji-shorthand");
    // Don't do anything if there are duplicate keys OR the key already doesn't exist.
    if (existingKeys.length > 2 || (existingKeys.length == 0 && !value))
        return source;
    /* No wrapper, add own spacing at start */
    const annotation = value ? ` ✅ ${value}` : "";
    let existingKey = existingKeys[0];
    if (existingKey) {
        const prefix = source.substring(0, existingKey.start);
        const suffix = source.substring(existingKey.end);
        return `${prefix.trimEnd()}${annotation}${suffix}`;
    }
    else {
        return `${source.trimEnd()}${annotation}`;
    }
}

/** All extracted markdown file metadata obtained from a file. */
class PageMetadata {
    /** The path this file exists at. */
    path;
    /** Obsidian-provided date this page was created. */
    ctime;
    /** Obsidian-provided date this page was modified. */
    mtime;
    /** Obsidian-provided size of this page in bytes. */
    size;
    /** The day associated with this page, if relevant. */
    day;
    /** The first H1/H2 header in the file. May not exist. */
    title;
    /** All of the fields contained in this markdown file - both frontmatter AND in-file links. */
    fields;
    /** All of the exact tags (prefixed with '#') in this file overall. */
    tags;
    /** All of the aliases defined for this file. */
    aliases;
    /** All OUTGOING links (including embeds, header + block links) in this file. */
    links;
    /** All list items contained within this page. Filter for tasks to get just tasks. */
    lists;
    /** The raw frontmatter for this document. */
    frontmatter;
    constructor(path, init) {
        this.path = path;
        this.fields = new Map();
        this.frontmatter = {};
        this.tags = new Set();
        this.aliases = new Set();
        this.links = [];
        Object.assign(this, init);
        this.lists = (this.lists || []).map(l => new ListItem$1(l));
    }
    /** Canonicalize raw links and other data in partial data with normalizers, returning a completed object. */
    static canonicalize(data, linkNormalizer) {
        // Mutate the data for now, which is probably a bad idea but... all well.
        if (data.frontmatter) {
            data.frontmatter = Values.mapLeaves(data.frontmatter, t => Values.isLink(t) ? linkNormalizer(t) : t);
        }
        if (data.fields) {
            for (let [key, value] of data.fields.entries()) {
                data.fields.set(key, Values.mapLeaves(value, t => (Values.isLink(t) ? linkNormalizer(t) : t)));
            }
        }
        if (data.lists) {
            for (let item of data.lists) {
                for (let [key, value] of item.fields.entries()) {
                    item.fields.set(key, value.map(x => Values.mapLeaves(x, t => (Values.isLink(t) ? linkNormalizer(t) : t))));
                }
            }
        }
        if (data.links) {
            data.links = data.links.map(l => linkNormalizer(l));
        }
        // This is pretty ugly, but it's not possible to normalize on the worker thread that does parsing.
        // The best way to improve this is to instead just canonicalize the entire data object; I can try to
        // optimize `Values.mapLeaves` to only mutate if it actually changes things.
        return new PageMetadata(data.path, data);
    }
    /** The name (based on path) of this file. */
    name() {
        return getFileTitle(this.path);
    }
    /** The containing folder (based on path) of this file. */
    folder() {
        return getParentFolder(this.path);
    }
    /** The extension of this file (likely 'md'). */
    extension() {
        return getExtension(this.path);
    }
    /** Return a set of tags AND all of their parent tags (so #hello/yes would become #hello, #hello/yes). */
    fullTags() {
        let result = new Set();
        for (let tag of this.tags) {
            for (let subtag of extractSubtags(tag))
                result.add(subtag);
        }
        return result;
    }
    /** Convert all links in this file to file links. */
    fileLinks() {
        // We want to make them distinct, but where links are not raw links we
        // now keep the additional metadata.
        let distinctLinks = new Set(this.links);
        return Array.from(distinctLinks);
    }
    /** Map this metadata to a full object; uses the index for additional data lookups.  */
    serialize(index, cache) {
        // Convert list items via the canonicalization cache.
        let realCache = cache ?? new ListSerializationCache(this.lists);
        let result = {
            file: {
                path: this.path,
                folder: this.folder(),
                name: this.name(),
                link: Link.file(this.path),
                outlinks: this.fileLinks(),
                inlinks: Array.from(index.links.getInverse(this.path)).map(l => Link.file(l)),
                etags: Array.from(this.tags),
                tags: Array.from(this.fullTags()),
                aliases: Array.from(this.aliases),
                lists: this.lists.map(l => realCache.get(l.line)),
                tasks: this.lists.filter(l => !!l.task).map(l => realCache.get(l.line)),
                ctime: this.ctime,
                cday: stripTime(this.ctime),
                mtime: this.mtime,
                mday: stripTime(this.mtime),
                size: this.size,
                starred: index.starred.starred(this.path),
                frontmatter: Values.deepCopy(this.frontmatter),
                ext: this.extension(),
            },
        };
        // Add the current day if present.
        if (this.day)
            result.file.day = this.day;
        // Then append the computed fields.
        for (let [key, value] of this.fields.entries()) {
            if (key in result)
                continue; // Don't allow fields to override existing keys.
            result[key] = value;
        }
        return result;
    }
}
/** A list item inside of a list. */
class ListItem$1 {
    /** The symbol ('*', '-', '1.') used to define this list item. */
    symbol;
    /** A link which points to this task, or to the closest block that this task is contained in. */
    link;
    /** A link to the section that contains this list element; could be a file if this is not in a section. */
    section;
    /** The text of this list item. This may be multiple lines of markdown. */
    text;
    /** The line that this list item starts on in the file. */
    line;
    /** The number of lines that define this list item. */
    lineCount;
    /** The line number for the first list item in the list this item belongs to. */
    list;
    /** Any links contained within this list item. */
    links;
    /** The tags contained within this list item. */
    tags;
    /** The raw Obsidian-provided position for where this task is. */
    position;
    /** The line number of the parent list item, if present; if this is undefined, this is a root item. */
    parent;
    /** The line numbers of children of this list item. */
    children;
    /** The block ID for this item, if one is present. */
    blockId;
    /** Any fields defined in this list item. For tasks, this includes fields underneath the task. */
    fields;
    task;
    constructor(init) {
        Object.assign(this, init);
        this.fields = this.fields || new Map();
        this.tags = this.tags || new Set();
        this.children = this.children || [];
        this.links = this.links || [];
    }
    id() {
        return `${this.file().path}-${this.line}`;
    }
    file() {
        return this.link.toFile();
    }
    markdown() {
        if (this.task)
            return `${this.symbol} [${this.task.completed ? "x" : " "}] ${this.text}`;
        else
            return `${this.symbol} ${this.text}`;
    }
    created() {
        return (this.fields.get("created") ?? this.fields.get("ctime") ?? this.fields.get("cday"))?.[0];
    }
    due() {
        return (this.fields.get("due") ?? this.fields.get("duetime") ?? this.fields.get("dueday"))?.[0];
    }
    completed() {
        return (this.fields.get("completed") ??
            this.fields.get("completion") ??
            this.fields.get("comptime") ??
            this.fields.get("compday"))?.[0];
    }
    start() {
        return this.fields.get("start")?.[0];
    }
    scheduled() {
        return this.fields.get("scheduled")?.[0];
    }
    /** Create an API-friendly copy of this list item. De-duplication is done via the provided cache. */
    serialize(cache) {
        // Map children to their serialized/de-duplicated equivalents right away.
        let children = this.children.map(l => cache.get(l)).filter((l) => l !== undefined);
        let result = {
            symbol: this.symbol,
            link: this.link,
            section: this.section,
            text: this.text,
            tags: Array.from(this.tags),
            line: this.line,
            lineCount: this.lineCount,
            list: this.list,
            outlinks: Array.from(this.links),
            path: this.link.path,
            children: children,
            task: !!this.task,
            annotated: this.fields.size > 0,
            position: Values.deepCopy(this.position),
            subtasks: children,
            real: !!this.task,
            header: this.section, // @deprecated, use 'item.section' instead.
        };
        if (this.parent || this.parent === 0)
            result.parent = this.parent;
        if (this.blockId)
            result.blockId = this.blockId;
        addFields(this.fields, result);
        if (this.task) {
            result.status = this.task.status;
            result.checked = this.task.checked;
            result.completed = this.task.completed;
            result.fullyCompleted = this.task.fullyCompleted;
            let created = this.created(), due = this.due(), completed = this.completed(), start = this.start(), scheduled = this.scheduled();
            if (created)
                result.created = Values.deepCopy(created);
            if (due)
                result.due = Values.deepCopy(due);
            if (completed)
                result.completion = Values.deepCopy(completed);
            if (start)
                result.start = Values.deepCopy(start);
            if (scheduled)
                result.scheduled = Values.deepCopy(scheduled);
        }
        return result;
    }
}
//////////////////////////////////////////
// Conversion / Serialization Utilities //
//////////////////////////////////////////
/** De-duplicates list items across section metadata and page metadata. */
class ListSerializationCache {
    listItems;
    cache;
    seen;
    constructor(listItems) {
        this.listItems = {};
        this.cache = {};
        this.seen = new Set();
        for (let item of listItems)
            this.listItems[item.line] = item;
    }
    get(lineno) {
        if (lineno in this.cache)
            return this.cache[lineno];
        else if (this.seen.has(lineno)) {
            console.log(`Dataview: Encountered a circular list (line number ${lineno}; children ${this.listItems[lineno].children.join(", ")})`);
            return undefined;
        }
        this.seen.add(lineno);
        let result = this.listItems[lineno].serialize(this);
        this.cache[lineno] = result;
        return result;
    }
}
function addFields(fields, target) {
    for (let [key, values] of fields.entries()) {
        if (key in target)
            continue;
        target[key] = values.length == 1 ? values[0] : values;
    }
    return target;
}

/** Importer for markdown documents. */
// TODO: Consider using an actual parser in lieu of a more expensive regex.
const LIST_ITEM_REGEX = /^[\s>]*(\d+\.|\d+\)|\*|-|\+)\s*(\[.{0,1}\])?\s*(.*)$/mu;
/** Recursively convert frontmatter into fields. We have to dance around YAML structure. */
function parseFrontmatter(value) {
    if (value == null) {
        return null;
    }
    else if (typeof value === "object") {
        if (Array.isArray(value)) {
            let result = [];
            for (let child of value) {
                result.push(parseFrontmatter(child));
            }
            return result;
        }
        else if (value instanceof Date) {
            let dateParse = DateTime.fromJSDate(value);
            return dateParse;
        }
        else {
            let object = value;
            let result = {};
            for (let key in object) {
                result[key] = parseFrontmatter(object[key]);
            }
            return result;
        }
    }
    else if (typeof value === "number") {
        return value;
    }
    else if (typeof value === "boolean") {
        return value;
    }
    else if (typeof value === "string") {
        let dateParse = EXPRESSION.date.parse(value);
        if (dateParse.status)
            return dateParse.value;
        let durationParse = EXPRESSION.duration.parse(value);
        if (durationParse.status)
            return durationParse.value;
        let linkParse = EXPRESSION.embedLink.parse(value);
        if (linkParse.status)
            return linkParse.value;
        return value;
    }
    // Backup if we don't understand the type.
    return null;
}

/** Parse a CSV file into a collection of data rows. */
function parseCsv(content) {
    let parsed = papaparse_minExports.parse(content, {
        header: true,
        skipEmptyLines: true,
        comments: "#",
        dynamicTyping: true,
    });
    let rows = [];
    for (let parsedRow of parsed.data) {
        let fields = parseFrontmatter(parsedRow);
        let result = {};
        for (let [key, value] of Object.entries(fields)) {
            result[key] = value;
            result[canonicalizeVarName(key)] = value;
        }
        rows.push(result);
    }
    return rows;
}

/** Simplifies passing dataview values across the JS web worker barrier. */
var Transferable;
(function (Transferable) {
    /** Convert a literal value to a serializer-friendly transferable value. */
    function transferable(value) {
        // Handle simple universal types first.
        if (value instanceof Map) {
            let copied = new Map();
            for (let [key, val] of value.entries())
                copied.set(transferable(key), transferable(val));
            return copied;
        }
        else if (value instanceof Set) {
            let copied = new Set();
            for (let val of value)
                copied.add(transferable(val));
            return copied;
        }
        let wrapped = Values.wrapValue(value);
        if (wrapped === undefined)
            throw Error("Unrecognized transferable value: " + value);
        switch (wrapped.type) {
            case "null":
            case "number":
            case "string":
            case "boolean":
                return wrapped.value;
            case "date":
                return {
                    "___transfer-type": "date",
                    value: transferable(wrapped.value.toObject()),
                    options: {
                        zone: wrapped.value.zone.equals(SystemZone.instance) ? undefined : wrapped.value.zoneName,
                    },
                };
            case "duration":
                return { "___transfer-type": "duration", value: transferable(wrapped.value.toObject()) };
            case "array":
                return wrapped.value.map(v => transferable(v));
            case "link":
                return { "___transfer-type": "link", value: transferable(wrapped.value.toObject()) };
            case "object":
                let result = {};
                for (let [key, value] of Object.entries(wrapped.value))
                    result[key] = transferable(value);
                return result;
        }
    }
    Transferable.transferable = transferable;
    /** Convert a transferable value back to a literal value we can work with. */
    function value(transferable) {
        if (transferable === null) {
            return null;
        }
        else if (transferable === undefined) {
            return undefined;
        }
        else if (transferable instanceof Map) {
            let real = new Map();
            for (let [key, val] of transferable.entries())
                real.set(value(key), value(val));
            return real;
        }
        else if (transferable instanceof Set) {
            let real = new Set();
            for (let val of transferable)
                real.add(value(val));
            return real;
        }
        else if (Array.isArray(transferable)) {
            return transferable.map(v => value(v));
        }
        else if (typeof transferable === "object") {
            if ("___transfer-type" in transferable) {
                switch (transferable["___transfer-type"]) {
                    case "date":
                        let dateOpts = value(transferable.options);
                        let dateData = value(transferable.value);
                        return DateTime.fromObject(dateData, { zone: dateOpts.zone });
                    case "duration":
                        return Duration.fromObject(value(transferable.value));
                    case "link":
                        return Link.fromObject(value(transferable.value));
                    default:
                        throw Error(`Unrecognized transfer type '${transferable["___transfer-type"]}'`);
                }
            }
            let result = {};
            for (let [key, val] of Object.entries(transferable))
                result[key] = value(val);
            return result;
        }
        return transferable;
    }
    Transferable.value = value;
})(Transferable || (Transferable = {}));

function commonjsRequire(path) {
	throw new Error('Could not dynamically require "' + path + '". Please configure the dynamicRequireTargets or/and ignoreDynamicRequires option of @rollup/plugin-commonjs appropriately for this require call to work.');
}

var localforage$1 = {exports: {}};

/*!
    localForage -- Offline Storage, Improved
    Version 1.10.0
    https://localforage.github.io/localForage
    (c) 2013-2017 Mozilla, Apache License 2.0
*/
localforage$1.exports;

(function (module, exports) {
	(function(f){{module.exports=f();}})(function(){return (function e(t,n,r){function s(o,u){if(!n[o]){if(!t[o]){var a=typeof commonjsRequire=="function"&&commonjsRequire;if(!u&&a)return a(o,!0);if(i)return i(o,!0);var f=new Error("Cannot find module '"+o+"'");throw (f.code="MODULE_NOT_FOUND", f)}var l=n[o]={exports:{}};t[o][0].call(l.exports,function(e){var n=t[o][1][e];return s(n?n:e)},l,l.exports,e,t,n,r);}return n[o].exports}var i=typeof commonjsRequire=="function"&&commonjsRequire;for(var o=0;o<r.length;o++)s(r[o]);return s})({1:[function(_dereq_,module,exports){
	(function (global){
	var Mutation = global.MutationObserver || global.WebKitMutationObserver;

	var scheduleDrain;

	{
	  if (Mutation) {
	    var called = 0;
	    var observer = new Mutation(nextTick);
	    var element = global.document.createTextNode('');
	    observer.observe(element, {
	      characterData: true
	    });
	    scheduleDrain = function () {
	      element.data = (called = ++called % 2);
	    };
	  } else if (!global.setImmediate && typeof global.MessageChannel !== 'undefined') {
	    var channel = new global.MessageChannel();
	    channel.port1.onmessage = nextTick;
	    scheduleDrain = function () {
	      channel.port2.postMessage(0);
	    };
	  } else if ('document' in global && 'onreadystatechange' in global.document.createElement('script')) {
	    scheduleDrain = function () {

	      // Create a <script> element; its readystatechange event will be fired asynchronously once it is inserted
	      // into the document. Do so, thus queuing up the task. Remember to clean up once it's been called.
	      var scriptEl = global.document.createElement('script');
	      scriptEl.onreadystatechange = function () {
	        nextTick();

	        scriptEl.onreadystatechange = null;
	        scriptEl.parentNode.removeChild(scriptEl);
	        scriptEl = null;
	      };
	      global.document.documentElement.appendChild(scriptEl);
	    };
	  } else {
	    scheduleDrain = function () {
	      setTimeout(nextTick, 0);
	    };
	  }
	}

	var draining;
	var queue = [];
	//named nextTick for less confusing stack traces
	function nextTick() {
	  draining = true;
	  var i, oldQueue;
	  var len = queue.length;
	  while (len) {
	    oldQueue = queue;
	    queue = [];
	    i = -1;
	    while (++i < len) {
	      oldQueue[i]();
	    }
	    len = queue.length;
	  }
	  draining = false;
	}

	module.exports = immediate;
	function immediate(task) {
	  if (queue.push(task) === 1 && !draining) {
	    scheduleDrain();
	  }
	}

	}).call(this,typeof commonjsGlobal !== "undefined" ? commonjsGlobal : typeof self !== "undefined" ? self : typeof window !== "undefined" ? window : {});
	},{}],2:[function(_dereq_,module,exports){
	var immediate = _dereq_(1);

	/* istanbul ignore next */
	function INTERNAL() {}

	var handlers = {};

	var REJECTED = ['REJECTED'];
	var FULFILLED = ['FULFILLED'];
	var PENDING = ['PENDING'];

	module.exports = Promise;

	function Promise(resolver) {
	  if (typeof resolver !== 'function') {
	    throw new TypeError('resolver must be a function');
	  }
	  this.state = PENDING;
	  this.queue = [];
	  this.outcome = void 0;
	  if (resolver !== INTERNAL) {
	    safelyResolveThenable(this, resolver);
	  }
	}

	Promise.prototype["catch"] = function (onRejected) {
	  return this.then(null, onRejected);
	};
	Promise.prototype.then = function (onFulfilled, onRejected) {
	  if (typeof onFulfilled !== 'function' && this.state === FULFILLED ||
	    typeof onRejected !== 'function' && this.state === REJECTED) {
	    return this;
	  }
	  var promise = new this.constructor(INTERNAL);
	  if (this.state !== PENDING) {
	    var resolver = this.state === FULFILLED ? onFulfilled : onRejected;
	    unwrap(promise, resolver, this.outcome);
	  } else {
	    this.queue.push(new QueueItem(promise, onFulfilled, onRejected));
	  }

	  return promise;
	};
	function QueueItem(promise, onFulfilled, onRejected) {
	  this.promise = promise;
	  if (typeof onFulfilled === 'function') {
	    this.onFulfilled = onFulfilled;
	    this.callFulfilled = this.otherCallFulfilled;
	  }
	  if (typeof onRejected === 'function') {
	    this.onRejected = onRejected;
	    this.callRejected = this.otherCallRejected;
	  }
	}
	QueueItem.prototype.callFulfilled = function (value) {
	  handlers.resolve(this.promise, value);
	};
	QueueItem.prototype.otherCallFulfilled = function (value) {
	  unwrap(this.promise, this.onFulfilled, value);
	};
	QueueItem.prototype.callRejected = function (value) {
	  handlers.reject(this.promise, value);
	};
	QueueItem.prototype.otherCallRejected = function (value) {
	  unwrap(this.promise, this.onRejected, value);
	};

	function unwrap(promise, func, value) {
	  immediate(function () {
	    var returnValue;
	    try {
	      returnValue = func(value);
	    } catch (e) {
	      return handlers.reject(promise, e);
	    }
	    if (returnValue === promise) {
	      handlers.reject(promise, new TypeError('Cannot resolve promise with itself'));
	    } else {
	      handlers.resolve(promise, returnValue);
	    }
	  });
	}

	handlers.resolve = function (self, value) {
	  var result = tryCatch(getThen, value);
	  if (result.status === 'error') {
	    return handlers.reject(self, result.value);
	  }
	  var thenable = result.value;

	  if (thenable) {
	    safelyResolveThenable(self, thenable);
	  } else {
	    self.state = FULFILLED;
	    self.outcome = value;
	    var i = -1;
	    var len = self.queue.length;
	    while (++i < len) {
	      self.queue[i].callFulfilled(value);
	    }
	  }
	  return self;
	};
	handlers.reject = function (self, error) {
	  self.state = REJECTED;
	  self.outcome = error;
	  var i = -1;
	  var len = self.queue.length;
	  while (++i < len) {
	    self.queue[i].callRejected(error);
	  }
	  return self;
	};

	function getThen(obj) {
	  // Make sure we only access the accessor once as required by the spec
	  var then = obj && obj.then;
	  if (obj && (typeof obj === 'object' || typeof obj === 'function') && typeof then === 'function') {
	    return function appyThen() {
	      then.apply(obj, arguments);
	    };
	  }
	}

	function safelyResolveThenable(self, thenable) {
	  // Either fulfill, reject or reject with error
	  var called = false;
	  function onError(value) {
	    if (called) {
	      return;
	    }
	    called = true;
	    handlers.reject(self, value);
	  }

	  function onSuccess(value) {
	    if (called) {
	      return;
	    }
	    called = true;
	    handlers.resolve(self, value);
	  }

	  function tryToUnwrap() {
	    thenable(onSuccess, onError);
	  }

	  var result = tryCatch(tryToUnwrap);
	  if (result.status === 'error') {
	    onError(result.value);
	  }
	}

	function tryCatch(func, value) {
	  var out = {};
	  try {
	    out.value = func(value);
	    out.status = 'success';
	  } catch (e) {
	    out.status = 'error';
	    out.value = e;
	  }
	  return out;
	}

	Promise.resolve = resolve;
	function resolve(value) {
	  if (value instanceof this) {
	    return value;
	  }
	  return handlers.resolve(new this(INTERNAL), value);
	}

	Promise.reject = reject;
	function reject(reason) {
	  var promise = new this(INTERNAL);
	  return handlers.reject(promise, reason);
	}

	Promise.all = all;
	function all(iterable) {
	  var self = this;
	  if (Object.prototype.toString.call(iterable) !== '[object Array]') {
	    return this.reject(new TypeError('must be an array'));
	  }

	  var len = iterable.length;
	  var called = false;
	  if (!len) {
	    return this.resolve([]);
	  }

	  var values = new Array(len);
	  var resolved = 0;
	  var i = -1;
	  var promise = new this(INTERNAL);

	  while (++i < len) {
	    allResolver(iterable[i], i);
	  }
	  return promise;
	  function allResolver(value, i) {
	    self.resolve(value).then(resolveFromAll, function (error) {
	      if (!called) {
	        called = true;
	        handlers.reject(promise, error);
	      }
	    });
	    function resolveFromAll(outValue) {
	      values[i] = outValue;
	      if (++resolved === len && !called) {
	        called = true;
	        handlers.resolve(promise, values);
	      }
	    }
	  }
	}

	Promise.race = race;
	function race(iterable) {
	  var self = this;
	  if (Object.prototype.toString.call(iterable) !== '[object Array]') {
	    return this.reject(new TypeError('must be an array'));
	  }

	  var len = iterable.length;
	  var called = false;
	  if (!len) {
	    return this.resolve([]);
	  }

	  var i = -1;
	  var promise = new this(INTERNAL);

	  while (++i < len) {
	    resolver(iterable[i]);
	  }
	  return promise;
	  function resolver(value) {
	    self.resolve(value).then(function (response) {
	      if (!called) {
	        called = true;
	        handlers.resolve(promise, response);
	      }
	    }, function (error) {
	      if (!called) {
	        called = true;
	        handlers.reject(promise, error);
	      }
	    });
	  }
	}

	},{"1":1}],3:[function(_dereq_,module,exports){
	(function (global){
	if (typeof global.Promise !== 'function') {
	  global.Promise = _dereq_(2);
	}

	}).call(this,typeof commonjsGlobal !== "undefined" ? commonjsGlobal : typeof self !== "undefined" ? self : typeof window !== "undefined" ? window : {});
	},{"2":2}],4:[function(_dereq_,module,exports){

	var _typeof = typeof Symbol === "function" && typeof Symbol.iterator === "symbol" ? function (obj) { return typeof obj; } : function (obj) { return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj; };

	function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

	function getIDB() {
	    /* global indexedDB,webkitIndexedDB,mozIndexedDB,OIndexedDB,msIndexedDB */
	    try {
	        if (typeof indexedDB !== 'undefined') {
	            return indexedDB;
	        }
	        if (typeof webkitIndexedDB !== 'undefined') {
	            return webkitIndexedDB;
	        }
	        if (typeof mozIndexedDB !== 'undefined') {
	            return mozIndexedDB;
	        }
	        if (typeof OIndexedDB !== 'undefined') {
	            return OIndexedDB;
	        }
	        if (typeof msIndexedDB !== 'undefined') {
	            return msIndexedDB;
	        }
	    } catch (e) {
	        return;
	    }
	}

	var idb = getIDB();

	function isIndexedDBValid() {
	    try {
	        // Initialize IndexedDB; fall back to vendor-prefixed versions
	        // if needed.
	        if (!idb || !idb.open) {
	            return false;
	        }
	        // We mimic PouchDB here;
	        //
	        // We test for openDatabase because IE Mobile identifies itself
	        // as Safari. Oh the lulz...
	        var isSafari = typeof openDatabase !== 'undefined' && /(Safari|iPhone|iPad|iPod)/.test(navigator.userAgent) && !/Chrome/.test(navigator.userAgent) && !/BlackBerry/.test(navigator.platform);

	        var hasFetch = typeof fetch === 'function' && fetch.toString().indexOf('[native code') !== -1;

	        // Safari <10.1 does not meet our requirements for IDB support
	        // (see: https://github.com/pouchdb/pouchdb/issues/5572).
	        // Safari 10.1 shipped with fetch, we can use that to detect it.
	        // Note: this creates issues with `window.fetch` polyfills and
	        // overrides; see:
	        // https://github.com/localForage/localForage/issues/856
	        return (!isSafari || hasFetch) && typeof indexedDB !== 'undefined' &&
	        // some outdated implementations of IDB that appear on Samsung
	        // and HTC Android devices <4.4 are missing IDBKeyRange
	        // See: https://github.com/mozilla/localForage/issues/128
	        // See: https://github.com/mozilla/localForage/issues/272
	        typeof IDBKeyRange !== 'undefined';
	    } catch (e) {
	        return false;
	    }
	}

	// Abstracts constructing a Blob object, so it also works in older
	// browsers that don't support the native Blob constructor. (i.e.
	// old QtWebKit versions, at least).
	// Abstracts constructing a Blob object, so it also works in older
	// browsers that don't support the native Blob constructor. (i.e.
	// old QtWebKit versions, at least).
	function createBlob(parts, properties) {
	    /* global BlobBuilder,MSBlobBuilder,MozBlobBuilder,WebKitBlobBuilder */
	    parts = parts || [];
	    properties = properties || {};
	    try {
	        return new Blob(parts, properties);
	    } catch (e) {
	        if (e.name !== 'TypeError') {
	            throw e;
	        }
	        var Builder = typeof BlobBuilder !== 'undefined' ? BlobBuilder : typeof MSBlobBuilder !== 'undefined' ? MSBlobBuilder : typeof MozBlobBuilder !== 'undefined' ? MozBlobBuilder : WebKitBlobBuilder;
	        var builder = new Builder();
	        for (var i = 0; i < parts.length; i += 1) {
	            builder.append(parts[i]);
	        }
	        return builder.getBlob(properties.type);
	    }
	}

	// This is CommonJS because lie is an external dependency, so Rollup
	// can just ignore it.
	if (typeof Promise === 'undefined') {
	    // In the "nopromises" build this will just throw if you don't have
	    // a global promise object, but it would throw anyway later.
	    _dereq_(3);
	}
	var Promise$1 = Promise;

	function executeCallback(promise, callback) {
	    if (callback) {
	        promise.then(function (result) {
	            callback(null, result);
	        }, function (error) {
	            callback(error);
	        });
	    }
	}

	function executeTwoCallbacks(promise, callback, errorCallback) {
	    if (typeof callback === 'function') {
	        promise.then(callback);
	    }

	    if (typeof errorCallback === 'function') {
	        promise["catch"](errorCallback);
	    }
	}

	function normalizeKey(key) {
	    // Cast the key to a string, as that's all we can set as a key.
	    if (typeof key !== 'string') {
	        console.warn(key + ' used as a key, but it is not a string.');
	        key = String(key);
	    }

	    return key;
	}

	function getCallback() {
	    if (arguments.length && typeof arguments[arguments.length - 1] === 'function') {
	        return arguments[arguments.length - 1];
	    }
	}

	// Some code originally from async_storage.js in
	// [Gaia](https://github.com/mozilla-b2g/gaia).

	var DETECT_BLOB_SUPPORT_STORE = 'local-forage-detect-blob-support';
	var supportsBlobs = void 0;
	var dbContexts = {};
	var toString = Object.prototype.toString;

	// Transaction Modes
	var READ_ONLY = 'readonly';
	var READ_WRITE = 'readwrite';

	// Transform a binary string to an array buffer, because otherwise
	// weird stuff happens when you try to work with the binary string directly.
	// It is known.
	// From http://stackoverflow.com/questions/14967647/ (continues on next line)
	// encode-decode-image-with-base64-breaks-image (2013-04-21)
	function _binStringToArrayBuffer(bin) {
	    var length = bin.length;
	    var buf = new ArrayBuffer(length);
	    var arr = new Uint8Array(buf);
	    for (var i = 0; i < length; i++) {
	        arr[i] = bin.charCodeAt(i);
	    }
	    return buf;
	}

	//
	// Blobs are not supported in all versions of IndexedDB, notably
	// Chrome <37 and Android <5. In those versions, storing a blob will throw.
	//
	// Various other blob bugs exist in Chrome v37-42 (inclusive).
	// Detecting them is expensive and confusing to users, and Chrome 37-42
	// is at very low usage worldwide, so we do a hacky userAgent check instead.
	//
	// content-type bug: https://code.google.com/p/chromium/issues/detail?id=408120
	// 404 bug: https://code.google.com/p/chromium/issues/detail?id=447916
	// FileReader bug: https://code.google.com/p/chromium/issues/detail?id=447836
	//
	// Code borrowed from PouchDB. See:
	// https://github.com/pouchdb/pouchdb/blob/master/packages/node_modules/pouchdb-adapter-idb/src/blobSupport.js
	//
	function _checkBlobSupportWithoutCaching(idb) {
	    return new Promise$1(function (resolve) {
	        var txn = idb.transaction(DETECT_BLOB_SUPPORT_STORE, READ_WRITE);
	        var blob = createBlob(['']);
	        txn.objectStore(DETECT_BLOB_SUPPORT_STORE).put(blob, 'key');

	        txn.onabort = function (e) {
	            // If the transaction aborts now its due to not being able to
	            // write to the database, likely due to the disk being full
	            e.preventDefault();
	            e.stopPropagation();
	            resolve(false);
	        };

	        txn.oncomplete = function () {
	            var matchedChrome = navigator.userAgent.match(/Chrome\/(\d+)/);
	            var matchedEdge = navigator.userAgent.match(/Edge\//);
	            // MS Edge pretends to be Chrome 42:
	            // https://msdn.microsoft.com/en-us/library/hh869301%28v=vs.85%29.aspx
	            resolve(matchedEdge || !matchedChrome || parseInt(matchedChrome[1], 10) >= 43);
	        };
	    })["catch"](function () {
	        return false; // error, so assume unsupported
	    });
	}

	function _checkBlobSupport(idb) {
	    if (typeof supportsBlobs === 'boolean') {
	        return Promise$1.resolve(supportsBlobs);
	    }
	    return _checkBlobSupportWithoutCaching(idb).then(function (value) {
	        supportsBlobs = value;
	        return supportsBlobs;
	    });
	}

	function _deferReadiness(dbInfo) {
	    var dbContext = dbContexts[dbInfo.name];

	    // Create a deferred object representing the current database operation.
	    var deferredOperation = {};

	    deferredOperation.promise = new Promise$1(function (resolve, reject) {
	        deferredOperation.resolve = resolve;
	        deferredOperation.reject = reject;
	    });

	    // Enqueue the deferred operation.
	    dbContext.deferredOperations.push(deferredOperation);

	    // Chain its promise to the database readiness.
	    if (!dbContext.dbReady) {
	        dbContext.dbReady = deferredOperation.promise;
	    } else {
	        dbContext.dbReady = dbContext.dbReady.then(function () {
	            return deferredOperation.promise;
	        });
	    }
	}

	function _advanceReadiness(dbInfo) {
	    var dbContext = dbContexts[dbInfo.name];

	    // Dequeue a deferred operation.
	    var deferredOperation = dbContext.deferredOperations.pop();

	    // Resolve its promise (which is part of the database readiness
	    // chain of promises).
	    if (deferredOperation) {
	        deferredOperation.resolve();
	        return deferredOperation.promise;
	    }
	}

	function _rejectReadiness(dbInfo, err) {
	    var dbContext = dbContexts[dbInfo.name];

	    // Dequeue a deferred operation.
	    var deferredOperation = dbContext.deferredOperations.pop();

	    // Reject its promise (which is part of the database readiness
	    // chain of promises).
	    if (deferredOperation) {
	        deferredOperation.reject(err);
	        return deferredOperation.promise;
	    }
	}

	function _getConnection(dbInfo, upgradeNeeded) {
	    return new Promise$1(function (resolve, reject) {
	        dbContexts[dbInfo.name] = dbContexts[dbInfo.name] || createDbContext();

	        if (dbInfo.db) {
	            if (upgradeNeeded) {
	                _deferReadiness(dbInfo);
	                dbInfo.db.close();
	            } else {
	                return resolve(dbInfo.db);
	            }
	        }

	        var dbArgs = [dbInfo.name];

	        if (upgradeNeeded) {
	            dbArgs.push(dbInfo.version);
	        }

	        var openreq = idb.open.apply(idb, dbArgs);

	        if (upgradeNeeded) {
	            openreq.onupgradeneeded = function (e) {
	                var db = openreq.result;
	                try {
	                    db.createObjectStore(dbInfo.storeName);
	                    if (e.oldVersion <= 1) {
	                        // Added when support for blob shims was added
	                        db.createObjectStore(DETECT_BLOB_SUPPORT_STORE);
	                    }
	                } catch (ex) {
	                    if (ex.name === 'ConstraintError') {
	                        console.warn('The database "' + dbInfo.name + '"' + ' has been upgraded from version ' + e.oldVersion + ' to version ' + e.newVersion + ', but the storage "' + dbInfo.storeName + '" already exists.');
	                    } else {
	                        throw ex;
	                    }
	                }
	            };
	        }

	        openreq.onerror = function (e) {
	            e.preventDefault();
	            reject(openreq.error);
	        };

	        openreq.onsuccess = function () {
	            var db = openreq.result;
	            db.onversionchange = function (e) {
	                // Triggered when the database is modified (e.g. adding an objectStore) or
	                // deleted (even when initiated by other sessions in different tabs).
	                // Closing the connection here prevents those operations from being blocked.
	                // If the database is accessed again later by this instance, the connection
	                // will be reopened or the database recreated as needed.
	                e.target.close();
	            };
	            resolve(db);
	            _advanceReadiness(dbInfo);
	        };
	    });
	}

	function _getOriginalConnection(dbInfo) {
	    return _getConnection(dbInfo, false);
	}

	function _getUpgradedConnection(dbInfo) {
	    return _getConnection(dbInfo, true);
	}

	function _isUpgradeNeeded(dbInfo, defaultVersion) {
	    if (!dbInfo.db) {
	        return true;
	    }

	    var isNewStore = !dbInfo.db.objectStoreNames.contains(dbInfo.storeName);
	    var isDowngrade = dbInfo.version < dbInfo.db.version;
	    var isUpgrade = dbInfo.version > dbInfo.db.version;

	    if (isDowngrade) {
	        // If the version is not the default one
	        // then warn for impossible downgrade.
	        if (dbInfo.version !== defaultVersion) {
	            console.warn('The database "' + dbInfo.name + '"' + " can't be downgraded from version " + dbInfo.db.version + ' to version ' + dbInfo.version + '.');
	        }
	        // Align the versions to prevent errors.
	        dbInfo.version = dbInfo.db.version;
	    }

	    if (isUpgrade || isNewStore) {
	        // If the store is new then increment the version (if needed).
	        // This will trigger an "upgradeneeded" event which is required
	        // for creating a store.
	        if (isNewStore) {
	            var incVersion = dbInfo.db.version + 1;
	            if (incVersion > dbInfo.version) {
	                dbInfo.version = incVersion;
	            }
	        }

	        return true;
	    }

	    return false;
	}

	// encode a blob for indexeddb engines that don't support blobs
	function _encodeBlob(blob) {
	    return new Promise$1(function (resolve, reject) {
	        var reader = new FileReader();
	        reader.onerror = reject;
	        reader.onloadend = function (e) {
	            var base64 = btoa(e.target.result || '');
	            resolve({
	                __local_forage_encoded_blob: true,
	                data: base64,
	                type: blob.type
	            });
	        };
	        reader.readAsBinaryString(blob);
	    });
	}

	// decode an encoded blob
	function _decodeBlob(encodedBlob) {
	    var arrayBuff = _binStringToArrayBuffer(atob(encodedBlob.data));
	    return createBlob([arrayBuff], { type: encodedBlob.type });
	}

	// is this one of our fancy encoded blobs?
	function _isEncodedBlob(value) {
	    return value && value.__local_forage_encoded_blob;
	}

	// Specialize the default `ready()` function by making it dependent
	// on the current database operations. Thus, the driver will be actually
	// ready when it's been initialized (default) *and* there are no pending
	// operations on the database (initiated by some other instances).
	function _fullyReady(callback) {
	    var self = this;

	    var promise = self._initReady().then(function () {
	        var dbContext = dbContexts[self._dbInfo.name];

	        if (dbContext && dbContext.dbReady) {
	            return dbContext.dbReady;
	        }
	    });

	    executeTwoCallbacks(promise, callback, callback);
	    return promise;
	}

	// Try to establish a new db connection to replace the
	// current one which is broken (i.e. experiencing
	// InvalidStateError while creating a transaction).
	function _tryReconnect(dbInfo) {
	    _deferReadiness(dbInfo);

	    var dbContext = dbContexts[dbInfo.name];
	    var forages = dbContext.forages;

	    for (var i = 0; i < forages.length; i++) {
	        var forage = forages[i];
	        if (forage._dbInfo.db) {
	            forage._dbInfo.db.close();
	            forage._dbInfo.db = null;
	        }
	    }
	    dbInfo.db = null;

	    return _getOriginalConnection(dbInfo).then(function (db) {
	        dbInfo.db = db;
	        if (_isUpgradeNeeded(dbInfo)) {
	            // Reopen the database for upgrading.
	            return _getUpgradedConnection(dbInfo);
	        }
	        return db;
	    }).then(function (db) {
	        // store the latest db reference
	        // in case the db was upgraded
	        dbInfo.db = dbContext.db = db;
	        for (var i = 0; i < forages.length; i++) {
	            forages[i]._dbInfo.db = db;
	        }
	    })["catch"](function (err) {
	        _rejectReadiness(dbInfo, err);
	        throw err;
	    });
	}

	// FF doesn't like Promises (micro-tasks) and IDDB store operations,
	// so we have to do it with callbacks
	function createTransaction(dbInfo, mode, callback, retries) {
	    if (retries === undefined) {
	        retries = 1;
	    }

	    try {
	        var tx = dbInfo.db.transaction(dbInfo.storeName, mode);
	        callback(null, tx);
	    } catch (err) {
	        if (retries > 0 && (!dbInfo.db || err.name === 'InvalidStateError' || err.name === 'NotFoundError')) {
	            return Promise$1.resolve().then(function () {
	                if (!dbInfo.db || err.name === 'NotFoundError' && !dbInfo.db.objectStoreNames.contains(dbInfo.storeName) && dbInfo.version <= dbInfo.db.version) {
	                    // increase the db version, to create the new ObjectStore
	                    if (dbInfo.db) {
	                        dbInfo.version = dbInfo.db.version + 1;
	                    }
	                    // Reopen the database for upgrading.
	                    return _getUpgradedConnection(dbInfo);
	                }
	            }).then(function () {
	                return _tryReconnect(dbInfo).then(function () {
	                    createTransaction(dbInfo, mode, callback, retries - 1);
	                });
	            })["catch"](callback);
	        }

	        callback(err);
	    }
	}

	function createDbContext() {
	    return {
	        // Running localForages sharing a database.
	        forages: [],
	        // Shared database.
	        db: null,
	        // Database readiness (promise).
	        dbReady: null,
	        // Deferred operations on the database.
	        deferredOperations: []
	    };
	}

	// Open the IndexedDB database (automatically creates one if one didn't
	// previously exist), using any options set in the config.
	function _initStorage(options) {
	    var self = this;
	    var dbInfo = {
	        db: null
	    };

	    if (options) {
	        for (var i in options) {
	            dbInfo[i] = options[i];
	        }
	    }

	    // Get the current context of the database;
	    var dbContext = dbContexts[dbInfo.name];

	    // ...or create a new context.
	    if (!dbContext) {
	        dbContext = createDbContext();
	        // Register the new context in the global container.
	        dbContexts[dbInfo.name] = dbContext;
	    }

	    // Register itself as a running localForage in the current context.
	    dbContext.forages.push(self);

	    // Replace the default `ready()` function with the specialized one.
	    if (!self._initReady) {
	        self._initReady = self.ready;
	        self.ready = _fullyReady;
	    }

	    // Create an array of initialization states of the related localForages.
	    var initPromises = [];

	    function ignoreErrors() {
	        // Don't handle errors here,
	        // just makes sure related localForages aren't pending.
	        return Promise$1.resolve();
	    }

	    for (var j = 0; j < dbContext.forages.length; j++) {
	        var forage = dbContext.forages[j];
	        if (forage !== self) {
	            // Don't wait for itself...
	            initPromises.push(forage._initReady()["catch"](ignoreErrors));
	        }
	    }

	    // Take a snapshot of the related localForages.
	    var forages = dbContext.forages.slice(0);

	    // Initialize the connection process only when
	    // all the related localForages aren't pending.
	    return Promise$1.all(initPromises).then(function () {
	        dbInfo.db = dbContext.db;
	        // Get the connection or open a new one without upgrade.
	        return _getOriginalConnection(dbInfo);
	    }).then(function (db) {
	        dbInfo.db = db;
	        if (_isUpgradeNeeded(dbInfo, self._defaultConfig.version)) {
	            // Reopen the database for upgrading.
	            return _getUpgradedConnection(dbInfo);
	        }
	        return db;
	    }).then(function (db) {
	        dbInfo.db = dbContext.db = db;
	        self._dbInfo = dbInfo;
	        // Share the final connection amongst related localForages.
	        for (var k = 0; k < forages.length; k++) {
	            var forage = forages[k];
	            if (forage !== self) {
	                // Self is already up-to-date.
	                forage._dbInfo.db = dbInfo.db;
	                forage._dbInfo.version = dbInfo.version;
	            }
	        }
	    });
	}

	function getItem(key, callback) {
	    var self = this;

	    key = normalizeKey(key);

	    var promise = new Promise$1(function (resolve, reject) {
	        self.ready().then(function () {
	            createTransaction(self._dbInfo, READ_ONLY, function (err, transaction) {
	                if (err) {
	                    return reject(err);
	                }

	                try {
	                    var store = transaction.objectStore(self._dbInfo.storeName);
	                    var req = store.get(key);

	                    req.onsuccess = function () {
	                        var value = req.result;
	                        if (value === undefined) {
	                            value = null;
	                        }
	                        if (_isEncodedBlob(value)) {
	                            value = _decodeBlob(value);
	                        }
	                        resolve(value);
	                    };

	                    req.onerror = function () {
	                        reject(req.error);
	                    };
	                } catch (e) {
	                    reject(e);
	                }
	            });
	        })["catch"](reject);
	    });

	    executeCallback(promise, callback);
	    return promise;
	}

	// Iterate over all items stored in database.
	function iterate(iterator, callback) {
	    var self = this;

	    var promise = new Promise$1(function (resolve, reject) {
	        self.ready().then(function () {
	            createTransaction(self._dbInfo, READ_ONLY, function (err, transaction) {
	                if (err) {
	                    return reject(err);
	                }

	                try {
	                    var store = transaction.objectStore(self._dbInfo.storeName);
	                    var req = store.openCursor();
	                    var iterationNumber = 1;

	                    req.onsuccess = function () {
	                        var cursor = req.result;

	                        if (cursor) {
	                            var value = cursor.value;
	                            if (_isEncodedBlob(value)) {
	                                value = _decodeBlob(value);
	                            }
	                            var result = iterator(value, cursor.key, iterationNumber++);

	                            // when the iterator callback returns any
	                            // (non-`undefined`) value, then we stop
	                            // the iteration immediately
	                            if (result !== void 0) {
	                                resolve(result);
	                            } else {
	                                cursor["continue"]();
	                            }
	                        } else {
	                            resolve();
	                        }
	                    };

	                    req.onerror = function () {
	                        reject(req.error);
	                    };
	                } catch (e) {
	                    reject(e);
	                }
	            });
	        })["catch"](reject);
	    });

	    executeCallback(promise, callback);

	    return promise;
	}

	function setItem(key, value, callback) {
	    var self = this;

	    key = normalizeKey(key);

	    var promise = new Promise$1(function (resolve, reject) {
	        var dbInfo;
	        self.ready().then(function () {
	            dbInfo = self._dbInfo;
	            if (toString.call(value) === '[object Blob]') {
	                return _checkBlobSupport(dbInfo.db).then(function (blobSupport) {
	                    if (blobSupport) {
	                        return value;
	                    }
	                    return _encodeBlob(value);
	                });
	            }
	            return value;
	        }).then(function (value) {
	            createTransaction(self._dbInfo, READ_WRITE, function (err, transaction) {
	                if (err) {
	                    return reject(err);
	                }

	                try {
	                    var store = transaction.objectStore(self._dbInfo.storeName);

	                    // The reason we don't _save_ null is because IE 10 does
	                    // not support saving the `null` type in IndexedDB. How
	                    // ironic, given the bug below!
	                    // See: https://github.com/mozilla/localForage/issues/161
	                    if (value === null) {
	                        value = undefined;
	                    }

	                    var req = store.put(value, key);

	                    transaction.oncomplete = function () {
	                        // Cast to undefined so the value passed to
	                        // callback/promise is the same as what one would get out
	                        // of `getItem()` later. This leads to some weirdness
	                        // (setItem('foo', undefined) will return `null`), but
	                        // it's not my fault localStorage is our baseline and that
	                        // it's weird.
	                        if (value === undefined) {
	                            value = null;
	                        }

	                        resolve(value);
	                    };
	                    transaction.onabort = transaction.onerror = function () {
	                        var err = req.error ? req.error : req.transaction.error;
	                        reject(err);
	                    };
	                } catch (e) {
	                    reject(e);
	                }
	            });
	        })["catch"](reject);
	    });

	    executeCallback(promise, callback);
	    return promise;
	}

	function removeItem(key, callback) {
	    var self = this;

	    key = normalizeKey(key);

	    var promise = new Promise$1(function (resolve, reject) {
	        self.ready().then(function () {
	            createTransaction(self._dbInfo, READ_WRITE, function (err, transaction) {
	                if (err) {
	                    return reject(err);
	                }

	                try {
	                    var store = transaction.objectStore(self._dbInfo.storeName);
	                    // We use a Grunt task to make this safe for IE and some
	                    // versions of Android (including those used by Cordova).
	                    // Normally IE won't like `.delete()` and will insist on
	                    // using `['delete']()`, but we have a build step that
	                    // fixes this for us now.
	                    var req = store["delete"](key);
	                    transaction.oncomplete = function () {
	                        resolve();
	                    };

	                    transaction.onerror = function () {
	                        reject(req.error);
	                    };

	                    // The request will be also be aborted if we've exceeded our storage
	                    // space.
	                    transaction.onabort = function () {
	                        var err = req.error ? req.error : req.transaction.error;
	                        reject(err);
	                    };
	                } catch (e) {
	                    reject(e);
	                }
	            });
	        })["catch"](reject);
	    });

	    executeCallback(promise, callback);
	    return promise;
	}

	function clear(callback) {
	    var self = this;

	    var promise = new Promise$1(function (resolve, reject) {
	        self.ready().then(function () {
	            createTransaction(self._dbInfo, READ_WRITE, function (err, transaction) {
	                if (err) {
	                    return reject(err);
	                }

	                try {
	                    var store = transaction.objectStore(self._dbInfo.storeName);
	                    var req = store.clear();

	                    transaction.oncomplete = function () {
	                        resolve();
	                    };

	                    transaction.onabort = transaction.onerror = function () {
	                        var err = req.error ? req.error : req.transaction.error;
	                        reject(err);
	                    };
	                } catch (e) {
	                    reject(e);
	                }
	            });
	        })["catch"](reject);
	    });

	    executeCallback(promise, callback);
	    return promise;
	}

	function length(callback) {
	    var self = this;

	    var promise = new Promise$1(function (resolve, reject) {
	        self.ready().then(function () {
	            createTransaction(self._dbInfo, READ_ONLY, function (err, transaction) {
	                if (err) {
	                    return reject(err);
	                }

	                try {
	                    var store = transaction.objectStore(self._dbInfo.storeName);
	                    var req = store.count();

	                    req.onsuccess = function () {
	                        resolve(req.result);
	                    };

	                    req.onerror = function () {
	                        reject(req.error);
	                    };
	                } catch (e) {
	                    reject(e);
	                }
	            });
	        })["catch"](reject);
	    });

	    executeCallback(promise, callback);
	    return promise;
	}

	function key(n, callback) {
	    var self = this;

	    var promise = new Promise$1(function (resolve, reject) {
	        if (n < 0) {
	            resolve(null);

	            return;
	        }

	        self.ready().then(function () {
	            createTransaction(self._dbInfo, READ_ONLY, function (err, transaction) {
	                if (err) {
	                    return reject(err);
	                }

	                try {
	                    var store = transaction.objectStore(self._dbInfo.storeName);
	                    var advanced = false;
	                    var req = store.openKeyCursor();

	                    req.onsuccess = function () {
	                        var cursor = req.result;
	                        if (!cursor) {
	                            // this means there weren't enough keys
	                            resolve(null);

	                            return;
	                        }

	                        if (n === 0) {
	                            // We have the first key, return it if that's what they
	                            // wanted.
	                            resolve(cursor.key);
	                        } else {
	                            if (!advanced) {
	                                // Otherwise, ask the cursor to skip ahead n
	                                // records.
	                                advanced = true;
	                                cursor.advance(n);
	                            } else {
	                                // When we get here, we've got the nth key.
	                                resolve(cursor.key);
	                            }
	                        }
	                    };

	                    req.onerror = function () {
	                        reject(req.error);
	                    };
	                } catch (e) {
	                    reject(e);
	                }
	            });
	        })["catch"](reject);
	    });

	    executeCallback(promise, callback);
	    return promise;
	}

	function keys(callback) {
	    var self = this;

	    var promise = new Promise$1(function (resolve, reject) {
	        self.ready().then(function () {
	            createTransaction(self._dbInfo, READ_ONLY, function (err, transaction) {
	                if (err) {
	                    return reject(err);
	                }

	                try {
	                    var store = transaction.objectStore(self._dbInfo.storeName);
	                    var req = store.openKeyCursor();
	                    var keys = [];

	                    req.onsuccess = function () {
	                        var cursor = req.result;

	                        if (!cursor) {
	                            resolve(keys);
	                            return;
	                        }

	                        keys.push(cursor.key);
	                        cursor["continue"]();
	                    };

	                    req.onerror = function () {
	                        reject(req.error);
	                    };
	                } catch (e) {
	                    reject(e);
	                }
	            });
	        })["catch"](reject);
	    });

	    executeCallback(promise, callback);
	    return promise;
	}

	function dropInstance(options, callback) {
	    callback = getCallback.apply(this, arguments);

	    var currentConfig = this.config();
	    options = typeof options !== 'function' && options || {};
	    if (!options.name) {
	        options.name = options.name || currentConfig.name;
	        options.storeName = options.storeName || currentConfig.storeName;
	    }

	    var self = this;
	    var promise;
	    if (!options.name) {
	        promise = Promise$1.reject('Invalid arguments');
	    } else {
	        var isCurrentDb = options.name === currentConfig.name && self._dbInfo.db;

	        var dbPromise = isCurrentDb ? Promise$1.resolve(self._dbInfo.db) : _getOriginalConnection(options).then(function (db) {
	            var dbContext = dbContexts[options.name];
	            var forages = dbContext.forages;
	            dbContext.db = db;
	            for (var i = 0; i < forages.length; i++) {
	                forages[i]._dbInfo.db = db;
	            }
	            return db;
	        });

	        if (!options.storeName) {
	            promise = dbPromise.then(function (db) {
	                _deferReadiness(options);

	                var dbContext = dbContexts[options.name];
	                var forages = dbContext.forages;

	                db.close();
	                for (var i = 0; i < forages.length; i++) {
	                    var forage = forages[i];
	                    forage._dbInfo.db = null;
	                }

	                var dropDBPromise = new Promise$1(function (resolve, reject) {
	                    var req = idb.deleteDatabase(options.name);

	                    req.onerror = function () {
	                        var db = req.result;
	                        if (db) {
	                            db.close();
	                        }
	                        reject(req.error);
	                    };

	                    req.onblocked = function () {
	                        // Closing all open connections in onversionchange handler should prevent this situation, but if
	                        // we do get here, it just means the request remains pending - eventually it will succeed or error
	                        console.warn('dropInstance blocked for database "' + options.name + '" until all open connections are closed');
	                    };

	                    req.onsuccess = function () {
	                        var db = req.result;
	                        if (db) {
	                            db.close();
	                        }
	                        resolve(db);
	                    };
	                });

	                return dropDBPromise.then(function (db) {
	                    dbContext.db = db;
	                    for (var i = 0; i < forages.length; i++) {
	                        var _forage = forages[i];
	                        _advanceReadiness(_forage._dbInfo);
	                    }
	                })["catch"](function (err) {
	                    (_rejectReadiness(options, err) || Promise$1.resolve())["catch"](function () {});
	                    throw err;
	                });
	            });
	        } else {
	            promise = dbPromise.then(function (db) {
	                if (!db.objectStoreNames.contains(options.storeName)) {
	                    return;
	                }

	                var newVersion = db.version + 1;

	                _deferReadiness(options);

	                var dbContext = dbContexts[options.name];
	                var forages = dbContext.forages;

	                db.close();
	                for (var i = 0; i < forages.length; i++) {
	                    var forage = forages[i];
	                    forage._dbInfo.db = null;
	                    forage._dbInfo.version = newVersion;
	                }

	                var dropObjectPromise = new Promise$1(function (resolve, reject) {
	                    var req = idb.open(options.name, newVersion);

	                    req.onerror = function (err) {
	                        var db = req.result;
	                        db.close();
	                        reject(err);
	                    };

	                    req.onupgradeneeded = function () {
	                        var db = req.result;
	                        db.deleteObjectStore(options.storeName);
	                    };

	                    req.onsuccess = function () {
	                        var db = req.result;
	                        db.close();
	                        resolve(db);
	                    };
	                });

	                return dropObjectPromise.then(function (db) {
	                    dbContext.db = db;
	                    for (var j = 0; j < forages.length; j++) {
	                        var _forage2 = forages[j];
	                        _forage2._dbInfo.db = db;
	                        _advanceReadiness(_forage2._dbInfo);
	                    }
	                })["catch"](function (err) {
	                    (_rejectReadiness(options, err) || Promise$1.resolve())["catch"](function () {});
	                    throw err;
	                });
	            });
	        }
	    }

	    executeCallback(promise, callback);
	    return promise;
	}

	var asyncStorage = {
	    _driver: 'asyncStorage',
	    _initStorage: _initStorage,
	    _support: isIndexedDBValid(),
	    iterate: iterate,
	    getItem: getItem,
	    setItem: setItem,
	    removeItem: removeItem,
	    clear: clear,
	    length: length,
	    key: key,
	    keys: keys,
	    dropInstance: dropInstance
	};

	function isWebSQLValid() {
	    return typeof openDatabase === 'function';
	}

	// Sadly, the best way to save binary data in WebSQL/localStorage is serializing
	// it to Base64, so this is how we store it to prevent very strange errors with less
	// verbose ways of binary <-> string data storage.
	var BASE_CHARS = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/';

	var BLOB_TYPE_PREFIX = '~~local_forage_type~';
	var BLOB_TYPE_PREFIX_REGEX = /^~~local_forage_type~([^~]+)~/;

	var SERIALIZED_MARKER = '__lfsc__:';
	var SERIALIZED_MARKER_LENGTH = SERIALIZED_MARKER.length;

	// OMG the serializations!
	var TYPE_ARRAYBUFFER = 'arbf';
	var TYPE_BLOB = 'blob';
	var TYPE_INT8ARRAY = 'si08';
	var TYPE_UINT8ARRAY = 'ui08';
	var TYPE_UINT8CLAMPEDARRAY = 'uic8';
	var TYPE_INT16ARRAY = 'si16';
	var TYPE_INT32ARRAY = 'si32';
	var TYPE_UINT16ARRAY = 'ur16';
	var TYPE_UINT32ARRAY = 'ui32';
	var TYPE_FLOAT32ARRAY = 'fl32';
	var TYPE_FLOAT64ARRAY = 'fl64';
	var TYPE_SERIALIZED_MARKER_LENGTH = SERIALIZED_MARKER_LENGTH + TYPE_ARRAYBUFFER.length;

	var toString$1 = Object.prototype.toString;

	function stringToBuffer(serializedString) {
	    // Fill the string into a ArrayBuffer.
	    var bufferLength = serializedString.length * 0.75;
	    var len = serializedString.length;
	    var i;
	    var p = 0;
	    var encoded1, encoded2, encoded3, encoded4;

	    if (serializedString[serializedString.length - 1] === '=') {
	        bufferLength--;
	        if (serializedString[serializedString.length - 2] === '=') {
	            bufferLength--;
	        }
	    }

	    var buffer = new ArrayBuffer(bufferLength);
	    var bytes = new Uint8Array(buffer);

	    for (i = 0; i < len; i += 4) {
	        encoded1 = BASE_CHARS.indexOf(serializedString[i]);
	        encoded2 = BASE_CHARS.indexOf(serializedString[i + 1]);
	        encoded3 = BASE_CHARS.indexOf(serializedString[i + 2]);
	        encoded4 = BASE_CHARS.indexOf(serializedString[i + 3]);

	        /*jslint bitwise: true */
	        bytes[p++] = encoded1 << 2 | encoded2 >> 4;
	        bytes[p++] = (encoded2 & 15) << 4 | encoded3 >> 2;
	        bytes[p++] = (encoded3 & 3) << 6 | encoded4 & 63;
	    }
	    return buffer;
	}

	// Converts a buffer to a string to store, serialized, in the backend
	// storage library.
	function bufferToString(buffer) {
	    // base64-arraybuffer
	    var bytes = new Uint8Array(buffer);
	    var base64String = '';
	    var i;

	    for (i = 0; i < bytes.length; i += 3) {
	        /*jslint bitwise: true */
	        base64String += BASE_CHARS[bytes[i] >> 2];
	        base64String += BASE_CHARS[(bytes[i] & 3) << 4 | bytes[i + 1] >> 4];
	        base64String += BASE_CHARS[(bytes[i + 1] & 15) << 2 | bytes[i + 2] >> 6];
	        base64String += BASE_CHARS[bytes[i + 2] & 63];
	    }

	    if (bytes.length % 3 === 2) {
	        base64String = base64String.substring(0, base64String.length - 1) + '=';
	    } else if (bytes.length % 3 === 1) {
	        base64String = base64String.substring(0, base64String.length - 2) + '==';
	    }

	    return base64String;
	}

	// Serialize a value, afterwards executing a callback (which usually
	// instructs the `setItem()` callback/promise to be executed). This is how
	// we store binary data with localStorage.
	function serialize(value, callback) {
	    var valueType = '';
	    if (value) {
	        valueType = toString$1.call(value);
	    }

	    // Cannot use `value instanceof ArrayBuffer` or such here, as these
	    // checks fail when running the tests using casper.js...
	    //
	    // TODO: See why those tests fail and use a better solution.
	    if (value && (valueType === '[object ArrayBuffer]' || value.buffer && toString$1.call(value.buffer) === '[object ArrayBuffer]')) {
	        // Convert binary arrays to a string and prefix the string with
	        // a special marker.
	        var buffer;
	        var marker = SERIALIZED_MARKER;

	        if (value instanceof ArrayBuffer) {
	            buffer = value;
	            marker += TYPE_ARRAYBUFFER;
	        } else {
	            buffer = value.buffer;

	            if (valueType === '[object Int8Array]') {
	                marker += TYPE_INT8ARRAY;
	            } else if (valueType === '[object Uint8Array]') {
	                marker += TYPE_UINT8ARRAY;
	            } else if (valueType === '[object Uint8ClampedArray]') {
	                marker += TYPE_UINT8CLAMPEDARRAY;
	            } else if (valueType === '[object Int16Array]') {
	                marker += TYPE_INT16ARRAY;
	            } else if (valueType === '[object Uint16Array]') {
	                marker += TYPE_UINT16ARRAY;
	            } else if (valueType === '[object Int32Array]') {
	                marker += TYPE_INT32ARRAY;
	            } else if (valueType === '[object Uint32Array]') {
	                marker += TYPE_UINT32ARRAY;
	            } else if (valueType === '[object Float32Array]') {
	                marker += TYPE_FLOAT32ARRAY;
	            } else if (valueType === '[object Float64Array]') {
	                marker += TYPE_FLOAT64ARRAY;
	            } else {
	                callback(new Error('Failed to get type for BinaryArray'));
	            }
	        }

	        callback(marker + bufferToString(buffer));
	    } else if (valueType === '[object Blob]') {
	        // Conver the blob to a binaryArray and then to a string.
	        var fileReader = new FileReader();

	        fileReader.onload = function () {
	            // Backwards-compatible prefix for the blob type.
	            var str = BLOB_TYPE_PREFIX + value.type + '~' + bufferToString(this.result);

	            callback(SERIALIZED_MARKER + TYPE_BLOB + str);
	        };

	        fileReader.readAsArrayBuffer(value);
	    } else {
	        try {
	            callback(JSON.stringify(value));
	        } catch (e) {
	            console.error("Couldn't convert value into a JSON string: ", value);

	            callback(null, e);
	        }
	    }
	}

	// Deserialize data we've inserted into a value column/field. We place
	// special markers into our strings to mark them as encoded; this isn't
	// as nice as a meta field, but it's the only sane thing we can do whilst
	// keeping localStorage support intact.
	//
	// Oftentimes this will just deserialize JSON content, but if we have a
	// special marker (SERIALIZED_MARKER, defined above), we will extract
	// some kind of arraybuffer/binary data/typed array out of the string.
	function deserialize(value) {
	    // If we haven't marked this string as being specially serialized (i.e.
	    // something other than serialized JSON), we can just return it and be
	    // done with it.
	    if (value.substring(0, SERIALIZED_MARKER_LENGTH) !== SERIALIZED_MARKER) {
	        return JSON.parse(value);
	    }

	    // The following code deals with deserializing some kind of Blob or
	    // TypedArray. First we separate out the type of data we're dealing
	    // with from the data itself.
	    var serializedString = value.substring(TYPE_SERIALIZED_MARKER_LENGTH);
	    var type = value.substring(SERIALIZED_MARKER_LENGTH, TYPE_SERIALIZED_MARKER_LENGTH);

	    var blobType;
	    // Backwards-compatible blob type serialization strategy.
	    // DBs created with older versions of localForage will simply not have the blob type.
	    if (type === TYPE_BLOB && BLOB_TYPE_PREFIX_REGEX.test(serializedString)) {
	        var matcher = serializedString.match(BLOB_TYPE_PREFIX_REGEX);
	        blobType = matcher[1];
	        serializedString = serializedString.substring(matcher[0].length);
	    }
	    var buffer = stringToBuffer(serializedString);

	    // Return the right type based on the code/type set during
	    // serialization.
	    switch (type) {
	        case TYPE_ARRAYBUFFER:
	            return buffer;
	        case TYPE_BLOB:
	            return createBlob([buffer], { type: blobType });
	        case TYPE_INT8ARRAY:
	            return new Int8Array(buffer);
	        case TYPE_UINT8ARRAY:
	            return new Uint8Array(buffer);
	        case TYPE_UINT8CLAMPEDARRAY:
	            return new Uint8ClampedArray(buffer);
	        case TYPE_INT16ARRAY:
	            return new Int16Array(buffer);
	        case TYPE_UINT16ARRAY:
	            return new Uint16Array(buffer);
	        case TYPE_INT32ARRAY:
	            return new Int32Array(buffer);
	        case TYPE_UINT32ARRAY:
	            return new Uint32Array(buffer);
	        case TYPE_FLOAT32ARRAY:
	            return new Float32Array(buffer);
	        case TYPE_FLOAT64ARRAY:
	            return new Float64Array(buffer);
	        default:
	            throw new Error('Unkown type: ' + type);
	    }
	}

	var localforageSerializer = {
	    serialize: serialize,
	    deserialize: deserialize,
	    stringToBuffer: stringToBuffer,
	    bufferToString: bufferToString
	};

	/*
	 * Includes code from:
	 *
	 * base64-arraybuffer
	 * https://github.com/niklasvh/base64-arraybuffer
	 *
	 * Copyright (c) 2012 Niklas von Hertzen
	 * Licensed under the MIT license.
	 */

	function createDbTable(t, dbInfo, callback, errorCallback) {
	    t.executeSql('CREATE TABLE IF NOT EXISTS ' + dbInfo.storeName + ' ' + '(id INTEGER PRIMARY KEY, key unique, value)', [], callback, errorCallback);
	}

	// Open the WebSQL database (automatically creates one if one didn't
	// previously exist), using any options set in the config.
	function _initStorage$1(options) {
	    var self = this;
	    var dbInfo = {
	        db: null
	    };

	    if (options) {
	        for (var i in options) {
	            dbInfo[i] = typeof options[i] !== 'string' ? options[i].toString() : options[i];
	        }
	    }

	    var dbInfoPromise = new Promise$1(function (resolve, reject) {
	        // Open the database; the openDatabase API will automatically
	        // create it for us if it doesn't exist.
	        try {
	            dbInfo.db = openDatabase(dbInfo.name, String(dbInfo.version), dbInfo.description, dbInfo.size);
	        } catch (e) {
	            return reject(e);
	        }

	        // Create our key/value table if it doesn't exist.
	        dbInfo.db.transaction(function (t) {
	            createDbTable(t, dbInfo, function () {
	                self._dbInfo = dbInfo;
	                resolve();
	            }, function (t, error) {
	                reject(error);
	            });
	        }, reject);
	    });

	    dbInfo.serializer = localforageSerializer;
	    return dbInfoPromise;
	}

	function tryExecuteSql(t, dbInfo, sqlStatement, args, callback, errorCallback) {
	    t.executeSql(sqlStatement, args, callback, function (t, error) {
	        if (error.code === error.SYNTAX_ERR) {
	            t.executeSql('SELECT name FROM sqlite_master ' + "WHERE type='table' AND name = ?", [dbInfo.storeName], function (t, results) {
	                if (!results.rows.length) {
	                    // if the table is missing (was deleted)
	                    // re-create it table and retry
	                    createDbTable(t, dbInfo, function () {
	                        t.executeSql(sqlStatement, args, callback, errorCallback);
	                    }, errorCallback);
	                } else {
	                    errorCallback(t, error);
	                }
	            }, errorCallback);
	        } else {
	            errorCallback(t, error);
	        }
	    }, errorCallback);
	}

	function getItem$1(key, callback) {
	    var self = this;

	    key = normalizeKey(key);

	    var promise = new Promise$1(function (resolve, reject) {
	        self.ready().then(function () {
	            var dbInfo = self._dbInfo;
	            dbInfo.db.transaction(function (t) {
	                tryExecuteSql(t, dbInfo, 'SELECT * FROM ' + dbInfo.storeName + ' WHERE key = ? LIMIT 1', [key], function (t, results) {
	                    var result = results.rows.length ? results.rows.item(0).value : null;

	                    // Check to see if this is serialized content we need to
	                    // unpack.
	                    if (result) {
	                        result = dbInfo.serializer.deserialize(result);
	                    }

	                    resolve(result);
	                }, function (t, error) {
	                    reject(error);
	                });
	            });
	        })["catch"](reject);
	    });

	    executeCallback(promise, callback);
	    return promise;
	}

	function iterate$1(iterator, callback) {
	    var self = this;

	    var promise = new Promise$1(function (resolve, reject) {
	        self.ready().then(function () {
	            var dbInfo = self._dbInfo;

	            dbInfo.db.transaction(function (t) {
	                tryExecuteSql(t, dbInfo, 'SELECT * FROM ' + dbInfo.storeName, [], function (t, results) {
	                    var rows = results.rows;
	                    var length = rows.length;

	                    for (var i = 0; i < length; i++) {
	                        var item = rows.item(i);
	                        var result = item.value;

	                        // Check to see if this is serialized content
	                        // we need to unpack.
	                        if (result) {
	                            result = dbInfo.serializer.deserialize(result);
	                        }

	                        result = iterator(result, item.key, i + 1);

	                        // void(0) prevents problems with redefinition
	                        // of `undefined`.
	                        if (result !== void 0) {
	                            resolve(result);
	                            return;
	                        }
	                    }

	                    resolve();
	                }, function (t, error) {
	                    reject(error);
	                });
	            });
	        })["catch"](reject);
	    });

	    executeCallback(promise, callback);
	    return promise;
	}

	function _setItem(key, value, callback, retriesLeft) {
	    var self = this;

	    key = normalizeKey(key);

	    var promise = new Promise$1(function (resolve, reject) {
	        self.ready().then(function () {
	            // The localStorage API doesn't return undefined values in an
	            // "expected" way, so undefined is always cast to null in all
	            // drivers. See: https://github.com/mozilla/localForage/pull/42
	            if (value === undefined) {
	                value = null;
	            }

	            // Save the original value to pass to the callback.
	            var originalValue = value;

	            var dbInfo = self._dbInfo;
	            dbInfo.serializer.serialize(value, function (value, error) {
	                if (error) {
	                    reject(error);
	                } else {
	                    dbInfo.db.transaction(function (t) {
	                        tryExecuteSql(t, dbInfo, 'INSERT OR REPLACE INTO ' + dbInfo.storeName + ' ' + '(key, value) VALUES (?, ?)', [key, value], function () {
	                            resolve(originalValue);
	                        }, function (t, error) {
	                            reject(error);
	                        });
	                    }, function (sqlError) {
	                        // The transaction failed; check
	                        // to see if it's a quota error.
	                        if (sqlError.code === sqlError.QUOTA_ERR) {
	                            // We reject the callback outright for now, but
	                            // it's worth trying to re-run the transaction.
	                            // Even if the user accepts the prompt to use
	                            // more storage on Safari, this error will
	                            // be called.
	                            //
	                            // Try to re-run the transaction.
	                            if (retriesLeft > 0) {
	                                resolve(_setItem.apply(self, [key, originalValue, callback, retriesLeft - 1]));
	                                return;
	                            }
	                            reject(sqlError);
	                        }
	                    });
	                }
	            });
	        })["catch"](reject);
	    });

	    executeCallback(promise, callback);
	    return promise;
	}

	function setItem$1(key, value, callback) {
	    return _setItem.apply(this, [key, value, callback, 1]);
	}

	function removeItem$1(key, callback) {
	    var self = this;

	    key = normalizeKey(key);

	    var promise = new Promise$1(function (resolve, reject) {
	        self.ready().then(function () {
	            var dbInfo = self._dbInfo;
	            dbInfo.db.transaction(function (t) {
	                tryExecuteSql(t, dbInfo, 'DELETE FROM ' + dbInfo.storeName + ' WHERE key = ?', [key], function () {
	                    resolve();
	                }, function (t, error) {
	                    reject(error);
	                });
	            });
	        })["catch"](reject);
	    });

	    executeCallback(promise, callback);
	    return promise;
	}

	// Deletes every item in the table.
	// TODO: Find out if this resets the AUTO_INCREMENT number.
	function clear$1(callback) {
	    var self = this;

	    var promise = new Promise$1(function (resolve, reject) {
	        self.ready().then(function () {
	            var dbInfo = self._dbInfo;
	            dbInfo.db.transaction(function (t) {
	                tryExecuteSql(t, dbInfo, 'DELETE FROM ' + dbInfo.storeName, [], function () {
	                    resolve();
	                }, function (t, error) {
	                    reject(error);
	                });
	            });
	        })["catch"](reject);
	    });

	    executeCallback(promise, callback);
	    return promise;
	}

	// Does a simple `COUNT(key)` to get the number of items stored in
	// localForage.
	function length$1(callback) {
	    var self = this;

	    var promise = new Promise$1(function (resolve, reject) {
	        self.ready().then(function () {
	            var dbInfo = self._dbInfo;
	            dbInfo.db.transaction(function (t) {
	                // Ahhh, SQL makes this one soooooo easy.
	                tryExecuteSql(t, dbInfo, 'SELECT COUNT(key) as c FROM ' + dbInfo.storeName, [], function (t, results) {
	                    var result = results.rows.item(0).c;
	                    resolve(result);
	                }, function (t, error) {
	                    reject(error);
	                });
	            });
	        })["catch"](reject);
	    });

	    executeCallback(promise, callback);
	    return promise;
	}

	// Return the key located at key index X; essentially gets the key from a
	// `WHERE id = ?`. This is the most efficient way I can think to implement
	// this rarely-used (in my experience) part of the API, but it can seem
	// inconsistent, because we do `INSERT OR REPLACE INTO` on `setItem()`, so
	// the ID of each key will change every time it's updated. Perhaps a stored
	// procedure for the `setItem()` SQL would solve this problem?
	// TODO: Don't change ID on `setItem()`.
	function key$1(n, callback) {
	    var self = this;

	    var promise = new Promise$1(function (resolve, reject) {
	        self.ready().then(function () {
	            var dbInfo = self._dbInfo;
	            dbInfo.db.transaction(function (t) {
	                tryExecuteSql(t, dbInfo, 'SELECT key FROM ' + dbInfo.storeName + ' WHERE id = ? LIMIT 1', [n + 1], function (t, results) {
	                    var result = results.rows.length ? results.rows.item(0).key : null;
	                    resolve(result);
	                }, function (t, error) {
	                    reject(error);
	                });
	            });
	        })["catch"](reject);
	    });

	    executeCallback(promise, callback);
	    return promise;
	}

	function keys$1(callback) {
	    var self = this;

	    var promise = new Promise$1(function (resolve, reject) {
	        self.ready().then(function () {
	            var dbInfo = self._dbInfo;
	            dbInfo.db.transaction(function (t) {
	                tryExecuteSql(t, dbInfo, 'SELECT key FROM ' + dbInfo.storeName, [], function (t, results) {
	                    var keys = [];

	                    for (var i = 0; i < results.rows.length; i++) {
	                        keys.push(results.rows.item(i).key);
	                    }

	                    resolve(keys);
	                }, function (t, error) {
	                    reject(error);
	                });
	            });
	        })["catch"](reject);
	    });

	    executeCallback(promise, callback);
	    return promise;
	}

	// https://www.w3.org/TR/webdatabase/#databases
	// > There is no way to enumerate or delete the databases available for an origin from this API.
	function getAllStoreNames(db) {
	    return new Promise$1(function (resolve, reject) {
	        db.transaction(function (t) {
	            t.executeSql('SELECT name FROM sqlite_master ' + "WHERE type='table' AND name <> '__WebKitDatabaseInfoTable__'", [], function (t, results) {
	                var storeNames = [];

	                for (var i = 0; i < results.rows.length; i++) {
	                    storeNames.push(results.rows.item(i).name);
	                }

	                resolve({
	                    db: db,
	                    storeNames: storeNames
	                });
	            }, function (t, error) {
	                reject(error);
	            });
	        }, function (sqlError) {
	            reject(sqlError);
	        });
	    });
	}

	function dropInstance$1(options, callback) {
	    callback = getCallback.apply(this, arguments);

	    var currentConfig = this.config();
	    options = typeof options !== 'function' && options || {};
	    if (!options.name) {
	        options.name = options.name || currentConfig.name;
	        options.storeName = options.storeName || currentConfig.storeName;
	    }

	    var self = this;
	    var promise;
	    if (!options.name) {
	        promise = Promise$1.reject('Invalid arguments');
	    } else {
	        promise = new Promise$1(function (resolve) {
	            var db;
	            if (options.name === currentConfig.name) {
	                // use the db reference of the current instance
	                db = self._dbInfo.db;
	            } else {
	                db = openDatabase(options.name, '', '', 0);
	            }

	            if (!options.storeName) {
	                // drop all database tables
	                resolve(getAllStoreNames(db));
	            } else {
	                resolve({
	                    db: db,
	                    storeNames: [options.storeName]
	                });
	            }
	        }).then(function (operationInfo) {
	            return new Promise$1(function (resolve, reject) {
	                operationInfo.db.transaction(function (t) {
	                    function dropTable(storeName) {
	                        return new Promise$1(function (resolve, reject) {
	                            t.executeSql('DROP TABLE IF EXISTS ' + storeName, [], function () {
	                                resolve();
	                            }, function (t, error) {
	                                reject(error);
	                            });
	                        });
	                    }

	                    var operations = [];
	                    for (var i = 0, len = operationInfo.storeNames.length; i < len; i++) {
	                        operations.push(dropTable(operationInfo.storeNames[i]));
	                    }

	                    Promise$1.all(operations).then(function () {
	                        resolve();
	                    })["catch"](function (e) {
	                        reject(e);
	                    });
	                }, function (sqlError) {
	                    reject(sqlError);
	                });
	            });
	        });
	    }

	    executeCallback(promise, callback);
	    return promise;
	}

	var webSQLStorage = {
	    _driver: 'webSQLStorage',
	    _initStorage: _initStorage$1,
	    _support: isWebSQLValid(),
	    iterate: iterate$1,
	    getItem: getItem$1,
	    setItem: setItem$1,
	    removeItem: removeItem$1,
	    clear: clear$1,
	    length: length$1,
	    key: key$1,
	    keys: keys$1,
	    dropInstance: dropInstance$1
	};

	function isLocalStorageValid() {
	    try {
	        return typeof localStorage !== 'undefined' && 'setItem' in localStorage &&
	        // in IE8 typeof localStorage.setItem === 'object'
	        !!localStorage.setItem;
	    } catch (e) {
	        return false;
	    }
	}

	function _getKeyPrefix(options, defaultConfig) {
	    var keyPrefix = options.name + '/';

	    if (options.storeName !== defaultConfig.storeName) {
	        keyPrefix += options.storeName + '/';
	    }
	    return keyPrefix;
	}

	// Check if localStorage throws when saving an item
	function checkIfLocalStorageThrows() {
	    var localStorageTestKey = '_localforage_support_test';

	    try {
	        localStorage.setItem(localStorageTestKey, true);
	        localStorage.removeItem(localStorageTestKey);

	        return false;
	    } catch (e) {
	        return true;
	    }
	}

	// Check if localStorage is usable and allows to save an item
	// This method checks if localStorage is usable in Safari Private Browsing
	// mode, or in any other case where the available quota for localStorage
	// is 0 and there wasn't any saved items yet.
	function _isLocalStorageUsable() {
	    return !checkIfLocalStorageThrows() || localStorage.length > 0;
	}

	// Config the localStorage backend, using options set in the config.
	function _initStorage$2(options) {
	    var self = this;
	    var dbInfo = {};
	    if (options) {
	        for (var i in options) {
	            dbInfo[i] = options[i];
	        }
	    }

	    dbInfo.keyPrefix = _getKeyPrefix(options, self._defaultConfig);

	    if (!_isLocalStorageUsable()) {
	        return Promise$1.reject();
	    }

	    self._dbInfo = dbInfo;
	    dbInfo.serializer = localforageSerializer;

	    return Promise$1.resolve();
	}

	// Remove all keys from the datastore, effectively destroying all data in
	// the app's key/value store!
	function clear$2(callback) {
	    var self = this;
	    var promise = self.ready().then(function () {
	        var keyPrefix = self._dbInfo.keyPrefix;

	        for (var i = localStorage.length - 1; i >= 0; i--) {
	            var key = localStorage.key(i);

	            if (key.indexOf(keyPrefix) === 0) {
	                localStorage.removeItem(key);
	            }
	        }
	    });

	    executeCallback(promise, callback);
	    return promise;
	}

	// Retrieve an item from the store. Unlike the original async_storage
	// library in Gaia, we don't modify return values at all. If a key's value
	// is `undefined`, we pass that value to the callback function.
	function getItem$2(key, callback) {
	    var self = this;

	    key = normalizeKey(key);

	    var promise = self.ready().then(function () {
	        var dbInfo = self._dbInfo;
	        var result = localStorage.getItem(dbInfo.keyPrefix + key);

	        // If a result was found, parse it from the serialized
	        // string into a JS object. If result isn't truthy, the key
	        // is likely undefined and we'll pass it straight to the
	        // callback.
	        if (result) {
	            result = dbInfo.serializer.deserialize(result);
	        }

	        return result;
	    });

	    executeCallback(promise, callback);
	    return promise;
	}

	// Iterate over all items in the store.
	function iterate$2(iterator, callback) {
	    var self = this;

	    var promise = self.ready().then(function () {
	        var dbInfo = self._dbInfo;
	        var keyPrefix = dbInfo.keyPrefix;
	        var keyPrefixLength = keyPrefix.length;
	        var length = localStorage.length;

	        // We use a dedicated iterator instead of the `i` variable below
	        // so other keys we fetch in localStorage aren't counted in
	        // the `iterationNumber` argument passed to the `iterate()`
	        // callback.
	        //
	        // See: github.com/mozilla/localForage/pull/435#discussion_r38061530
	        var iterationNumber = 1;

	        for (var i = 0; i < length; i++) {
	            var key = localStorage.key(i);
	            if (key.indexOf(keyPrefix) !== 0) {
	                continue;
	            }
	            var value = localStorage.getItem(key);

	            // If a result was found, parse it from the serialized
	            // string into a JS object. If result isn't truthy, the
	            // key is likely undefined and we'll pass it straight
	            // to the iterator.
	            if (value) {
	                value = dbInfo.serializer.deserialize(value);
	            }

	            value = iterator(value, key.substring(keyPrefixLength), iterationNumber++);

	            if (value !== void 0) {
	                return value;
	            }
	        }
	    });

	    executeCallback(promise, callback);
	    return promise;
	}

	// Same as localStorage's key() method, except takes a callback.
	function key$2(n, callback) {
	    var self = this;
	    var promise = self.ready().then(function () {
	        var dbInfo = self._dbInfo;
	        var result;
	        try {
	            result = localStorage.key(n);
	        } catch (error) {
	            result = null;
	        }

	        // Remove the prefix from the key, if a key is found.
	        if (result) {
	            result = result.substring(dbInfo.keyPrefix.length);
	        }

	        return result;
	    });

	    executeCallback(promise, callback);
	    return promise;
	}

	function keys$2(callback) {
	    var self = this;
	    var promise = self.ready().then(function () {
	        var dbInfo = self._dbInfo;
	        var length = localStorage.length;
	        var keys = [];

	        for (var i = 0; i < length; i++) {
	            var itemKey = localStorage.key(i);
	            if (itemKey.indexOf(dbInfo.keyPrefix) === 0) {
	                keys.push(itemKey.substring(dbInfo.keyPrefix.length));
	            }
	        }

	        return keys;
	    });

	    executeCallback(promise, callback);
	    return promise;
	}

	// Supply the number of keys in the datastore to the callback function.
	function length$2(callback) {
	    var self = this;
	    var promise = self.keys().then(function (keys) {
	        return keys.length;
	    });

	    executeCallback(promise, callback);
	    return promise;
	}

	// Remove an item from the store, nice and simple.
	function removeItem$2(key, callback) {
	    var self = this;

	    key = normalizeKey(key);

	    var promise = self.ready().then(function () {
	        var dbInfo = self._dbInfo;
	        localStorage.removeItem(dbInfo.keyPrefix + key);
	    });

	    executeCallback(promise, callback);
	    return promise;
	}

	// Set a key's value and run an optional callback once the value is set.
	// Unlike Gaia's implementation, the callback function is passed the value,
	// in case you want to operate on that value only after you're sure it
	// saved, or something like that.
	function setItem$2(key, value, callback) {
	    var self = this;

	    key = normalizeKey(key);

	    var promise = self.ready().then(function () {
	        // Convert undefined values to null.
	        // https://github.com/mozilla/localForage/pull/42
	        if (value === undefined) {
	            value = null;
	        }

	        // Save the original value to pass to the callback.
	        var originalValue = value;

	        return new Promise$1(function (resolve, reject) {
	            var dbInfo = self._dbInfo;
	            dbInfo.serializer.serialize(value, function (value, error) {
	                if (error) {
	                    reject(error);
	                } else {
	                    try {
	                        localStorage.setItem(dbInfo.keyPrefix + key, value);
	                        resolve(originalValue);
	                    } catch (e) {
	                        // localStorage capacity exceeded.
	                        // TODO: Make this a specific error/event.
	                        if (e.name === 'QuotaExceededError' || e.name === 'NS_ERROR_DOM_QUOTA_REACHED') {
	                            reject(e);
	                        }
	                        reject(e);
	                    }
	                }
	            });
	        });
	    });

	    executeCallback(promise, callback);
	    return promise;
	}

	function dropInstance$2(options, callback) {
	    callback = getCallback.apply(this, arguments);

	    options = typeof options !== 'function' && options || {};
	    if (!options.name) {
	        var currentConfig = this.config();
	        options.name = options.name || currentConfig.name;
	        options.storeName = options.storeName || currentConfig.storeName;
	    }

	    var self = this;
	    var promise;
	    if (!options.name) {
	        promise = Promise$1.reject('Invalid arguments');
	    } else {
	        promise = new Promise$1(function (resolve) {
	            if (!options.storeName) {
	                resolve(options.name + '/');
	            } else {
	                resolve(_getKeyPrefix(options, self._defaultConfig));
	            }
	        }).then(function (keyPrefix) {
	            for (var i = localStorage.length - 1; i >= 0; i--) {
	                var key = localStorage.key(i);

	                if (key.indexOf(keyPrefix) === 0) {
	                    localStorage.removeItem(key);
	                }
	            }
	        });
	    }

	    executeCallback(promise, callback);
	    return promise;
	}

	var localStorageWrapper = {
	    _driver: 'localStorageWrapper',
	    _initStorage: _initStorage$2,
	    _support: isLocalStorageValid(),
	    iterate: iterate$2,
	    getItem: getItem$2,
	    setItem: setItem$2,
	    removeItem: removeItem$2,
	    clear: clear$2,
	    length: length$2,
	    key: key$2,
	    keys: keys$2,
	    dropInstance: dropInstance$2
	};

	var sameValue = function sameValue(x, y) {
	    return x === y || typeof x === 'number' && typeof y === 'number' && isNaN(x) && isNaN(y);
	};

	var includes = function includes(array, searchElement) {
	    var len = array.length;
	    var i = 0;
	    while (i < len) {
	        if (sameValue(array[i], searchElement)) {
	            return true;
	        }
	        i++;
	    }

	    return false;
	};

	var isArray = Array.isArray || function (arg) {
	    return Object.prototype.toString.call(arg) === '[object Array]';
	};

	// Drivers are stored here when `defineDriver()` is called.
	// They are shared across all instances of localForage.
	var DefinedDrivers = {};

	var DriverSupport = {};

	var DefaultDrivers = {
	    INDEXEDDB: asyncStorage,
	    WEBSQL: webSQLStorage,
	    LOCALSTORAGE: localStorageWrapper
	};

	var DefaultDriverOrder = [DefaultDrivers.INDEXEDDB._driver, DefaultDrivers.WEBSQL._driver, DefaultDrivers.LOCALSTORAGE._driver];

	var OptionalDriverMethods = ['dropInstance'];

	var LibraryMethods = ['clear', 'getItem', 'iterate', 'key', 'keys', 'length', 'removeItem', 'setItem'].concat(OptionalDriverMethods);

	var DefaultConfig = {
	    description: '',
	    driver: DefaultDriverOrder.slice(),
	    name: 'localforage',
	    // Default DB size is _JUST UNDER_ 5MB, as it's the highest size
	    // we can use without a prompt.
	    size: 4980736,
	    storeName: 'keyvaluepairs',
	    version: 1.0
	};

	function callWhenReady(localForageInstance, libraryMethod) {
	    localForageInstance[libraryMethod] = function () {
	        var _args = arguments;
	        return localForageInstance.ready().then(function () {
	            return localForageInstance[libraryMethod].apply(localForageInstance, _args);
	        });
	    };
	}

	function extend() {
	    for (var i = 1; i < arguments.length; i++) {
	        var arg = arguments[i];

	        if (arg) {
	            for (var _key in arg) {
	                if (arg.hasOwnProperty(_key)) {
	                    if (isArray(arg[_key])) {
	                        arguments[0][_key] = arg[_key].slice();
	                    } else {
	                        arguments[0][_key] = arg[_key];
	                    }
	                }
	            }
	        }
	    }

	    return arguments[0];
	}

	var LocalForage = function () {
	    function LocalForage(options) {
	        _classCallCheck(this, LocalForage);

	        for (var driverTypeKey in DefaultDrivers) {
	            if (DefaultDrivers.hasOwnProperty(driverTypeKey)) {
	                var driver = DefaultDrivers[driverTypeKey];
	                var driverName = driver._driver;
	                this[driverTypeKey] = driverName;

	                if (!DefinedDrivers[driverName]) {
	                    // we don't need to wait for the promise,
	                    // since the default drivers can be defined
	                    // in a blocking manner
	                    this.defineDriver(driver);
	                }
	            }
	        }

	        this._defaultConfig = extend({}, DefaultConfig);
	        this._config = extend({}, this._defaultConfig, options);
	        this._driverSet = null;
	        this._initDriver = null;
	        this._ready = false;
	        this._dbInfo = null;

	        this._wrapLibraryMethodsWithReady();
	        this.setDriver(this._config.driver)["catch"](function () {});
	    }

	    // Set any config values for localForage; can be called anytime before
	    // the first API call (e.g. `getItem`, `setItem`).
	    // We loop through options so we don't overwrite existing config
	    // values.


	    LocalForage.prototype.config = function config(options) {
	        // If the options argument is an object, we use it to set values.
	        // Otherwise, we return either a specified config value or all
	        // config values.
	        if ((typeof options === 'undefined' ? 'undefined' : _typeof(options)) === 'object') {
	            // If localforage is ready and fully initialized, we can't set
	            // any new configuration values. Instead, we return an error.
	            if (this._ready) {
	                return new Error("Can't call config() after localforage " + 'has been used.');
	            }

	            for (var i in options) {
	                if (i === 'storeName') {
	                    options[i] = options[i].replace(/\W/g, '_');
	                }

	                if (i === 'version' && typeof options[i] !== 'number') {
	                    return new Error('Database version must be a number.');
	                }

	                this._config[i] = options[i];
	            }

	            // after all config options are set and
	            // the driver option is used, try setting it
	            if ('driver' in options && options.driver) {
	                return this.setDriver(this._config.driver);
	            }

	            return true;
	        } else if (typeof options === 'string') {
	            return this._config[options];
	        } else {
	            return this._config;
	        }
	    };

	    // Used to define a custom driver, shared across all instances of
	    // localForage.


	    LocalForage.prototype.defineDriver = function defineDriver(driverObject, callback, errorCallback) {
	        var promise = new Promise$1(function (resolve, reject) {
	            try {
	                var driverName = driverObject._driver;
	                var complianceError = new Error('Custom driver not compliant; see ' + 'https://mozilla.github.io/localForage/#definedriver');

	                // A driver name should be defined and not overlap with the
	                // library-defined, default drivers.
	                if (!driverObject._driver) {
	                    reject(complianceError);
	                    return;
	                }

	                var driverMethods = LibraryMethods.concat('_initStorage');
	                for (var i = 0, len = driverMethods.length; i < len; i++) {
	                    var driverMethodName = driverMethods[i];

	                    // when the property is there,
	                    // it should be a method even when optional
	                    var isRequired = !includes(OptionalDriverMethods, driverMethodName);
	                    if ((isRequired || driverObject[driverMethodName]) && typeof driverObject[driverMethodName] !== 'function') {
	                        reject(complianceError);
	                        return;
	                    }
	                }

	                var configureMissingMethods = function configureMissingMethods() {
	                    var methodNotImplementedFactory = function methodNotImplementedFactory(methodName) {
	                        return function () {
	                            var error = new Error('Method ' + methodName + ' is not implemented by the current driver');
	                            var promise = Promise$1.reject(error);
	                            executeCallback(promise, arguments[arguments.length - 1]);
	                            return promise;
	                        };
	                    };

	                    for (var _i = 0, _len = OptionalDriverMethods.length; _i < _len; _i++) {
	                        var optionalDriverMethod = OptionalDriverMethods[_i];
	                        if (!driverObject[optionalDriverMethod]) {
	                            driverObject[optionalDriverMethod] = methodNotImplementedFactory(optionalDriverMethod);
	                        }
	                    }
	                };

	                configureMissingMethods();

	                var setDriverSupport = function setDriverSupport(support) {
	                    if (DefinedDrivers[driverName]) {
	                        console.info('Redefining LocalForage driver: ' + driverName);
	                    }
	                    DefinedDrivers[driverName] = driverObject;
	                    DriverSupport[driverName] = support;
	                    // don't use a then, so that we can define
	                    // drivers that have simple _support methods
	                    // in a blocking manner
	                    resolve();
	                };

	                if ('_support' in driverObject) {
	                    if (driverObject._support && typeof driverObject._support === 'function') {
	                        driverObject._support().then(setDriverSupport, reject);
	                    } else {
	                        setDriverSupport(!!driverObject._support);
	                    }
	                } else {
	                    setDriverSupport(true);
	                }
	            } catch (e) {
	                reject(e);
	            }
	        });

	        executeTwoCallbacks(promise, callback, errorCallback);
	        return promise;
	    };

	    LocalForage.prototype.driver = function driver() {
	        return this._driver || null;
	    };

	    LocalForage.prototype.getDriver = function getDriver(driverName, callback, errorCallback) {
	        var getDriverPromise = DefinedDrivers[driverName] ? Promise$1.resolve(DefinedDrivers[driverName]) : Promise$1.reject(new Error('Driver not found.'));

	        executeTwoCallbacks(getDriverPromise, callback, errorCallback);
	        return getDriverPromise;
	    };

	    LocalForage.prototype.getSerializer = function getSerializer(callback) {
	        var serializerPromise = Promise$1.resolve(localforageSerializer);
	        executeTwoCallbacks(serializerPromise, callback);
	        return serializerPromise;
	    };

	    LocalForage.prototype.ready = function ready(callback) {
	        var self = this;

	        var promise = self._driverSet.then(function () {
	            if (self._ready === null) {
	                self._ready = self._initDriver();
	            }

	            return self._ready;
	        });

	        executeTwoCallbacks(promise, callback, callback);
	        return promise;
	    };

	    LocalForage.prototype.setDriver = function setDriver(drivers, callback, errorCallback) {
	        var self = this;

	        if (!isArray(drivers)) {
	            drivers = [drivers];
	        }

	        var supportedDrivers = this._getSupportedDrivers(drivers);

	        function setDriverToConfig() {
	            self._config.driver = self.driver();
	        }

	        function extendSelfWithDriver(driver) {
	            self._extend(driver);
	            setDriverToConfig();

	            self._ready = self._initStorage(self._config);
	            return self._ready;
	        }

	        function initDriver(supportedDrivers) {
	            return function () {
	                var currentDriverIndex = 0;

	                function driverPromiseLoop() {
	                    while (currentDriverIndex < supportedDrivers.length) {
	                        var driverName = supportedDrivers[currentDriverIndex];
	                        currentDriverIndex++;

	                        self._dbInfo = null;
	                        self._ready = null;

	                        return self.getDriver(driverName).then(extendSelfWithDriver)["catch"](driverPromiseLoop);
	                    }

	                    setDriverToConfig();
	                    var error = new Error('No available storage method found.');
	                    self._driverSet = Promise$1.reject(error);
	                    return self._driverSet;
	                }

	                return driverPromiseLoop();
	            };
	        }

	        // There might be a driver initialization in progress
	        // so wait for it to finish in order to avoid a possible
	        // race condition to set _dbInfo
	        var oldDriverSetDone = this._driverSet !== null ? this._driverSet["catch"](function () {
	            return Promise$1.resolve();
	        }) : Promise$1.resolve();

	        this._driverSet = oldDriverSetDone.then(function () {
	            var driverName = supportedDrivers[0];
	            self._dbInfo = null;
	            self._ready = null;

	            return self.getDriver(driverName).then(function (driver) {
	                self._driver = driver._driver;
	                setDriverToConfig();
	                self._wrapLibraryMethodsWithReady();
	                self._initDriver = initDriver(supportedDrivers);
	            });
	        })["catch"](function () {
	            setDriverToConfig();
	            var error = new Error('No available storage method found.');
	            self._driverSet = Promise$1.reject(error);
	            return self._driverSet;
	        });

	        executeTwoCallbacks(this._driverSet, callback, errorCallback);
	        return this._driverSet;
	    };

	    LocalForage.prototype.supports = function supports(driverName) {
	        return !!DriverSupport[driverName];
	    };

	    LocalForage.prototype._extend = function _extend(libraryMethodsAndProperties) {
	        extend(this, libraryMethodsAndProperties);
	    };

	    LocalForage.prototype._getSupportedDrivers = function _getSupportedDrivers(drivers) {
	        var supportedDrivers = [];
	        for (var i = 0, len = drivers.length; i < len; i++) {
	            var driverName = drivers[i];
	            if (this.supports(driverName)) {
	                supportedDrivers.push(driverName);
	            }
	        }
	        return supportedDrivers;
	    };

	    LocalForage.prototype._wrapLibraryMethodsWithReady = function _wrapLibraryMethodsWithReady() {
	        // Add a stub for each driver API method that delays the call to the
	        // corresponding driver method until localForage is ready. These stubs
	        // will be replaced by the driver methods as soon as the driver is
	        // loaded, so there is no performance impact.
	        for (var i = 0, len = LibraryMethods.length; i < len; i++) {
	            callWhenReady(this, LibraryMethods[i]);
	        }
	    };

	    LocalForage.prototype.createInstance = function createInstance(options) {
	        return new LocalForage(options);
	    };

	    return LocalForage;
	}();

	// The actual localForage object that we expose as a module or via a
	// global. It's extended by pulling in one of our other libraries.


	var localforage_js = new LocalForage();

	module.exports = localforage_js;

	},{"3":3}]},{},[4])(4)
	}); 
} (localforage$1, localforage$1.exports));

var localforageExports = localforage$1.exports;
var localforage = /*@__PURE__*/getDefaultExportFromCjs(localforageExports);

/** Simpler wrapper for a file-backed cache for arbitrary metadata. */
class LocalStorageCache {
    appId;
    version;
    persister;
    constructor(appId, version) {
        this.appId = appId;
        this.version = version;
        this.persister = localforage.createInstance({
            name: "dataview/cache/" + appId,
            driver: [localforage.INDEXEDDB],
            description: "Cache metadata about files and sections in the dataview index.",
        });
    }
    /** Drop the entire cache instance and re-create a new fresh instance. */
    async recreate() {
        await localforage.dropInstance({ name: "dataview/cache/" + this.appId });
        this.persister = localforage.createInstance({
            name: "dataview/cache/" + this.appId,
            driver: [localforage.INDEXEDDB],
            description: "Cache metadata about files and sections in the dataview index.",
        });
    }
    /** Load file metadata by path. */
    async loadFile(path) {
        return this.persister.getItem(this.fileKey(path)).then(raw => {
            let result = raw;
            if (result)
                result.data = Transferable.value(result.data);
            return result;
        });
    }
    /** Store file metadata by path. */
    async storeFile(path, data) {
        await this.persister.setItem(this.fileKey(path), {
            version: this.version,
            time: Date.now(),
            data: Transferable.transferable(data),
        });
    }
    /** Drop old file keys that no longer exist. */
    async synchronize(existing) {
        let keys = new Set(await this.allFiles());
        for (let exist of existing)
            keys.delete(exist);
        // Any keys remaining after deleting existing keys are non-existent keys that should be cleared from cache.
        for (let key of keys)
            await this.persister.removeItem(this.fileKey(key));
        return keys;
    }
    /** Obtain a list of all metadata keys. */
    async allKeys() {
        return this.persister.keys();
    }
    /** Obtain a list of all persisted files. */
    async allFiles() {
        let keys = await this.allKeys();
        return keys.filter(k => k.startsWith("file:")).map(k => k.substring(5));
    }
    fileKey(path) {
        return "file:" + path;
    }
}

function decodeBase64(base64, enableUnicode) {
    var binaryString = atob(base64);
    if (enableUnicode) {
        var binaryView = new Uint8Array(binaryString.length);
        for (var i = 0, n = binaryString.length; i < n; ++i) {
            binaryView[i] = binaryString.charCodeAt(i);
        }
        return String.fromCharCode.apply(null, new Uint16Array(binaryView.buffer));
    }
    return binaryString;
}

function createURL(base64, sourcemapArg, enableUnicodeArg) {
    var sourcemap = sourcemapArg === undefined ? null : sourcemapArg;
    var enableUnicode = enableUnicodeArg === undefined ? false : enableUnicodeArg;
    var source = decodeBase64(base64, enableUnicode);
    var start = source.indexOf('\n', 10) + 1;
    var body = source.substring(start) + (sourcemap ? '\/\/# sourceMappingURL=' + sourcemap : '');
    var blob = new Blob([body], { type: 'application/javascript' });
    return URL.createObjectURL(blob);
}

function createBase64WorkerFactory(base64, sourcemapArg, enableUnicodeArg) {
    var url;
    return function WorkerFactory(options) {
        url = url || createURL(base64, sourcemapArg, enableUnicodeArg);
        return new Worker(url, options);
    };
}

var WorkerFactory = createBase64WorkerFactory('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', null, false);
/* eslint-enable */

/** Controls and creates Dataview file importers, allowing for asynchronous loading and parsing of files. */
/** Multi-threaded file parser which debounces rapid file requests automatically. */
class FileImporter extends obsidian.Component {
    numWorkers;
    vault;
    metadataCache;
    /* Background workers which do the actual file parsing. */
    workers;
    /** Tracks which workers are actively parsing a file, to make sure we properly delegate results. */
    busy;
    /** List of files which have been queued for a reload. */
    reloadQueue;
    /** Fast-access set which holds the list of files queued to be reloaded; used for debouncing. */
    reloadSet;
    /** Paths -> promises for file reloads which have not yet been queued. */
    callbacks;
    constructor(numWorkers, vault, metadataCache) {
        super();
        this.numWorkers = numWorkers;
        this.vault = vault;
        this.metadataCache = metadataCache;
        this.workers = [];
        this.busy = [];
        this.reloadQueue = [];
        this.reloadSet = new Set();
        this.callbacks = new Map();
        for (let index = 0; index < numWorkers; index++) {
            let worker = new WorkerFactory({ name: "Dataview Indexer " + (index + 1) });
            worker.onmessage = evt => this.finish(evt.data.path, Transferable.value(evt.data.result), index);
            this.workers.push(worker);
            this.register(() => worker.terminate());
            this.busy.push(false);
        }
    }
    /**
     * Queue the given file for reloading. Multiple reload requests for the same file in a short time period will be de-bounced
     * and all be resolved by a single actual file reload.
     */
    reload(file) {
        let promise = new Promise((resolve, reject) => {
            if (this.callbacks.has(file.path))
                this.callbacks.get(file.path)?.push([resolve, reject]);
            else
                this.callbacks.set(file.path, [[resolve, reject]]);
        });
        // De-bounce repeated requests for the same file.
        if (this.reloadSet.has(file.path))
            return promise;
        this.reloadSet.add(file.path);
        // Immediately run this task if there are available workers; otherwise, add it to the queue.
        let workerId = this.nextAvailableWorker();
        if (workerId !== undefined) {
            this.send(file, workerId);
        }
        else {
            this.reloadQueue.push(file);
        }
        return promise;
    }
    /** Finish the parsing of a file, potentially queueing a new file. */
    finish(path, data, index) {
        // Cache the callbacks before we do book-keeping.
        let calls = [].concat(this.callbacks.get(path) ?? []);
        // Book-keeping to clear metadata & allow the file to be re-loaded again.
        this.reloadSet.delete(path);
        this.callbacks.delete(path);
        // Notify the queue this file is available for new work.
        this.busy[index] = false;
        // Queue a new job onto this worker.
        let job = this.reloadQueue.shift();
        if (job !== undefined)
            this.send(job, index);
        // Resolve promises to let users know this file has finished.
        if ("$error" in data) {
            for (let [_, reject] of calls)
                reject(data["$error"]);
        }
        else {
            for (let [callback, _] of calls)
                callback(data);
        }
    }
    /** Send a new task to the given worker ID. */
    send(file, workerId) {
        this.busy[workerId] = true;
        this.vault.cachedRead(file).then(c => this.workers[workerId].postMessage({
            path: file.path,
            contents: c,
            stat: file.stat,
            metadata: this.metadataCache.getFileCache(file),
        }));
    }
    /** Find the next available, non-busy worker; return undefined if all workers are busy. */
    nextAvailableWorker() {
        let index = this.busy.indexOf(false);
        return index == -1 ? undefined : index;
    }
}

/** Stores various indices on all files in the vault to make dataview generation fast. */
/** Aggregate index which has several sub-indices and will initialize all of them. */
class FullIndex extends obsidian.Component {
    app;
    indexVersion;
    onChange;
    /** Generate a full index from the given vault. */
    static create(app, indexVersion, onChange) {
        return new FullIndex(app, indexVersion, onChange);
    }
    /** Whether all files in the vault have been indexed at least once. */
    initialized;
    /** I/O access to the Obsidian vault contents. */
    vault;
    /** Access to in-memory metadata, useful for parsing and metadata lookups. */
    metadataCache;
    /** Persistent IndexedDB backing store, used for faster startup. */
    persister;
    /* Maps path -> markdown metadata for all markdown pages. */
    pages;
    /** Map files -> tags in that file, and tags -> files. This version includes subtags. */
    tags;
    /** Map files -> exact tags in that file, and tags -> files. This version does not automatically add subtags. */
    etags;
    /** Map files -> linked files in that file, and linked file -> files that link to it. */
    links;
    /** Search files by path prefix. */
    prefix;
    /** Allows for efficient lookups of whether a file is starred or not. */
    starred;
    /** Caches data in CSV files. */
    // TODO: CSV parsing should be done by a worker thread asynchronously to avoid frontend stalls.
    csv;
    /**
     * The current "revision" of the index, which monotonically increases for every index change. Use this to determine
     * if you are up to date.
     */
    revision;
    /** Asynchronously parses files in the background using web workers. */
    importer;
    /** Construct a new index using the app data and a current data version. */
    constructor(app, indexVersion, onChange) {
        super();
        this.app = app;
        this.indexVersion = indexVersion;
        this.onChange = onChange;
        this.initialized = false;
        this.vault = app.vault;
        this.metadataCache = app.metadataCache;
        this.pages = new Map();
        this.tags = new ValueCaseInsensitiveIndexMap();
        this.etags = new ValueCaseInsensitiveIndexMap();
        this.links = new IndexMap();
        this.revision = 0;
        // Caches metadata via durable storage to speed up cache initialization when Obsidian restarts.
        this.persister = new LocalStorageCache(app.appId || "shared", indexVersion);
        // Handles asynchronous reloading of files on web workers.
        this.addChild((this.importer = new FileImporter(2, this.vault, this.metadataCache)));
        // Prefix listens to file creation/deletion/rename, and not modifies, so we let it set up it's own listeners.
        this.addChild((this.prefix = PrefixIndex.create(this.vault, () => this.touch())));
        // The CSV cache also needs to listen to filesystem events for cache invalidation.
        this.addChild((this.csv = new CsvCache(this.vault)));
        // The starred cache fetches starred entries semi-regularly via an interval.
        this.addChild((this.starred = new StarredCache(this.app, () => this.touch())));
    }
    /** Trigger a metadata event on the metadata cache. */
    trigger(...args) {
        this.metadataCache.trigger("dataview:metadata-change", ...args);
    }
    /** "Touch" the index, incrementing the revision number and causing downstream views to reload. */
    touch() {
        this.revision += 1;
        this.onChange();
    }
    /** Runs through the whole vault to set up initial file metadata. */
    initialize() {
        // The metadata cache is updated on initial file index and file loads.
        this.registerEvent(this.metadataCache.on("resolve", file => this.reload(file)));
        // Renames do not set off the metadata cache; catch these explicitly.
        this.registerEvent(this.vault.on("rename", this.rename, this));
        // File creation does cause a metadata change, but deletes do not. Clear the caches for this.
        this.registerEvent(this.vault.on("delete", af => {
            if (!(af instanceof obsidian.TFile) || !PathFilters.markdown(af.path))
                return;
            let file = af;
            this.pages.delete(file.path);
            this.tags.delete(file.path);
            this.etags.delete(file.path);
            this.links.delete(file.path);
            this.touch();
            this.trigger("delete", file);
        }));
        // Asynchronously initialize actual content in the background.
        this._initialize(this.vault.getMarkdownFiles());
    }
    /** Drops the local storage cache and re-indexes all files; this should generally be used if you expect cache issues. */
    async reinitialize() {
        await this.persister.recreate();
        const files = this.vault.getMarkdownFiles();
        const start = Date.now();
        let promises = files.map(file => this.reload(file));
        await Promise.all(promises);
        console.log(`Dataview: re-initialized index with ${files.length} files (${(Date.now() - start) / 1000.0}s)`);
    }
    /** Internal asynchronous initializer. */
    async _initialize(files) {
        let reloadStart = Date.now();
        let promises = files.map(l => this.reload(l));
        let results = await Promise.all(promises);
        let cached = 0, skipped = 0;
        for (let item of results) {
            if (item.skipped) {
                skipped += 1;
                continue;
            }
            if (item.cached)
                cached += 1;
        }
        this.initialized = true;
        this.metadataCache.trigger("dataview:index-ready");
        console.log(`Dataview: all ${files.length} files have been indexed in ${(Date.now() - reloadStart) / 1000.0}s (${cached} cached, ${skipped} skipped).`);
        // Drop keys for files which do not exist anymore.
        let remaining = await this.persister.synchronize(files.map(l => l.path));
        if (remaining.size > 0) {
            console.log(`Dataview: Dropped cache entries for ${remaining.size} deleted files.`);
        }
    }
    rename(file, oldPath) {
        if (!(file instanceof obsidian.TFile) || !PathFilters.markdown(file.path))
            return;
        if (this.pages.has(oldPath)) {
            const oldMeta = this.pages.get(oldPath);
            this.pages.delete(oldPath);
            if (oldMeta) {
                oldMeta.path = file.path;
                this.pages.set(file.path, oldMeta);
            }
        }
        this.tags.rename(oldPath, file.path);
        this.links.rename(oldPath, file.path);
        this.etags.rename(oldPath, file.path);
        this.touch();
        this.trigger("rename", file, oldPath);
    }
    /** Queue a file for reloading; this is done asynchronously in the background and may take a few seconds. */
    async reload(file) {
        if (!PathFilters.markdown(file.path))
            return { cached: false, skipped: true };
        // The first load of a file is attempted from persisted cache; subsequent loads just use the importer.
        if (this.pages.has(file.path) || this.initialized) {
            await this.import(file);
            return { cached: false, skipped: false };
        }
        else {
            // Check the cache for the latest data; if it is out of date or non-existent, then reload.
            return this.persister.loadFile(file.path).then(async (cached) => {
                if (!cached || cached.time < file.stat.mtime || cached.version != this.indexVersion) {
                    // This cache value is out of data, reload via the importer and update the cache.
                    // We will skip files with no active file metadata - they will be caught by a later reload
                    // via the 'resolve' metadata event.
                    let fileCache = this.metadataCache.getFileCache(file);
                    if (fileCache === undefined || fileCache === null)
                        return { cached: false, skipped: true };
                    await this.import(file);
                    return { cached: false, skipped: false };
                }
                else {
                    // Use the cached data since it is up to date and on the same version.
                    this.finish(file, cached.data);
                    return { cached: true, skipped: false };
                }
            });
        }
    }
    /** Import a file directly from disk, skipping the cache. */
    async import(file) {
        return this.importer.reload(file).then(r => {
            this.finish(file, r);
            this.persister.storeFile(file.path, r);
        });
    }
    /** Finish the reloading of file metadata by adding it to in memory indexes. */
    finish(file, parsed) {
        let meta = PageMetadata.canonicalize(parsed, link => {
            let realPath = this.metadataCache.getFirstLinkpathDest(link.path, file.path);
            if (realPath)
                return link.withPath(realPath.path);
            else
                return link;
        });
        this.pages.set(file.path, meta);
        this.tags.set(file.path, meta.fullTags());
        this.etags.set(file.path, meta.tags);
        this.links.set(file.path, new Set(meta.links.map(l => l.path)));
        this.touch();
        this.trigger("update", file);
    }
}
/** Indexes files by their full prefix - essentially a simple prefix tree. */
class PrefixIndex extends obsidian.Component {
    vault;
    updateRevision;
    static create(vault, updateRevision) {
        return new PrefixIndex(vault, updateRevision);
    }
    constructor(vault, updateRevision) {
        super();
        this.vault = vault;
        this.updateRevision = updateRevision;
    }
    *walk(folder, filter) {
        for (const file of folder.children) {
            if (file instanceof obsidian.TFolder) {
                yield* this.walk(file, filter);
            }
            else if (filter ? filter(file.path) : true) {
                yield file.path;
            }
        }
    }
    /** Get the list of all files under the given path. */
    get(prefix, filter) {
        let folder = this.vault.getAbstractFileByPath(prefix || "/");
        return new Set(folder instanceof obsidian.TFolder ? this.walk(folder, filter) : []);
    }
    /** Determines if the given path exists in the prefix index. */
    pathExists(path) {
        return this.vault.getAbstractFileByPath(path || "/") != null;
    }
    /** Determines if the given prefix exists in the prefix index. */
    nodeExists(prefix) {
        return this.vault.getAbstractFileByPath(prefix || "/") instanceof obsidian.TFolder;
    }
    /**
     * Use the in-memory prefix index to convert a relative path to an absolute one.
     */
    resolveRelative(path, origin) {
        if (!origin)
            return path;
        else if (path.startsWith("/"))
            return path.substring(1);
        let relativePath = getParentFolder(origin) + "/" + path;
        if (this.pathExists(relativePath))
            return relativePath;
        else
            return path;
    }
}
/** Simple path filters which filter file types. */
var PathFilters;
(function (PathFilters) {
    function csv(path) {
        return path.toLowerCase().endsWith(".csv");
    }
    PathFilters.csv = csv;
    function markdown(path) {
        let lcPath = path.toLowerCase();
        return lcPath.endsWith(".md") || lcPath.endsWith(".markdown");
    }
    PathFilters.markdown = markdown;
})(PathFilters || (PathFilters = {}));
/**
 * Caches in-use CSVs to make high-frequency reloads (such as actively looking at a document
 * that uses CSV) fast.
 */
class CsvCache extends obsidian.Component {
    vault;
    static CACHE_EXPIRY_SECONDS = 5 * 60;
    // Cache of loaded CSVs; old entries will periodically be removed
    cache;
    // Periodic job which clears out the cache based on time.
    cacheClearInterval;
    constructor(vault) {
        super();
        this.vault = vault;
        this.cache = new Map();
        // Force-flush the cache on CSV file deletions or modifications.
        this.registerEvent(this.vault.on("modify", file => {
            if (file instanceof obsidian.TFile && PathFilters.csv(file.path))
                this.cache.delete(file.path);
        }));
        this.registerEvent(this.vault.on("delete", file => {
            if (file instanceof obsidian.TFile && PathFilters.csv(file.path))
                this.cache.delete(file.path);
        }));
    }
    /** Load a CSV file from the cache, doing a fresh load if it has not been loaded. */
    async get(path) {
        // Clear old entries on every fresh load, since the path being loaded may be stale.
        this.clearOldEntries();
        let existing = this.cache.get(path);
        if (existing)
            return Result.success(existing.data);
        else {
            let value = await this.loadInternal(path);
            if (value.successful)
                this.cache.set(path, { data: value.value, loadTime: DateTime.now() });
            return value;
        }
    }
    /** Do the actual raw loading of a CSV path (which is either local or an HTTP request). */
    async loadInternal(path) {
        // Allow http://, https://, and file:// prefixes which use AJAX.
        if (path.startsWith("http://") || path.startsWith("https://") || path.startsWith("file://")) {
            try {
                let result = await fetch(path, {
                    method: "GET",
                    mode: "no-cors",
                    redirect: "follow",
                });
                return Result.success(parseCsv(await result.text()));
            }
            catch (ex) {
                return Result.failure("" + ex + "\n\n" + ex.stack);
            }
        }
        // Otherwise, assume it is a fully-qualified file path.
        try {
            let fileData = await this.vault.adapter.read(path);
            return Result.success(parseCsv(fileData));
        }
        catch (ex) {
            return Result.failure(`Failed to load data from path '${path}'.`);
        }
    }
    /** Clear old entries in the cache (as measured by insertion time). */
    clearOldEntries() {
        let currentTime = DateTime.now();
        let keysToRemove = new Set();
        for (let [key, value] of this.cache.entries()) {
            let entryAge = Math.abs(currentTime.diff(value.loadTime, "seconds").seconds);
            if (entryAge > CsvCache.CACHE_EXPIRY_SECONDS)
                keysToRemove.add(key);
        }
        keysToRemove.forEach(key => this.cache.delete(key));
    }
}
/** Optional connector to the Obsidian 'Starred' plugin which allows for efficiently querying if a file is starred or not. */
class StarredCache extends obsidian.Component {
    app;
    onUpdate;
    /** Initial delay before checking the cache; we need to wait for it to asynchronously load the initial stars. */
    static INITIAL_DELAY = 4 * 1000;
    /** How frequently to check for star updates. */
    static REFRESH_INTERVAL = 30 * 1000;
    /** Set of all starred file paths. */
    stars;
    constructor(app, onUpdate) {
        super();
        this.app = app;
        this.onUpdate = onUpdate;
        this.stars = StarredCache.fetch(this.app);
        this.registerInterval(window.setInterval(() => this.reload(), StarredCache.REFRESH_INTERVAL));
        const initialHandler = window.setTimeout(() => this.reload(), StarredCache.INITIAL_DELAY);
        this.register(() => window.clearTimeout(initialHandler));
    }
    /** Determines if the given path is starred. */
    starred(path) {
        return this.stars.has(path);
    }
    reload() {
        let newStars = StarredCache.fetch(this.app);
        if (!setsEqual(this.stars, newStars)) {
            this.stars = newStars;
            this.onUpdate();
        }
    }
    /** Fetch all starred files from the stars plugin, if present. */
    static fetch(app) {
        let items = app?.internalPlugins?.plugins?.bookmarks?.instance?.items;
        if (items == undefined)
            return new Set();
        // Retrieve all grouped (nested) items, returning a flat array
        const flattenItems = (items) => {
            let children = [];
            return items
                .map(i => {
                if (i.type == "group" && i.items && i.items.length) {
                    children = [...children, ...i.items];
                }
                return i;
            })
                .concat(children.length ? flattenItems(children) : children);
        };
        items = flattenItems(items);
        return new Set(items.filter((l) => l.type === "file").map(l => l.path));
    }
}
/** A generic index which indexes variables of the form key -> value[], allowing both forward and reverse lookups. */
class IndexMap {
    /** Maps key -> values for that key. */
    map;
    /** Cached inverse map; maps value -> keys that reference that value. */
    invMap;
    /** Create a new, empty index map. */
    constructor() {
        this.map = new Map();
        this.invMap = new Map();
    }
    /** Returns all values for the given key. */
    get(key) {
        let result = this.map.get(key);
        if (result) {
            return new Set(result);
        }
        else {
            return new Set();
        }
    }
    /** Returns all keys that reference the given key. Mutating the returned set is not allowed. */
    getInverse(value) {
        return this.invMap.get(value) || IndexMap.EMPTY_SET;
    }
    /** Sets the key to the given values; this will delete the old mapping for the key if one was present. */
    set(key, values) {
        if (!values.size) {
            // no need to store if no values
            this.delete(key);
            return this;
        }
        let oldValues = this.map.get(key);
        if (oldValues) {
            for (let value of oldValues) {
                // Only delete the ones we're not adding back
                if (!values.has(key))
                    this.invMap.get(value)?.delete(key);
            }
        }
        this.map.set(key, values);
        for (let value of values) {
            if (!this.invMap.has(value))
                this.invMap.set(value, new Set([key]));
            else
                this.invMap.get(value)?.add(key);
        }
        return this;
    }
    /** Clears all values for the given key so they can be re-added. */
    delete(key) {
        let oldValues = this.map.get(key);
        if (!oldValues)
            return false;
        this.map.delete(key);
        for (let value of oldValues) {
            this.invMap.get(value)?.delete(key);
        }
        return true;
    }
    /** Rename all references to the given key to a new value. */
    rename(oldKey, newKey) {
        let oldValues = this.map.get(oldKey);
        if (!oldValues)
            return false;
        this.delete(oldKey);
        this.set(newKey, oldValues);
        return true;
    }
    /** Clear the entire index. */
    clear() {
        this.map.clear();
        this.invMap.clear();
    }
    static EMPTY_SET = Object.freeze(new Set());
}
/** Index map wrapper which is case-insensitive in the key. */
class ValueCaseInsensitiveIndexMap {
    delegate;
    /** Create a new, empty case insensitive index map. */
    constructor(delegate = new IndexMap()) {
        this.delegate = delegate;
    }
    /** Returns all values for the given key. */
    get(key) {
        return this.delegate.get(key);
    }
    /** Returns all keys that reference the given value. Mutating the returned set is not allowed. */
    getInverse(value) {
        return this.delegate.getInverse(value.toLocaleLowerCase());
    }
    /** Sets the key to the given values; this will delete the old mapping for the key if one was present. */
    set(key, values) {
        this.delegate.set(key, new Set(Array.from(values).map(v => v.toLocaleLowerCase())));
        return this;
    }
    /** Clears all values for the given key so they can be re-added. */
    delete(key) {
        return this.delegate.delete(key);
    }
    /** Rename all references to the given key to a new value. */
    rename(oldKey, newKey) {
        return this.delegate.rename(oldKey, newKey);
    }
    /** Clear the entire index. */
    clear() {
        this.delegate.clear();
    }
}

/** Collect data matching a source query. */
/** Find source paths which match the given source. */
function matchingSourcePaths(source, index, originFile = "") {
    switch (source.type) {
        case "empty":
            return Result.success(new Set());
        case "tag":
            return Result.success(index.tags.getInverse(source.tag));
        case "csv":
            return Result.success(new Set([index.prefix.resolveRelative(source.path, originFile)]));
        case "folder":
            // Prefer loading from the folder at the given path.
            if (index.prefix.nodeExists(source.folder))
                return Result.success(index.prefix.get(source.folder, PathFilters.markdown));
            // But allow for loading individual files if they exist.
            if (index.prefix.pathExists(source.folder))
                return Result.success(new Set([source.folder]));
            else if (index.prefix.pathExists(source.folder + ".md"))
                return Result.success(new Set([source.folder + ".md"]));
            // For backwards-compat, return an empty result even if the folder does not exist.
            return Result.success(new Set());
        case "link":
            let fullPath = index.metadataCache.getFirstLinkpathDest(source.file, originFile)?.path;
            if (!fullPath) {
                // Look in links which includes unresolved links
                return Result.success(index.links.getInverse(source.file));
            }
            if (source.direction === "incoming") {
                // To find all incoming links (i.e., things that link to this), use the index that Obsidian provides.
                // TODO: Use an actual index so this isn't a fullscan.
                let resolved = index.metadataCache.resolvedLinks;
                let incoming = new Set();
                for (let [key, value] of Object.entries(resolved)) {
                    if (fullPath in value)
                        incoming.add(key);
                }
                return Result.success(incoming);
            }
            else {
                let resolved = index.metadataCache.resolvedLinks;
                if (!(fullPath in resolved))
                    return Result.failure(`Could not find file "${source.file}" during link lookup - does it exist?`);
                return Result.success(new Set(Object.keys(index.metadataCache.resolvedLinks[fullPath])));
            }
        case "binaryop":
            return Result.flatMap2(matchingSourcePaths(source.left, index, originFile), matchingSourcePaths(source.right, index, originFile), (left, right) => {
                if (source.op == "&") {
                    let result = new Set();
                    for (let elem of right) {
                        if (left.has(elem))
                            result.add(elem);
                    }
                    return Result.success(result);
                }
                else if (source.op == "|") {
                    let result = new Set(left);
                    for (let elem of right)
                        result.add(elem);
                    return Result.success(result);
                }
                else {
                    return Result.failure(`Unrecognized operator '${source.op}'.`);
                }
            });
        case "negate":
            return matchingSourcePaths(source.child, index, originFile).map(child => {
                // TODO: This is obviously very inefficient. Can be improved by complicating the
                // return type of this function & optimizing 'and' / 'or'.
                let allFiles = new Set(index.vault.getMarkdownFiles().map(f => f.path));
                child.forEach(f => allFiles.delete(f));
                return allFiles;
            });
    }
}
/** Convert a path to the data for that path; usually markdown pages, but could also be other file types (like CSV).  */
async function resolvePathData(path, index) {
    if (PathFilters.csv(path))
        return resolveCsvData(path, index);
    else
        return resolveMarkdownData(path, index);
}
// TODO: We shouldn't be doing path normalization here relative to an origin file,
/** Convert a CSV path to the data in the CSV (in dataview format). */
async function resolveCsvData(path, index) {
    let rawData = await index.csv.get(path);
    return rawData.map(rows => {
        return rows.map((row, index) => {
            return {
                id: `${path}#${index}`,
                data: row,
            };
        });
    });
}
/** Convert a path pointing to a markdown page, into the associated metadata. */
function resolveMarkdownData(path, index) {
    let page = index.pages.get(path);
    if (!page)
        return Result.success([]);
    return Result.success([
        {
            id: Link.file(path),
            data: page.serialize(index),
        },
    ]);
}
/** Resolve a source to the collection of data rows that it matches. */
async function resolveSource(source, index, originFile = "") {
    let paths = matchingSourcePaths(source, index, originFile);
    if (!paths.successful)
        return Result.failure(paths.error);
    let result = [];
    for (let path of paths.value) {
        let resolved = await resolvePathData(path, index);
        if (!resolved.successful)
            return resolved;
        for (let val of resolved.value)
            result.push(val);
    }
    return Result.success(result);
}

// cyrb53 (c) 2018 bryc (github.com/bryc). License: Public domain. Attribution appreciated.
// A fast and simple 64-bit (or 53-bit) string hash function with decent collision resistance.
// Largely inspired by MurmurHash2/3, but with a focus on speed/simplicity.
// See https://stackoverflow.com/questions/7616461/generate-a-hash-from-string-in-javascript/52171480#52171480
// https://github.com/bryc/code/blob/master/jshash/experimental/cyrb53.js
function cyrb53(str, seed = 0) {
    let h1 = 0xdeadbeef ^ seed, h2 = 0x41c6ce57 ^ seed;
    for (let i = 0, ch; i < str.length; i++) {
        ch = str.charCodeAt(i);
        h1 = Math.imul(h1 ^ ch, 2654435761);
        h2 = Math.imul(h2 ^ ch, 1597334677);
    }
    h1 = Math.imul(h1 ^ (h1 >>> 16), 2246822507);
    h1 ^= Math.imul(h2 ^ (h2 >>> 13), 3266489909);
    h2 = Math.imul(h2 ^ (h2 >>> 16), 2246822507);
    h2 ^= Math.imul(h1 ^ (h1 >>> 13), 3266489909);
    // For a full 64-bit value we could return
    //  [h2>>>0, h1>>>0]
    return 4294967296 * (2097151 & h2) + (h1 >>> 0); // ;
}

/** Default function implementations for the expression evaluator. */
/**
 * Allows for the creation of functions that check the number and type of their arguments, and dispatch
 * to different implementations based on the types of the inputs.
 */
class FunctionBuilder {
    name;
    variants;
    vectorized;
    constructor(name) {
        this.name = name;
        this.variants = [];
        this.vectorized = {};
    }
    /** Add a general function variant which accepts any number of arguments of any type. */
    vararg(impl) {
        this.variants.push({ args: [], varargs: true, impl });
        return this;
    }
    /** Add a function variant which takes in a single argument. */
    add1(argType, impl) {
        this.variants.push({
            args: [argType],
            varargs: false,
            impl: (c, ...rest) => impl(rest[0], c),
        });
        return this;
    }
    /** Add a function variant which takes in two typed arguments. */
    add2(arg1, arg2, impl) {
        this.variants.push({
            args: [arg1, arg2],
            varargs: false,
            impl: (c, ...rest) => impl(rest[0], rest[1], c),
        });
        return this;
    }
    /** Add a function variant which takes in three typed arguments. */
    add3(arg1, arg2, arg3, impl) {
        this.variants.push({
            args: [arg1, arg2, arg3],
            varargs: false,
            impl: (c, ...rest) => impl(rest[0], rest[1], rest[2], c),
        });
        return this;
    }
    /** Add vectorized variants which accept the given number of arguments and delegate. */
    vectorize(numArgs, positions) {
        this.vectorized[numArgs] = positions;
        return this;
    }
    /** Return a function which checks the number and type of arguments, passing them on to the first matching variant. */
    build() {
        let self = (context, ...args) => {
            let types = [];
            for (let arg of args) {
                let argType = Values.typeOf(arg);
                if (!argType)
                    throw Error(`Unrecognized argument type for argument '${arg}'`);
                types.push(argType);
            }
            // Handle vectorization, possibly in multiple fields.
            if (this.vectorized[types.length]) {
                let vectorizedPositions = this.vectorized[types.length].filter(k => types[k] == "array");
                if (vectorizedPositions.length > 0) {
                    let minLength = vectorizedPositions
                        .map(p => args[p].length)
                        .reduce((p, c) => Math.min(p, c));
                    // Call the subfunction for each element in the longest array.
                    // If you call a vectorized function with different-length arrays,
                    // the output is limited by the length of the shortest array.
                    let result = [];
                    for (let vpos = 0; vpos < minLength; vpos++) {
                        let subargs = [];
                        for (let index = 0; index < args.length; index++) {
                            if (vectorizedPositions.includes(index)) {
                                let arr = args[index];
                                subargs.push(arr[vpos]);
                            }
                            else {
                                subargs.push(args[index]);
                            }
                        }
                        result.push(self(context, ...subargs));
                    }
                    return result;
                }
            }
            outer: for (let variant of this.variants) {
                if (variant.varargs)
                    return variant.impl(context, ...args);
                if (variant.args.length != types.length)
                    continue;
                for (let index = 0; index < variant.args.length; index++) {
                    if (variant.args[index] != "*" && variant.args[index] != types[index])
                        continue outer;
                }
                return variant.impl(context, ...args);
            }
            throw Error(`No implementation of '${this.name}' found for arguments: ${types.join(", ")}`);
        };
        return self;
    }
}
/** Utilities for managing function implementations. */
var Functions;
(function (Functions) {
    /** Bind a context to