package models

import (
	"github.com/google/uuid"
	"github.com/guregu/null/v5"
)

const APSchema = `
CREATE TABLE IF NOT EXISTS aps (
    id TEXT PRIMARY KEY,
    name TEXT UNIQUE NOT NULL,
    firstConnectedAt DATETIME DEFAULT NULL,
    lastConnectedAt DATETIME DEFAULT NULL
);
`

type AP struct {
	ID               uuid.UUID        `db:"id" json:"id"`
	Name             string           `db:"name" json:"name"`
	Mac              string           `json:"mac"`
	FirstConnectedAt null.Time        `db:"firstConnectedAt" json:"firstConnectedAt"`
	LastConnectedAt  null.Time        `db:"lastConnectedAt" json:"lastConnectedAt"`
	MeshConnections  []MeshConnection `json:"meshConnections"`
	ConnectedDevices int              `json:"connectedDevices"`
	Online           bool             `json:"online"`
	SignalStrength   int              `json:"signal"`
}

const MeshConnectionSchema = `
CREATE TABLE IF NOT EXISTS ap_mesh_connections (
    ap UUID NOT NULL,
    otherAP UUID NOT NULL,
    signalStrength TEXT NOT NULL,
    PRIMARY KEY(ap, otherAP),
    FOREIGN KEY(ap) REFERENCES aps(id) ON DELETE CASCADE,
    FOREIGN KEY(otherAP) REFERENCES aps(id) ON DELETE CASCADE
);

CREATE UNIQUE INDEX IF NOT EXISTS idx_unique_links ON ap_mesh_connections (
    min(ap, otherAP),
    max(ap, otherAP)
);`

type MeshConnection struct {
	AP             uuid.UUID      `db:"ap" json:"-"`
	OtherAP        uuid.UUID      `db:"otherAP" json:"otherAP"`
	SignalStrength SignalStrength `db:"signalStrength" json:"signalStrength"`
}

const ConnectedDeviceSchema = `
CREATE TABLE IF NOT EXISTS ap_connected_devices (
    ap UUID NOT NULL,
    device UUID NOT NULL,
    signalStrength TEXT NOT NULL,
    PRIMARY KEY(ap, device),
    FOREIGN KEY(ap) REFERENCES aps(id) ON DELETE CASCADE,
    FOREIGN KEY(device) REFERENCES devices(id) ON DELETE CASCADE
);`

type ConnectedDevice struct {
	AP             uuid.UUID      `db:"ap" json:"-"`
	Device         uuid.UUID      `db:"device" json:"device"`
	SignalStrength SignalStrength `db:"signalStrength" json:"signalStrength"`
}

type SignalStrength string

const (
	SignalStrengthBad      SignalStrength = "bad"
	SignalStrengthModerate SignalStrength = "moderate"
	SignalStrengthGreat    SignalStrength = "great"
)
