package models

import (
	"edge-api/internal/database"
	"fmt"
	"time"

	"github.com/google/uuid"
	"github.com/guregu/null/v5"
)

const DeviceSchema = `
CREATE TABLE IF NOT EXISTS devices (
    id TEXT PRIMARY KEY,
    name TEXT UNIQUE NOT NULL,
    type TEXT NOT NULL,
    password TEXT DEFAULT NULL,
    mac TEXT DEFAULT NULL,
    ip TEXT DEFAULT NULL,
    currentAP TEXT DEFAULT NULL,
    addedBy TEXT DEFAULT NULL,
    addedAt DATETIME DEFAULT CURRENT_TIMESTAMP,
    firstConnectedAt DATETIME DEFAULT NULL,
    lastConnectedAt DATETIME DEFAULT NULL,
    network TEXT DEFAULT NULL,
    online BOOLEAN DEFAULT FALSE,
    last_checked_at DATETIME DEFAULT NULL,
    auth BOOLEAN DEFAULT FALSE,
    authorized BOOLEAN DEFAULT FALSE,
    SignalStrength INTEGER DEFAULT NULL,
    bytes_rx INTEGER DEFAULT NULL,
    bytes_tx INTEGER DEFAULT NULL,
    packets_rx INTEGER DEFAULT NULL,
    packets_tx INTEGER DEFAULT NULL,
    rate_rx INTEGER DEFAULT NULL,
    rate_tx INTEGER DEFAULT NULL,
    FOREIGN KEY (currentAP) REFERENCES aps(id) ON DELETE SET NULL,
    FOREIGN KEY (addedBy) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (network) REFERENCES networks(id) ON DELETE CASCADE
);
`

type DeviceType string

const (
	DeviceTypePhone   DeviceType = "phone"
	DeviceTypeLaptop  DeviceType = "laptop"
	DeviceTypePC      DeviceType = "pc"
	DeviceTypePrinter DeviceType = "printer"
	DeviceTypeIoT     DeviceType = "iot"
)

type Device struct {
	ID               uuid.UUID             `db:"id" json:"id"`
	Name             string                `db:"name" json:"name"`
	Type             DeviceType            `db:"type" json:"type"`
	Password         null.String           `db:"password" json:"password"`
	MAC              null.String           `db:"mac" json:"mac"`
	IP               null.String           `db:"ip" json:"ip"`
	CurrentAP        null.String           `db:"currentAP" json:"currentAP"`
	CurrentAPName    *string               `db:"currentAP_name" json:"currentAP_name"`
	AddedBy          null.Value[uuid.UUID] `db:"addedBy" json:"addedBy"`
	AddedAt          time.Time             `db:"addedAt" json:"addedAt"`
	FirstConnectedAt null.Time             `db:"firstConnectedAt" json:"firstConnectedAt"`
	LastConnectedAt  null.Time             `db:"lastConnectedAt" json:"lastConnectedAt"`
	Network          uuid.UUID             `db:"network" json:"network"`
	Online           bool                  `db:"online" json:"online"`
	LastCheckedAt    null.Time             `db:"last_checked_at" json:"last_checked_at"`
	Auth             bool                  `db:"auth" json:"auth"`
	Authorized       bool                  `db:"authorized" json:"authorized"`
	SignalStrength   null.Int              `db:"SignalStrength" json:"signal"`
	BytesRX          null.Int              `db:"bytes_rx" json:"bytes_rx"`
	BytesTX          null.Int              `db:"bytes_tx" json:"bytes_tx"`
	PacketsRX        null.Int              `db:"packets_rx" json:"packets_rx"`
	PacketsTX        null.Int              `db:"packets_tx" json:"packets_tx"`
	RateRX           null.Int              `db:"rate_rx" json:"rate_rx"`
	RateTX           null.Int              `db:"rate_tx" json:"rate_tx"`
}

type DeviceNew struct {
	Name string     `json:"name" binding:"required"`
	Type DeviceType `json:"type" binding:"required"`
}

type ClientHostapd struct {
	Clients map[string]Client `json:"clients"`
	Freq    int               `json:"freq"`
}

type Client struct {
	AID                  int          `json:"aid"`
	AirTime              AirTime      `json:"airtime"`
	Assoc                bool         `json:"assoc"`
	Auth                 bool         `json:"auth"`
	Authorized           bool         `json:"authorized"`
	Bytes                Bytes        `json:"bytes"`
	Capabilities         Capabilities `json:"capabilities"`
	ExtendedCapabilities []int        `json:"extended_capabilities"`
	HE                   bool         `json:"he"`
	HT                   bool         `json:"ht"`
	MBO                  bool         `json:"mbo"`
	MFP                  bool         `json:"mfp"`
	Packets              Packets      `json:"packets"`
	PreAuth              bool         `json:"preauth"`
	PSK                  string       `json:"psk"`
	Rate                 Rate         `json:"rate"`
	RRM                  []int        `json:"rrm"`
	Signal               int          `json:"signal"`
	VHT                  bool         `json:"vht"`
	WDS                  bool         `json:"wds"`
	WMM                  bool         `json:"wmm"`
	WPS                  bool         `json:"wps"`
}

type AirTime struct {
	RX int `json:"rx"`
	TX int `json:"tx"`
}

type Bytes struct {
	RX int `json:"rx"`
	TX int `json:"tx"`
}

type Capabilities struct {
	VHT VHTCapabilities `json:"vht"`
}

type VHTCapabilities struct {
	MCSMap       MCSMap `json:"mcs_map"`
	MUBeamformee bool   `json:"mu_beamformee"`
	SUBeamformee bool   `json:"su_beamformee"`
}

type MCSMap struct {
	RX MCS `json:"rx"`
	TX MCS `json:"tx"`
}

type MCS struct {
	OneSS   int `json:"1ss"`
	TwoSS   int `json:"2ss"`
	ThreeSS int `json:"3ss"`
	FourSS  int `json:"4ss"`
	FiveSS  int `json:"5ss"`
	SixSS   int `json:"6ss"`
	SevenSS int `json:"7ss"`
	EightSS int `json:"8ss"`
}

type Packets struct {
	RX int `json:"rx"`
	TX int `json:"tx"`
}

type Rate struct {
	RX int `json:"rx"`
	TX int `json:"tx"`
}

func DeviceGetByNetwork(networkID uuid.UUID) ([]Device, error) {
	devices := []Device{}
	err := database.DB.Select(&devices, "SELECT * FROM devices WHERE network = ?", networkID)
	if err != nil {
		return nil, fmt.Errorf("error getting devices from database: %w", err)
	}
	return devices, nil
}

func (d *Device) GetByMAC(mac string) error {
	return database.DB.Get(d, "SELECT * FROM devices WHERE mac = ?", mac)
}

func (d *Device) GetByPassword(password string) error {
	return database.DB.Get(d, "SELECT * FROM devices WHERE password = ?", password)
}

func (d *Device) GetAllActivated() ([]Device, error) {
	devices := []Device{}
	err := database.DB.Select(&devices, "SELECT * FROM devices WHERE authorized = 1")
	if err != nil {
		return nil, fmt.Errorf("error getting activated devices from database: %w", err)
	}
	return devices, nil
}

func (d *Device) GetAllOnline() ([]Device, error) {
	devices := []Device{}
	err := database.DB.Select(&devices, "SELECT * FROM devices WHERE online = 1")
	if err != nil {
		return nil, fmt.Errorf("error getting online devices from database: %w", err)
	}
	return devices, nil
}

func (d *Device) SetDeviceOffline() error {

	d.Online = false
	d.SignalStrength = null.Int{}
	d.CurrentAP = null.String{}
	d.BytesTX = null.Int{}
	d.BytesRX = null.Int{}

	_, err := database.DB.NamedExec(`UPDATE devices SET online = :online, SignalStrength = :SignalStrength, currentAP = :currentAP, bytes_tx = :bytes_tx, bytes_rx = :bytes_rx  WHERE id = :id`, d)

	if err != nil {
		return fmt.Errorf("error updating device online status to offline in database: %w", err)
	}
	return nil
}
