package models

import (
	"edge-api/internal/database"
	"fmt"

	"github.com/google/uuid"
	"github.com/guregu/null/v5"
)

const NetworkSchema = `
CREATE TABLE IF NOT EXISTS networks (
    id TEXT PRIMARY KEY,
    name TEXT UNIQUE NOT NULL,
    type TEXT NOT NULL,
    vlan INTEGER UNIQUE NOT NULL,
    password TEXT DEFAULT NULL
);
`

type NetworkType string

const (
	NetworkTypeGuest    NetworkType = "guest"
	NetworkTypeBusiness NetworkType = "business"
)

type Network struct {
	ID                   uuid.UUID   `db:"id" json:"id"`
	Name                 string      `db:"name" json:"name"`
	Type                 NetworkType `db:"type" json:"type"`
	VLAN                 int         `db:"vlan" json:"-"`
	Password             null.String `db:"password" json:"-"`
	TotalClients         int         `json:"total_clients"`
	AuthenticatedClients int         `json:"authenticated_clients"`
	OnlineClients        int         `json:"online_clients"`
	OfflineClients       int         `json:"offline_clients"`
	TotalDownload        int64       `json:"total_download"`
	TotalUpload          int64       `json:"total_upload"`
}

type NetworkNew struct {
	Name string      `json:"name" binding:"required"`
	Type NetworkType `json:"type" binding:"required"`
}

func (network *Network) AddClientInfo() error {
	var err error

	network.TotalClients, err = network.GetTotalClients()
	if err != nil {
		return fmt.Errorf("failed to get total clients for network %s: %w", network.ID, err)
	}

	network.AuthenticatedClients, err = network.GetAutenthicatedClients()
	if err != nil {
		return fmt.Errorf("failed to get authenticated clients for network %s: %w", network.ID, err)
	}

	network.OnlineClients, err = network.GetOnlineClients()
	if err != nil {
		return fmt.Errorf("failed to get online clients for network %s: %w", network.ID, err)
	}

	network.OfflineClients, err = network.GetOfflineClients()
	if err != nil {
		return fmt.Errorf("failed to get offline clients for network %s: %w", network.ID, err)
	}

	network.TotalDownload, err = network.GetTotalDownload()
	if err != nil {
		return fmt.Errorf("failed to get total download for network %s: %w", network.ID, err)
	}

	network.TotalUpload, err = network.GetTotalUpload()
	if err != nil {
		return fmt.Errorf("failed to get total upload for network %s: %w", network.ID, err)
	}

	return nil
}

func (network *Network) GetTotalClients() (int, error) {
	var count int

	err := database.DB.Get(&count, "SELECT count(id) FROM devices WHERE network = ?", network.ID)
	if err != nil {
		return 0, fmt.Errorf("error getting total clients for network %s: %w", network.ID, err)
	}

	return count, nil
}

func (network *Network) GetAutenthicatedClients() (int, error) {
	var count int

	err := database.DB.Get(&count, "SELECT count(id) FROM devices WHERE network = ? AND authorized = 1", network.ID)
	if err != nil {
		return 0, fmt.Errorf("error getting total clients for network %s: %w", network.ID, err)
	}

	return count, nil
}

func (network *Network) GetOnlineClients() (int, error) {
	var count int

	err := database.DB.Get(&count, "SELECT count(id) FROM devices WHERE network = ? AND online = 1", network.ID)
	if err != nil {
		return 0, fmt.Errorf("error getting total clients for network %s: %w", network.ID, err)
	}

	return count, nil
}

func (network *Network) GetOfflineClients() (int, error) {
	var count int

	err := database.DB.Get(&count, "SELECT count(id) FROM devices WHERE network = ? AND online = 0 AND authorized = 1", network.ID)
	if err != nil {
		return 0, fmt.Errorf("error getting total clients for network %s: %w", network.ID, err)
	}

	return count, nil
}

func (network *Network) GetTotalDownload() (int64, error) {
	var totalBytes null.Int

	err := database.DB.Get(&totalBytes, "SELECT COALESCE(SUM(bytes_rx), 0) FROM devices WHERE network = ?", network.ID)
	if err != nil {
		return 0, fmt.Errorf("error getting total download for network %s: %w", network.ID, err)
	}
	// if NULL
	if !totalBytes.Valid {
		return 0, nil
	}

	return totalBytes.Int64, nil
}

func (network *Network) GetTotalUpload() (int64, error) {
	var totalBytes null.Int

	err := database.DB.Get(&totalBytes, "SELECT COALESCE(SUM(bytes_tx), 0) FROM devices WHERE network = ?", network.ID)
	if err != nil {
		return 0, fmt.Errorf("error getting total download for network %s: %w", network.ID, err)
	}
	// if NULL
	if !totalBytes.Valid {
		return 0, nil
	}

	return totalBytes.Int64, nil
}
