package models

import (
	"encoding/json"
	"fmt"
	"time"
)

type NodeResponse struct {
	IPAddress string  `json:"ip_address"`
	MAC       string  `json:"mac"`
	Name      string  `json:"name"`
	Radios    []Radio `json:"radios"`
	Serial    string  `json:"serial"`
}

type Radio struct {
	Band       string `json:"band"`
	Bandwidth  string `json:"bandwidth"`
	Generation string `json:"generation"`
}

type NodeResponseList []*NodeResponse
type NodeResponseMap map[string]*NodeResponse // Map where node names map to node responses

func (n *NodeResponseMap) Names() []string {
	// return empty string instead of null
	if n == nil || *n == nil {
		return []string{}
	}

	nodeMap := *n

	names := make([]string, 0, len(nodeMap))

	for nodeName := range nodeMap {
		names = append(names, nodeName)
	}

	fmt.Println("access points:", names)
	return names
}

type NodeTimeoutError struct {
	Node NodeResponse
}

func (e *NodeTimeoutError) Error() string {
	return fmt.Sprintf("Edge controller /nodes/%s/metrics request timed out", e.Node.MAC)
}

type NodeMetricsResponse struct {
	Neighbors      []NodeMetricsResponseNeighbor  `json:"neighbors"`
	Interfaces     []NodeMetricsResponseInterface `json:"interfaces"`
	ClientsHostapd []ClientHostapd                `json:"clients_hostapd"` // Match JSON key from API response

	// TODO: Clients
	// TODO: Originators
}

type NodeMetricsResponseNeighbor struct {
	HardwareInterfaceIndex int            `json:"hard_ifindex"`
	HardwareInterfaceName  string         `json:"hard_ifname"`
	LastSeen               DurationMillis `json:"last_seen_msecs"`
	NeighborAddress        string         `json:"neigh_address"`
}

type NodeMetricsResponseInterface struct {
	MeshInterfaceIndex       int    `json:"mesh_ifindex"`
	MeshInterfaceName        string `json:"mesh_ifname"`
	HardwareInterfaceIndex   int    `json:"hard_ifindex"`
	HardwareInterfaceName    string `json:"hard_ifname"`
	HardwareInterfaceAddress string `json:"hard_address"`
	Active                   bool   `json:"active"`
	HopPenalty               int    `json:"hop_penalty"`
	ELPInterval              int    `json:"elp_interval"`
	ThroughputOverride       int    `json:"throughput_override"`
}

func (nmr *NodeMetricsResponse) FindInInterfaces(address string) bool {
	for _, iface := range nmr.Interfaces {
		if iface.HardwareInterfaceAddress == address {
			return true
		}
	}
	return false
}

type NodeMetricsMap map[string]*NodeMetricsResponse // Map where node names map to metric responses

type NodeNeighbor struct {
	Name    string
	Metrics *NodeMetricsResponseNeighbor
}

func (m *NodeMetricsMap) FindNeighbors() *map[string]*NodeNeighbor {
	neighbors := make(map[string]*NodeNeighbor)

	for nodeName, nodeMetrics := range *m {
		if nodeMetrics == nil {
			continue
		}

		for _, neighbor := range nodeMetrics.Neighbors {
			for potentialNeighborNodeName, potentialNeighborMetrics := range *m {
				if potentialNeighborMetrics == nil {
					continue
				}
				if potentialNeighborMetrics.FindInInterfaces(neighbor.NeighborAddress) {
					neighbors[nodeName] = &NodeNeighbor{
						Name:    potentialNeighborNodeName,
						Metrics: &neighbor,
					}
				}
			}
		}
	}

	return &neighbors
}

type DurationMillis time.Duration

func (d *DurationMillis) UnmarshalJSON(b []byte) error {
	var ms int64
	if err := json.Unmarshal(b, &ms); err != nil {
		return fmt.Errorf("failed to unmarshal duration: %v", err)
	}
	*d = DurationMillis(time.Duration(ms) * time.Millisecond)
	return nil
}

func (d DurationMillis) Duration() time.Duration {
	return time.Duration(d)
}
