package services

import (
	"crypto/rand"
	"edge-api/internal/models"
	"math/big"
	"time"

	"github.com/dgrijalva/jwt-go"
	"golang.org/x/crypto/bcrypt"
)

// HashPassword hashes a plain password
func HashPassword(password string) (string, error) {
	bytes, err := bcrypt.GenerateFromPassword([]byte(password), 14)
	return string(bytes), err
}

// CheckPasswordHash compares a hashed password with its plain version
func CheckPasswordHash(password, hash string) bool {
	err := bcrypt.CompareHashAndPassword([]byte(hash), []byte(password))
	return err == nil
}

// GenerateJWT generates a JWT token
func GenerateJWT(secret string, expiration time.Duration, user models.User) (string, error) {

	claims := &models.Claims{
		Email: user.Email,
		StandardClaims: jwt.StandardClaims{
			ExpiresAt: time.Now().Add(expiration).Unix(),
		},
	}

	token := jwt.NewWithClaims(jwt.SigningMethodHS256, claims)
	tokenString, err := token.SignedString([]byte(secret))
	if err != nil {
		return "", err
	}
	return tokenString, nil
}

func RandomLexicalPassword(length int) (string, error) {
	attributes := []string{
		"Happy",
		"Sad",
		"Angry",
		"Excited",
		"Curious",
		"Brave",
		"Smart",
		"Kind",
		"Funny",
		"Creative",
		"Friendly",
		"Fast",
		"Strong",
		"Wise",
		"Bold",
		"Calm",
		"Joyful",
		"Hopeful",
		"Peaceful",
		"Generous",
		"Thoughtful",
		"Adventurous",
		"Playful",
		"Imaginative",
		"Confident",
		"Determined",
		"Optimistic",
		"Respectful",
		"Trustworthy",
		"Empathetic",
		"Supportive",
		"Honest",
		"Reliable",
		"Patient",
	}

	individuals := []string{
		"Cat",
		"Dog",
		"Bird",
		"Fish",
		"Elephant",
		"Lion",
		"Tiger",
		"Giraffe",
		"Monkey",
		"Bear",
		"Penguin",
		"Koala",
		"Zebra",
		"Panda",
		"Human",
		"Robot",
		"Alien",
		"Dragon",
		"Unicorn",
		"Phoenix",
		"Mermaid",
		"Fairy",
		"Vampire",
		"Witch",
		"Zombie",
		"Ghost",
		"Monster",
		"Superhero",
		"Villain",
		"Wizard",
		"Knight",
		"Princess",
		"Prince",
		"King",
		"Queen",
		"Pirate",
		"Explorer",
		"Adventurer",
		"Detective",
		"Scientist",
	}

	// Get a random attribute and individual
	randomAttribute := attributes[randInt(len(attributes))]
	randomIndividual := individuals[randInt(len(individuals))]

	// Add a random number of length "length" to the end of the password
	randomNumber := ""
	for i := 0; i < length; i++ {
		// Get a random Number
		randomDigit, err := rand.Int(rand.Reader, big.NewInt(9))
		if err != nil {
			return "", err
		}

		randomNumber += randomDigit.String()
	}

	return randomAttribute + randomIndividual + randomNumber, nil
}

func RandomPassword(length int) (string, error) {
	// Define character sets to use in the password
	lowercase := "abcdefghijkmnopqrstuvwxyz"
	uppercase := "ABCDEFGHJKLMNPQRSTUVWXYZ"
	numbers := "0123456789"
	special := "!@"

	allChars := lowercase + uppercase + numbers + special
	password := make([]byte, length)

	// Ensure the password contains at least one character from each set
	password[0] = lowercase[randInt(len(lowercase))]
	password[1] = uppercase[randInt(len(uppercase))]
	password[2] = numbers[randInt(len(numbers))]
	password[3] = special[randInt(len(special))]

	// Fill the rest of the password with random characters
	for i := 4; i < length; i++ {
		randomIndex, err := rand.Int(rand.Reader, big.NewInt(int64(len(allChars))))
		if err != nil {
			return "", err
		}
		password[i] = allChars[randomIndex.Int64()]
	}

	// Shuffle the password to ensure randomness
	shuffle(password)

	return string(password), nil

}

// randInt generates a random integer in [0, n)
func randInt(n int) int {
	b := make([]byte, 1) // 1 byte for random value
	_, err := rand.Read(b)
	if err != nil {
		panic(err)
	}
	return int(b[0]) % n
}

// Shuffle function to randomize the password characters
func shuffle(password []byte) {
	for i := len(password) - 1; i > 0; i-- {
		j := randInt(i + 1)
		password[i], password[j] = password[j], password[i]
	}
}
