package services

import (
	"bytes"
	"context"
	"edge-api/internal/database"
	"edge-api/internal/models"
	"encoding/json"
	"fmt"
	"io"
	"log"
	"net/http"
	"net/url"
	"os"
	"strings"
	"time"

	"github.com/google/uuid"
)

func getControllerApi() string {
	controllerApi := os.Getenv("EDGE_CONTROLLER_API")
	if controllerApi != "" {
		return controllerApi
	}

	/* Check if secret is in files in data-directory */
	secretFilePath := "/lib/uniberg/edge-api/data/edge_controller_api"
	if _, err := os.Stat(secretFilePath); err == nil {
		data, err := os.ReadFile(secretFilePath)
		if err != nil {
			log.Fatal("Failed to get edge controller api from file")
		}

		return strings.Replace(string(data), "\n", "", -1)
	}

	return ""
}

func generateNetworkConfig(networks []models.Network, devices []models.Device) models.Config {
	config := models.Config{
		UUID:             time.Now().Unix(), // Current Linux epoch
		Networks:         []models.ConfigNetwork{},
		WirelessNetworks: []models.WirelessNetwork{},
	}

	for _, network := range networks {
		configNetwork := models.ConfigNetwork{
			UUID:   network.ID.String(),
			Name:   network.Name,
			VLANID: network.VLAN,
		}
		config.Networks = append(config.Networks, configNetwork)

		// Set up the wireless network entry
		wirelessNet := models.WirelessNetwork{
			SSID:        network.Name,
			NetworkUUID: network.ID.String(),
			Encryption:  models.EncryptionConfig{},
		}

		if network.Type == models.NetworkTypeGuest {
			// Use "psk2" encryption for guest networks and set passphrase if available
			wirelessNet.Encryption.Method = "psk2"
			if network.Password.Valid {
				wirelessNet.Encryption.Passphrase = network.Password.String
			}
		} else if network.Type == models.NetworkTypeBusiness {
			// Use "ppsk2" encryption for business networks and add device passkeys
			wirelessNet.Encryption.Method = "ppsk2"
			wirelessNet.Encryption.Passkeys = []models.PasskeyConfig{}
			for _, device := range devices {
				if device.Network == network.ID && device.Password.Valid {
					passkey := models.PasskeyConfig{
						Passkey:     device.Password.String,
						NetworkUUID: network.ID.String(),
					}
					if device.MAC.Valid {
						passkey.MACAddress = device.MAC.String
					}
					wirelessNet.Encryption.Passkeys = append(wirelessNet.Encryption.Passkeys, passkey)
				}
			}
		}
		config.WirelessNetworks = append(config.WirelessNetworks, wirelessNet)
	}
	return config
}

func uploadNetworkConfig(api *url.URL, config models.Config) error {
	ctx, cancel := context.WithTimeout(context.Background(), 4*time.Second)
	defer cancel()

	body, err := config.JSON()
	if err != nil {
		fmt.Errorf("Error generating edge controller config: %v", err)
	}

	req, err := http.NewRequestWithContext(ctx, http.MethodPost, api.JoinPath("config").String(), bytes.NewBuffer(body))
	if err != nil {
		fmt.Errorf("Error creating an edge controller /config request: %v", err)
	}
	req.Header.Set("Content-Type", "application/json")

	resp, err := http.DefaultClient.Do(req)
	if err != nil {
		if ctx.Err() == context.DeadlineExceeded {
			return fmt.Errorf("Edge controller /config request timed out")
		}
		fmt.Errorf("Error with edge controller /config request: %v", err)
		return err
	}
	defer resp.Body.Close()

	if resp.StatusCode != http.StatusOK {
		return fmt.Errorf("Edge controller /config request failed with status %v", resp.StatusCode)
	}

	_, err = io.ReadAll(resp.Body)
	if err != nil {
		return fmt.Errorf("Error reading edge controller /config response body: %v", err)
	}

	return nil
}

type PSKRequestBody struct {
	Id         string `json:"id"`
	MacAddress string `json:"mac_address"`
	Password   string `json:"password"`
}

func UpdatePSKForNetwork(network_id uuid.UUID) error {
	// Get all passwords for Network
	var devices []models.Device
	devices, err := models.DeviceGetByNetwork(network_id)
	if err != nil {
		log.Fatalf("Error getting devices for network %v: %v", network_id, err)
	}

	// Prepare RequestBody
	var requestBodyDevices []PSKRequestBody
	for _, device := range devices {
		requestBodyDevices = append(requestBodyDevices, PSKRequestBody{
			Id:         device.ID.String(),
			MacAddress: device.MAC.String,
			Password:   device.Password.String,
		})
	}

	// Send Request
	ctx, cancel := context.WithTimeout(context.Background(), 4*time.Second)
	defer cancel()

	edge_controller_api := getControllerApi()
	if edge_controller_api == "" {
		log.Fatalf("EDGE_CONTROLLER_API not set")
	}
	edge_controller_api_url, err := url.Parse(edge_controller_api)
	if err != nil {
		log.Fatalf("Failed to parse EDGE_CONTROLLER_API: %v", err)
	}

	b, err := json.Marshal(requestBodyDevices)
	if err != nil {
		fmt.Errorf("Error marshalling edge controller /password request body: %v", err)
	}

	req, err := http.NewRequestWithContext(ctx, http.MethodPost, edge_controller_api_url.JoinPath("network").JoinPath(network_id.String()).JoinPath("password").String(), bytes.NewBuffer(b))
	if err != nil {
		fmt.Errorf("Error creating an edge controller /password creation request: %v", err)
	}
	req.Header.Set("Content-Type", "application/json")

	resp, err := http.DefaultClient.Do(req)
	if err != nil {
		if ctx.Err() == context.DeadlineExceeded {
			return fmt.Errorf("Edge controller /password request timed out")
		}
		fmt.Errorf("Error with edge controller /password request: %v", err)
		return err
	}
	defer resp.Body.Close()

	if resp.StatusCode != http.StatusOK {
		return fmt.Errorf("Edge controller /password request failed with status %v", resp.StatusCode)
	}

	return nil
}

func UpdateConfig() error {
	edge_controller_api := getControllerApi()
	if edge_controller_api == "" {
		log.Fatalf("EDGE_CONTROLLER_API not set")
	}
	edge_controller_api_url, err := url.Parse(edge_controller_api)
	if err != nil {
		log.Fatalf("Failed to parse EDGE_CONTROLLER_API: %v", err)
	}

	var networks []models.Network
	var devices []models.Device

	if err := database.DB.Select(&networks, "SELECT * FROM networks"); err != nil {
		return err
	}

	if err := database.DB.Select(&devices, "SELECT * FROM devices"); err != nil {
		return err
	}

	config := generateNetworkConfig(networks, devices)

	if err := uploadNetworkConfig(edge_controller_api_url, config); err != nil {
		return err
	}

	return nil
}

func UpdateAllPSK() error {
	var networks []models.Network
	if err := database.DB.Select(&networks, "SELECT * FROM networks"); err != nil {
		return err
	}

	for _, network := range networks {
		if network.Type == models.NetworkTypeBusiness {
			if err := UpdatePSKForNetwork(network.ID); err != nil {
				return err
			}
		}
	}

	return nil
}

func ConfigSyncWithController() error {
	if err := UpdateConfig(); err != nil {
		return err
	}

	if err := UpdateAllPSK(); err != nil {
		return err
	}

	return nil
}

func StartInitialConfigSync(ctx context.Context) {
	// Check if a network is created
	var network models.Network
	err := database.DB.Get(&network, "SELECT * FROM networks LIMIT 1")
	if err != nil {
		log.Printf("No networks found, skipping initial config sync")
		return
	}

	ticker := time.NewTicker(5 * time.Second)
	defer ticker.Stop()

	for {
		select {
		case <-ctx.Done():
			return
		case <-ticker.C:
			err := ConfigSyncWithController()
			if err != nil {
				log.Printf("Error syncing config with controller: %v", err)
				time.Sleep(5 * time.Second)
			} else {
				log.Printf("Config synced with controller")
				return
			}
		}
	}
}
