package services

import (
	"context"
	"edge-api/internal/models"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log" // Import the log package
	"net/http"
	"net/url"
	"time"
)

type ControllerAPI struct {
	URL url.URL
}

func (api *ControllerAPI) getNodes() (*models.NodeResponseMap, error) {
	ctx, cancel := context.WithTimeout(context.Background(), 4*time.Second)
	defer cancel()

	nodesURL := api.URL.JoinPath("nodes").String() // Get the full URL string

	req, err := http.NewRequestWithContext(ctx, http.MethodGet, nodesURL, nil)
	if err != nil {
		log.Printf("Error creating edge controller /nodes request: %v", err) // Log error
		return nil, fmt.Errorf("Error creating edge controller /nodes request: %v", err)
	}

	resp, err := http.DefaultClient.Do(req)
	if err != nil {
		if ctx.Err() == context.DeadlineExceeded {
			log.Println("Edge controller /nodes request timed out") // Log timeout
			return nil, fmt.Errorf("Edge controller /nodes request timed out")
		}
		log.Printf("Edge controller /nodes request failed: %v", err) // Log request failure
		return nil, fmt.Errorf("Edge controller /nodes request failed %v", err)
	}
	defer resp.Body.Close()

	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("Edge controller /nodes request failed with status %v", resp.StatusCode)
	}

	body, err := io.ReadAll(resp.Body)
	if err != nil {
		log.Printf("Error reading edge controller /nodes response body: %v", err) // Log body read error
		return nil, fmt.Errorf("Error reading edge controller /nodes response body: %v", err)
	}

	// Parse the JSON response
	var nodes models.NodeResponseList
	if err := json.Unmarshal(body, &nodes); err != nil {
		log.Printf("Error parsing edge controller /nodes response body: %v", err) // Log JSON parsing error
		return nil, fmt.Errorf("Error parsing edge controller /nodes response body: %v", err)
	}

	result := make(models.NodeResponseMap)
	for _, node := range nodes {
		result[node.Name] = node
	}

	return &result, nil
}

func (api *ControllerAPI) getNodeMetrics(node *models.NodeResponse) (*models.NodeMetricsResponse, error) {
	ctx, cancel := context.WithTimeout(context.Background(), 4*time.Second)
	defer cancel()

	metricsURL := api.URL.JoinPath("nodes", node.MAC, "metrics").String() // Get the full metrics URL

	req, err := http.NewRequestWithContext(ctx, http.MethodGet, metricsURL, nil)
	if err != nil {
		log.Printf("Error creating edge controller /nodes/%s/metrics request: %v", node.MAC, err) // Log metrics request error
		return nil, fmt.Errorf("Error creating edge controller /nodes/%s/metrics request: %v", node.MAC, err)
	}

	resp, err := http.DefaultClient.Do(req)
	if err != nil {
		if errors.Is(ctx.Err(), context.DeadlineExceeded) {
			log.Printf("Edge controller /nodes/%s/metrics request timed out", node.MAC) // Log metrics timeout
			return nil, &models.NodeTimeoutError{Node: *node}
		}
		log.Printf("Edge controller /nodes/%s/metrics request failed: %v", node.MAC, err) // Log metrics request failure
		return nil, fmt.Errorf("Edge controller /nodes/%s/metrics request failed %v", node.MAC, err)
	}
	defer resp.Body.Close()

	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("Edge controller /nodes/%s/metrics request failed with status %v", node.MAC, resp.StatusCode)
	}

	body, err := io.ReadAll(resp.Body)
	if err != nil {
		log.Printf("Error reading edge controller /nodes/%s/metrics response body: %v", node.MAC, err) // Log metrics body read error
		return nil, fmt.Errorf("Error reading edge controller /nodes/%s/metrics response body: %v", node.MAC, err)
	}

	var result models.NodeMetricsResponse
	if err := json.Unmarshal(body, &result); err != nil {
		log.Printf("Error parsing edge controller /nodes/%s/metrics response body: %v", node.MAC, err) // Log metrics JSON parsing error
		return nil, fmt.Errorf("Error parsing edge controller /nodes/%s/metrics response body: %v", node.MAC, err)
	}

	return &result, nil
}
