"use client"

// React Dependencies
import React, { useState, ReactNode } from 'react';
// Custom Hooks
import AuthCheck from "../components/AuthCheck";
import { useAuth } from '@/hooks/useAuth';
import { useDomReady } from '@/hooks/useDomReady';
import { useToast } from '../components/ToastProvider';
import { useLoader } from '@/context/LoaderContext';
// Services
import { getAccessPoints } from '@/services/accessPoints';
// External Libraries Dependencies
import {
  formatDate,
  Comparators,
  EuiBasicTable,
  EuiBasicTableColumn,
  EuiTableSortingType,
  Criteria,
  EuiButtonIcon,
  EuiHealth,
  EuiFlexGroup,
  EuiTitle,
  EuiSwitch,
  EuiFlexItem,
  EuiDescriptionList,
  EuiScreenReaderOnly,
} from '@elastic/eui';
// Components
import Navigation from "../components/navigation/Navigation";
import { CheckQuality } from '../components/QualityCheck';
// Own Dependencies
import { ApiError, getErrorCode } from '@/lib/errorsCodes';
import getDict from '@/lib/dict';
import AP from '../types/AP';
// Own Styles
import styles from "./page.module.css";
import { checkResponseShape } from '@/lib/checkResponseShape';

const dict = getDict("de");

export default function Antennas() {
  const [auto, setAuto] = React.useState(true);
  const domReady = useDomReady();
  const { isAuthenticated } = useAuth();
  const { setLoading } = useLoader();
  const { addToast } = useToast();
  const [aps, setAPs] = React.useState<AP[]>([]);

  async function fetchAPs() {
    try {
      setLoading(true);
      const response = await getAccessPoints();
      if (!response.ok) {
        const errorCode = getErrorCode('ACCESS_POINT_FETCH', response.status);
        addToast({
          title: errorCode,
          color: "danger",
          text: dict.apFetchToastFail,
        });
        setLoading(false);
        return;
      }

      const data = await response.json();
      if (checkResponseShape(data)) {
        setAPs(data);
      } else {
        addToast({
          title: 'Fetch Error',
          color: "danger",
          text: dict.apFetchToastFail,
        });
      }
      setLoading(false);
    } catch (err) {
      const error = err as ApiError;
      const errorCode = getErrorCode('ACCESS_POINT_EXCEPTION', error.status);
      setLoading(false);
      addToast({
        title: errorCode,
        color: "danger",
        text: dict.apFetchToastFail,
      });
    }
  }
  React.useEffect(() => {
    if (!domReady || !isAuthenticated) return;
    let interval: NodeJS.Timeout | null = null;
    fetchAPs();
    if (auto) {
      interval = setInterval(fetchAPs, 5000);
    }
    return () => {
      if (interval) {
        clearInterval(interval);
      }
    }
  }, [auto, domReady])
  const columns: Array<EuiBasicTableColumn<AP>> = [
    {
      field: 'name',
      name: dict.name,
      sortable: true,
    },
    {
      field: 'online',
      name: 'Status',
      sortable: true,
      render: (online: AP['online'], ap: AP) => <EuiHealth color={online ? 'green' : 'danger'}>{online ? dict.online : dict.offline } <CheckQuality value={ap.signal || 0} /></EuiHealth>
    },
    {
      field: 'firstConnectedAt',
      name: dict.firstConnectedAt,
      dataType: 'date',
      render: (firstConnectedAt: AP['firstConnectedAt']) =>
        formatDate(firstConnectedAt, 'dobLong'),
    },
    {
      field: 'lastConnectedAt',
      name: dict.lastConnectedAt,
      dataType: 'date',
      render: (lastConnectedAt: AP['lastConnectedAt']) =>
        formatDate(lastConnectedAt, 'dobLong'),
    },
    {
      field: 'connectedDevices',
      name: dict.connectedDevices,
      align: 'center',
      render: (connectedDevices: AP['connectedDevices']) =>
        connectedDevices ? connectedDevices : '0'
    },
  ];
  const [itemIdToExpandedRowMap, setItemIdToExpandedRowMap] = useState<
      Record<string, ReactNode>
      >({});

  const toggleDetails = (accessPoint: AP) => {
    const itemIdToExpandedRowMapValues = { ...itemIdToExpandedRowMap };

    if (itemIdToExpandedRowMapValues[accessPoint.id]) {
      delete itemIdToExpandedRowMapValues[accessPoint.id];
    } else {
      const { name, signal, connectedDevices, mac_address, online } = accessPoint;

      const listItems = [
        {
          title: dict.name,
          description: name,
        },
        {
          title: dict.signal,
          description: <EuiHealth color={online ? 'green' : 'danger'}>{online ? dict.online : dict.offline } <CheckQuality value={signal || 0} /></EuiHealth> ,
        },
        {
          title: dict.macAddress,
          description: `${mac_address || '-'}`,
        },
        {
          title: dict.connectedDevices,
          description: `${connectedDevices || '-'}`,
        },
      ];
      itemIdToExpandedRowMapValues[accessPoint.id] = (
        <div className={styles.page}>

          <EuiDescriptionList listItems={listItems} type="column"
            columnWidths={[1, 3]}
            style={{ maxInlineSize: '400px' }} /></div>
      );
    }
    setItemIdToExpandedRowMap(itemIdToExpandedRowMapValues);
  };

  const columnsWithExpandingRowToggle: Array<EuiBasicTableColumn<AP>> = [
    ...columns,
    {
      align: 'right',
      width: '40px',
      isExpander: true,
      name: (
        <EuiScreenReaderOnly>
          <span>Expand row</span>
        </EuiScreenReaderOnly>
      ),
      mobileOptions: { header: false },
      render: (accessPoint: AP) => {
        const itemIdToExpandedRowMapValues = { ...itemIdToExpandedRowMap };

        return (
          <EuiButtonIcon
            onClick={() => toggleDetails(accessPoint)}
            aria-label={
              itemIdToExpandedRowMapValues[accessPoint.id] ? 'Collapse' : 'Expand'
            }
            iconType={
              itemIdToExpandedRowMapValues[accessPoint.id] ? 'arrowDown' : 'arrowRight'
            }
          />
        );
      },
    },
  ];

  const [pageIndex, setPageIndex] = useState(0);
  const [pageSize, setPageSize] = useState(5);
  const [sortField, setSortField] = useState<keyof AP>('name');
  const [sortDirection, setSortDirection] = useState<'asc' | 'desc'>('asc');

  const onTableChange = ({ page, sort }: Criteria<AP>) => {
    if (page) {
      const { index: pageIndex, size: pageSize } = page;
      setPageIndex(pageIndex);
      setPageSize(pageSize);
    }
    if (sort) {
      const { field: sortField, direction: sortDirection } = sort;
      setSortField(sortField);
      setSortDirection(sortDirection);
    }
  };

  const findAps = (
    aps: AP[],
    pageIndex: number,
    pageSize: number,
    sortField: keyof AP,
    sortDirection: 'asc' | 'desc'
  ) => {
    let items;

    if (sortField) {
      items = aps
        .slice(0)
        .sort(
          Comparators.property(sortField, Comparators.default(sortDirection))
        );
    } else {
      items = aps;
    }

    let pageOfItems;

    if (!pageIndex && !pageSize) {
      pageOfItems = items;
    } else {
      const startIndex = pageIndex * pageSize;
      pageOfItems = items.slice(
        startIndex,
        Math.min(startIndex + pageSize, aps.length)
      );
    }

    return {
      pageOfItems,
    };
  };

  const { pageOfItems } = findAps(
    aps,
    pageIndex,
    pageSize,
    sortField,
    sortDirection
  );

  const sorting: EuiTableSortingType<AP> = {
    sort: {
      field: sortField,
      direction: sortDirection,
    },
  };
  return (
    <AuthCheck>
      <Navigation>
        <div className={styles.page}>
          <EuiFlexGroup justifyContent="spaceBetween">
            <EuiFlexItem grow={false}>
              <EuiTitle>
                <h2>{dict.accessPoint}</h2>
              </EuiTitle>
            </EuiFlexItem>
          </EuiFlexGroup>
          <EuiSwitch
            label={dict.refreshData}
            checked={auto}
            onChange={() => {
              setAuto(!auto);
              addToast({
                title: auto ? dict.refreshDataDisabled: dict.refreshDataEnabled,
                color: "primary",
              });
            }}
          />
          <EuiBasicTable
            tableCaption="APS LIST"
            items={pageOfItems}
            itemId="id"
            itemIdToExpandedRowMap={itemIdToExpandedRowMap}
            columns={columnsWithExpandingRowToggle}
            sorting={sorting}
            onChange={onTableChange}
          />
        </div>
      </Navigation>
    </AuthCheck>
  );
}
