"use client"
// React Dependencies
import React from "react";
// External Libraries Dependencies
import {
  EuiButton,
  EuiButtonEmpty,
  EuiFieldText,
  EuiForm,
  EuiFormRow, EuiLoadingSpinner, EuiModal,
  EuiModalBody, EuiModalFooter,
  EuiModalHeader,
  EuiModalHeaderTitle,
  EuiSelect, EuiSelectOption
} from "@elastic/eui";
// Own Dependencies
import getDict from "@/lib/dict";
import { DeviceTypes } from "@/app/types/Device";
import Network from "@/app/types/Network";
import { useToast } from "@/app/components/ToastProvider";
import { ApiError, getErrorCode } from "@/lib/errorsCodes";
import { apiFetch } from "@/lib/api";
const dict = getDict("de");

export const AddModal = (props: { closeModal: () => void; networkId?: string}) => {
  const [name, setName] = React.useState('');
  const [type, setType] = React.useState('');
  const [networkType, setNetworkType] = React.useState('');
  const [networks, setNetworks] = React.useState<EuiSelectOption[]>([]);
  const [loading, setLoading] = React.useState(false);

  const [errors, setErrors] = React.useState<{ select?: string[]; text?: string[]; networkType?: string[]; }>({});

  const [isModalVisible, setIsModalVisible] = React.useState(false);

  const { addToast } = useToast();
  React.useEffect(() => {
    const fetchNetworks = async () => {
      try {
        const response = await apiFetch("/networks");

        if (!response.ok) {
          const errorCode = getErrorCode('NETWORK_FETCH', response.status);
          addToast({
            title: errorCode,
            color: "danger",
            text: `${dict.networkFetchToastFail}`,
          });
        }
        const data = await response.json();
        const options = data.filter((element: Network) => element.type !== 'guest').map((item: Network) => ({
          value: item.id,
          text: item.name,
        }));
        setNetworks(options);
      } catch (err) {
        const error = err as ApiError;
        const errorCode = getErrorCode('NETWORK_EXCEPTION', error.status);
        addToast({
          title: errorCode,
          color: "danger",
          text: `${dict.networkFetchToastFail}`,
        });
      }
    };
    fetchNetworks();
  }, [isModalVisible]);
  const validate = () => {
    const newErrors: { select?: string[]; text?: string[]; networkType?: string[] } = {};

    if (!type) {
      newErrors.select = [dict.required];
    }
    if (name.length === 0) {
      newErrors.text = [dict.required];
    }
    if (!props.networkId) {
      if (networkType.length === 0) {
        newErrors.networkType = [dict.required];
      }
    }

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  React.useEffect(() => {
    validate();
  }, [name, type, networkType]);

  const [isFormValid, setIsFormValid] = React.useState(false);

  React.useEffect(() => {
    setIsFormValid(Object.keys(errors).length === 0);
  }, [errors]);

  const onNameChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setName(e.target.value);
  };
  const onTypeChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
    setType(e.target.value);
  };
  const onNetworkTypeChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
    setNetworkType(e.target.value);
  };
  const deviceTypeOptions: EuiSelectOption[] = DeviceTypes.map(type => ({
    value: type,
    text: dict[type] || type,
  }));
  const closeModal = () => {
    setLoading(false);
    setName('');
    setType('');
    setNetworkType('');
    setIsModalVisible(false);
    setErrors({});
    props.closeModal();
  };

  async function handleSubmit() {
    setLoading(true);
    if (validate()) {
      const response = await apiFetch(`/networks/${props.networkId ? props.networkId : networkType}/devices`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ name, type }),
      });
      if (response.ok) {
        setLoading(!loading);
        addToast({
          title: "Success",
          color: "success",
          text: `${dict.deviceAddToast}`,
        });
        closeModal();
      } else {
        const errorCode = getErrorCode('DEVICE_ADD_EXCEPTION', response.status);
        setLoading(!loading);
        addToast({
          title: errorCode,
          color: "danger",
          text: `${dict.deviceAddError}`,
        });
        closeModal();
      }
      setErrors({});
    }
  }

  return (
    <EuiModal
      aria-labelledby={'modalTitleId'}
      onClose={closeModal}
      initialFocus="[name=popswitch]"
    >
      <EuiModalHeader>
        <EuiModalHeaderTitle id={'modalTitleId'}>
          {dict.addDevice}
        </EuiModalHeaderTitle>
      </EuiModalHeader>

      <EuiModalBody>
        <EuiForm component="form">

          <EuiFormRow label={dict.selectDeviceType} helpText={dict.type} error={errors.select} isInvalid={!!errors.select}>
            <EuiSelect
              options={deviceTypeOptions}
              value={type}
              isInvalid={!!errors.select}
              onChange={onTypeChange}
              aria-label={dict.type}
              hasNoInitialSelection={true}
            />
          </EuiFormRow>
          {!props.networkId ? <EuiFormRow label={dict.selectNetworkType} helpText={dict.type} error={errors.networkType} isInvalid={!!errors.networkType}>
            <EuiSelect
              options={networks}
              value={networkType}
              isInvalid={!!errors.networkType}
              onChange={onNetworkTypeChange}
              aria-label={dict.type}
              hasNoInitialSelection={true}
            />
          </EuiFormRow> : undefined}
          <EuiFormRow label={dict.name} helpText={dict.name} error={errors.text} isInvalid={!!errors.text}>
            <EuiFieldText
              name={"name"}
              placeholder={dict.name}
              value={name}
              isInvalid={!!errors.text}
              onChange={(e: React.ChangeEvent<HTMLInputElement>) => onNameChange(e)}
              aria-label={dict.name}
            />
          </EuiFormRow>
        </EuiForm>
      </EuiModalBody>

      <EuiModalFooter>
        <EuiButtonEmpty onClick={closeModal}>Cancel</EuiButtonEmpty>

        <EuiButton
          type="submit"
          color={"success"}
          onClick={handleSubmit}
          form={'modalFormId'}
          isDisabled={!isFormValid || loading}
          fill
        >
          {dict.add} {loading ? <EuiLoadingSpinner size="l" /> : undefined}
        </EuiButton>
      </EuiModalFooter>
    </EuiModal>)
}
