import React, { useState, useReducer, useRef, useCallback, useEffect } from 'react';
import {
  EuiPage,
  EuiPageBody,
  EuiPageHeader,
  EuiTitle,
  EuiFlexGroup,
  EuiFlexItem,
  EuiPanel,
  EuiButton,
  EuiFilePicker,
  EuiText,
  EuiIcon,
  EuiSpacer,
  EuiButtonIcon,
  EuiHealth,
  EuiCallOut,
  EuiLoadingSpinner,
  EuiEmptyPrompt,
  EuiToolTip,
  EuiCard,
  EuiAccordion,
  EuiListGroup,
  EuiListGroupItem,
  EuiBadge,
  EuiOverlayMask,
  EuiModal,
  EuiModalHeader,
  EuiModalHeaderTitle,
  EuiModalBody,
  EuiModalFooter,
  EuiFieldText,
  EuiContextMenu,
  EuiPopover,
  EuiFormRow,
  EuiNotificationBadge,
  EuiSwitch,
  EuiCode,
  EuiDescriptionList,
  EuiDescriptionListTitle,
  EuiDescriptionListDescription,
  EuiProgress
} from '@elastic/eui';
import { AnimatePresence, motion } from 'framer-motion';
import { useLoader } from '@/context/LoaderContext';
import MultiStepDemo from '@/setupApp/components/ui/StepProgress';
import Test from '@/setupApp/Test';
import { useScreenManager } from '@/setupApp/hooks/useScreenManager';
import { SCREEN_TYPES } from '@/setupApp/utils/constants';
import { useProjectManager } from '@/hooks/useProjectManager';
import { useToast } from './ToastProvider';
import getDict from '@/lib/dict';
import { FileUploadComponent } from '@/setupApp/components/forms/FileUploadComponent';
import { processPdfFromFile, uploadPdfToBackend, generatePDF as generatePdfFile, cropImageToViewport, transformAccessPointsAfterCrop } from '@/lib/pdfUtils';
import { ProjectSetupScreen } from '@/setupApp/components/screens/ProjectSetupScreen';
import { EuiButton as Btn, EuiTitle as Title } from '@/setupApp/components/ui';
import { animations } from '@/setupApp/utils/animations';
import { useAccessPoints } from '@/hooks/useAccessPoints';
import { FileUploader } from './FileUploader';
import { Error } from '@/setupApp/components/ui/Error';
import { NodeDetails } from '../NodeDetails';
import { usePDFLoader } from '@/hooks/usePDFLoader';
import { CurrentPageAccessPoints } from './CurrentPageAccessPoints';
import { migrateToRelativeCoordinates, relativeToPixel, pixelToRelative, MIN_ZOOM, MAX_ZOOM, ZOOM_STEP, screenToMapCoordinates, getCurrentImageDimensions, isTypeAtLimit, getTypeCount, getTouchDistance, generateUniqueId } from '@/lib/nodeScaling';
import AccessPointMarkerPoint from '../types/AccessPointMarker';
import { useResponsiveLayout } from '@/hooks/useResponsiveLayout';
import { NodeMovementToggle, ZoomComponent } from '@/setupApp/components/ui';
import { RotationComponent } from '@/setupApp/components/ui/Rotation';
import { useProjectLoader } from '@/hooks/useProjectLoader';
import { DropZoneIndicator } from '@/setupApp/components/ui/DropZoneIndicator';
import { AccessPointLayer } from '@/setupApp/components/ui/AccessPointLayer';
import { AccessPointTypes } from './AccessPointTypes';
import { viewportReducer, initialState } from '@/store/reducer';
import ZoomPanRotationEditor from './ZoomPanRotationEditor';

const dict = getDict("de");
export default function PDFMapEditor({ aps }) {
  const [state, dispatch] = useReducer(viewportReducer, initialState);
  // Window check state
  const { generateDeviceDetails } = useAccessPoints(aps);
  const [pdfFile, setPdfFile] = useState({
    uploadedPath: '',
    name: '',
    size: 0,
  });
  const [pdfPages, setPdfPages] = useState([]);
  const { pdfJsLoaded, jsPdfLoaded, loadingError, setLoadingError, isClient } = usePDFLoader();

  const [currentPage, setCurrentPage] = useState(0);
  const [accessPoints, setAccessPoints] = useState<AccessPointMarkerPoint[]>([]);
  const [draggedItem, setDraggedItem] = useState<{isNew: boolean, type: string, accessPoint?: object}| null>(null);
  const [isGenerating, setIsGenerating] = useState(false);
  const [projectUpdated, setProjectUpdated] = useState(false);
  const [showLegendModal, setShowLegendModal] = useState(false);
  const [editingNodeId, setEditingNodeId] = useState(null);
  const [editingNodeName, setEditingNodeName] = useState('');
  const [projectName, setProjectName] = useState('');
  const [showSaveModal, setShowSaveModal] = useState(false);
  const [showLoadModal, setShowLoadModal] = useState(false);
  const [apiEndpoint, setApiEndpoint] = useState('');
  const [apiLoading, setApiLoading] = useState(false);
  const [loading, setLoading] = useState(false);
  const [newProject, setNewProject] = useState(false);
  const [selectedNodeDetails, setSelectedNodeDetails] = useState(null);
  const [showNodeDetailsModal, setShowNodeDetailsModal] = useState(false);
  const [status, setStatus] = useState(false);
  // Image dimensions tracking
  const { loadProjectsFromApi , savedProjects, saveProject: saveProjectToApi, saved } = useProjectManager();
  // Zoom state
  const { addToast } = useToast();

  const [projectFound, setProjectFound] = useState(false);
  // Touch/mobile drag state
  const [touchDragItem, setTouchDragItem] = useState<{isNew: boolean, type: string, accessPoint?: object}| null>(null);
  const [isDraggingTouch, setIsDraggingTouch] = useState(false);
  const [touchDragPreview, setTouchDragPreview] = useState({ x: 0, y: 0, visible: false });

  // Node movement control
  const [allowNodeMovement, setAllowNodeMovement] = useState(true);

  // Limit notification state
  const [limitMessage, setLimitMessage] = useState('');

  // Access point types with enhanced properties
  const { globalAps, setCurrentScreen } = useLoader();
  const accessPointTypes = globalAps;

  const mapContainerRef = useRef<HTMLImageElement | null>(null);
  const mainRef = useRef<HTMLImageElement | null>(null);
  const containerRef = useRef<HTMLImageElement | null>(null);
  const mapImageRef = useRef<HTMLImageElement | null>(null);

  const { isMobile } = useResponsiveLayout();

  // Update image dimensions when image loads
  const handleImageLoad = () => {
    if (mapImageRef.current) {
      const { offsetWidth, offsetHeight } = mapImageRef.current;
      dispatch({
        type: 'SET_IMAGE_DIMENSIONS',
        payload: { width: offsetWidth, height: offsetHeight }
      });
      // Migrate existing access points to relative coordinates
      setAccessPoints(prevPoints =>
          prevPoints.map(ap => migrateToRelativeCoordinates(ap, state.imageDimensions.width, state.imageDimensions.height))
      );
    }
  };

  // Handle window resize to update image dimensions
  useEffect(() => {
    const handleResize = () => {
      if (mapImageRef.current) {
        const { width, height } = mapImageRef.current;

        dispatch({
          type: 'SET_IMAGE_DIMENSIONS',
          payload: { width, height }
        });
      }
    };

    window.addEventListener('resize', handleResize);
    return () => window.removeEventListener('resize', handleResize);
  }, []);




  const { loadFirstProjectFromBackend } = useProjectLoader();

  // Load saved projects and API config on component mount
  useEffect(() => {
    const loadSavedProjects = async () => {
      // Only run on client side
      if (!isClient || !pdfJsLoaded) return;

      try {
        await loadFirstProjectFromBackend({
          setStatus,
          loadProjectsFromApi,
          setAccessPoints,
          setCurrentPage,
          setPageViewports,
          setPdfPages,
          setPdfFile,
          setProjectFound,
          setLoadingError,
          addToast,
          generateDeviceDetails,
          processPdfFromFile
        });
      } catch (error) {
        addToast({ title: 'Error', color: 'danger', text: `Failed to load: ${error}`})
      }
    };

    loadSavedProjects();
  }, [isClient, pdfJsLoaded]);

  // Monitor PDF loading state to control node movement
  useEffect(() => {
    if (pdfPages.length > 0) {
      // PDF is loaded, disable node movement
      if (projectFound) {
        setAllowNodeMovement(false);
      }
    } else {
      // No PDF loaded, allow node movement
      setAllowNodeMovement(true);
    }
  }, [pdfPages]);




  // Zoom functions
  const handleZoomIn = () => {
    dispatch({ type: 'ZOOM_IN'});
  };

  const handleZoomOut = () => {
    dispatch({ type: 'ZOOM_OUT'});
  };

  const handleZoomReset = () => {
    dispatch({ type: 'SET_ZOOM', payload: 1 });
    dispatch({ type: 'SET_PAN', payload: { x: 0, y: 0 } });
  };

  // Rotation functions
  const handleRotateLeft = () => {
    dispatch({ type: 'ROTATE_LEFT' });
  };

  const handleRotateRight = () => {
    dispatch({ type: 'ROTATE_RIGHT' });
  };
  const [originalImageSrc, setOriginalImageSrc] = React.useState(null);

  const [dragOffset, setDragOffset] = useState({ x: 0, y: 0 });

  // Mouse wheel zoom
  const handleWheel = useCallback((e) => {
    if (e.ctrlKey || e.metaKey) {
      e.preventDefault();
      const delta = e.deltaY > 0 ? -ZOOM_STEP : ZOOM_STEP;
      dispatch({ type: 'SET_ZOOM', payload: Math.max(MIN_ZOOM, Math.min(MAX_ZOOM, state.viewport.zoomLevel + delta)) });
    }
  }, []);

  // Pan functions
  const handleMouseDown = useCallback((e) => {
    // Left click with Ctrl or middle mouse button
    if (e.button === 0 || ( e.button === 1)) {
      dispatch({
        type: 'START_PAN',
        payload: { x: e.clientX, y: e.clientY }
      });
      e.preventDefault();
    }
  }, []);

  const handleMouseMove = useCallback((e: MouseEvent) => {
    if (state.viewport.isPanning) {
      const deltaX = e.clientX - state.viewport.lastPanPosition.x;
      const deltaY = e.clientY - state.viewport.lastPanPosition.y;

      dispatch({ type: 'SET_PAN', payload: { x: state.viewport.panOffset.x + deltaX, y: state.viewport.panOffset.y + deltaY } });

      dispatch({
        type: 'UPDATE_PAN',
        payload: { x: e.clientX, y: e.clientY }
      });
    }
  }, [state.viewport.isPanning, state.viewport.lastPanPosition]);

  const handleMouseUp = useCallback(() => {
    dispatch({ type: 'END_PAN' });
  }, []);

  const {completeCurrentStep, setCurrentStep, currentStep } = useLoader();
  const [pageViewports, setPageViewports] = useState({});
  const { navigateTo, isCurrentScreen } = useScreenManager();
  const { currentScreen } = useLoader();
  const [ goToStep, setgoToStep ]= useState(SCREEN_TYPES.NETWORK_SETUP);
  const [isLoading, setIsLoading] = React.useState(true);

  // Add mouse event listeners
  useEffect(() => {
    document.addEventListener('mousemove', handleMouseMove);
    document.addEventListener('mouseup', handleMouseUp);
    return () => {
      document.removeEventListener('mousemove', handleMouseMove);
      document.removeEventListener('mouseup', handleMouseUp);
    };
  }, [handleMouseMove, handleMouseUp]);


  // Handle PDF file upload
  const handleFileUpload = async (files) => {
    const file = files[0];
    if (!file || file.type !== 'application/pdf') {
      setLoadingError('Please upload a valid PDF file');
      return;
    }

    // @ts-ignore
    if (!pdfJsLoaded || !window.pdfjsLib) {
      setLoadingError('PDF.js is still loading. Please wait a moment and try again.');
      return;
    }

    setPdfFile(file);
    setLoadingError(null);
    setLimitMessage('');
    setApiLoading(true);
    setLoading(true);

    try {
      setLoading(true);
      const pages = await processPdfFromFile(file);
      setPdfPages(pages);
      setCurrentPage(0);
      setAccessPoints([]);

      // Reset zoom and pan when new PDF is loaded
      dispatch({ type: 'SET_ZOOM', payload: 1 });
      dispatch({ type: 'SET_PAN', payload: { x: 0, y: 0 } });

      // Update image size after PDF is processed
      setTimeout(() => {
        // updateImageSize();
      }, 100);

      // Upload to backend
      const uploadedPath = await uploadPdfToBackend(file);

      if (savedProjects.length < 0) {
        completeCurrentStep();
        setCurrentStep(prev => prev + 1);
      }

      navigateTo(SCREEN_TYPES.PDF_CROP);
      setStatus(false);
      if (uploadedPath) {
        // Store the uploaded path with the file for later saving
        setPdfFile(prev => ({ ...prev, uploadedPath }));
        console.log('✅ PDF uploaded and ready for project saving');
      }
    } catch (error) {
      console.error('Error processing PDF:', error);
      setLoadingError('Error processing PDF file. Please make sure it\'s a valid PDF.');
    } finally {
      setLoading(false);
    }
  };

  // Handle drag start for access point types
  const handleDragStart = (e, type: string) => {

    if (isTypeAtLimit(type, accessPoints)) {
      e.preventDefault();
      setLimitMessage(`Limit reached (1 max per map)`);
      setTimeout(() => setLimitMessage(''), 3000);
      return;
    }
    setDragOffset({ x: 0, y: 0 });
    setDraggedItem({ type, isNew: true });
    e.dataTransfer.effectAllowed = 'copy';
  };

  // Handle touch start for access point types (mobile)
  const handleTouchStartType = (e, type: string) => {
    if (isTypeAtLimit(type, accessPoints)) {
      e.preventDefault();
      setLimitMessage(`Limit reached (1 max per map)`);
      setTimeout(() => setLimitMessage(''), 3000);
      return;
    }
    e.preventDefault();
    setDragOffset({ x: 0, y: 0 });
    setTouchDragItem({ type, isNew: true });
    setIsDraggingTouch(true);
    const touch = e.touches[0];
    setTouchDragPreview({
      x: touch.clientX - 20,
      y: touch.clientY - 40,
      visible: true
    });
  };

  // Handle drag start for existing access points
  const handleAccessPointDragStart = (e, accessPoint) => {
    // Prevent dragging if node movement is disabled
    if (!allowNodeMovement) {
      e.preventDefault();
      return;
    }

    if (isDraggingTouch) {
      e.preventDefault();
      return;
    }
    // Calculate the offset from where user clicked on the element
    const elementRect = e.target.getBoundingClientRect();
    const offsetX = e.clientX - elementRect.left - (elementRect.width / 2);
    const offsetY = e.clientY - elementRect.top - (elementRect.height / 2);

    setDragOffset({ x: offsetX, y: offsetY });
    setDraggedItem({ ...accessPoint, isNew: false });
    setProjectUpdated(true);

    e.dataTransfer.effectAllowed = 'move';
  };

  // Handle touch start for existing access points (mobile)
  const handleTouchStartAccessPoint = (e, accessPoint) => {
    if (!allowNodeMovement) {
      e.preventDefault();
      return;
    }

    e.preventDefault();
    e.stopPropagation();

    const touch = e.touches[0];
    const elementRect = e.target.getBoundingClientRect();
    const offsetX = touch.clientX - elementRect.left - (elementRect.width / 2);
    const offsetY = touch.clientY - elementRect.top - (elementRect.height / 2);

    setDragOffset({ x: offsetX, y: offsetY });
    setTouchDragItem({ ...accessPoint, isNew: false });
    setIsDraggingTouch(true);

    setTouchDragPreview({
      x: touch.clientX - 20,
      y: touch.clientY - 40,
      visible: true
    });
  };

  const handleDrop = useCallback((e) => {
    e.preventDefault();
    if (!draggedItem || !mapContainerRef.current || !mapImageRef.current) return;
    const adjustedClientX = e.clientX - dragOffset.x;
    const adjustedClientY = e.clientY - dragOffset.y;

    const { x, y } = screenToMapCoordinates(e.clientX, e.clientY, mapContainerRef, mapImageRef, state.viewport.zoomLevel, state.viewport.panOffset, state.viewport.rotation, state.imageDimensions);
    const currentDimensions = getCurrentImageDimensions(mapImageRef, state.imageDimensions);
    if (draggedItem.isNew) {
      // Check if type is at limit before adding
      if (isTypeAtLimit(draggedItem.type, accessPoints)) {
        setDraggedItem(null);
        setDragOffset({ x: 0, y: 0 });
        return;
      }

      // Convert to relative coordinates
      const { relativeX, relativeY } = pixelToRelative(x, y, currentDimensions.width, currentDimensions.height);

      const newAccessPoint = {
        id: generateUniqueId(draggedItem.type, accessPoints),
        type: draggedItem.type,
        x: x, // Keep for backwards compatibility
        y: y, // Keep for backwards compatibility
        relativeX, // Store relative coordinates
        relativeY, // Store relative coordinates
        page: currentPage,
        name: `${accessPointTypes.find(t => t.id === draggedItem.type)?.name}`,
        deviceDetails: generateDeviceDetails(draggedItem.type)
      };
      setAccessPoints(prev => [...prev, newAccessPoint]);
    } else if (allowNodeMovement) {
      // Only allow moving existing nodes if movement is enabled
      const { relativeX, relativeY } = pixelToRelative(x, y, currentDimensions.width, currentDimensions.height);
      setAccessPoints(prev => prev.map(ap =>
          ap.id === draggedItem.id
              ? {
                ...ap,
                x: x, // Keep for backwards compatibility
                y: y, // Keep for backwards compatibility
                relativeX, // Update relative coordinates
                relativeY, // Update relative coordinates
                page: currentPage
              }
              : ap
      ));
    }

    setDraggedItem(null);
    setDragOffset({ x: 0, y: 0 });
  }, [draggedItem, dragOffset, accessPoints, currentPage, state.viewport.zoomLevel, state.viewport.panOffset, state.viewport.rotation, allowNodeMovement]);
  const handleDragOver = (e) => {
    e.preventDefault();
    e.dataTransfer.dropEffect = draggedItem?.isNew ? 'copy' : 'move';
  };

  // Handle touch move (mobile)
  const handleTouchMove = useCallback((e) => {
    if (!isDraggingTouch || !touchDragItem) return;

    e.preventDefault();
    const touch = e.touches[0];
    setTouchDragPreview({
      x: touch.clientX - 20, // Offset so finger doesn't cover the preview
      y: touch.clientY - 40,
      visible: true
    });
  }, [isDraggingTouch, touchDragItem]);

  // Handle touch end (mobile)
  const handleTouchEnd = useCallback((e) => {
    if (!isDraggingTouch || !touchDragItem) return;

    e.preventDefault();
    const touch = e.changedTouches[0];

    // Check if touch ended over the map container
    const mapElement = mapContainerRef.current;
    if (mapElement) {
      const rect = mapElement.getBoundingClientRect();
      const isOverMap = touch.clientX >= rect.left &&
          touch.clientX <= rect.right &&
          touch.clientY >= rect.top &&
          touch.clientY <= rect.bottom;

      if (isOverMap) {
        // Apply drag offset compensation for touch
        const adjustedClientX = touch.clientX - dragOffset.x;
        const adjustedClientY = touch.clientY - dragOffset.y;

        const { x, y } = screenToMapCoordinates(
            adjustedClientX,
            adjustedClientY,
            mapContainerRef,
            mapImageRef,
            state.viewport.zoomLevel,
            state.viewport.panOffset,
            state.viewport.rotation,
            state.imageDimensions
        );

        const currentDimensions = getCurrentImageDimensions(mapImageRef, state.imageDimensions);

        if (touchDragItem.isNew) {
          const { relativeX, relativeY } = pixelToRelative(x, y, currentDimensions.width, currentDimensions.height);

          const newAccessPoint = {
            id: generateUniqueId(touchDragItem.type, accessPoints),
            type: touchDragItem.type,
            x: x,
            y: y,
            relativeX,
            relativeY,
            page: currentPage,
            name: `${accessPointTypes.find(t => t.id === touchDragItem.type)?.name}`,
            deviceDetails: generateDeviceDetails(touchDragItem.type)
          };
          setAccessPoints(prev => [...prev, newAccessPoint]);
        } else if (allowNodeMovement) {
          // Move existing access point
          const { relativeX, relativeY } = pixelToRelative(x, y, currentDimensions.width, currentDimensions.height);

          setAccessPoints(prev => prev.map(ap =>
              ap.id === touchDragItem.id
                  ? {
                    ...ap,
                    x: x,
                    y: y,
                    relativeX,
                    relativeY,
                    page: currentPage
                  }
                  : ap
          ));
        }
      }
    }

    // Clean up
    setIsDraggingTouch(false);
    setTouchDragItem(null);
    setTouchDragPreview({ x: 0, y: 0, visible: false });
    setDragOffset({ x: 0, y: 0 });
  }, [isDraggingTouch, touchDragItem, dragOffset, accessPoints, currentPage, allowNodeMovement]);
  // Add touch event listeners
  useEffect(() => {
    document.addEventListener('touchmove', handleTouchMove, { passive: false });
    document.addEventListener('touchend', handleTouchEnd, { passive: false });
    return () => {
      document.removeEventListener('touchmove', handleTouchMove);
      document.removeEventListener('touchend', handleTouchEnd);
    };
  }, [handleTouchMove, handleTouchEnd]);

  // Remove access point
  const removeAccessPoint = (id: string) => {
    setAccessPoints(accessPoints.filter(ap => ap.id !== id));
  };

  // Start editing node name
  const startEditingNode = (accessPoint: AccessPointMarkerPoint) => {
    setEditingNodeId(accessPoint.id);
    setProjectUpdated(true);
    setEditingNodeName(accessPoint.floor);
  };



  // Clear current project
  const clearProject = () => {
    setAccessPoints([]);
    setLimitMessage('');
    setAllowNodeMovement(true);
  };

  // Save project
  const saveProject = async () => {
    if (!projectName.trim()) {
      setLoadingError('Please enter a project name');
      return;
    }

    // If no PDF is loaded, show error asking to upload PDF first
    if (!pdfFile) {
      setLoadingError('Please upload a PDF file before saving the project');
      return;
    }
    const currentViewportState = {
      zoomLevel: state.viewport.zoomLevel,
      panOffset: state.viewport.panOffset,
      pageViewports: {
        ...pageViewports,
        [currentPage]: {
          zoomLevel: state.viewport.zoomLevel,
          panOffset: state.viewport.panOffset
        }
      }
    };
    const projectData = {
      name: projectName.trim(),
      accessPoints: accessPoints.map(ap => ({
        id: ap.id,
        type: ap.type,
        name: ap.name,
        floor: ap.floor,
        x: ap.relativeX,
        y: ap.relativeY,
        page: ap.page,
      })),
      currentPage,
      timestamp: Date.now(),
      pageRotation: state.viewport.rotation || 0,
      pdfFile: {
        name: pdfFile.name,
        size: pdfFile.size,
        originalName: pdfFile.name,
        uploadedPath: pdfFile.uploadedPath || null
      },
      viewportState: currentViewportState
    };

    setApiLoading(true);

    try {
      // If we have a PDF but no uploaded path, upload it first
      if ((pdfFile && !pdfFile.uploadedPath) || (state.pdfFile && !state.pdfFile.uploadedPath)) {
        console.log('📤 Uploading PDF before saving project...');
        const uploadedPath = await uploadPdfToBackend(state.pdfFile ?? pdfFile);
        if (uploadedPath) {
          setStatus(true);
          projectData.pdfFile.uploadedPath = uploadedPath;
          setPdfFile(prev => ({...prev, uploadedPath}));
        } else {
          setStatus(false);
          addToast({
            title: 'Error',
            color: 'danger',
            text: 'Failed to upload PDF to backend',
          });
        }
      }

      // Save to backend API
      const savedProjectResult = await saveProjectToApi(projectData);

      setShowSaveModal(false);
      setProjectName('');
      setProjectUpdated(!projectUpdated);

      if (!savedProjectResult) {
        return addToast({
          title: 'error',
          color: "danger",
          text: 'error'
        })
      }

      return addToast({
        title: '',
        color: "success",
        text: dict.projectSaved
      });

    } catch (error) {
      console.error('Error saving project:', error);
      setLoadingError(`Error saving project: ${error.message}`);
    } finally {
      setApiLoading(false);
    }
  };

  // Load project
  const loadProject = async (project) => {
    try {
      setLimitMessage('');

      const currentDimensions = getCurrentImageDimensions(mapImageRef, state.imageDimensions);
      const accessPointsWithDetails = (project.accessPoints || []).map(ap => {
        let migratedAp = {
          ...ap,
          deviceDetails: generateDeviceDetails(ap.type)
        };

        // If image dimensions are available and point doesn't have relative coordinates, migrate them
        if (currentDimensions.width > 0 && currentDimensions.height > 0 &&
            (typeof migratedAp.relativeX !== 'number' || typeof migratedAp.relativeY !== 'number')) {
          migratedAp = migrateToRelativeCoordinates(migratedAp, currentDimensions.width, currentDimensions.height);
        }

        return migratedAp;
      });

      setAccessPoints(accessPointsWithDetails);
      setCurrentPage(project.currentPage || 0);

      if (project.viewportState) {
        console.log('🔍 Restoring viewport state from project');

        // Restore page viewports
        if (project.viewportState.pageViewports) {
          setPageViewports(project.viewportState.pageViewports);
        }

        // Restore current page viewport or global viewport
        const currentPageViewport = project.viewportState.pageViewports?.[project.currentPage || 0];
        if (currentPageViewport) {
          dispatch({ type: 'SET_ZOOM', payload: currentPageViewport.zoomLevel });
          dispatch({ type: 'SET_PAN', payload: currentPageViewport.panOffset });

          console.log(`🔍 Restored page ${project.currentPage || 0} viewport: zoom ${Math.round(currentPageViewport.zoomLevel * 100)}%`);
        } else if (project.viewportState.zoomLevel && project.viewportState.panOffset) {
          dispatch({ type: 'SET_ZOOM', payload: project.viewportState.zoomLevel });
          dispatch({ type: 'SET_PAN', payload: project.viewportState.panOffset });
          console.log(`🔍 Restored global viewport: zoom ${Math.round(project.viewportState.zoomLevel * 100)}%`);
        }
      } else {
        // Reset viewport if no saved state
        dispatch({ type: 'SET_ZOOM', payload: 1 });
        dispatch({ type: 'SET_PAN', payload: { x: 0, y: 0 } });

        setPageViewports({});
        console.log('🔍 No viewport state found, using defaults');
      }



      setShowLoadModal(false);

      // Load PDF from backend if available
      if (project.pdfFile?.uploadedPath) {
        try {
          setApiLoading(true);
          console.log('📥 Loading PDF from backend:', project.pdfFile.uploadedPath);

          const pdfResponse = await fetch(`/api/${project.pdfFile.uploadedPath}`);
          if (pdfResponse.ok) {
            const pdfBlob = await pdfResponse.blob();
            console.log(pdfBlob)
            const pdfFile = new File([pdfBlob], project.pdfFile.originalName || project.pdfFile.name, {
              type: 'application/pdf'
            });

            // Add the uploaded path to the file object
            pdfFile.uploadedPath = project.pdfFile.uploadedPath;

            // Process the PDF
            const pages = await processPdfFromFile(pdfFile);
            setPdfPages(pages);
            setPdfFile(pdfFile);
            addToast({
              title: '',
              color: "success",
              text: dict.projectFound,
            });
          } else {
            addToast({
              title: 'Error',
              color: "danger",
              text: `Failed to load PDF: ${pdfResponse.status}`,
            });
          }
        } catch (pdfError) {
          console.warn('Could not load PDF from backend:', pdfError);
          setLoadingError(`Project loaded, but could not load PDF: ${project.pdfFile?.originalName || project.pdfFile?.name || 'Unknown'}. Please re-upload the PDF file.`);
        } finally {
          setApiLoading(false);
        }
      } else {
        // No PDF in project, show message
        console.log(`Loaded project "${project.name}" - No PDF file associated`);
        setLoadingError(`Project "${project.name}" loaded successfully. Please upload a PDF file to continue editing.`);
      }

    } catch (error) {
      console.error('Error loading project:', error);
      setLoadingError('Error loading project');
    }
  };

  // Show node details modal
  const showNodeDetails = (accessPoint) => {
    setSelectedNodeDetails(accessPoint);
    setShowNodeDetailsModal(true);
  };

  const [progress, setProgress] = useState(0);

  const onClose = () => {
    setShowNodeDetailsModal(false);
  }

  if (currentScreen === SCREEN_TYPES.MODULE_LOADING) {
    const interval = setInterval(() => {
      if (!saved) {
        setProgress(prev => {
          if (prev >= 100) {
            clearInterval(interval);
            setTimeout(() => {
              setgoToStep(SCREEN_TYPES.PROJECT_MODAL)
              navigateTo(SCREEN_TYPES.FAIL)
            }, 800);
            return 100;
          }
          return prev + Math.random() * 8 + 2;
        });
      } else {
        setProgress(prev => {
          if (prev >= 100) {
            clearInterval(interval);
            setTimeout(() => navigateTo(SCREEN_TYPES.COMPLETE), 800);
            return 100;
          }
          return prev + Math.random() * 8 + 2;
        });
      }

    }, 150);
    interval;
  }
  // Get current page access points with proper positioning
  const getCurrentPageAccessPoints = () => {
    const currentDimensions = getCurrentImageDimensions(mapImageRef, state.imageDimensions);

    return accessPoints
        .filter(ap => ap.page === currentPage)
        .map(ap => {
          // Use relative coordinates if available, otherwise fallback to absolute
          if (typeof ap.relativeX === 'number' && typeof ap.relativeY === 'number') {
            const { x, y } = relativeToPixel(ap.relativeX, ap.relativeY, currentDimensions.width, currentDimensions.height);
            return { ...ap, x, y };
          }
          // Fallback to stored absolute coordinates
          return ap;
        });
  };

  const currentPageAccessPoints = getCurrentPageAccessPoints();

  const handleSaveModalOpen = async () => {
    setShowSaveModal(true);
  };

  React.useEffect(() => {
    const timer = setTimeout(() => {
      setIsLoading(false);
    }, 4000);

    return () => clearTimeout(timer);
  }, []);
  const [touchStart, setTouchStart] = useState(null);
  const [initialDistance, setInitialDistance] = useState(null);
  const [initialZoom, setInitialZoom] = useState(1);
  const [isTouchPanning, setIsTouchPanning] = useState(false);
  const [lastTouchPanPosition, setLastTouchPanPosition] = useState({ x: 0, y: 0 });

  // Handle touch start for zoom and pan
  const handleMobileTouchStart = useCallback((e) => {
    if (e.touches.length === 2) {
      // Two finger zoom
      e.preventDefault();
      const distance = getTouchDistance(e.touches[0], e.touches[1]);
      setInitialDistance(distance);
      setInitialZoom(state.viewport.zoomLevel);
      setTouchStart({
        x1: e.touches[0].clientX,
        y1: e.touches[0].clientY,
        x2: e.touches[1].clientX,
        y2: e.touches[1].clientY
      });
      setIsTouchPanning(false); // Disable panning during zoom
    } else if (e.touches.length === 1) {
      // Single finger pan
      const touch = e.touches[0];
      setIsTouchPanning(true);
      setLastTouchPanPosition({ x: touch.clientX, y: touch.clientY });
      setInitialDistance(null); // Disable zoom during pan
    }
  }, [state.viewport.zoomLevel]);

  // Handle touch move for zoom and pan
  const handleMobileTouchMove = useCallback((e) => {
    if (e.touches.length === 2 && initialDistance && touchStart) {
      // Two finger zoom
      e.preventDefault();
      const currentDistance = getTouchDistance(e.touches[0], e.touches[1]);
      const scale = currentDistance / initialDistance;
      const newZoom = Math.max(MIN_ZOOM, Math.min(MAX_ZOOM, initialZoom * scale));
      dispatch({ type: 'SET_ZOOM', payload: newZoom });
    } else if (e.touches.length === 1 && isTouchPanning) {
      // Single finger pan
      e.preventDefault();
      const touch = e.touches[0];
      const deltaX = touch.clientX - lastTouchPanPosition.x;
      const deltaY = touch.clientY - lastTouchPanPosition.y;
      dispatch({ type: 'SET_PAN', payload: { x: state.viewport.panOffset.x + deltaX, y: state.viewport.panOffset.y + deltaY } });

      setLastTouchPanPosition({ x: touch.clientX, y: touch.clientY });
    }
  }, [initialDistance, touchStart, initialZoom, isTouchPanning, lastTouchPanPosition]);

  // Handle touch end for zoom and pan
  const handleMobileTouchEnd = useCallback((e) => {
    if (e.touches.length < 2) {
      setInitialDistance(null);
      setTouchStart(null);
    }
    if (e.touches.length === 0) {
      setIsTouchPanning(false);
    }
  }, []);

  // Add touch event listeners for mobile zoom
  useEffect(() => {
    if(!isMobile) return;
    const mapContainer = mapContainerRef.current;
    if (mapContainer) {
      mapContainer.addEventListener('touchstart', handleMobileTouchStart, { passive: false });
      mapContainer.addEventListener('touchmove', handleMobileTouchMove, { passive: false });
      mapContainer.addEventListener('touchend', handleMobileTouchEnd, { passive: false });
    }

    return () => {
      if (mapContainer) {
        mapContainer.removeEventListener('touchstart', handleMobileTouchStart);
        mapContainer.removeEventListener('touchmove', handleMobileTouchMove);
        mapContainer.removeEventListener('touchend', handleMobileTouchEnd);
      }
    };
  }, [handleMobileTouchStart, handleMobileTouchMove, handleMobileTouchEnd]);

  if (!pdfJsLoaded) {
    return <></>;
  }


  return (
      <EuiPage>
        <EuiPageBody>
          <EuiPageHeader>
            <EuiTitle size="l">
              <h1></h1>
            </EuiTitle>
          </EuiPageHeader>
          {(savedProjects.length === 0 && !isLoading) && <MultiStepDemo/>}
          <Test setCurrentScreen={(step) => navigateTo(step)} status={status} savedProjects={savedProjects}/>
          <>
            <>

              {((currentScreen === SCREEN_TYPES.PROJECT_SETUP || (currentScreen === SCREEN_TYPES.UPLOAD)) && (!projectFound)) && (<EuiPanel>
                <EuiTitle size="m">
                  <h2>{currentScreen === SCREEN_TYPES.PROJECT_SETUP ? dict.mapConfig : dict.uploadText + dict.accessPoint}</h2>
                </EuiTitle>
                <EuiSpacer size="m"/>
                {/* Upload Section */}
                <EuiFlexGroup alignItems="center" gutterSize="m">
                  <>
                    {(currentScreen === SCREEN_TYPES.UPLOAD) && (
                      <>
                        <FileUploader
                            pdfFile={pdfFile}
                            pdfJsLoaded={pdfJsLoaded}
                            loadingError={loadingError}
                            apiLoading={loading}
                            accessPoints={accessPoints}
                            savedProjects={savedProjects}
                            onFileUpload={handleFileUpload}
                            onSave={saveProject}
                            onLoad={undefined}
                            onViewLegend={undefined}
                            onGeneratePDF={undefined}
                            isGenerating={isGenerating}
                            jsPdfLoaded={jsPdfLoaded}
                            configSetup={false}
                        />
                      </>
                    )}
                  </>
                </EuiFlexGroup>

                {!pdfJsLoaded && !loadingError && (
                    <>
                      <EuiSpacer size="m"/>
                      <EuiFlexGroup alignItems="center" gutterSize="s">
                        <EuiFlexItem grow={false}>
                          <EuiLoadingSpinner size="m"/>
                        </EuiFlexItem>
                        <EuiFlexItem grow={false}>
                          <EuiText size="s" color="subdued">
                            Loading...
                          </EuiText>
                        </EuiFlexItem>
                      </EuiFlexGroup>
                    </>
                )}
              </EuiPanel>)}

              {currentScreen === SCREEN_TYPES.PDF_CROP && (
                  <ZoomPanRotationEditor currentPageAccessPoints={currentPageAccessPoints}
                                         accessPointTypes={accessPointTypes} imgsrc={pdfPages[currentPage]}
                                         navigateTo={navigateTo} dd={dispatch} zz={state} setPdfFile={setPdfFile}/>
              )}
              {currentScreen === SCREEN_TYPES.PROJECT_SETUP && (
                <EuiFlexGroup gutterSize="l">
                  {(
                    <EuiFlexItem >
                      {/*Access Point Types */}
                      <AccessPointTypes
                          accessPointTypes={accessPointTypes}
                          isTypeAtLimit={isTypeAtLimit}
                          getTypeCount={getTypeCount}
                          handleDragStart={handleDragStart}
                          handleTouchStartType={handleTouchStartType}
                          accessPoints={accessPoints} />

                      {/* Current Access Points */}
                      <CurrentPageAccessPoints
                          currentPageAccessPoints={currentPageAccessPoints}
                          accessPointTypes={accessPointTypes}
                          handleAccessPointDragStart={handleAccessPointDragStart}
                          handleTouchStartAccessPoint={handleTouchStartAccessPoint}
                          editingNodeName={editingNodeName}
                          editingNodeId={editingNodeId}
                          setEditingNodeName={setEditingNodeName}
                          startEditingNode={startEditingNode}
                          setAccessPoints={setAccessPoints}
                          setEditingNodeId={setEditingNodeId}
                      />
                    </EuiFlexItem>
                  )}

                {/* Map Display */}
                <EuiFlexItem grow={4} style={{ overflow: 'hidden'}} ref={mainRef}>
                  {pdfPages.length > 0 ? (
                      <EuiPanel>
                        <EuiFlexGroup alignItems="center" justifyContent="spaceBetween">
                          <EuiFlexItem grow={false}>
                            <EuiTitle size="m">
                              <h3>{projectFound ? savedProjects[0].name : ''} {projectUpdated ? '*': ''}</h3>
                            </EuiTitle>
                          </EuiFlexItem>

                          {/* Node Movement Toggle */}
                          <NodeMovementToggle allowNodeMovement={allowNodeMovement} setAllowNodeMovement={setAllowNodeMovement} />

                          {/* Zoom and Rotation Controls */}
                          <EuiFlexItem grow={false}>
                            {projectFound && currentScreen === SCREEN_TYPES.PROJECT_SETUP &&(
                                <EuiButton
                                    color={'success'}
                                    onClick={handleSaveModalOpen}
                                    disabled={!projectUpdated}
                                >
                                  {dict.saveProject}
                                </EuiButton>
                            )}
                          </EuiFlexItem>
                        </EuiFlexGroup>

                        <EuiSpacer size="m" />

                        {/* Map Container with Zoom and Pan */}
                        <div
                            ref={mapContainerRef}
                            style={{
                              position: 'relative',
                              border: '2px dashed #D3DAE6',
                              borderRadius: '6px',
                              backgroundColor: '#F5F7FA',
                              overflow: 'hidden',
                              cursor: state.viewport.isPanning ? 'grabbing' : (state.viewport.zoomLevel > 1 ? 'grab' : 'default')
                            }}
                            onDrop={handleDrop}
                            onDragOver={handleDragOver}
                            onDragEnter={(e) => e.preventDefault()}
                            onWheel={handleWheel}
                            onMouseDown={handleMouseDown}
                        >
                          <ViewportControls
                              state={state}
                              dispatch={dispatch}
                              containerRef={containerRef}
                          />
                          {pdfPages[currentPage] && (
                              <div
                                  ref={containerRef}
                                  style={{
                                    position: 'relative',
                                    transform: `translate(${state.viewport.panOffset.x}px, ${state.viewport.panOffset.y}px) scale(${state.viewport.zoomLevel})`,
                                    transformOrigin: '0 0',
                                    transition: state.viewport.isPanning ? 'none' : 'transform 0.1s ease-out'
                                  }}
                              >
                                <img
                                    ref={mapImageRef}
                                    src={state.src ?? pdfPages[currentPage]}
                                    alt={`PDF Page ${currentPage + 1}`}
                                    style={{
                                      width: '100%',
                                      display: 'block',
                                      position: 'relative',
                                      zIndex: 0,
                                      userSelect: 'none',
                                      transform: `rotate(${state.viewport.rotation}deg)`,
                                      transformOrigin: 'center center'
                                    }}
                                    draggable={false}
                                    onLoad={handleImageLoad}
                                />

                                {/* Access Points Layer */}
                                <AccessPointLayer
                                    accessPointTypes={accessPointTypes}
                                    rotation={state.viewport.rotation}
                                    currentPageAccessPoints={currentPageAccessPoints}
                                    editingNodeId={editingNodeId}
                                    allowNodeMovement={allowNodeMovement}
                                    zoomLevel={state.viewport.zoomLevel}
                                    handleAccessPointDragStart={handleAccessPointDragStart}
                                    startEditingNode={startEditingNode}
                                    showNodeDetails={showNodeDetails}
                                    handleTouchStartAccessPoint={handleTouchStartAccessPoint}
                                />
                              </div>
                          )}

                          {/* Touch drag preview for mobile */}
                          {touchDragPreview.visible && touchDragItem && (
                              <div
                                  style={{
                                    position: 'fixed',
                                    left: `${touchDragPreview.x}px`,
                                    top: `${touchDragPreview.y}px`,
                                    transform: 'translate(-50%, -50%)',
                                    pointerEvents: 'none',
                                    zIndex: 1000,
                                    opacity: 0.8
                                  }}
                              >
                                <div
                                    style={{
                                      width: '24px',
                                      height: '24px',
                                      borderRadius: '50%',
                                      border: '2px solid white',
                                      backgroundColor: touchDragItem.isNew
                                          ? accessPointTypes.find(t => t.id === touchDragItem.type)?.color || '#006BB4'
                                          : accessPointTypes.find(t => t.id === touchDragItem.type)?.color || '#006BB4',
                                      display: 'flex',
                                      alignItems: 'center',
                                      justifyContent: 'center',
                                      color: 'white',
                                      fontSize: '14px',
                                      fontWeight: 'bold',
                                      boxShadow: '0 4px 12px rgba(0,0,0,0.4), 0 2px 4px rgba(0,0,0,0.2)'
                                    }}
                                >
                                </div>
                                <div style={{
                                  position: 'absolute',
                                  top: '28px',
                                  left: '50%',
                                  transform: 'translateX(-50%)',
                                  backgroundColor: '#343741',
                                  color: 'white',
                                  fontSize: '10px',
                                  padding: '2px 8px',
                                  borderRadius: '8px',
                                  whiteSpace: 'nowrap'
                                }}>
                                  {touchDragItem.isNew
                                      ? `New ${accessPointTypes.find(t => t.id === touchDragItem.type)?.name || 'Access Point'}`
                                      : touchDragItem.name || 'Access Point'}
                                </div>
                              </div>
                          )}
                        </div>

                        <EuiSpacer size="s" />
                        <EuiText size="s" color="subdued">
                          <strong>Controls:</strong>
                          <>
                            <div>• {dict.dragText}. </div>
                            <div>• {allowNodeMovement ? dict.nodeMovementText2 : dict.nodeMovementText} </div>
                            <div>• {state.viewport.zoomLevel > 1 ? dict.panText : dict.zoomText} </div>
                            <div>• {dict.rigthClick} </div>
                          </>
                        </EuiText>
                        {!projectFound && !newProject ?<motion.div variants={animations.itemVariants} style={{textAlign: 'center'}}>
                          <div style={{display: 'flex', gap: '12px', justifyContent: 'center'}}>
                            {!loading ?(<Btn
                                fill
                                onClick={() => {
                                  setLoading(true);
                                  navigateTo(SCREEN_TYPES.PROJECT_MODAL);
                                  completeCurrentStep();
                                  setLoading(false);
                                }}
                            >
                              {dict.continue}
                            </Btn>): <EuiLoadingSpinner size="l"/>}
                          </div>
                        </motion.div>: ''}
                      </EuiPanel>
                  ) : (
                      <EuiEmptyPrompt
                          icon={<EuiIcon type="document" size="xxl" />}
                          title={<h2><EuiLoadingSpinner /></h2>}
                          body={<p>{dict.wait}</p>}
                      />
                  )}
                </EuiFlexItem>
              </EuiFlexGroup> )}
            </>
          </>
          {/* Node Details Modal */}
          {showNodeDetailsModal && selectedNodeDetails && (
              <NodeDetails setShowNodeDetailsModal={setShowNodeDetailsModal} accessPointTypes={accessPointTypes} selectedNodeDetails={selectedNodeDetails} globalAps={globalAps} startEditingNode={startEditingNode} />
          )}

          {/* Save Project Modal */}
          {showSaveModal && (
              <EuiOverlayMask>
                <EuiModal onClose={() => setShowSaveModal(false)} maxWidth={400}>
                  <EuiModalHeader>
                    <EuiModalHeaderTitle>{dict.saveProject}</EuiModalHeaderTitle>
                  </EuiModalHeader>

                  <EuiModalBody>
                    <EuiFormRow label={dict.name} helpText={dict.projectNameText}>
                      <EuiFieldText
                          value={projectName}
                          onChange={(e: { target: { value: React.SetStateAction<string>; }; }) => setProjectName(e.target.value)}
                          placeholder="My Network Map"
                          onKeyDown={(e: { key: string; }) => e.key === 'Enter' && saveProject()}
                      />
                    </EuiFormRow>

                    <EuiSpacer size="m" />
                  </EuiModalBody>

                  <EuiModalFooter>
                    <EuiButton color={"danger"} onClick={() => setShowSaveModal(false)}>
                      {dict.cancelBtn}
                    </EuiButton>
                    <EuiButton
                        onClick={saveProject}
                        color={"success"}
                        disabled={!projectName.trim() || !pdfFile}
                        isLoading={apiLoading}
                    >
                      {apiLoading ? dict.saving+'...' : dict.saveProject}
                    </EuiButton>
                  </EuiModalFooter>
                </EuiModal>
              </EuiOverlayMask>
          )}

          {/* Load Project Modal */}
          {showLoadModal && (
              <EuiOverlayMask>
                <EuiModal onClose={() => setShowLoadModal(false)} maxWidth={600}>
                  <EuiModalHeader>
                    <EuiModalHeaderTitle>
                      {dict.loadProject}
                    </EuiModalHeaderTitle>
                  </EuiModalHeader>

                  <EuiModalBody>
                    {apiLoading ? (
                        <EuiFlexGroup alignItems="center" justifyContent="center">
                          <EuiFlexItem grow={false}>
                            <EuiLoadingSpinner size="l" />
                          </EuiFlexItem>
                          <EuiFlexItem grow={false}>
                            <EuiText>{dict.loadProject}...</EuiText>
                          </EuiFlexItem>
                        </EuiFlexGroup>
                    ) : savedProjects.length === 0 ? (
                        <EuiEmptyPrompt
                            icon={<EuiIcon type="folderOpen" size="l" />}
                            title={<h3>No Saved Projects</h3>}
                            body={<p>Save your current project to see it here.</p>}
                        />
                    ) : (
                        <EuiListGroup maxWidth={false} flush>
                          {savedProjects
                              .sort((a: { timestamp: number }, b: { timestamp: number }) => b.timestamp - a.timestamp)
                              .map((project) => (
                                  <EuiListGroupItem
                                      key={project.name}
                                      label={
                                        <EuiFlexGroup alignItems="center" justifyContent="spaceBetween">
                                          <EuiFlexItem>
                                            <EuiFlexGroup direction="column" gutterSize="xs">
                                              <EuiFlexItem>
                                                <EuiText size="s">
                                                  <strong>{project.name}</strong>
                                                </EuiText>
                                              </EuiFlexItem>
                                              <EuiFlexItem>
                                                <EuiText size="xs" color="subdued">
                                                  {project.accessPoints?.length || 0} {dict.accessPoint} • {' '}
                                                  {project.pdfFile?.name || 'No PDF'} • {' '}
                                                  {new Date(project.timestamp).toLocaleDateString()}
                                                </EuiText>
                                              </EuiFlexItem>
                                            </EuiFlexGroup>
                                          </EuiFlexItem>
                                          <EuiFlexItem grow={false}>
                                            <EuiFlexGroup gutterSize="xs">
                                              <EuiFlexItem grow={false}>
                                                <EuiButton
                                                    onClick={() => loadProject(project)}
                                                    size="s"
                                                    fill
                                                    disabled={apiLoading}
                                                >
                                                  {dict.load}
                                                </EuiButton>
                                              </EuiFlexItem>
                                            </EuiFlexGroup>
                                          </EuiFlexItem>
                                        </EuiFlexGroup>
                                      }
                                  />
                              ))}
                        </EuiListGroup>
                    )}
                  </EuiModalBody>

                  <EuiModalFooter>
                    <EuiButton onClick={() => setShowLoadModal(false)} fill>
                      {dict.close}
                    </EuiButton>
                  </EuiModalFooter>
                </EuiModal>
              </EuiOverlayMask>
          )}
              {currentScreen === SCREEN_TYPES.PROJECT_MODAL && (
                <ProjectSetupScreen
                  projectName={projectName}
                  setProjectName={setProjectName}
                  onContinue={async () => {
                    await saveProject();
                    navigateTo(SCREEN_TYPES.MODULE_LOADING);
                  }}
                />
              )}

          {currentScreen === SCREEN_TYPES.MODULE_LOADING && (
              <motion.div
                  key="loading-module"
                  variants={animations.pageVariants}
                  initial="initial"
                  animate="in"
                  exit="out"
              >
                <motion.div
                    variants={animations.containerVariants}
                    initial="hidden"
                    animate="visible"
                >
                  <EuiPanel paddingSize="l" style={{ margin: '0 12px', textAlign: 'center' }}>
                    <motion.div
                        variants={animations.itemVariants}
                        style={{
                          fontSize: '48px',
                          marginBottom:  '20px',
                          filter: 'drop-shadow(0 2px 4px rgba(0,0,0,0.1))'
                        }}
                    >

                    </motion.div>

                    <motion.div variants={animations.itemVariants}>
                      <Title size="m">{dict.saving}</Title>
                    </motion.div>

                    <EuiSpacer size="s" />

                    <motion.div variants={animations.itemVariants}>
                      <EuiText color="subdued" size="s">
                        {dict.wait}...
                      </EuiText>
                    </motion.div>

                    <EuiSpacer size="l" />

                    <motion.div variants={animations.itemVariants}>
                      <EuiProgress value={progress} max={100} />
                    </motion.div>

                    <EuiSpacer size="m" />



                    <motion.div variants={animations.itemVariants} style={{ display: 'flex', justifyContent: 'center', alignItems: 'center', gap: '12px' }}>
                      <EuiLoadingSpinner size="m" />
                      <EuiText size="s" style={{ fontWeight: '500', color: '#dc2626' }}>
                        {Math.round(progress)}%
                      </EuiText>
                    </motion.div>
                  </EuiPanel>
                </motion.div>
              </motion.div>
          )}
          {currentScreen === SCREEN_TYPES.FAIL && (
              <Error loadingError={loadingError} step={goToStep}/>
          )}
          {currentScreen === SCREEN_TYPES.COMPLETE && (
              <motion.div
                  key="complete"
                  variants={animations.pageVariants}
                  initial="initial"
                  animate="in"
                  exit="out"
              >
                <motion.div
                    variants={animations.containerVariants}
                    initial="hidden"
                    animate="visible"
                >
                  <EuiPanel paddingSize="l" style={{ margin: '0 12px', textAlign: 'center' }}>
                    <motion.div
                        variants={animations.itemVariants}
                        initial={{ scale: 0, rotate: -180 }}
                        animate={{ scale: 1, rotate: 0 }}
                        transition={{
                          type: "spring",
                          stiffness: 200,
                          damping: 15,
                          delay: 0.2
                        }}
                        style={{
                          fontSize: false ? '56px' : '72px',
                          marginBottom: false ? '20px' : '24px',
                          filter: 'drop-shadow(0 4px 12px rgba(220, 38, 38, 0.3))'
                        }}
                    >
                      ✨
                    </motion.div>

                    <motion.div variants={animations.itemVariants}>
                      <Title size="l"> {dict.mapSavedText}</Title>
                    </motion.div>

                    <EuiSpacer size="s" />

                    <motion.div variants={animations.itemVariants}>
                      <EuiText color="subdued" style={{ padding: '0 12px' }}>
                        {dict.redirecting}..
                      </EuiText>
                    </motion.div>

                    <EuiSpacer size="l" />
                    <motion.div variants={animations.itemVariants} style={{textAlign: 'center'}}>
                      <div style={{display: 'flex', gap: '12px', justifyContent: 'center'}}>
                        <Btn
                            fill
                            onClick={async () => {
                              await loadFirstProjectFromBackend({
                                setStatus,
                                loadProjectsFromApi,
                                setAccessPoints,
                                setCurrentPage,
                                setPageViewports,
                                setPdfPages,
                                setPdfFile,
                                setProjectFound,
                                setLoadingError,
                                addToast,
                                generateDeviceDetails,
                                processPdfFromFile
                              });
                              navigateTo(SCREEN_TYPES.PROJECT_SETUP)
                            }}
                        >
                          Finish
                        </Btn>
                      </div>
                    </motion.div>
                  </EuiPanel>
                </motion.div>
              </motion.div>
          )}
        </EuiPageBody>
      </EuiPage>
  );
}
const ViewportControls = React.memo(({ state, dispatch, containerRef }) => {
  const handleFitToView = useCallback(() => {
    if (containerRef.current) {
      const rect = containerRef.current.parentElement.getBoundingClientRect();
      dispatch({
        type: 'FIT_TO_VIEW',
        payload: {
          containerWidth: rect.width,
          containerHeight: rect.height
        }
      });
    }
  }, [dispatch, containerRef]);

  const zoomPercentage = Math.round(state.viewport.zoomLevel * 100);
  const canZoomIn = state.viewport.zoomLevel < MAX_ZOOM;
  const canZoomOut = state.viewport.zoomLevel > MIN_ZOOM;

  return (
      <div style={{
        position: 'absolute',
        top: '12px',
        right: '12px',
        display: 'flex',
        flexDirection: 'column',
        gap: '8px',
        zIndex: 1000
      }}>
        {/* Zoom Controls */}
        <div style={{
          display: 'flex',
          backgroundColor: 'white',
          borderRadius: '6px',
          boxShadow: '0 2px 8px rgba(0,0,0,0.15)',
          overflow: 'hidden',
          border: '1px solid #e1e5e9'
        }}>
          <button
              onClick={() => dispatch({ type: 'ZOOM_OUT' })}
              disabled={!canZoomOut}
              style={{
                padding: '8px 12px',
                border: 'none',
                backgroundColor: 'white',
                cursor: canZoomOut ? 'pointer' : 'not-allowed',
                opacity: canZoomOut ? 1 : 0.5,
                fontSize: '16px',
                fontWeight: 'bold'
              }}
              title="Zoom Out (Ctrl + -)"
          >
            −
          </button>

          <div style={{
            padding: '8px 16px',
            backgroundColor: '#f8f9fa',
            display: 'flex',
            alignItems: 'center',
            fontSize: '13px',
            fontWeight: '500',
            minWidth: '60px',
            justifyContent: 'center',
            borderLeft: '1px solid #e1e5e9',
            borderRight: '1px solid #e1e5e9'
          }}>
            {zoomPercentage}%
          </div>

          <button
              onClick={() => dispatch({ type: 'ZOOM_IN' })}
              disabled={!canZoomIn}
              style={{
                padding: '8px 12px',
                border: 'none',
                backgroundColor: 'white',
                cursor: canZoomIn ? 'pointer' : 'not-allowed',
                opacity: canZoomIn ? 1 : 0.5,
                fontSize: '16px',
                fontWeight: 'bold'
              }}
              title="Zoom In (Ctrl + +)"
          >
            +
          </button>
        </div>

        {/* Rotation Controls */}
        <div style={{
          display: 'flex',
          backgroundColor: 'white',
          borderRadius: '6px',
          boxShadow: '0 2px 8px rgba(0,0,0,0.15)',
          overflow: 'hidden',
          border: '1px solid #e1e5e9'
        }}>
          <button
              onClick={() => dispatch({ type: 'ROTATE_LEFT' })}
              style={{
                padding: '8px 12px',
                border: 'none',
                backgroundColor: 'white',
                cursor: 'pointer',
                fontSize: '16px'
              }}
              title="Rotate Left (Ctrl + ←)"
          >
            ↻
          </button>

          <div style={{
            padding: '8px 12px',
            backgroundColor: '#f8f9fa',
            display: 'flex',
            alignItems: 'center',
            fontSize: '12px',
            fontWeight: '500',
            minWidth: '45px',
            justifyContent: 'center',
            borderLeft: '1px solid #e1e5e9',
            borderRight: '1px solid #e1e5e9'
          }}>
            {state.viewport.rotation}°
          </div>

          <button
              onClick={() => dispatch({ type: 'ROTATE_RIGHT' })}
              style={{
                padding: '8px 12px',
                border: 'none',
                backgroundColor: 'white',
                cursor: 'pointer',
                fontSize: '16px'
              }}
              title="Rotate Right (Ctrl + →)"
          >
            ↺
          </button>
        </div>
      </div>
  );
});

