import { cropImageToViewport } from '@/lib/pdfUtils';
import { SCREEN_TYPES } from '@/setupApp/utils/constants';
import {
    EuiTourStep,
    EuiButton,
    EuiFlexGroup,
    EuiFlexItem,
    EuiText,
    EuiButtonIcon
} from '@elastic/eui';
import React, { useReducer, useCallback, useMemo, useRef, useEffect } from 'react';
import getDict from '@/lib/dict';
const dict = getDict("de");
// ========================
// CONSTANTS
// ========================
const MIN_ZOOM = 0.1;
const MAX_ZOOM = 5;
const ZOOM_STEP = 0.1;
const ZOOM_SENSITIVITY = 0.001;

const initialState = {
    viewport: {
        zoomLevel: 1,
        panOffset: { x: 0, y: 0 },
        rotation: 0,
        isPanning: false,
        lastPanPosition: { x: 0, y: 0 }
    },
    pdfFile: null,
    accessPoints: [],
    imageDimensions: { width: 0, height: 0 },
    draggedItem: null,
    touch: {
        initialDistance: null,
        initialZoom: 1,
        isPinching: false,
        touchCount: 0
    },
    ui: {
        showControls: false,
    }
};

function viewportReducer(state, action) {
    switch (action.type) {
        case 'SET_ZOOM':
            return {
                ...state,
                viewport: {
                    ...state.viewport,
                    zoomLevel: Math.max(MIN_ZOOM, Math.min(MAX_ZOOM, action.payload))
                }
            };

        case 'ZOOM_IN':
            return {
                ...state,
                viewport: {
                    ...state.viewport,
                    zoomLevel: Math.min(MAX_ZOOM, state.viewport.zoomLevel + ZOOM_STEP)
                }
            };

        case 'ZOOM_OUT':
            return {
                ...state,
                viewport: {
                    ...state.viewport,
                    zoomLevel: Math.max(MIN_ZOOM, state.viewport.zoomLevel - ZOOM_STEP)
                }
            };

        case 'SET_PAN':
            return {
                ...state,
                viewport: {
                    ...state.viewport,
                    panOffset: action.payload
                }
            };

        case 'START_PAN':
            return {
                ...state,
                viewport: {
                    ...state.viewport,
                    isPanning: true,
                    lastPanPosition: action.payload
                }
            };

        case 'UPDATE_PAN':
            if (!state.viewport.isPanning) return state;

            const deltaX = action.payload.x - state.viewport.lastPanPosition.x;
            const deltaY = action.payload.y - state.viewport.lastPanPosition.y;

            return {
                ...state,
                viewport: {
                    ...state.viewport,
                    panOffset: {
                        x: state.viewport.panOffset.x + deltaX,
                        y: state.viewport.panOffset.y + deltaY
                    },
                    lastPanPosition: action.payload
                }
            };

        case 'END_PAN':
            return {
                ...state,
                viewport: {
                    ...state.viewport,
                    isPanning: false
                }
            };

        case 'SET_ROTATION':
            return {
                ...state,
                viewport: {
                    ...state.viewport,
                    rotation: ((action.payload % 360) + 360) % 360
                }
            };

        case 'ROTATE_LEFT':
            return {
                ...state,
                viewport: {
                    ...state.viewport,
                    rotation: ((state.viewport.rotation - 90) + 360) % 360
                }
            };

        case 'ROTATE_RIGHT':
            return {
                ...state,
                viewport: {
                    ...state.viewport,
                    rotation: (state.viewport.rotation + 90) % 360
                }
            };

        case 'RESET_VIEW':
            return {
                ...state,
                viewport: {
                    ...state.viewport,
                    zoomLevel: 1,
                    panOffset: { x: 0, y: 0 },
                    rotation: 0
                }
            };

        case 'FIT_TO_VIEW':
            // Calculate zoom to fit image in container
            const containerPadding = 40;
            const containerWidth = action.payload.containerWidth - containerPadding;
            const containerHeight = action.payload.containerHeight - containerPadding;
            const imageWidth = state.imageDimensions.width;
            const imageHeight = state.imageDimensions.height;

            if (imageWidth === 0 || imageHeight === 0) return state;

            const scaleX = containerWidth / imageWidth;
            const scaleY = containerHeight / imageHeight;
            const fitZoom = Math.min(scaleX, scaleY, MAX_ZOOM);

            // Center the image
            const panX = (containerWidth - imageWidth * fitZoom) / 2;
            const panY = (containerHeight - imageHeight * fitZoom) / 2;

            return {
                ...state,
                viewport: {
                    ...state.viewport,
                    zoomLevel: fitZoom,
                    panOffset: { x: panX, y: panY },
                    rotation: 0
                }
            };

        case 'SET_DRAGGED_ITEM':
            return {
                ...state,
                draggedItem: action.payload
            };
        case 'SET_PDF_FILE':
            return {
                ...state,
                pdfFile: action.payload
            };
        case 'SET_IMAGE_DIMENSIONS':
            return {
                ...state,
                imageDimensions: action.payload
            };

        case 'SET_TOUCH':
            return {
                ...state,
                touch: { ...state.touch, ...action.payload }
            };

        case 'CROP_TO_VIEWPORT':
            // Calculate visible area and create cropped image
            return {
                ...state,
                viewport: {
                    ...state.viewport,
                    zoomLevel: 1,
                    panOffset: { x: 0, y: 0 },
                },
                imageDimensions: action.payload.newDimensions
            };

        case 'UPDATE_AFTER_CROP':
            return {
                ...state,
                imageDimensions: action.payload.imageDimensions
            };

        default:
            return state;
    }
}

const calculateVisibleRegion = (viewport, imageDimensions, containerWidth, containerHeight) => {
    const { zoomLevel, panOffset } = viewport;

    // Calculate the visible area in original image coordinates
    const visibleWidth = containerWidth / zoomLevel;
    const visibleHeight = containerHeight / zoomLevel;

    // Calculate top-left corner of visible area
    const visibleX = Math.max(0, -panOffset.x / zoomLevel);
    const visibleY = Math.max(0, -panOffset.y / zoomLevel);

    // Clamp to image bounds
    const clampedX = Math.max(0, Math.min(imageDimensions.width - visibleWidth, visibleX));
    const clampedY = Math.max(0, Math.min(imageDimensions.height - visibleHeight, visibleY));
    const clampedWidth = Math.min(visibleWidth, imageDimensions.width - clampedX);
    const clampedHeight = Math.min(visibleHeight, imageDimensions.height - clampedY);

    return {
        x: clampedX,
        y: clampedY,
        width: clampedWidth,
        height: clampedHeight
    };
};

const getImageRelativePosition = (clientX, clientY, imageRef, viewport, imageDimensions) => {
    if (!imageRef.current) return { x: 0, y: 0 };

    const imageRect = imageRef.current.getBoundingClientRect();

    // Step 1: Convert screen coordinates to container coordinates
    const containerX = clientX - imageRect.left;
    const containerY = clientY - imageRect.top;

    // Step 2: Account for pan offset
    const panAdjustedX = (containerX - viewport.panOffset.x) / viewport.zoomLevel;
    const panAdjustedY = (containerY - viewport.panOffset.y) / viewport.zoomLevel;

    // Step 3: Account for rotation (inverse transform)
    if (viewport.rotation !== 0) {
        // Get image center
        const centerX = imageDimensions.width / 2;
        const centerY = imageDimensions.height / 2;

        // Translate to origin (center of image)
        const translatedX = panAdjustedX - centerX;
        const translatedY = panAdjustedY - centerY;

        // Apply inverse rotation
        const radians = (-viewport.rotation * Math.PI) / 180; // Negative for inverse
        const cos = Math.cos(radians);
        const sin = Math.sin(radians);

        const rotatedX = translatedX * cos - translatedY * sin;
        const rotatedY = translatedX * sin + translatedY * cos;

        // Translate back
        return {
            x: rotatedX + centerX,
            y: rotatedY + centerY
        };
    }

    return { x: panAdjustedX, y: panAdjustedY };
};

const ViewportControls = React.memo(({ state, dispatch, containerRef, onCropToViewport, isCropping }) => {
    const handleFitToView = useCallback(() => {
        if (containerRef.current) {
            const rect = containerRef.current.getBoundingClientRect();
            dispatch({
                type: 'FIT_TO_VIEW',
                payload: {
                    containerWidth: rect.width,
                    containerHeight: rect.height
                }
            });
        }
    }, [dispatch, containerRef]);
    const [isTourActive, setIsTourActive] = React.useState(true);
    const [currentStep, setCurrentStep] = React.useState(0);
    const zoomControlsRef = useRef(null);
    const rotationControlsRef = useRef(null);
    const fitButtonRef = useRef(null);
    const cropButtonRef = useRef(null);
    const handleFinishTour = () => {
        setIsTourActive(false);
        setCurrentStep(0);
    };

    const handleNextStep = () => {
        if (currentStep < tourSteps.length - 1) {
            setCurrentStep(currentStep + 1);
        } else {
            handleFinishTour();
        }
    };

    const handlePrevStep = () => {
        if (currentStep > 0) {
            setCurrentStep(currentStep - 1);
        }
    };
    const tourSteps = [
        {
            title: 'Zoom-Steuerung',
            content: 'Verwenden Sie diese Tasten zum Vergrößern und Verkleinern. Sie können auch den Mausrad verwenden.',
        },
        {
            title: 'Rotations-Steuerung',
            content: 'Drehen Sie das Bild um 90° nach links oder rechts.',
        },
        {
            title: 'Ansicht anpassen',
            content: 'Passt das Bild automatisch an die Fenstergröße an.',
        },
        {
            title: 'Zuschneiden',
            content: 'Schneidet das Bild auf den aktuell sichtbaren Bereich zu.',
        },
    ];
    const zoomPercentage = Math.round(state.viewport.zoomLevel * 100);
    const canZoomIn = state.viewport.zoomLevel < MAX_ZOOM;
    const canZoomOut = state.viewport.zoomLevel > MIN_ZOOM;

    return (
        <div style={{
            position: 'absolute',
            top: '12px',
            right: '12px',
            display: 'flex',
            flexDirection: 'column',
            gap: '8px',
            zIndex: 1000
        }}>
            {/* Help Button */}
            <div style={{ display: 'flex', justifyContent: 'flex-end' }}>
                <EuiButtonIcon
                    iconType="questionInCircle"
                    color="primary"
                    aria-label="Hilfe anzeigen"
                    onClick={() => {
                        setIsTourActive(true);
                        setCurrentStep(0);
                    }}
                    display="fill"
                    size="s"
                />
            </div>

            {/* Zoom Controls with Tour */}
            <EuiTourStep
                content={
                    <div>
                        <EuiText size="s">
                            <p>{tourSteps[0].content}</p>
                        </EuiText>
                    </div>
                }
                isStepOpen={isTourActive && currentStep === 0}
                minWidth={300}
                onFinish={handleFinishTour}
                step={1}
                stepsTotal={tourSteps.length}
                title={tourSteps[0].title}
                anchorPosition="leftCenter"
                footerAction={
                    <EuiFlexGroup justifyContent="spaceBetween">
                        <EuiFlexItem grow={false}>
                            <EuiButton size="s" color="text" onClick={handleFinishTour}>
                                Überspringen
                            </EuiButton>
                        </EuiFlexItem>
                        <EuiFlexItem grow={false}>
                            <EuiButton size="s" fill onClick={handleNextStep}>
                                Weiter
                            </EuiButton>
                        </EuiFlexItem>
                    </EuiFlexGroup>
                }
            >
                <div
                    ref={zoomControlsRef}
                    style={{
                        display: 'flex',
                        backgroundColor: 'white',
                        borderRadius: '6px',
                        boxShadow: '0 2px 8px rgba(0,0,0,0.15)',
                        overflow: 'hidden',
                        border: '1px solid #e1e5e9'
                    }}
                >
                    <button
                        onClick={() => dispatch({ type: 'ZOOM_OUT' })}
                        disabled={!canZoomOut}
                        style={{
                            padding: '8px 12px',
                            border: 'none',
                            backgroundColor: 'white',
                            cursor: canZoomOut ? 'pointer' : 'not-allowed',
                            opacity: canZoomOut ? 1 : 0.5,
                            fontSize: '16px',
                            fontWeight: 'bold'
                        }}
                        title="Zoom Out (Ctrl + -)"
                    >
                        −
                    </button>

                    <div style={{
                        padding: '8px 16px',
                        backgroundColor: '#f8f9fa',
                        display: 'flex',
                        alignItems: 'center',
                        fontSize: '13px',
                        fontWeight: '500',
                        minWidth: '60px',
                        justifyContent: 'center',
                        borderLeft: '1px solid #e1e5e9',
                        borderRight: '1px solid #e1e5e9'
                    }}>
                        {zoomPercentage}%
                    </div>

                    <button
                        onClick={() => dispatch({ type: 'ZOOM_IN' })}
                        disabled={!canZoomIn}
                        style={{
                            padding: '8px 12px',
                            border: 'none',
                            backgroundColor: 'white',
                            cursor: canZoomIn ? 'pointer' : 'not-allowed',
                            opacity: canZoomIn ? 1 : 0.5,
                            fontSize: '16px',
                            fontWeight: 'bold'
                        }}
                        title="Zoom In (Ctrl + +)"
                    >
                        +
                    </button>
                </div>
            </EuiTourStep>

            {/* Rotation Controls with Tour */}
            <EuiTourStep
                content={
                    <div>
                        <EuiText size="s">
                            <p>{tourSteps[1].content}</p>
                        </EuiText>
                    </div>
                }
                isStepOpen={isTourActive && currentStep === 1}
                minWidth={300}
                onFinish={handleFinishTour}
                step={2}
                stepsTotal={tourSteps.length}
                title={tourSteps[1].title}
                anchorPosition="leftCenter"
                footerAction={
                    <EuiFlexGroup justifyContent="spaceBetween">
                        <EuiFlexItem grow={false}>
                            <EuiButton size="s" color="text" onClick={handlePrevStep}>
                                Zurück
                            </EuiButton>
                        </EuiFlexItem>
                        <EuiFlexItem grow={false}>
                            <EuiButton size="s" fill onClick={handleNextStep}>
                                Weiter
                            </EuiButton>
                        </EuiFlexItem>
                    </EuiFlexGroup>
                }
            >
                <div
                    ref={rotationControlsRef}
                    style={{
                        display: 'flex',
                        backgroundColor: 'white',
                        borderRadius: '6px',
                        boxShadow: '0 2px 8px rgba(0,0,0,0.15)',
                        overflow: 'hidden',
                        border: '1px solid #e1e5e9'
                    }}
                >
                    <button
                        onClick={() => dispatch({ type: 'ROTATE_LEFT' })}
                        style={{
                            padding: '8px 12px',
                            border: 'none',
                            backgroundColor: 'white',
                            cursor: 'pointer',
                            fontSize: '16px'
                        }}
                        title="Rotate Left (Ctrl + ←)"
                    >
                        ↻
                    </button>

                    <div style={{
                        padding: '8px 12px',
                        backgroundColor: '#f8f9fa',
                        display: 'flex',
                        alignItems: 'center',
                        fontSize: '12px',
                        fontWeight: '500',
                        minWidth: '45px',
                        justifyContent: 'center',
                        borderLeft: '1px solid #e1e5e9',
                        borderRight: '1px solid #e1e5e9'
                    }}>
                        {state.viewport.rotation}°
                    </div>

                    <button
                        onClick={() => dispatch({ type: 'ROTATE_RIGHT' })}
                        style={{
                            padding: '8px 12px',
                            border: 'none',
                            backgroundColor: 'white',
                            cursor: 'pointer',
                            fontSize: '16px'
                        }}
                        title="Rotate Right (Ctrl + →)"
                    >
                        ↺
                    </button>
                </div>
            </EuiTourStep>

            {/* Action Controls */}
            <div style={{
                display: 'flex',
                flexDirection: 'column',
                gap: '4px'
            }}>
                {/* Fit to View Button with Tour */}
                <EuiTourStep
                    content={
                        <div>
                            <EuiText size="s">
                                <p>{tourSteps[2].content}</p>
                            </EuiText>
                        </div>
                    }
                    isStepOpen={isTourActive && currentStep === 2}
                    minWidth={300}
                    onFinish={handleFinishTour}
                    step={3}
                    stepsTotal={tourSteps.length}
                    title={tourSteps[2].title}
                    anchorPosition="leftCenter"
                    footerAction={
                        <EuiFlexGroup justifyContent="spaceBetween">
                            <EuiFlexItem grow={false}>
                                <EuiButton size="s" color="text" onClick={handlePrevStep}>
                                    Zurück
                                </EuiButton>
                            </EuiFlexItem>
                            <EuiFlexItem grow={false}>
                                <EuiButton size="s" fill onClick={handleNextStep}>
                                    Weiter
                                </EuiButton>
                            </EuiFlexItem>
                        </EuiFlexGroup>
                    }
                >
                    <button
                        ref={fitButtonRef}
                        onClick={handleFitToView}
                        style={{
                            padding: '8px 12px',
                            border: '1px solid #e1e5e9',
                            backgroundColor: 'white',
                            borderRadius: '6px',
                            cursor: 'pointer',
                            fontSize: '12px',
                            fontWeight: '500',
                            boxShadow: '0 2px 8px rgba(0,0,0,0.15)'
                        }}
                        title="Fit to View (Ctrl + 0)"
                    >
                        {dict.fitToView}
                    </button>
                </EuiTourStep>

                {/* Crop Button with Tour */}
                <EuiTourStep
                    content={
                        <div>
                            <EuiText size="s">
                                <p>{tourSteps[3].content}</p>
                            </EuiText>
                        </div>
                    }
                    isStepOpen={isTourActive && currentStep === 3}
                    minWidth={300}
                    onFinish={handleFinishTour}
                    step={4}
                    stepsTotal={tourSteps.length}
                    title={tourSteps[3].title}
                    anchorPosition="leftCenter"
                    footerAction={
                        <EuiFlexGroup justifyContent="spaceBetween">
                            <EuiFlexItem grow={false}>
                                <EuiButton size="s" color="text" onClick={handlePrevStep}>
                                    Zurück
                                </EuiButton>
                            </EuiFlexItem>
                            <EuiFlexItem grow={false}>
                                <EuiButton size="s" fill onClick={handleFinishTour}>
                                    Fertig
                                </EuiButton>
                            </EuiFlexItem>
                        </EuiFlexGroup>
                    }
                >
                    <button
                        ref={cropButtonRef}
                        onClick={onCropToViewport}
                        disabled={isCropping || (state.viewport.zoomLevel === 1 && state.viewport.panOffset.x === 0 && state.viewport.panOffset.y === 0)}
                        style={{
                            padding: '8px 12px',
                            border: '1px solid #e1e5e9',
                            backgroundColor: 'white',
                            borderRadius: '6px',
                            cursor: (isCropping || (state.viewport.zoomLevel === 1 && state.viewport.panOffset.x === 0 && state.viewport.panOffset.y === 0)) ? 'not-allowed' : 'pointer',
                            fontSize: '12px',
                            fontWeight: '500',
                            boxShadow: '0 2px 8px rgba(0,0,0,0.15)',
                            opacity: (isCropping || (state.viewport.zoomLevel === 1 && state.viewport.panOffset.x === 0 && state.viewport.panOffset.y === 0)) ? 0.5 : 1
                        }}
                        title="Crop"
                    >
                        {dict.crop}
                    </button>
                </EuiTourStep>
            </div>
        </div>
    );
});

export default function ZoomPanRotationEditor({currentPageAccessPoints, accessPointTypes, imgsrc, navigateTo, dd, zz, setPdfFile}) {
    const [state, dispatch] = useReducer(viewportReducer, initialState);
    const imageRef = useRef(null);
    const containerRef = useRef(null);
    const dragOffsetRef = useRef({ x: 0, y: 0 });

    // Handle image load
    const handleImageLoad = useCallback(() => {
        if (imageRef.current) {
            const { offsetWidth, offsetHeight } = imageRef.current;
            dispatch({
                type: 'SET_IMAGE_DIMENSIONS',
                payload: { width: offsetWidth, height: offsetHeight }
            });
            if (containerRef.current) {
                const rect = containerRef.current.getBoundingClientRect();
                dispatch({
                    type: 'FIT_TO_VIEW',
                    payload: {
                        containerWidth: rect.width,
                        containerHeight: rect.height
                    }
                });
            }
        }
    }, [state.accessPoints.length]);

    // Mouse wheel zoom
    const handleWheel = useCallback((e) => {
        if (e.ctrlKey || e.metaKey) {
            e.preventDefault();

            const rect = containerRef.current.getBoundingClientRect();
            const mouseX = e.clientX - rect.left;
            const mouseY = e.clientY - rect.top;

            // Calculate zoom factor
            const zoomFactor = -e.deltaY * ZOOM_SENSITIVITY;
            const newZoom = Math.max(MIN_ZOOM, Math.min(MAX_ZOOM, state.viewport.zoomLevel + zoomFactor));

            if (newZoom !== state.viewport.zoomLevel) {
                // Zoom towards mouse position
                const zoomRatio = newZoom / state.viewport.zoomLevel;
                const newPanX = mouseX - (mouseX - state.viewport.panOffset.x) * zoomRatio;
                const newPanY = mouseY - (mouseY - state.viewport.panOffset.y) * zoomRatio;

                dispatch({ type: 'SET_ZOOM', payload: newZoom });
                dispatch({ type: 'SET_PAN', payload: { x: newPanX, y: newPanY } });
            }
        }
    }, [state.viewport.zoomLevel, state.viewport.panOffset]);

    // Pan controls
    const handleMouseDown = useCallback((e) => {
        if (e.button === 1 || (e.button === 0 )) {
            e.preventDefault();
            dispatch({
                type: 'START_PAN',
                payload: { x: e.clientX, y: e.clientY }
            });
        }
    }, []);

    const handleMouseMove = useCallback((e) => {
        if (state.viewport.isPanning) {
            dispatch({
                type: 'UPDATE_PAN',
                payload: { x: e.clientX, y: e.clientY }
            });
        }
    }, [state.viewport.isPanning]);

    const handleMouseUp = useCallback(() => {
        if (state.viewport.isPanning) {
            dispatch({ type: 'END_PAN' });
        }
    }, [state.viewport.isPanning]);

    // Touch controls
    const getTouchDistance = useCallback((touch1, touch2) => {
        const dx = touch1.clientX - touch2.clientX;
        const dy = touch1.clientY - touch2.clientY;
        return Math.sqrt(dx * dx + dy * dy);
    }, []);

    const handleTouchStart = useCallback((e) => {
        if (e.touches.length === 2) {
            e.preventDefault();
            const distance = getTouchDistance(e.touches[0], e.touches[1]);
            dispatch({
                type: 'SET_TOUCH',
                payload: {
                    initialDistance: distance,
                    initialZoom: state.viewport.zoomLevel,
                    isPinching: true,
                    touchCount: 2
                }
            });
        } else if (e.touches.length === 1) {
            const touch = e.touches[0];
            dispatch({
                type: 'START_PAN',
                payload: { x: touch.clientX, y: touch.clientY }
            });
            dispatch({
                type: 'SET_TOUCH',
                payload: { touchCount: 1 }
            });
        }
    }, [getTouchDistance, state.viewport.zoomLevel]);

    const handleTouchMove = useCallback((e) => {
        if (e.touches.length === 2 && state.touch.isPinching) {
            e.preventDefault();
            const distance = getTouchDistance(e.touches[0], e.touches[1]);
            const scale = distance / state.touch.initialDistance;
            const newZoom = Math.max(MIN_ZOOM, Math.min(MAX_ZOOM, state.touch.initialZoom * scale));

            dispatch({ type: 'SET_ZOOM', payload: newZoom });
        } else if (e.touches.length === 1 && state.viewport.isPanning) {
            e.preventDefault();
            const touch = e.touches[0];
            dispatch({
                type: 'UPDATE_PAN',
                payload: { x: touch.clientX, y: touch.clientY }
            });
        }
    }, [getTouchDistance, state.touch, state.viewport.isPanning]);

    const handleTouchEnd = useCallback((e) => {
        if (e.touches.length < 2) {
            dispatch({
                type: 'SET_TOUCH',
                payload: { isPinching: false, initialDistance: null }
            });
        }
        if (e.touches.length === 0) {
            dispatch({ type: 'END_PAN' });
            dispatch({
                type: 'SET_TOUCH',
                payload: { touchCount: 0 }
            });
        }
    }, []);

    // Handle crop to viewport
    const [originalImageSrc, setOriginalImageSrc] = React.useState(null);
    const [isCropping, setIsCropping] = React.useState(false);

    const handleCropToViewport = useCallback(async () => {
        if (isCropping) return;

        setIsCropping(true);

        try {
            // Store original image source if not already stored
            if (!originalImageSrc && imageRef.current) {
                setOriginalImageSrc(imageRef.current.src);
            }

            const cropResult = await cropImageToViewport(
                imageRef,
                state.viewport,
                state.imageDimensions,
                containerRef,
                dd,
                setPdfFile
            );

            if (cropResult) {

                // Update the image source
                if (imageRef.current) {
                    imageRef.current.src = cropResult.dataUrl;
                }
                dd({
                    type: 'UPDATE_SRC',
                    payload: cropResult.dataUrl
                });
                // Update state
                dispatch({
                    type: 'UPDATE_AFTER_CROP',
                    payload: {
                        imageDimensions: cropResult.newDimensions
                    }
                });

                // Reset viewport
                dispatch({ type: 'RESET_VIEW' });
            }
        } catch (error) {
            console.error('Error cropping image:', error);
        } finally {
            setIsCropping(false);
        }
    }, [state.viewport, state.imageDimensions, state.accessPoints, originalImageSrc, isCropping]);

    // Event listeners
    useEffect(() => {
        document.addEventListener('mousemove', handleMouseMove);
        document.addEventListener('mouseup', handleMouseUp);

        return () => {
            document.removeEventListener('mousemove', handleMouseMove);
            document.removeEventListener('mouseup', handleMouseUp);
        };
    }, [handleMouseMove, handleMouseUp]);
    return (
        <div style={{
            fontFamily: 'Inter, -apple-system, BlinkMacSystemFont, sans-serif',
            height: '100vh',
            display: 'flex',
            flexDirection: 'column',
            backgroundColor: '#f5f7fa'
        }}>
            {/* Header */}
            <div style={{
                padding: '16px',
                backgroundColor: 'white',
                borderBottom: '1px solid #e1e5e9',
                display: 'flex',
                justifyContent: 'space-between',
                alignItems: 'center'
            }}>
                <div>
                    <h1 style={{ margin: '0 0 4px 0', fontSize: '20px', fontWeight: '600' }}>
                        {dict.cropText}
                    </h1>
                </div>
                <EuiButton
                    color={'success'}
                    onClick={()=> {
                        navigateTo(SCREEN_TYPES.PROJECT_SETUP);
                    }}>
                    {dict.continue}
                </EuiButton>
            </div>

            {/* Main container */}
            <div
                ref={containerRef}
                style={{
                    flex: 1,
                    position: 'relative',
                    overflow: 'hidden',
                    cursor: state.viewport.isPanning ? 'grabbing' : 'grab'
                }}
                onWheel={handleWheel}
                onMouseDown={handleMouseDown}
                onTouchStart={handleTouchStart}
                onTouchMove={handleTouchMove}
                onTouchEnd={handleTouchEnd}
            >
                {/* Transformed content */}
                <div
                    style={{
                        position: 'absolute',
                        top: 0,
                        left: 0,
                        transform: `translate(${state.viewport.panOffset.x}px, ${state.viewport.panOffset.y}px) scale(${state.viewport.zoomLevel})`,
                        transformOrigin: '0 0',
                        transition: state.viewport.isPanning ? 'none' : 'transform 0.1s ease-out'
                    }}
                >
                    <div style={{
                        transform: `rotate(${state.viewport.rotation}deg)`,
                        transformOrigin: 'center center'
                    }}>
                        <img
                            ref={imageRef}
                            src={imgsrc}
                            alt="Floor Plan"
                            style={{
                                display: 'block',
                                userSelect: 'none',
                                maxWidth: 'none'
                            }}
                            onLoad={handleImageLoad}
                            draggable={false}
                        />
                    </div>
                </div>

                {/* Viewport Controls */}
                <ViewportControls
                    state={state}
                    dispatch={dispatch}
                    containerRef={containerRef}
                    onCropToViewport={handleCropToViewport}
                    isCropping={isCropping}
                />
            </div>
        </div>
    );
}
