"use client";

// React Dependencies
import React, { useEffect, useRef, useState } from "react";
// NextJs Dependencies
import Image from "next/image";
// Own Dependencies
import { DataSet } from "vis-data";
import { Network } from "vis-network";
import getDict from "@/lib/dict";

export type NodeType = "server" | "antenna" | "device";
export type ConnectionQuality = "bad" | "moderate" | "good";

export interface Node {
  id: string;
  label: string;
  type: NodeType;
  source: string;
  name: string;
  uuid: string;
}

export interface Link {
  from: string;
  to: string;
  width: number;
  color: {
    color: ConnectionQuality;
    highlight: string;
    hover: string;
    inherit: boolean;
  };
  source: string;
  target: string;
  type?: ConnectionQuality;
}

export interface Data {
  nodes: Node[];
  links: Link[];
}
interface Position {
  x: number;
  y: number;
}

interface MapProps {
  data: Data;
  title: string;
}

export default function Map({ data, title }: MapProps) {
  const dict = getDict("de");
  const containerRef = useRef<HTMLDivElement | null>(null);
  const networkRef = useRef<Network | null>(null);
  const nodesRef = useRef<DataSet<any>>(new DataSet());
  const edgesRef = useRef<DataSet<any>>(new DataSet());
  const [dimensions, setDimensions] = useState({ width: 0, height: 0 });
  const [isInitialRender, setIsInitialRender] = useState(true);

  const colors = {
    node: {
      server: "var(--primary)",
      antenna: "var(--primary-lighter-2)",
      device: "var(--primary-lighter-5)",
    },
    link: {
      bad: "#F4343D",
      moderate: "#EEB868",
      good: "#06D6A0",
    },
  };
  const legendDict = {
    server: dict.server,
    antenna: dict.antennas,
    device: dict.devices,
  };

  const positionServersInCenter = () => {
    if (!networkRef.current || !containerRef.current) return;

    const { width, height } = dimensions;
    const serverNodes = data.nodes.filter((node) => node.type === "server");

    if (serverNodes.length > 0) {
      const centerX = width / 2;
      const centerY = height / 2;
      const radius = Math.min(50, Math.max(20, 20 * serverNodes.length));

      serverNodes.forEach((node, index) => {
        const angle = (index * 2 * Math.PI) / Math.max(1, serverNodes.length);
        const x = centerX + (serverNodes.length > 1 ? radius * Math.cos(angle) : 0);
        const y = centerY + (serverNodes.length > 1 ? radius * Math.sin(angle) : 0);

        nodesRef.current.update({
          id: node.uuid,
          x,
          y,
          fixed: { x: true, y: true },
        });
      });
    }
  };

  useEffect(() => {
    const updateDimensions = () => {
      if (containerRef.current) {
        setDimensions({
          width: containerRef.current.clientWidth,
          height: containerRef.current.clientHeight,
        });
      }
    };

    updateDimensions();
    window.addEventListener("resize", updateDimensions);
    return () => window.removeEventListener("resize", updateDimensions);
  }, []);

  useEffect(() => {
    if (!containerRef.current || networkRef.current) return;

    const options = {
      physics: { enabled: false },
      layout: { improvedLayout: true },
      nodes: {
        shape: "image",
      },
      edges: {
        smooth: true,
      },
      interaction: {
        dragNodes: true,
        dragView: true,
        zoomView: true,
        navigationButtons: true,
      },
    };

    networkRef.current = new Network(
      containerRef.current,
      { nodes: nodesRef.current, edges: edgesRef.current },
      options
    );

    networkRef.current.on("doubleClick", () => {
      networkRef.current?.fit({
        animation: { duration: 1000, easingFunction: "easeInOutQuad" },
      });
    });
  }, []);

  useEffect(() => {
    if (!data || !networkRef.current) return;

    const nodePositions: Record<string, Position> = {};
    nodesRef.current.forEach((node: { id: string | number }) => {
      const id = String(node.id);
      const pos = networkRef.current!.getPositions(id)[id];
      if (pos) nodePositions[id] = pos;
    });

    const newNodes = data.nodes.map((node) => {
      const id = node.uuid;
      const existingPos = nodePositions[id];
      const size =
          node.type === "server" ? 35 : node.type === "antenna" ? 25 : 20;

      return {
        id,
        label: node.name,
        group: node.type,
        shape: "image",
        image: `/icons/${node.type}s-filled.svg`,
        size,
        font: { size: 14 },
        fixed: node.type === "server" ? { x: true, y: true } : false,
        level: node.type === "server" ? 0 : node.type === "antenna" ? 1 : 2,
        mass: node.type === "server" ? 3 : node.type === "antenna" ? 2 : 1,
        ...(existingPos && node.type !== "server"
          ? { x: existingPos.x, y: existingPos.y }
          : {}),
      };
    });

    const newEdges = data.links.map((link) => {
      const type = link.type || link.color?.color || "moderate";
      return {
        id: `${link.source}-${link.target}`,
        from: link.source,
        to: link.target,
        color: {
          color: colors.link[type],
          highlight: colors.link[type],
          hover: colors.link[type],
          inherit: false,
        },
        width: link.width || 2,
        smooth: true,
      };
    });

    const currentNodeIds = nodesRef.current.getIds();
    const currentEdgeIds = edgesRef.current.getIds();

    const newNodeIds = new Set(newNodes.map((n) => n.id));
    const newEdgeIds = new Set(newEdges.map((e) => e.id));

    currentNodeIds.forEach((id: string | number) => {
      if (!newNodeIds.has(String(id))) nodesRef.current.remove(id);
    });

    currentEdgeIds.forEach((id: string | number) => {
      if (!newEdgeIds.has(String(id))) edgesRef.current.remove(id);
    });

    nodesRef.current.update(newNodes);
    edgesRef.current.update(newEdges);

    // Position servers immediately
    positionServersInCenter();

    const hasNewNodes = newNodes.some((n) => !nodePositions[n.id]);
    const hasChange = hasNewNodes || newNodes.length !== Object.keys(nodePositions).length;

    if (hasChange || isInitialRender) {
      networkRef.current.setOptions({ physics: { enabled: true } });
      networkRef.current.stabilize();

      setTimeout(() => {
        networkRef.current?.setOptions({ physics: { enabled: false } });
        networkRef.current?.fit({
          animation: {
            duration: 500,
            easingFunction: "easeOutQuad",
          },
        });
        if (isInitialRender) setIsInitialRender(false);
      }, 1000);
    }

    networkRef.current.on("dragEnd", function (params: { nodes: any[]; }) {
      if (!networkRef.current || !containerRef.current) return;

      const positions = networkRef.current.getPositions(params.nodes);
      const container = containerRef.current;
      const padding = 20;

      const minX = -container.clientWidth / 2 + padding;
      const maxX = container.clientWidth / 2 - padding;
      const minY = -container.clientHeight / 2 + padding;
      const maxY = container.clientHeight / 2 - padding;

      params.nodes.forEach((nodeId) => {
        const pos = positions[nodeId];
        if (pos) {
          const clampedX = Math.max(minX, Math.min(maxX, pos.x));
          const clampedY = Math.max(minY, Math.min(maxY, pos.y));
          nodesRef.current.update({
            id: nodeId,
            x: clampedX,
            y: clampedY,
          });
        }
      });
    });
  }, [data, isInitialRender, dimensions]);

  useEffect(() => {
    if (dimensions.width > 0 && dimensions.height > 0) {
      positionServersInCenter();
    }
  }, [dimensions]);

  const fitNetworkView = () => {
    positionServersInCenter();
    networkRef.current?.fit({
      animation: {
        duration: 1000,
        easingFunction: "easeInOutQuad",
      },
    });
  };

  return (
    <div className="flex flex-col gap-4 bg-background-darker-1 p-8 w-full">
      <div className="flex justify-between items-center">
        <h3>{title}</h3>
        <button
          onClick={fitNetworkView}
          className="text-xs bg-background-darker-3 hover:bg-background-darker-4 text-white py-1 px-3 rounded-md transition-colors"
        >
          {dict.fit}
        </button>
      </div>
      <div id="legend" style={{ display: "flex", gap: "1rem", marginTop: "1rem" }}>
        {Object.keys(colors.node).map((type) => (
          <div key={type} className="flex items-center gap-1">
            <Image
              src={`/icons/${type}s-filled.svg`}
              alt={type}
              width={20}
              height={20}
            />
            <span style={{ color: colors.node[type as NodeType] }}>
              {legendDict[type as NodeType]}
            </span>
          </div>
        ))}
      </div>
      <div
        ref={containerRef}
        style={{ height: "400px", width: "100%" }}
        className="h-96 w-full"
      />
    </div>
  );
}
