import Image from "next/image";
import Link from "next/link";
import styles from "./Navbar.module.css";
import branding from "@/lib/branding";
import { getFirstPathElement, MenuItem } from "./Navigation";
import getDict from "@/lib/dict";
import Icon from "../Icon";
import React from "react";
import { usePathname } from "next/navigation";
import Cookies from "js-cookie";
import Button from "../form/Button";
import { useAuth } from "@/hooks/useAuth";
import { EuiButtonEmpty } from "@elastic/eui";

export default function Navbar(props: { menuItems: MenuItem[] }) {
  const dict = getDict("de");
  const pathName = usePathname();
  const { logout } = useAuth();

  const [menuOpen, setMenuOpen] = React.useState(false);
  const [userEmail, setUserEmail] = React.useState<string | null>(null);

  function toggleMenu() {
    setMenuOpen(!menuOpen);
  }

  // Get user email from cookie
  React.useEffect(() => {
    const savedEmail = Cookies.get('userEmail'); // Access email from cookies
    if (savedEmail) {
      setUserEmail(savedEmail);
    }
  }, []);

  // Create an event listener that closes the menu when screen size is larger than 800.
  // Reason: If a user opens the menu and then resizes the window the menu close button would disappear,
  // making it impossible to close the menu.
  React.useEffect(() => {

    const handleResize = () => {
      if (window.innerWidth > 800) {
        setMenuOpen(false);
      }
    }

    window.addEventListener('resize', handleResize);

    return () => {
      window.removeEventListener('resize', handleResize);
    };

  }, []);

  return (
    <nav className={`${styles.nav} ${menuOpen ? styles.navMenuOpen : ""}`}>
      <div className={styles.topBar}>
        <Link href="/" className={styles.logo} scroll={false}>
          <Image
            src="/logo.svg"
            alt="Logo"
            width={30}
            height={30}
            priority
          />
          <h1>{branding.productName}</h1>
        </Link>
        <div className={styles.rightHand}>
          <div className={styles.userInfo}>
            <p>{dict.loggedInAs}: {userEmail}</p>
            <EuiButtonEmpty iconType={"exit"} color={"text"} onClick={logout}>{dict.logout}</EuiButtonEmpty>
          </div>
          <button className={styles.menuButton} onClick={toggleMenu}>
            {menuOpen ? <Icon name="close" /> : <Icon name="menu" />}
          </button>
        </div>
      </div>
      {menuOpen ?
        <div className={styles.contentMenuOpen}>

          <div className={styles.userInfo}>
            <p>{dict.loggedInAs}: {userEmail}</p>
          </div>
          <Button onClick={logout}>{dict.logout}</Button>
          <hr />

          {props.menuItems.map(item => {
            const isActive = getFirstPathElement(pathName) === getFirstPathElement(item.href);

            return (
              <Link
                key={item.href}
                href={item.href}
                className={`${styles.menuLink} ${isActive ? styles.activeMenuLink : ""}`}
                scroll={false}
              >
                <Icon name={item.icon.name} color={isActive ? "var(--primary-text)" : undefined}/>
                {item.text}
              </Link>
            );
          }
          )}
        </div>
        : null}
    </nav>
  );
}
