"use client"

// React Dependencies
import React from "react";
// NextJs Dependencies
import { useRouter } from 'next/navigation';
// External Libraries Dependencies
import { EuiBadge, EuiBasicTableColumn, EuiHealth, EuiFlexGroup, EuiFlexItem, EuiLink } from "@elastic/eui";
// Own Dependencies
import Device from "@/app/types/Device";
import Icon from "@/app/components/Icon";
import { CheckQuality}  from "@/app/components/QualityCheck";
import { TrafficConversion } from "@/app/components/TrafficConversion";
import { capitalize } from "@/lib/capitalize";
import getDict from "@/lib/dict";

const dict = getDict("de");
export const DeviceColumns = (showNetworkColumn: boolean) => {
  const router = useRouter();
  const [items, setItems] = React.useState<EuiBasicTableColumn<Device>[]>([]);

  React.useEffect(() => {
    const baseColumns: EuiBasicTableColumn<Device>[] = [
      {
        field: 'type',
        name: dict.type,
        sortable: true,
        render: (item: string, device: Device) => (
          <EuiFlexGroup gutterSize="none">
            <EuiFlexItem grow={false}>
              <EuiHealth color={device.online ? 'green' : 'danger'}>
                <Icon name={device.type} />
              </EuiHealth>
            </EuiFlexItem>
            <EuiFlexItem grow={false}>
              {capitalize(device.type)}
            </EuiFlexItem>
          </EuiFlexGroup>
        ),
      },
      {
        field: 'name',
        name: dict.name,
        sortable: true,
        render: (name: string, device: Device) => (
          <span style={{ display: 'flex', alignItems: 'center', gap: '8px' }}>
            <CheckQuality value={device.signal} /> {name}
          </span>
        ),
      },
      {
        field: 'bytes_rx',
        name: dict.up,
        render: (item: number) => TrafficConversion(item, 'sortUp'),
        sortable: (item: Device) => item.bytes_rx,
      },
      {
        field: 'bytes_tx',
        name: dict.down,
        render: (item: number) => TrafficConversion(item, 'sortDown'),
        sortable: (item: Device) => item.bytes_tx,
      },
      {
        field: 'accessPointName',
        name: dict.accessPoint,
        sortable: true,
        render: (item: string, obj: Device) => obj.appName ?? '-',
      },
      {
        field: 'lastConnectedAt',
        name: dict.lastConnectedAt,
        sortable: true,
        render: (formatedDate: string, device: Device) => {
          const dateString = device.lastConnectedAt;

          // if empty
          if (!dateString) {
            return <EuiBadge>{dict.unregistered}</EuiBadge>;
          }

          const date = new Date(dateString);

          const formattedDate = date.toLocaleDateString(undefined, {
            year: 'numeric',
            month: 'long',
            day: 'numeric',
          });

          const formattedTime = date.toLocaleTimeString(undefined, {
            hour: 'numeric',
            minute: 'numeric',
            second: 'numeric',
          });

          return `${formattedDate} ${formattedTime}`;
        },
      },
      {
        name: dict.actions,
        actions: [
          {
            name: 'Clone',
            description: 'Edit',
            icon: 'pencil',
            type: 'icon',
            onClick: (device: Device) => {
              router.push(`/devices/edit/?id=${device.id}`);
            },
          },
        ],
      },
    ];

    if (showNetworkColumn) {
      const networkColumn = {
        field: 'network',
        name: dict.network,
        render: (item: string, obj: Device) => {
          if (obj.networkName !== undefined) {
            return (<EuiLink href={`/networks/edit/?id=${obj.network}`} external>{obj.networkName} </EuiLink>);
          }
          return  '-';
        },
      };

      baseColumns.splice(5, 0, networkColumn);
    }

    setItems(baseColumns);
  }, [showNetworkColumn, router]);

  return items;
};
