"use client"
// React Dependencies
import React, { Suspense, useRef } from "react";
// NextJs Dependencies
import { useRouter, useSearchParams } from "next/navigation";
// Service
import { deleteDevice, getDevice } from "@/services/device";
import { getNetwork } from "@/services/network";
// External Libraries Dependencies
import { useReactToPrint } from "react-to-print";
import QRCode from "react-qr-code";
import {
  EuiDescriptionList, EuiTitle, EuiConfirmModal, useGeneratedHtmlId,
  EuiHealth, EuiButton
} from "@elastic/eui";
import { Printer } from 'lucide-react';
// Components
import AuthCheck from "@/app/components/AuthCheck";
import Navigation from "@/app/components/navigation/Navigation";
import Icon from "@/app/components/Icon";
import { useToast } from "@/app/components/ToastProvider";
import { CheckQuality } from "@/app/components/QualityCheck";
import { TrafficConversion } from "@/app/components/TrafficConversion";
// Own Dependencies
import getDict from "@/lib/dict";
import Device from "@/app/types/Device";
import Network from "@/app/types/Network";
import { ApiError, getErrorCode } from "@/lib/errorsCodes";
import { formatDateTime } from "@/lib/formatDate";

import styles from './page.module.css';
import { PrintComponent } from "./print";

function DeviceDetailsContent() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const id = searchParams.get("id");
  const dict = getDict("de");
  const [device, setDevice] = React.useState<Device | null>(null);
  const [network, setNetwork] = React.useState<Network | null>(null);
  const { addToast } = useToast();

  React.useEffect(() => {

    async function fetchDevice() {
      if (!id) return;

      try {
        const response = await getDevice(id);

        if (!response.ok) {
          const errorCode = getErrorCode('DEVICE_EDIT_EXCEPTION', response.status);
          addToast({
            title: errorCode,
            color: "danger",
            text: `${dict.deviceFetchToast} ${id}`,
          });
        }

        const data = await response.json();
        setDevice(data);
        if (data.network) fetchNetwork(data.network);
      } catch (err) {
        const error = err as ApiError;
        const errorCode = getErrorCode('DEVICE_EXCEPTION', error.status);
        addToast({
          title: errorCode,
          color: "danger",
          text: `${dict.deviceFetchToast} ${id}`,
        });
        router.push("/devices");
      }
    }

    async function fetchNetwork(id: string) {
      if (!id) return;

      try {
        const response = await getNetwork(id);

        if (!response.ok) {
          const errorCode = getErrorCode('NETWORK_FETCH', response.status);
          addToast({
            title: errorCode,
            color: "danger",
            text: `${dict.networkFetchToastFail} ${id}`,
          });
        }

        const data = await response.json();
        setNetwork(data);
      } catch (err) {
        const error = err as ApiError;
        const errorCode = getErrorCode('NETWORK_EXCEPTION', error.status);
        addToast({
          title: errorCode,
          color: "danger",
          text: `${dict.networkFetchToastFail} ${id}`,
        });
      }
    }

    const interval = setInterval(() => {
      fetchDevice();
    }, 5000);

    fetchDevice();

    return () => clearInterval(interval);
  }, [id, router]);

  async function removeDevice() {
    if (!id) return;

    try {
      const response = await deleteDevice(id);
      setIsDestroyModalVisible(false);
      if (!response.ok) {
        const errorCode = getErrorCode('DEVICE_FETCH', response.status);
        addToast({
          title: errorCode,
          color: "danger",
          text: `${dict.deviceDeleteToast} ${id}`,
        });
      } else {
        addToast({
          title: "Success",
          color: "success",
          text: `${dict.deviceDeleted}`,
        });
      }
      router.push("/devices");
    } catch (err) {
      const error = err as ApiError;
      const errorCode = getErrorCode('DEVICE_EXCEPTION', error.status);
      addToast({
        title: errorCode,
        color: "danger",
        text: `${dict.deviceFetchToastFail} ${id}`,
      });
    }
  }
  const items = device?.type ? ([
    {
      title: dict.type,
      description:
        <>
          <EuiHealth color={device.online ? 'green' : 'danger'}>
            <Icon name={device.type} /></EuiHealth>
        </>
    },
    {
      title: dict.signal,
      description: <CheckQuality value={device.signal} />,
    },
    {
      title: dict.name,
      description: device.name ?? "",
    },
    {
      title: dict.up,
      description: TrafficConversion(device.bytes_rx, 'sortUp') ?? "",
    },
    {
      title: dict.down,
      description: TrafficConversion(device.bytes_tx, 'sortDown') ?? "",
    },
    {
      title: dict.network,
      description: network?.name ?? "",
    },
    // {
    //   title: dict.addedBy,
    //   description: device.addedBy ?? "",
    // },
    {
      title: dict.addedAt,
      description: formatDateTime(device.addedAt)
    },
    {
      title: dict.password,
      description: device.password ?? "",
    },
    {
      title: dict.password + ' QR-Code:',
      description: <>{device.password && network ?
        <QRCode
          value={`WIFI:S:${network.name};T:WPA;P:${device.password};H:false;`}
          style={{maxWidth: "10rem", height: "auto"}}
        />
        : ""}</>,
    },
    {
      title: dict.firstConnectedAt,
      description: formatDateTime(device.firstConnectedAt)
    },
    {
      title: dict.lastConnectedAt,
      description: formatDateTime(device.lastConnectedAt),
    },
    {
      title: dict.macAddress,
      description: device.mac || "N/A",
    },
    // {
    //   title: dict.ipAddress,
    //   description: device.ipAddress ?? "",
    // },
  ]) : [];
  const [isDestroyModalVisible, setIsDestroyModalVisible] = React.useState(false);
  const destroyModalTitleId = useGeneratedHtmlId();
  const closeDestroyModal = () => setIsDestroyModalVisible(false);
  const showDestroyModal = () => setIsDestroyModalVisible(true);

  const contentRef = useRef<HTMLDivElement>(null);
  const reactToPrintFn = useReactToPrint({ contentRef });

  return (
    <div>
      <EuiTitle>
        <h2>{dict.editDevice}</h2>
      </EuiTitle>
      <EuiButton iconType={"arrowLeft"} color={"text"} href={'/devices'}/>


      {device && <> <EuiButton iconType={() => <Printer />} onClick={() => reactToPrintFn()} color={"text"}/> </>}
      <div ref={contentRef} className={styles.printOnly} >
        <PrintComponent data={items}/>
      </div>

      <EuiDescriptionList
        type="column"
        className="print-section"
        listItems={items}
        style={{ maxWidth: '400px' }}
      />
      {device && <>
        <EuiButton onClick={showDestroyModal} color={'danger'} fill>{dict.remove}</EuiButton>
      </>}
      {isDestroyModalVisible &&
        <EuiConfirmModal
          aria-labelledby={destroyModalTitleId}
          title={`${dict.remove} ${device?.name} ?`}
          titleProps={{id: destroyModalTitleId}}
          onCancel={closeDestroyModal}
          onConfirm={removeDevice}
          cancelButtonText={`${dict.cancel}`}
          confirmButtonText={`${dict.confirm}`}
          buttonColor="danger"
          defaultFocusedButton="confirm"
        >
        </EuiConfirmModal>}
    </div>
  );
}

export default function DeviceDetails() {
  return (
    <AuthCheck>
      <Navigation>
        <Suspense fallback={<div>Loading...</div>}>
          <DeviceDetailsContent />
        </Suspense>
      </Navigation>
    </AuthCheck>
  );
}
