"use client"

// React Dependencies
import React from "react";
// Custom Hooks
import { useDomReady } from "@/hooks/useDomReady";
import { useAuth } from "@/hooks/useAuth";
// Services
import { getDevices } from "@/services/device";
import { getNetworks } from "@/services/network";
import { getAccessPoints } from "@/services/accessPoints";
// Context
import { useLoader } from "@/context/LoaderContext";
// External Libraries Dependencies
import {
  EuiTitle,
  EuiFlexItem,
  EuiButton, EuiFlexGroup,
  EuiLoadingSpinner, EuiSwitch
} from '@elastic/eui';
// Components
import AuthCheck from "@/app/components/AuthCheck";
import Navigation from "@/app/components/navigation/Navigation";
import { useToast } from "@/app/components/ToastProvider";
import { AddModal } from "@/app/components/AddModal";
import { DeviceTable } from "@/app/devices/deviceTable";
// Own Dependencies
import { ApiError, getErrorCode } from "@/lib/errorsCodes";
import getDict from "@/lib/dict";
import Device  from "../types/Device";
// Own Styles
import styles from "./page.module.css";
import Network from "../types/Network";
import AP from "../types/AP";


const dict = getDict("de");

export default function Devices() {

  const [sortedData, setSortedData] = React.useState<Device[]>([]);
  const { addToast } = useToast();
  const { setLoading } = useLoader();
  const domReady = useDomReady();
  const { isAuthenticated } = useAuth();
  const [auto, setAuto] = React.useState(true);

  React.useEffect(() => {
    if (!domReady || !isAuthenticated) return;
    // Prevents state updates if the component unmounts
    let isMounted = true;
    const fetchDeviceWithAccessPoint = async () => {
      try {
        setLoading(true);
        const response = await getDevices();
        if (!response.ok) {
          const errorCode = getErrorCode('DEVICE_FETCH', response.status);
          addToast({
            title: errorCode,
            color: "danger",
            text: `${dict.deviceFetchToastFail}`,
          });
          setLoading(false);
          return;
        } else {
          const deviceData = await response.json();

          const networkResponse = await getNetworks();
          const networkList = await networkResponse.json();

          const accessPointResponses = await getAccessPoints();
          const accessPointList = await accessPointResponses.json();


          const deviceWithAPs = deviceData.map((device: Device) => ({
            ...device,
            networkName: networkList.find((item: Network) => item.id === device.network).name,
            appName: device.currentAP ? accessPointList.find((item: AP) => item.id === device.currentAP).name : '-',
          }));

          setLoading(false);
          if (isMounted) {
            setSortedData(deviceWithAPs);
          }
        }
      } catch (err) {
        const error = err as ApiError;
        const errorCode = getErrorCode('DEVICE_EXCEPTION', error.status);
        setLoading(false);
        addToast({
          title: errorCode,
          color: "danger",
          text: `${dict.deviceFetchToastFail}`,
        });
      }
    };
    fetchDeviceWithAccessPoint();
    let interval: NodeJS.Timeout | null = null;

    if (auto) {
      interval = setInterval(fetchDeviceWithAccessPoint, 5000);
    }
    return () => {
      isMounted = false;
      if (interval) {
        clearInterval(interval);
      }
    };
  }, [auto, domReady]);

  const [isModalVisible, setIsModalVisible] = React.useState(false);

  const showModal = () => setIsModalVisible(true);
  const closeModal = () => setIsModalVisible(false);

  if (!domReady) {
    return <EuiLoadingSpinner size="s"/>
  }
  return (
    <AuthCheck>
      <Navigation>
        <div className={styles.page}>
          <EuiFlexGroup justifyContent="spaceBetween">
            <EuiFlexItem grow={false}>
              <EuiTitle>
                <h2>{dict.devices}</h2>
              </EuiTitle>
            </EuiFlexItem>
            {sortedData && <EuiFlexItem grow={false}>
              <EuiButton color="success" iconType="plusInCircleFilled" onClick={showModal}>{dict.addDevice}</EuiButton>
            </EuiFlexItem>}
          </EuiFlexGroup>
          {isModalVisible && <AddModal closeModal={closeModal}/>}
          <EuiSwitch
            label={dict.refreshData}
            checked={auto}
            onChange={() => {
              setAuto(!auto);
              addToast({
                title: auto ? dict.refreshDataDisabled: dict.refreshDataEnabled,
                color: "primary",
              });
            }}
          />
          <DeviceTable data={sortedData} showNetworkColumn={true}/>
        </div>
      </Navigation>
    </AuthCheck>
  );
}
