"use client"
// React Dependencies
import React from "react";
// Custum hooks
import { useAuth } from "@/hooks/useAuth";
import { useDomReady } from "@/hooks/useDomReady";
// Services
import { getDevices } from "@/services/device";;
// External Libraries Dependencies
import {
  EuiFlexGroup, EuiFlexItem, EuiPanel,
  EuiStat, EuiText, EuiTextColor, EuiLoadingSpinner
} from '@elastic/eui';
import { Download, ArrowDownUp, Group, Upload, Wifi, WifiOff } from "lucide-react";
// Components
import Navigation from "@/app/components/navigation/Navigation";
import AuthCheck from "@/app/components/AuthCheck";
import { useToast } from "@/app/components/ToastProvider";
// Own Dependencies
import Device from "@/app/types/Device";
import getDict from "@/lib/dict";
import { ApiError, getErrorCode } from "@/lib/errorsCodes";
import { getUserStats } from "@/lib/userStats";
import { capitalize } from "@/lib/capitalize";

export default function Internet() {

  const dict = getDict("de");

  const domReady = useDomReady();

  const [traffic, setTrafic] = React.useState(Number);
  const [up, setUp] = React.useState(Number);
  const [down, setDown] = React.useState(Number);
  const [devices, setDevices] = React.useState<Device[]>([]);

  const { isAuthenticated } = useAuth();
  const { addToast } = useToast();
  
  const { onlineCount, totalCount } = getUserStats(devices);
  async function fetchDevice() {
    try {
      const response = await getDevices();
      if (!response.ok) {
        const errorCode = getErrorCode('DEVICE_FETCH', response.status);
        addToast({
          title: errorCode,
          color: "danger",
          text: `${dict.deviceFetchToastFail}`,
        });
      } else {
        const data = await response.json();
        const sumKb = data.reduce((acc: number, obj: Device) => acc + (obj.bytes_rx ?? 0) + (obj.bytes_tx ?? 0), 0)/1024;
        const sumUp = data.reduce((acc: number, obj: Device) => acc + (obj.bytes_tx ?? 0), 0)/1024;
        const sumDown = data.reduce((acc: number, obj: Device) => acc + (obj.bytes_rx ?? 0), 0)/1024;
        setTrafic(sumKb);
        setUp(sumUp);
        setDown(sumDown);
        setDevices(data);
      }
    } catch (err) {
      const error = err as ApiError;
      const errorCode = getErrorCode('DEVICE_EXCEPTION', error.status);
      addToast({
        title: errorCode,
        color: "danger",
        text: `${dict.deviceFetchToastFail}`,
      });
    }
  }

  React.useEffect(() => {
    if (!domReady || !isAuthenticated) return;

    const interval = setInterval(() => {
      fetchDevice();
    }, 5000);
    fetchDevice();

    return () => clearInterval(interval);
  }, [domReady]);
  if (!domReady) {
    return <EuiLoadingSpinner size="s"/>
  }
  return (
    <AuthCheck>
      <Navigation>
        <EuiFlexGroup>
          <EuiFlexItem>
            <EuiPanel hasBorder={true}>
              <EuiFlexGroup>
                <EuiText>Data</EuiText>
                <EuiFlexItem>
                  <EuiStat
                    title={(traffic).toFixed(1) + 'Mb'}
                    textAlign="center"
                    description={ <ArrowDownUp />}
                  >
                    {capitalize(dict.download)}
                  </EuiStat>
                </EuiFlexItem>
                <EuiFlexItem>
                  <EuiStat
                    title={(up).toFixed(1) + 'Mb'}
                    textAlign="center"
                    description={<Download />}
                  >
                    {capitalize(dict.download)}
                  </EuiStat>
                </EuiFlexItem>
                <EuiFlexItem>
                  <EuiStat
                    title={(down).toFixed(1) + 'Mb'}
                    description={<Upload />}
                    textAlign="center"
                  >
                    {capitalize(dict.upload)}
                  </EuiStat>
                </EuiFlexItem>
              </EuiFlexGroup>
            </EuiPanel>
          </EuiFlexItem>
          <EuiFlexItem>
            <EuiPanel hasBorder={true}>
              <EuiFlexGroup>
                <EuiText>{capitalize(dict.clientStats)}</EuiText>
                <EuiFlexItem>
                  <EuiStat
                    title={totalCount}
                    description={<EuiTextColor color={'green'}><Group /></EuiTextColor>}
                    textAlign="center"
                  >
                    Total
                  </EuiStat>
                </EuiFlexItem>
                <EuiFlexItem>
                  <EuiStat
                    title={onlineCount}
                    titleColor="success"
                    description={<EuiTextColor color={'green'}><Wifi /></EuiTextColor>}
                    textAlign="center"
                  >
                    {capitalize(dict.online)}
                  </EuiStat>
                </EuiFlexItem>
                <EuiFlexItem>
                  <EuiStat
                    title={totalCount - onlineCount}
                    titleColor="danger"
                    description={<EuiTextColor color={'danger'}><WifiOff /></EuiTextColor>}
                    textAlign="center"
                  >
                    {capitalize(dict.offline)}
                  </EuiStat>
                </EuiFlexItem>
              </EuiFlexGroup>
            </EuiPanel>
          </EuiFlexItem>
        </EuiFlexGroup>
      </Navigation>
    </AuthCheck>
  );
}
