"use client"
// React Dependencies
import React from 'react';
// NextJs Dependencies
import { useRouter } from "next/navigation";
import Image from "next/image";
// External Libraries Dependencies
import { EuiButton, EuiFieldText, EuiLoadingSpinner, EuiFieldPassword, EuiText, EuiTitle } from "@elastic/eui";
// Own Dependencies
import styles from "./page.module.css";
import getDict from "@/lib/dict";
import branding from "@/lib/branding";
import { useToast } from "@/app/components/ToastProvider";
import { getErrorCode} from "@/lib/errorsCodes";

export default function Login() {
  const dict = getDict("de");
  const router = useRouter();
  const [loading, setLoading] = React.useState(false);
  const { addToast } = useToast();
  async function handleSubmit(event?: React.MouseEvent<HTMLButtonElement>) {
    event?.preventDefault()
    setLoading(true);
    const response = await fetch("/api/auth/login", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ email, password }),
    })

    if (response.ok) {
      setLoading(false);
      router.push("/")
    } else {
      const errorCode = getErrorCode('LOGIN_EXCEPTION', response.status);
      setLoading(false);
      addToast({
        title: errorCode,
        color: "danger",
        text: dict.loginFailed,
      });
    }
  }
  const [email, setEmail] = React.useState('');
  const [password, setPassword] = React.useState('');

  const onEmailChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setEmail(e.target.value);
  };
  const onPasswordChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setPassword(e.target.value);
  };
  React.useEffect(() => {
    const listener = (event: KeyboardEvent) => {
      const target = event.target as HTMLInputElement;
      if (
        (event.code === 'Enter' || event.code === 'NumpadEnter') &&
        target.type === 'password'
      ) {
        handleSubmit();
      }
    };
    if (password.length > 6 && email.length > 4) {
      document.addEventListener('keydown', listener);
      return () => {
        document.removeEventListener('keydown', listener);
      };
    }
  }, [password, email]);
  return (
    <form className={styles.form}>
      <div className={styles.branding}>
        <Image
          src="/logo.svg"
          alt="Logo"
          width={60}
          height={60}
          priority
        />
        <div>
          <EuiTitle>
            <h1>{branding.productName}</h1>
          </EuiTitle>
          <EuiTitle>
            <EuiText>
              <h5>{branding.companyName}</h5>
            </EuiText>
          </EuiTitle>
        </div>
      </div>
      <EuiText>
        <h3 style={{textTransform: "capitalize"}}>{dict.login}</h3>
      </EuiText>
      <EuiFieldText
        placeholder={dict.email}
        value={email}
        onChange={(e: React.ChangeEvent<HTMLInputElement>) => onEmailChange(e)}
        aria-label={dict.email}
      />
      <EuiFieldPassword
        placeholder={dict.password}
        value={password}
        onChange={(e: React.ChangeEvent<HTMLInputElement>) => onPasswordChange(e)}
        aria-label={dict.password}
      />
      <EuiButton
        color={'danger'}
        isDisabled={password.length < 6 || email.length < 4}
        fill
        onClick={handleSubmit}
      >
        Login {loading ? <EuiLoadingSpinner size="l" /> : undefined}
      </EuiButton>
    </form>
  )
}
