"use client"

// React Dependencies
import React from "react";
// External Libraries Dependencies
import {
  EuiConfirmModal,
  EuiTitle,
  EuiDescriptionList,
  useGeneratedHtmlId,
  EuiText,
  EuiBetaBadge,
  EuiButton,
  EuiPanel
} from "@elastic/eui";
// Own Dependencies
import styles from "./page.module.css";
import getDict from "@/lib/dict";
import Network from "@/app/types/Network";
import { NetworkManagementFunctions } from "./page";
import Device from "@/app/types/Device";
import { useToast } from "@/app/components/ToastProvider";
import Stats from "@/app/components/Stats";
import { DeviceTable } from "@/app/devices/deviceTable";
import { AddModal } from "@/app/components/AddModal";
import {ApiError, getErrorCode} from "@/lib/errorsCodes";
import { useLoader } from "@/context/LoaderContext";
import { apiFetch } from "@/lib/api";

const dict = getDict("de");
export default function BusinessNetworkDetails(props: { network: Network, managementFunctions: NetworkManagementFunctions }) {

  const { addToast } = useToast();
  const [devices, setDevices] = React.useState<Device[]>([]);
  const { setLoading } = useLoader();

  React.useEffect(() => {
    let isMounted = true;
    const fetchDeviceWithAccessPoint = async () => {
      try {
        setLoading(true);
        const response = await apiFetch(`/networks/${props.network.id}/devices`);
        if (!response.ok) {
          const errorCode = getErrorCode('NETWORK_FETCH', response.status);
          setLoading(false);
          addToast({
            title: errorCode,
            color: "danger",
            text: `${dict.deviceFetchToastFail}`,
          });
        } else {
          const deviceData = await response.json();

          const accessPointResponses = await apiFetch("/aps");
          const accessPointList = await accessPointResponses.json();

          const deviceWithAPs = deviceData.map((device: Device, index: number) => ({
            ...device,
            appName: accessPointList[index]?.name || "-",
          }));

          setLoading(false);
          if (isMounted) {
            setDevices(deviceWithAPs);
          }
        }
      } catch (err) {
        const error = err as ApiError;
        const errorCode = getErrorCode('DEVICE_FETCH', error.status);
        setLoading(false);
        addToast({
          title: errorCode,
          color: "danger",
          text: dict.deviceFetchToastFail,
        });
      }
    };
    if (props.network.id) {
      fetchDeviceWithAccessPoint();
    }
    const interval = setInterval(fetchDeviceWithAccessPoint, 5000);

    return () => {
      isMounted = false;
      clearInterval(interval);
    };
  }, [props.network.id]);

  const items = props.network.name ? ([
    {
      title: dict.type,
      description:<EuiBetaBadge
        label={dict[props.network.type]}
        color={'hollow'}
        size="m"
      /> ,
    },
    {
      title: dict.name,
      description: <EuiText>{props.network.name}</EuiText>,
    },
  ]) : [];
  const [isAddModalVisible, setIsAddModalVisible] = React.useState(false);
  const [isDestroyModalVisible, setIsDestroyModalVisible] = React.useState(false);
  const destroyModalTitleId = useGeneratedHtmlId();
  const closeDestroyModal = () => setIsDestroyModalVisible(false);
  const showDestroyModal = () => setIsDestroyModalVisible(true);

  return (
    <>
      <EuiPanel>
        <EuiDescriptionList
          align="left"
          listItems={items}
        />
      </EuiPanel>

      <Stats network={props.network}/>
      <div className={styles.actions}>
        <EuiButton disabled color="success" iconType="pencil"> {dict.editName}</EuiButton>

        <EuiButton onClick={() => setIsAddModalVisible(true)} color="success" iconType="plusInCircleFilled"> {dict.addDevice}</EuiButton>
        {isAddModalVisible && <AddModal closeModal={() => setIsAddModalVisible(false)} networkId={props.network.id}/>}

        <EuiButton onClick={showDestroyModal} color="danger" iconType="trash"> {dict.delete}</EuiButton>
        {isDestroyModalVisible ?
          <EuiConfirmModal
            aria-labelledby={destroyModalTitleId}
            title={`${dict.remove} ${props.network.name} ?`}
            titleProps={{id: destroyModalTitleId}}
            onCancel={closeDestroyModal}
            onConfirm={async () => {
              await props.managementFunctions.deleteNetwork(props.network.id);
              setIsDestroyModalVisible(false);
            }}
            cancelButtonText={`${dict.cancel}`}
            confirmButtonText={`${dict.confirm}`}
            buttonColor="danger"
            defaultFocusedButton="confirm"
          >
          </EuiConfirmModal>: null}
      </div>
      <EuiTitle>
        <h2>{dict.devicesInNetwork}:</h2>
      </EuiTitle>
      <DeviceTable data={devices} showNetworkColumn={false} />
    </>
  );
}
