"use client"

// React Dependencies
import React, { Suspense } from "react";
import { useRouter, useSearchParams } from "next/navigation";
// External Libraries Dependencies
import { EuiTitle } from "@elastic/eui";
// Components
import AuthCheck from "@/app/components/AuthCheck";
import Navigation from "@/app/components/navigation/Navigation";
import Button from "@/app/components/form/Button";
import Icon from "@/app/components/Icon";
import { useToast } from "@/app/components/ToastProvider";
import BusinessNetworkDetails from "./business";
import GuestNetworkDetails from "./guest";
// Own Styles
import styles from "./page.module.css";
// Other Dependencies
import getDict from "@/lib/dict";
import Network, { GuestPassword } from "@/app/types/Network";
import { ApiError, getErrorCode } from "@/lib/errorsCodes";
import { apiFetch } from "@/lib/api";

export type NetworkManagementFunctions = {
  renameNetwork: (id: string) => Promise<void>
  deleteNetwork: (id: string) => Promise<void>
}

function NetworkDetailsContent() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const id = searchParams.get("id");
  const dict = getDict("de");
  const [network, setNetwork] = React.useState<Network | null>(null);
  const [guestPassword, setGuestPassword] = React.useState<GuestPassword | null>(null);
  const { addToast } = useToast();

  const managementFunctions: NetworkManagementFunctions = {
    renameNetwork: async function (id: string): Promise<void> {
      addToast({
        title: "Warning!",
        color: "warning",
        text: `Function ${id} not implemented`,
      });
    },
    deleteNetwork: async function (id: string): Promise<void> {
      if (!id) return;

      try {
        const response = await apiFetch(`/networks/${id}`, { method: "DELETE" });

        if (!response.ok) {
          const errorCode = getErrorCode('NETWORK_DELETE', response.status);
          addToast({
            title: errorCode,
            color: "danger",
            text: `${dict.networkDeleteToast}: ${id}`,
          });
        } else {
          addToast({
            title: "",
            color: "success",
            text: `${dict.networkDeleted}`,
          });
        }
        router.push("/networks");
      } catch (err) {
        const error = err as ApiError;
        const errorCode = getErrorCode('NETWORK_EXCEPTION', error.status);
        addToast({
          title: errorCode,
          color: "danger",
          text: `${dict.networkFetchToastFail} `,
        });
      }
    }
  }

  React.useEffect(() => {

    async function fetchNetwork() {
      if (!id) return;

      try {
        const response = await apiFetch(`/networks/${id}`);

        if (!response.ok) {
          const errorCode = getErrorCode('NETWORK_FETCH', response.status);
          addToast({
            title: errorCode,
            color: "danger",
            text: `${dict.networkFetchToast}: ${id}`,
          });
        }

        const data = await response.json();
        if (data.type === "guest") {
          await fetchGuestPassword();
        }
        setNetwork(data);
      } catch (err) {
        const error = err as ApiError;
        const errorCode = getErrorCode('NETWORK_EXCEPTION', error.status);
        addToast({
          title: errorCode,
          color: "danger",
          text: `${dict.networkFetchToast}: ${id}`,
        });
        router.push("/networks");
      }
    }

    async function fetchGuestPassword() {
      if (!id) return;

      try {
        const response = await apiFetch(`/networks/${id}/guest-password`);

        if (!response.ok) {
          const errorCode = getErrorCode('NETWORK_FETCH', response.status);
          addToast({
            title: errorCode,
            color: "danger",
            text: `Failed to fetch guest password for network with ID: ${id}`,
          });
        }

        const data = await response.json();
        setGuestPassword(data);
      } catch (err) {
        const error = err as ApiError;
        const errorCode = getErrorCode('NETWORK_EXCEPTION', error.status);
        addToast({
          title: errorCode,
          color: "danger",
          text: `Failed to fetch guest password for network with ID: ${id}`,
        });
      }
    }

    const interval = setInterval(() => {
      fetchNetwork();
    }, 5000);

    fetchNetwork();

    return () => clearInterval(interval);
  }, [id, router]);

  return (
    <div className={styles.page}>
      <Button href="/networks"><Icon name="back" /></Button>
      <EuiTitle>
        <h2>{dict.editNetwork}</h2>
      </EuiTitle>
      {
        network ? <>
          {
            network.type === "guest" ?
              <>
                {
                  guestPassword ? <GuestNetworkDetails network={network} guestPassword={guestPassword} managementFunctions={managementFunctions} /> : null
                }
              </>

              : <BusinessNetworkDetails network={network} managementFunctions={managementFunctions} />
          }
        </> : null
      }
    </div>
  );
}

export default function NetworkDetails() {
  return (
    <AuthCheck>
      <Navigation>
        <Suspense fallback={<div>Loading...</div>}>
          <NetworkDetailsContent />
        </Suspense>
      </Navigation>
    </AuthCheck>
  );
}
