"use client"
// React Dependencies
import React from "react";
// Custum hooks
import { useAuth } from "@/hooks/useAuth";
import { useDomReady } from "@/hooks/useDomReady";
// Context
import { useLoader } from "@/context/LoaderContext";
// Serivce
import { getNetworks } from '@/services/network';
// External Libraries Dependencies
import {
  EuiTitle, EuiFlexItem, EuiButton,
  EuiFlexGroup, EuiLoadingSpinner, EuiSwitch
} from '@elastic/eui';
// Components
import AuthCheck from "@/app/components/AuthCheck";
import Navigation from "@/app/components/navigation/Navigation";
import { useToast } from "@/app/components/ToastProvider";
import { NetworkTable } from "@/app/networks/networkTable";
import { AddNetwork } from "@/app/networks/addNetwork";
// Own Dependencies
import getDict from "@/lib/dict";
import Network from "@/app/types/Network";
// Own Styles
import styles from "./page.module.css";
import { ApiError, getErrorCode } from "@/lib/errorsCodes";

export default function Networks() {

  const dict = getDict("de");
  
  const [networks, setNetworks] = React.useState<Network[]>([]);
  const [auto, setAuto] = React.useState(true);
  
  const domReady = useDomReady();
  
  const { addToast } = useToast();
  const { setLoading } = useLoader();
  const { isAuthenticated } = useAuth();

  async function fetchNetworks() {
    try {
      setLoading(true);
      const response = await getNetworks();
      if (!response.ok) {
        const errorCode = getErrorCode('NETWORK_FETCH', response.status);
        addToast({
          title: errorCode,
          color: "danger",
          text: `${dict.networkFetchToastFail}`,
        });
        setLoading(false);
        return;
      }

      const data = await response.json();
      setNetworks(data);
      setLoading(false);
    } catch (err) {
      const error = err as ApiError;
      setLoading(false);
      const errorCode = getErrorCode('NETWORK_EXCEPTION', error.status);
      addToast({
        title: errorCode,
        color: "danger",
        text: `${dict.networkFetchToastFail}`,
      });
    }
  }

  React.useEffect(() => {
    if (!domReady || !isAuthenticated) return;
    
    const fetchAndSetNetworks = async () => {
      await fetchNetworks();
    };
    fetchAndSetNetworks(); // Initial fetch

    let interval: NodeJS.Timeout | null = null;
    if (auto) {
      interval = setInterval(fetchAndSetNetworks, 5000);
    }
    return () => {
      if (interval) {
        clearInterval(interval)
      }
    };

  }, [auto, domReady]);

  const [isModalVisible, setIsModalVisible] = React.useState(false);

  const showModal = () => setIsModalVisible(true);
  const closeModal = () => setIsModalVisible(false);
  if (!domReady) {
    return <EuiLoadingSpinner size="s"/>
  }
  return <AuthCheck>
    <Navigation>
      <div className={styles.page}>
        <EuiFlexGroup justifyContent="spaceBetween">
          <EuiFlexItem grow={false}>
            <EuiTitle>
              <h2>{dict.networks}</h2>
            </EuiTitle>
          </EuiFlexItem>
          {networks && <EuiFlexItem grow={false}>
            <EuiButton color="success" iconType="plusInCircleFilled" onClick={showModal}>
              {dict.addNetwork}
            </EuiButton>
          </EuiFlexItem>}
        </EuiFlexGroup>
        {isModalVisible && <AddNetwork closeModal={closeModal}/>}
        <EuiSwitch
          label={dict.refreshData}
          checked={auto}
          onChange={() => {
            setAuto(!auto);
            addToast({
              title: auto ? dict.refreshDataDisabled: dict.refreshDataEnabled,
              color: "primary",
            });
          }}
        />
        <NetworkTable data={networks} />
      </div>
    </Navigation>
  </AuthCheck>
}
