import AP from '@/app/types/AP';
import { useLoader } from '@/context/LoaderContext';
import { useState, useEffect } from 'react';

export const useAccessPoints = (aps) => {
    const [accessPoints, setAccessPoints] = useState([]);
    const [draggedItem, setDraggedItem] = useState(null);
    const [allowNodeMovement, setAllowNodeMovement] = useState(true);
    const [editingNodeId, setEditingNodeId] = useState(null);
    const [editingNodeName, setEditingNodeName] = useState('');
    const [contextMenuNodeId, setContextMenuNodeId] = useState(null);
    const { globalAps } = useLoader();
    const uniqueIds = [...new Set(aps.map(ap => ap.type))];
    const accessPointLimits = {
        total: aps.length,
        perType: Object.fromEntries(uniqueIds.map(type => [type, 1])),
    };
    const accessPointTypes = aps;
    const canAddAccessPoint = (type) => {
        const totalCount = accessPoints.length;
        const typeCount = accessPoints.filter(ap => ap.type === type).length;
        const perTypeLimit = accessPointLimits.perType[type] ?? 1;
        return {
            canAdd: totalCount < accessPointLimits.total && typeCount < perTypeLimit,
            limits: {
                total: { current: totalCount, max: accessPointLimits.total },
                type: { current: typeCount, max: perTypeLimit },
            },
        };
    };
    const getLimitStatus = (type) => {
        const { canAdd, limits } = canAddAccessPoint(type);
        return {
            canAdd,
            totalProgress: (limits.total.current / limits.total.max) * 100,
            typeProgress: (limits.type.current / limits.type.max) * 100,
            messages: {
                total: `${limits.total.current}/${limits.total.max}`,
                type: `${limits.type.current}/${limits.type.max} ${type}`,
            }
        };
    };

    const generateDeviceDetails = async (id) => {
        await globalAps;
        // Find the current access point with updated status
        const currentAccessPoint = globalAps?.find(ap => id === ap.id);
        const isOnline = currentAccessPoint?.online ?? false;
        return {
            online: isOnline
        }
    };

    const removeAccessPoint = (id) => {
        setAccessPoints(accessPoints.filter(ap => ap.id !== id));
    };

    const startEditingNode = (accessPoint) => {
        setEditingNodeId(accessPoint.id);
        setEditingNodeName(accessPoint.name);
    };

    const saveNodeName = () => {
        if (editingNodeName.trim()) {
            setAccessPoints(prev => prev.map(ap =>
                ap.id === editingNodeId
                    ? { ...ap, floor: editingNodeName.trim() }
                    : ap
            ));
        }
        setEditingNodeId(null);
        setEditingNodeName('');
    };

    const cancelEditingNode = () => {
        setEditingNodeId(null);
        setEditingNodeName('');
    };

    return {
        accessPoints,
        setAccessPoints,
        accessPointTypes,
        draggedItem,
        setDraggedItem,
        allowNodeMovement,
        setAllowNodeMovement,
        editingNodeId,
        editingNodeName,
        setEditingNodeName,
        contextMenuNodeId,
        setContextMenuNodeId,
        generateDeviceDetails,
        removeAccessPoint,
        startEditingNode,
        saveNodeName,
        cancelEditingNode,
        canAddAccessPoint,
        accessPointLimits,
        getLimitStatus,
    };
};
