import { useState, useReducer } from 'react';
import { initialState, viewportReducer } from '@/store/reducer';

export const useProjectLoader = () => {
    const [state, dispatch] = useReducer(viewportReducer, initialState);

    const [isLoading, setIsLoading] = useState(false);
    const [error, setError] = useState(null);

    const loadFirstProjectFromBackend = async ({
       setStatus,
       loadProjectsFromApi,
       setAccessPoints,
       setCurrentPage,
       setPageViewports,
       setPdfPages,
       setPdfFile,
       setProjectFound,
       setLoadingError,
       addToast,
       generateDeviceDetails,
       processPdfFromFile
   }) => {
        try {
            setIsLoading(true);
            setError(null);
            setStatus(true);

            const response = await fetch(`/api/projects/first/load`);
            if (response.ok) {
                const data = await response.json();

                if (data.project) {
                    await loadProjectsFromApi();
                    dispatch({ type: 'SET_ROTATION', payload: data.project.pageRotation || 0 });

                    // Load project data and ensure all access points have device details
                    const accessPointsWithDetails = (data.project.accessPoints).map(ap => ({
                        ...ap,
                        deviceDetails: generateDeviceDetails(ap.type)
                    }));

                    setAccessPoints(accessPointsWithDetails);
                    setCurrentPage(data.project.currentPage || 0);

                    if (data.project.viewportState) {
                        // Restore page viewports
                        if (data.project.viewportState.pageViewports) {
                            setPageViewports(data.project.viewportState.pageViewports);
                        }

                        // Restore current page viewport or global viewport
                        const currentPageViewport = data.project.viewportState.pageViewports?.[data.project.currentPage || 0];
                        if (currentPageViewport) {
                            dispatch({ type: 'SET_ZOOM', payload: currentPageViewport.zoomLevel });
                            dispatch({ type: 'SET_PAN', payload: currentPageViewport.panOffset });
                        } else if (data.project.viewportState.zoomLevel && data.project.viewportState.panOffset) {
                            dispatch({ type: 'SET_ZOOM', payload: data.project.viewportState.zoomLevel });
                            dispatch({ type: 'SET_PAN', payload: data.project.viewportState.panOffset });
                            
                        }
                    } else {
                        // Reset viewport if no saved state
                        dispatch({ type: 'SET_ZOOM', payload: 1 });
                        dispatch({ type: 'SET_PAN', payload: { x: 0, y:0 } });

                        setPageViewports({});
                        console.log('🔍 No viewport state found, using defaults');
                    }

                    // Load PDF if available
                    if (data.project.pdfFile.uploadedPath) {
                        try {
                            const pdfResponse = await fetch(`/api${data.project.pdfFile.uploadedPath}`);
                            if (pdfResponse.ok) {
                                const pdfBlob = await pdfResponse.blob();
                                const pdfFile = new File([pdfBlob], data.project.pdfFile.originalName || 'project.pdf', {
                                    type: 'application/pdf'
                                });

                                pdfFile.uploadedPath = data.project.pdfFile.uploadedPath;

                                // Process the PDF
                                const pages = await processPdfFromFile(pdfFile);
                                setPdfPages(pages);
                                setPdfFile(pdfFile);
                                setProjectFound(true);
                                setStatus(false);
                                console.log('📄 PDF loaded successfully:', pdfFile.name);
                            }
                        } catch (pdfError) {
                            setStatus(false);
                            setProjectFound(false);
                            console.warn('Could not load PDF file:', pdfError);
                            setLoadingError(`Project loaded, but could not load PDF: ${data.project.pdfFile?.originalName || 'Unknown'}`);
                        }
                    }

                    // Show success message
                    if (data.hasAccessPoints) {
                        console.log(`✅ Loaded ${accessPointsWithDetails.length} access points`);
                    }
                } else {
                    setProjectFound(false);
                    setStatus(false);
                    console.log('📭 No projects found in backend');
                }
            } else {
                setStatus(false);
                addToast({
                    title: '',
                    color: 'warning',
                    text: 'Backend not responding or no projects endpoint',
                });
            }
        } catch (error) {
            console.warn('Could not load first project from backend:', error.message);
            setError(error.message);
        } finally {
            setIsLoading(false);
        }
    };

    return {
        loadFirstProjectFromBackend,
        isLoading,
        error
    };
};
