import { useLoader } from '@/context/LoaderContext';
import { useState } from 'react';

interface Project {
  name: string,
  pdfFile: { name: string },
  timestamp: number,
  accessPoints: []
}
export const useProjectManager = () => {
    const [projectName, setProjectName] = useState('');
    const [savedProjects, setSavedProjects] = useState<Project[]>([]);
    const [showSaveModal, setShowSaveModal] = useState(false);
    const [showLoadModal, setShowLoadModal] = useState(false);
    const [apiEndpoint, setApiEndpoint] = useState('');
    const [apiLoading, setApiLoading] = useState(false);
    const [saved, setSaved] = useState(false);

    const { setLoading } = useLoader();
    const [currentProjectName, setCurrentProjectName] = useState('');
    const [isProjectModified, setIsProjectModified] = useState(false);

    const makeApiRequest = async (endpoint: string, method = 'GET', data = null) => {
        const options:RequestInit = {
            method,
            headers: {
                'Content-Type': 'application/json',
            },
        };

        if (data) {
            options.body = JSON.stringify(data);
        }

        const response = await fetch(endpoint, options);

        if (!response.ok) {
            throw new Error(`API request failed: ${response.status} ${response.statusText}`);
        }

        return response.json();
    };

    const loadProjectsFromApi = async () => {
        try {
            setApiLoading(true);
            setLoading(true);
            const projects = await makeApiRequest('/api/projects');
            setSavedProjects(projects);
        } catch (error) {
            throw error;
        } finally {
            setApiLoading(false);
            setLoading(false);
        }
    };

    const saveProject = async (projectData: Project) => {
        try {
            setApiLoading(true);
            setLoading(true);

            await makeApiRequest('/api/projects', 'POST', projectData);

            setShowSaveModal(false);
            setProjectName('');
            setSaved(true);
            return true;
        } catch (error) {
            setSaved(false);
            return false;
        } finally {
            setApiLoading(false);
            setLoading(false);
        }
    };

    return {
        projectName,
        setProjectName,
        savedProjects,
        setSavedProjects,
        showSaveModal,
        setShowSaveModal,
        showLoadModal,
        setShowLoadModal,
        apiEndpoint,
        setApiEndpoint,
        apiLoading,
        setApiLoading,
        makeApiRequest,
        loadProjectsFromApi,
        saveProject,
        currentProjectName,
        setCurrentProjectName,
        isProjectModified,
        setIsProjectModified,
        saved
    };
};
