// hooks/useResponsiveNodes.js
import { useState, useEffect, useCallback } from 'react';

export const useResponsiveNodes = (mapImageRef, pdfPages, currentPage) => {
    const [imageSize, setImageSize] = useState({ width: 0, height: 0 });
    const [originalImageSize, setOriginalImageSize] = useState({ width: 0, height: 0 });

    // Update image size when window resizes or image loads
    const updateImageSize = useCallback(() => {
        if (mapImageRef.current) {
            const rect = mapImageRef.current.getBoundingClientRect();
            setImageSize({
                width: rect.width,
                height: rect.height
            });

            // Store original (natural) dimensions for scaling calculations
            if (mapImageRef.current.naturalWidth && mapImageRef.current.naturalHeight) {
                setOriginalImageSize({
                    width: mapImageRef.current.naturalWidth,
                    height: mapImageRef.current.naturalHeight
                });
            }
        }
    }, [mapImageRef]);

    // Set up resize observer and load listener
    useEffect(() => {
        const imageElement = mapImageRef.current;
        if (!imageElement) return;

        // Update size when image loads
        const handleImageLoad = () => {
            updateImageSize();
        };

        // Update size when window resizes
        const handleResize = () => {
            updateImageSize();
        };

        // Set up event listeners
        imageElement.addEventListener('load', handleImageLoad);
        window.addEventListener('resize', handleResize);

        // Initial size update
        updateImageSize();

        // Cleanup
        return () => {
            imageElement.removeEventListener('load', handleImageLoad);
            window.removeEventListener('resize', handleResize);
        };
    }, [mapImageRef, updateImageSize, pdfPages, currentPage]);

    // Calculate responsive position for a node
    const getResponsivePosition = useCallback((originalX, originalY) => {
        if (!originalImageSize.width || !originalImageSize.height || !imageSize.width || !imageSize.height) {
            return { x: originalX, y: originalY };
        }

        // Calculate scale factors
        const scaleX = imageSize.width / originalImageSize.width;
        const scaleY = imageSize.height / originalImageSize.height;

        return {
            x: originalX * scaleX,
            y: originalY * scaleY
        };
    }, [imageSize, originalImageSize]);

    // Convert responsive position back to original coordinates
    const getOriginalPosition = useCallback((responsiveX, responsiveY) => {
        if (!originalImageSize.width || !originalImageSize.height || !imageSize.width || !imageSize.height) {
            return { x: responsiveX, y: responsiveY };
        }

        // Calculate inverse scale factors
        const scaleX = originalImageSize.width / imageSize.width;
        const scaleY = originalImageSize.height / imageSize.height;

        return {
            x: responsiveX * scaleX,
            y: responsiveY * scaleY
        };
    }, [imageSize, originalImageSize]);

    return {
        imageSize,
        originalImageSize,
        getResponsivePosition,
        getOriginalPosition,
        updateImageSize
    };
};
