import { useState, useCallback, useEffect } from 'react';
import { useResponsiveLayout } from './useResponsiveLayout';

export const useResponsiveZoom = () => {
    const { isMobile, isTablet, screenSize } = useResponsiveLayout();

    const [zoomLevel, setZoomLevel] = useState(1);
    const [panOffset, setPanOffset] = useState({ x: 0, y: 0 });
    const [isPanning, setIsPanning] = useState(false);
    const [lastPanPosition, setLastPanPosition] = useState({ x: 0, y: 0 });
    const [lastTouchDistance, setLastTouchDistance] = useState(0);

    // Responsive zoom limits based on device type
    const getZoomLimits = useCallback(() => {
        if (isMobile) {
            return {
                MIN_ZOOM: 0.5,   // Allow more zoom out on mobile
                MAX_ZOOM: 8,     // Allow more zoom in on mobile
                ZOOM_STEP: 0.5   // Larger steps for touch
            };
        } else if (isTablet) {
            return {
                MIN_ZOOM: 0.25,
                MAX_ZOOM: 6,
                ZOOM_STEP: 0.25
            };
        } else {
            return {
                MIN_ZOOM: 0.25,
                MAX_ZOOM: 5,
                ZOOM_STEP: 0.25
            };
        }
    }, [isMobile, isTablet]);

    const { MIN_ZOOM, MAX_ZOOM, ZOOM_STEP } = getZoomLimits();

    // Responsive initial zoom - fit content to screen
    const getInitialZoom = useCallback((containerWidth, containerHeight, contentWidth, contentHeight) => {
        if (!containerWidth || !containerHeight || !contentWidth || !contentHeight) {
            return 1;
        }

        const scaleX = containerWidth / contentWidth;
        const scaleY = containerHeight / contentHeight;
        const scale = Math.min(scaleX, scaleY);

        // Apply device-specific scaling
        if (isMobile) {
            // On mobile, fit with some padding
            return Math.max(MIN_ZOOM, Math.min(scale * 0.9, 1));
        } else if (isTablet) {
            return Math.max(MIN_ZOOM, Math.min(scale * 0.95, 1));
        } else {
            // Desktop - full size or fit
            return Math.max(MIN_ZOOM, Math.min(scale, 1));
        }
    }, [isMobile, isTablet, MIN_ZOOM]);

    // Zoom functions with responsive limits
    const handleZoomIn = useCallback(() => {
        setZoomLevel(prev => Math.min(MAX_ZOOM, prev + ZOOM_STEP));
    }, [MAX_ZOOM, ZOOM_STEP]);

    const handleZoomOut = useCallback(() => {
        setZoomLevel(prev => Math.max(MIN_ZOOM, prev - ZOOM_STEP));
    }, [MIN_ZOOM, ZOOM_STEP]);

    const handleZoomReset = useCallback(() => {
        setZoomLevel(1);
        setPanOffset({ x: 0, y: 0 });
    }, []);

    // Fit to screen zoom - responsive
    const handleZoomToFit = useCallback((containerWidth, containerHeight, contentWidth, contentHeight) => {
        const fitZoom = getInitialZoom(containerWidth, containerHeight, contentWidth, contentHeight);
        setZoomLevel(fitZoom);
        setPanOffset({ x: 0, y: 0 });
    }, [getInitialZoom]);

    // Mouse wheel zoom (desktop/tablet)
    const handleWheel = useCallback((e) => {
        if (!isMobile && (e.ctrlKey || e.metaKey)) {
            e.preventDefault();
            const delta = e.deltaY > 0 ? -ZOOM_STEP : ZOOM_STEP;
            setZoomLevel(prev => Math.max(MIN_ZOOM, Math.min(MAX_ZOOM, prev + delta)));
        }
    }, [isMobile, ZOOM_STEP, MIN_ZOOM, MAX_ZOOM]);

    // Mouse pan (desktop/tablet)
    const handleMouseDown = useCallback((e) => {
        if (!isMobile && (
            e.button === 1 ||
            (e.button === 0 && (e.ctrlKey || e.metaKey)) ||
            (e.button === 0 && zoomLevel > 1)
        )) {
            setIsPanning(true);
            setLastPanPosition({ x: e.clientX, y: e.clientY });
            e.preventDefault();
            document.body.style.cursor = 'grabbing';
        }
    }, [isMobile, zoomLevel]);

    // Touch events for mobile pinch-zoom and pan
    const handleTouchStart = useCallback((e) => {
        if (!isMobile) return;

        if (e.touches.length === 1) {
            // Single touch - pan
            const touch = e.touches[0];
            setIsPanning(true);
            setLastPanPosition({ x: touch.clientX, y: touch.clientY });
        } else if (e.touches.length === 2) {
            // Two fingers - pinch zoom
            const touch1 = e.touches[0];
            const touch2 = e.touches[1];
            const distance = Math.sqrt(
                Math.pow(touch2.clientX - touch1.clientX, 2) +
                Math.pow(touch2.clientY - touch1.clientY, 2)
            );
            setLastTouchDistance(distance);
            setIsPanning(false);
        }
        e.preventDefault();
    }, [isMobile]);

    const handleTouchMove = useCallback((e) => {
        if (!isMobile) return;

        if (e.touches.length === 1 && isPanning) {
            // Single touch pan
            const touch = e.touches[0];
            const deltaX = touch.clientX - lastPanPosition.x;
            const deltaY = touch.clientY - lastPanPosition.y;

            setPanOffset(prev => ({
                x: prev.x + deltaX,
                y: prev.y + deltaY
            }));

            setLastPanPosition({ x: touch.clientX, y: touch.clientY });
        } else if (e.touches.length === 2) {
            // Pinch zoom
            const touch1 = e.touches[0];
            const touch2 = e.touches[1];
            const distance = Math.sqrt(
                Math.pow(touch2.clientX - touch1.clientX, 2) +
                Math.pow(touch2.clientY - touch1.clientY, 2)
            );

            if (lastTouchDistance > 0) {
                const scale = distance / lastTouchDistance;
                const newZoom = Math.max(MIN_ZOOM, Math.min(MAX_ZOOM, zoomLevel * scale));
                setZoomLevel(newZoom);
            }

            setLastTouchDistance(distance);
        }
        e.preventDefault();
    }, [isMobile, isPanning, lastPanPosition, lastTouchDistance, zoomLevel, MIN_ZOOM, MAX_ZOOM]);

    const handleTouchEnd = useCallback((e) => {
        if (!isMobile) return;

        if (e.touches.length === 0) {
            setIsPanning(false);
            setLastTouchDistance(0);
        }
    }, [isMobile]);

    // Mouse events (desktop/tablet)
    const handleMouseMove = useCallback((e) => {
        if (!isMobile && isPanning) {
            const deltaX = e.clientX - lastPanPosition.x;
            const deltaY = e.clientY - lastPanPosition.y;

            setPanOffset(prev => ({
                x: prev.x + deltaX,
                y: prev.y + deltaY
            }));

            setLastPanPosition({ x: e.clientX, y: e.clientY });
        }
    }, [isMobile, isPanning, lastPanPosition]);

    const handleMouseUp = useCallback(() => {
        if (isPanning) {
            setIsPanning(false);
            document.body.style.cursor = '';
        }
    }, [isPanning]);

    // Event listeners
    useEffect(() => {
        document.addEventListener('mousemove', handleMouseMove);
        document.addEventListener('mouseup', handleMouseUp);
        document.addEventListener('touchmove', handleTouchMove, { passive: false });
        document.addEventListener('touchend', handleTouchEnd);

        return () => {
            document.removeEventListener('mousemove', handleMouseMove);
            document.removeEventListener('mouseup', handleMouseUp);
            document.removeEventListener('touchmove', handleTouchMove);
            document.removeEventListener('touchend', handleTouchEnd);
            document.body.style.cursor = '';
        };
    }, [handleMouseMove, handleMouseUp, handleTouchMove, handleTouchEnd]);

    // Responsive cursor
    const getMapCursor = useCallback(() => {
        if (isMobile) return 'default'; // No cursor indication on touch devices
        if (isPanning) return 'grabbing';
        if (zoomLevel > 1) return 'grab';
        return 'default';
    }, [isMobile, isPanning, zoomLevel]);

    // Zoom level validation when device changes
    useEffect(() => {
        const { MIN_ZOOM: newMin, MAX_ZOOM: newMax } = getZoomLimits();
        setZoomLevel(prev => Math.max(newMin, Math.min(newMax, prev)));
    }, [getZoomLimits]);

    return {
        zoomLevel,
        panOffset,
        isPanning,
        handleZoomIn,
        handleZoomOut,
        handleZoomReset,
        handleZoomToFit,
        handleWheel,
        handleMouseDown,
        handleTouchStart,
        handleTouchMove,
        handleTouchEnd,
        getInitialZoom,
        setZoomLevel,
        setPanOffset,
        getMapCursor,
        MIN_ZOOM,
        MAX_ZOOM,
        ZOOM_STEP,
        isMobile,
        isTablet
    };
};
