import { useState, useCallback, useRef } from 'react';

export const useTouchEvents = () => {
    const [isTouching, setIsTouching] = useState(false);
    const [touchStartPos, setTouchStartPos] = useState({ x: 0, y: 0 });
    const [lastTouchPos, setLastTouchPos] = useState({ x: 0, y: 0 });
    const touchStartTime = useRef(0);
    const longPressTimer = useRef(null);

    const handleTouchStart = useCallback((e, onLongPress) => {
        const touch = e.touches[0];
        setIsTouching(true);
        setTouchStartPos({ x: touch.clientX, y: touch.clientY });
        setLastTouchPos({ x: touch.clientX, y: touch.clientY });
        touchStartTime.current = Date.now();

        // Set up long press detection
        if (onLongPress) {
            longPressTimer.current = setTimeout(() => {
                onLongPress(e);
            }, 500); // 500ms for long press
        }
    }, []);

    const handleTouchMove = useCallback((e) => {
        if (longPressTimer.current) {
            clearTimeout(longPressTimer.current);
            longPressTimer.current = null;
        }

        if (isTouching) {
            const touch = e.touches[0];
            setLastTouchPos({ x: touch.clientX, y: touch.clientY });
        }
    }, [isTouching]);

    const handleTouchEnd = useCallback((e, onTap, onPan) => {
        if (longPressTimer.current) {
            clearTimeout(longPressTimer.current);
            longPressTimer.current = null;
        }

        if (isTouching) {
            const touchEndTime = Date.now();
            const touchDuration = touchEndTime - touchStartTime.current;
            const deltaX = lastTouchPos.x - touchStartPos.x;
            const deltaY = lastTouchPos.y - touchStartPos.y;
            const distance = Math.sqrt(deltaX * deltaX + deltaY * deltaY);

            // Determine if it's a tap or pan
            if (distance < 10 && touchDuration < 300) {
                // It's a tap
                onTap && onTap(e);
            } else if (distance > 10) {
                // It's a pan
                onPan && onPan(deltaX, deltaY);
            }
        }

        setIsTouching(false);
    }, [isTouching, touchStartPos, lastTouchPos]);

    return {
        handleTouchStart,
        handleTouchMove,
        handleTouchEnd,
        isTouching
    };
};
