// hooks/useZoomPan.js
import { useState, useCallback, useEffect } from 'react';

export const useZoomPan = () => {
    const [zoomLevel, setZoomLevel] = useState(1);
    const [panOffset, setPanOffset] = useState({ x: 0, y: 0 });
    const [isPanning, setIsPanning] = useState(false);
    const [lastPanPosition, setLastPanPosition] = useState({ x: 0, y: 0 });
    const [lastTouchDistance, setLastTouchDistance] = useState(0);
    const [touchStartTime, setTouchStartTime] = useState(0);

    const MIN_ZOOM = 0.25;
    const MAX_ZOOM = 5;
    const ZOOM_STEP = 0.25;
    const handleTouchStart = useCallback((e) => {
        const now = Date.now();
        setTouchStartTime(now);

        if (e.touches.length === 1) {
            // Single touch - always allow panning on mobile
            const touch = e.touches[0];
            setIsPanning(true);
            setLastPanPosition({ x: touch.clientX, y: touch.clientY });
            setLastTouchDistance(0);

            // Prevent default to stop browser scrolling/zooming
            e.preventDefault();
            console.log('🖐️ Touch pan started');
        } else if (e.touches.length === 2) {
            // Two touches - pinch zoom
            const touch1 = e.touches[0];
            const touch2 = e.touches[1];
            const distance = Math.sqrt(
                Math.pow(touch2.clientX - touch1.clientX, 2) +
                Math.pow(touch2.clientY - touch1.clientY, 2)
            );
            setLastTouchDistance(distance);
            setIsPanning(false); // Stop panning when pinching
            e.preventDefault();
            console.log('🤏 Pinch zoom started');
        }
    }, []);

    // Enhanced touch move - smooth mobile panning
    const handleTouchMove = useCallback((e) => {
        if (e.touches.length === 1 && isPanning) {
            // Single touch panning
            const touch = e.touches[0];
            const deltaX = touch.clientX - lastPanPosition.x;
            const deltaY = touch.clientY - lastPanPosition.y;

            // Apply pan offset
            setPanOffset(prev => ({
                x: prev.x + deltaX,
                y: prev.y + deltaY
            }));

            setLastPanPosition({ x: touch.clientX, y: touch.clientY });

            // Prevent default scrolling
            e.preventDefault();

        } else if (e.touches.length === 2 && lastTouchDistance > 0) {
            // Pinch zoom
            const touch1 = e.touches[0];
            const touch2 = e.touches[1];
            const distance = Math.sqrt(
                Math.pow(touch2.clientX - touch1.clientX, 2) +
                Math.pow(touch2.clientY - touch1.clientY, 2)
            );

            const scale = distance / lastTouchDistance;
            const newZoom = Math.max(MIN_ZOOM, Math.min(MAX_ZOOM, zoomLevel * scale));
            setZoomLevel(newZoom);
            setLastTouchDistance(distance);

            // Prevent default browser zoom
            e.preventDefault();
        }
    }, [isPanning, lastPanPosition, lastTouchDistance, zoomLevel, MIN_ZOOM, MAX_ZOOM]);

    // Enhanced touch end - proper cleanup
    const handleTouchEnd = useCallback((e) => {
        const touchDuration = Date.now() - touchStartTime;

        if (e.touches.length === 0) {
            // All touches ended
            setIsPanning(false);
            setLastTouchDistance(0);
            setLastPanPosition({ x: 0, y: 0 });
            console.log('✋ Touch ended, pan stopped');
        } else if (e.touches.length === 1 && lastTouchDistance > 0) {
            // One finger lifted during pinch - resume panning
            const touch = e.touches[0];
            setIsPanning(true);
            setLastPanPosition({ x: touch.clientX, y: touch.clientY });
            setLastTouchDistance(0);
            console.log('🖐️ Resumed panning after pinch');
        }
    }, [touchStartTime, lastTouchDistance]);

    const handleZoomIn = () => {
        setZoomLevel(prev => Math.min(MAX_ZOOM, prev + ZOOM_STEP));
    };

    const handleZoomOut = () => {
        setZoomLevel(prev => Math.max(MIN_ZOOM, prev - ZOOM_STEP));
    };

    const handleZoomReset = () => {
        setZoomLevel(1);
        setPanOffset({ x: 0, y: 0 });
    };

    const handleWheel = useCallback((e) => {
        if (e.ctrlKey || e.metaKey) {
            e.preventDefault();
            const delta = e.deltaY > 0 ? -ZOOM_STEP : ZOOM_STEP;
            setZoomLevel(prev => Math.max(MIN_ZOOM, Math.min(MAX_ZOOM, prev + delta)));
        }
    }, []);

    const handleMouseDown = useCallback((e) => {
        // Allow panning with:
        // 1. Middle mouse button (button === 1)
        // 2. Left click with Ctrl/Cmd
        // 3. Left click when zoomed in (zoomLevel > 1)
        if (e.button === 1 ||
            (e.button === 0 && (e.ctrlKey || e.metaKey)) ||
            (e.button === 0 && zoomLevel > 1)) {
            setIsPanning(true);
            setLastPanPosition({ x: e.clientX, y: e.clientY });
            e.preventDefault();

            // Change cursor to grabbing when panning starts
            document.body.style.cursor = 'grabbing';
        }
    }, [zoomLevel]);

    const handleMouseMove = useCallback((e) => {
        if (isPanning) {
            const deltaX = e.clientX - lastPanPosition.x;
            const deltaY = e.clientY - lastPanPosition.y;

            setPanOffset(prev => ({
                x: prev.x + deltaX,
                y: prev.y + deltaY
            }));

            setLastPanPosition({ x: e.clientX, y: e.clientY });
        }
    }, [isPanning, lastPanPosition]);

    const handleMouseUp = useCallback(() => {
        if (isPanning) {
            setIsPanning(false);
            // Reset cursor when panning ends
            document.body.style.cursor = '';
        }
    }, [isPanning]);

    useEffect(() => {
        document.addEventListener('mousemove', handleMouseMove);
        document.addEventListener('mouseup', handleMouseUp);
        // ADD these mobile touch listeners
        document.addEventListener('touchmove', handleTouchMove, { passive: false });
        document.addEventListener('touchend', handleTouchEnd, { passive: false });

        return () => {
            document.removeEventListener('mousemove', handleMouseMove);
            document.removeEventListener('mouseup', handleMouseUp);
            // REMOVE mobile touch listeners
            document.removeEventListener('touchmove', handleTouchMove);
            document.removeEventListener('touchend', handleTouchEnd);
            document.body.style.cursor = '';
        };
    }, [handleMouseMove, handleMouseUp, handleTouchMove, handleTouchEnd]);

    // Update cursor based on zoom level
    const getMapCursor = useCallback(() => {
        if (isPanning) return 'grabbing';
        if (zoomLevel > 1) return 'grab';
        return 'default';
    }, [isPanning, zoomLevel]);

    return {
        zoomLevel,
        panOffset,
        isPanning,
        handleZoomIn,
        handleZoomOut,
        handleZoomReset,
        handleWheel,
        handleMouseDown,
        handleTouchStart,
        handleTouchMove,
        handleTouchEnd,
        setZoomLevel,
        setPanOffset,
        getMapCursor,
        MIN_ZOOM,
        MAX_ZOOM,
        setIsPanning
    };
};
