import AccessPointMarkerPoint from "@/app/types/AccessPointMarker";
import { MutableRefObject } from "react";
// Zoom constants
export const MIN_ZOOM = 0.25;
export const MAX_ZOOM = 5;
export const ZOOM_STEP = 0.25;

export const calculateResponsiveNodeSize = (zoomLevel: number, isMobile = false) => {
  // Base sizes for different devices
  const baseSizes = {
    mobile: 50,
    desktop: 40
  };

  const baseSize = isMobile ? baseSizes.mobile : baseSizes.desktop;

  // Inverse scaling: as zoom increases, node size decreases
  // This keeps nodes visually consistent relative to the PDF content
  const scaleFactor = 1 / Math.max(zoomLevel, 0.5); // Prevent too small nodes

  // Apply scaling with minimum and maximum limits
  const minSize = isMobile ? 30 : 24;  // Minimum readable size
  const maxSize = isMobile ? 60 : 48;  // Maximum size (when zoomed out)

  const calculatedSize = baseSize * scaleFactor;
  return Math.max(minSize, Math.min(maxSize, calculatedSize));
};

export const calculateResponsiveElements = (zoomLevel: number, isMobile = false) => {
  const nodeSize = calculateResponsiveNodeSize(zoomLevel, isMobile);

  return {
    nodeSize,
    statusSize: Math.max(8, nodeSize * 0.3),           // Status indicator
    fontSize: Math.max(10, nodeSize * 0.32) + 'px',    // Text inside node
    labelFontSize: Math.max(10, nodeSize * 0.28) + 'px', // Label below node
    borderWidth: Math.max(2, nodeSize * 0.08),          // Border thickness
    labelPadding: Math.max(4, nodeSize * 0.12),         // Label padding
    touchTarget: Math.max(nodeSize, isMobile ? 44 : 32) // Minimum touch target
  };
};

export const transformCoordinates = (x: number, y: number, rotation: number, imageWidth: number, imageHeight: number) => {
  if (!rotation || rotation === 0) return { x, y };

  // The CSS transform origin is "center center" for the image
  // So we rotate around the center of the original image coordinates
  const centerX = imageWidth / 2;
  const centerY = imageHeight / 2;

  // Translate to origin (center of image)
  const translatedX = x - centerX;
  const translatedY = y - centerY;

  // Convert rotation to radians
  const radians = (rotation * Math.PI) / 180;

  // Apply rotation matrix
  const cos = Math.cos(radians);
  const sin = Math.sin(radians);

  const rotatedX = translatedX * cos - translatedY * sin;
  const rotatedY = translatedX * sin + translatedY * cos;

  // Translate back
  return {
    x: rotatedX + centerX,
    y: rotatedY + centerY
  };
};

export const getImageDimensions = (mapImageRef: MutableRefObject<HTMLImageElement | null>) => {
  if (!mapImageRef.current) return { width: 800, height: 600 }; // fallback
  return {
    width: mapImageRef.current.naturalWidth || mapImageRef.current.offsetWidth,
    height: mapImageRef.current.naturalHeight || mapImageRef.current.offsetHeight
  };
};
// Inverse transform for drag operations
const inverseTransformCoordinates = (x: number, y: number, rotation: number, imageWidth: number, imageHeight: number) => {
  if (!rotation || rotation === 0) return { x, y };

  const centerX = imageWidth / 2;
  const centerY = imageHeight / 2;

  // Translate to origin
  const translatedX = x - centerX;
  const translatedY = y - centerY;

  // Convert rotation to radians (negative for inverse)
  const radians = (-rotation * Math.PI) / 180;

  // Apply inverse rotation
  const rotatedX = translatedX * Math.cos(radians) - translatedY * Math.sin(radians);
  const rotatedY = translatedX * Math.sin(radians) + translatedY * Math.cos(radians);

  // Translate back
  return {
    x: rotatedX + centerX,
    y: rotatedY + centerY
  };
};

// Check if a type has reached its limit
export const isTypeAtLimit = (typeId: string, accessPoints:AccessPointMarkerPoint[]) => {
  return getTypeCount(typeId, accessPoints) > 0;
};

// Get count for a specific type
export const getTypeCount = (typeId: string, accessPoints: AccessPointMarkerPoint []) => {
  return accessPoints.filter((ap: AccessPointMarkerPoint) => ap.type === typeId).length;
};

// ===============================
// COORDINATE CONVERSION UTILITIES
// ===============================

// Convert absolute pixel coordinates to relative coordinates (0-1)
export const pixelToRelative = (pixelX: number, pixelY: number, imageWidth: number, imageHeight: number) => {
  return {
    relativeX: pixelX / imageWidth,
    relativeY: pixelY / imageHeight
  };
};

// Convert relative coordinates (0-1) to absolute pixel coordinates
export const relativeToPixel = (relativeX: number, relativeY: number, imageWidth: number, imageHeight: number) => {
  return {
    x: relativeX * imageWidth,
    y: relativeY * imageHeight
  };
};
// Migrate existing absolute coordinates to relative coordinates
export const migrateToRelativeCoordinates = (accessPoint: AccessPointMarkerPoint, imageWidth: number, imageHeight: number) => {
  // If already has relative coordinates, return as is
  if (typeof accessPoint.relativeX === 'number' && typeof accessPoint.relativeY === 'number') {
    return accessPoint;
  }

  if (accessPoint.x < 1) {
    const { x, y } = relativeToPixel(
        accessPoint.x || 0,
        accessPoint.y || 0,
        imageWidth,
        imageHeight
    );

    return {
      ...accessPoint,
      relativeX: accessPoint.x,
      relativeY: accessPoint.y,
      x: x,
      y: y
    };
  }
  // Convert absolute coordinates to relative
  const { x, y } = relativeToPixel(
      accessPoint.x || 0,
      accessPoint.y || 0,
      imageWidth,
      imageHeight
  );

  return {
    ...accessPoint,
    relativeX: x,
    relativeY: y,
    // Keep the old coordinates for backwards compatibility during transition
    x: accessPoint.x,
    y: accessPoint.y
  };
};

// Get current image dimensions
export const getCurrentImageDimensions = (mapImageRef: MutableRefObject<HTMLImageElement|null>, imageDimensions: { width: number, height: number}) => {
  if (mapImageRef.current) {
    return {
      width: mapImageRef.current.offsetWidth,
      height: mapImageRef.current.offsetHeight
    };
  }
  return imageDimensions;
};
// Convert screen coordinates to map coordinates (accounting for zoom, pan, and rotation)
export const screenToMapCoordinates = (clientX, clientY, mapContainer, mapImage, zoom, pan, rotation, imageDimensions) => {
  if (!mapContainer.current || !mapImage.current) {
    return { x: 0, y: 0 };
  }

  const containerRect = mapContainer.current.getBoundingClientRect();
  const imageRect = mapImage.current.getBoundingClientRect();
  console.log(mapImage.current)
  // console.log(mapContainer.current)
  // Get the actual image dimensions after scaling
  const actualImageWidth = mapImage.current.width;
  const actualImageHeight = mapImage.current.height;

  // Calculate position relative to the container
  const containerX = clientX - containerRect.left;
  const containerY = clientY - containerRect.top;

  // Account for the transform container offset
  const transformedContainerX = containerX - pan.x;
  const transformedContainerY = containerY - pan.y;

  // Reverse the zoom transformation
  const unzoomedX = transformedContainerX / zoom;
  const unzoomedY = transformedContainerY / zoom;

  // Handle rotation - get center point
  const centerX = actualImageWidth / 2;
  const centerY = actualImageHeight / 2;

  // Translate to origin (center)
  const translatedX = unzoomedX - centerX;
  const translatedY = unzoomedY - centerY;

  // Reverse rotation
  const radians = (-rotation * Math.PI) / 180;
  const cos = Math.cos(radians);
  const sin = Math.sin(radians);

  const rotatedX = translatedX * cos - translatedY * sin;
  const rotatedY = translatedX * sin + translatedY * cos;

  // Translate back
  const finalX = rotatedX + centerX;
  const finalY = rotatedY + centerY;

  // Clamp to image bounds
  return {
    x: Math.max(0, Math.min(actualImageWidth, finalX)),
    y: Math.max(0, Math.min(actualImageHeight, finalY))
  };
};
// Calculate distance between two touch points
export const getTouchDistance = (
    touch1: { clientX: number, clientY: number },
    touch2: { clientX: number, clientY: number }
) => {
  const dx = touch1.clientX - touch2.clientX;
  const dy = touch1.clientY - touch2.clientY;
  return Math.sqrt(dx * dx + dy * dy);
};
export const generateUniqueId = (type, existingAccessPoints) => {
  const timestamp = Date.now();
  const random = Math.random().toString(36).substr(2, 9);
  let id = `${type}-${timestamp}-${random}`;

  // Ensure uniqueness (though collision is extremely unlikely)
  while (existingAccessPoints.some(ap => ap.id === id)) {
    id = `${type}-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
  }

  return id;
};
