export const processPdfFromFile = async (file) => {
  try {
    console.log('🔄 Starting PDF processing...');
    const arrayBuffer = await file.arrayBuffer();
    const pdf = await window.pdfjsLib.getDocument({ data: arrayBuffer }).promise;

    const pages = [];

    for (let i = 1; i <= pdf.numPages; i++) {
      const page = await pdf.getPage(i);
      const viewport = page.getViewport({ scale: 2 });
      const canvas = document.createElement('canvas');
      const context = canvas.getContext('2d');
      canvas.height = viewport.height;
      canvas.width = viewport.width;

      await page.render({
        canvasContext: context,
        viewport: viewport
      }).promise;

      pages.push(canvas.toDataURL('image/png'));
      console.log(`✅ Page ${i} processed successfully`);
    }

    console.log(`📄 Successfully processed ${pages.length} pages from PDF`);
    return pages;
  } catch (error) {
    console.error('Error processing PDF with PDF.js:', error);
    throw new Error(`Failed to process PDF: ${error.message}`);
  }
};

export const uploadPdfToBackend = async (file) => {
  try {
    const formData = new FormData();
    formData.append('pdf', file);
    formData.append('originalName', file.name);

    const response = await fetch(`/api/projects/uploads/pdf`, {
      method: 'POST',
      body: formData
    });

    if (!response.ok) {
      throw new Error(`Upload failed: ${response.status} ${response.statusText}`);
    }

    const result = await response.json();
    console.log('📤 PDF uploaded to backend:', result.filePath);
    return result.filePath;
  } catch (error) {
    console.error('Failed to upload PDF to backend:', error);
    throw error;
  }
};

export const generatePDF = async (pdfPages, accessPoints, accessPointTypes, mapImageRef) => {
  const { jsPDF } = window.jsPDF;
  const pdf = new jsPDF('portrait', 'mm', 'a4');

  for (let pageIndex = 0; pageIndex < pdfPages.length; pageIndex++) {
    if (pageIndex > 0) {
      pdf.addPage();
    }

    const img = new Image();
    img.src = pdfPages[pageIndex];
    await new Promise(resolve => {
      img.onload = resolve;
    });

    const pdfWidth = pdf.internal.pageSize.getWidth();
    const pdfHeight = pdf.internal.pageSize.getHeight();
    const imgAspect = img.width / img.height;
    const pdfAspect = pdfWidth / pdfHeight;

    let imgWidth, imgHeight;
    if (imgAspect > pdfAspect) {
      imgWidth = pdfWidth;
      imgHeight = pdfWidth / imgAspect;
    } else {
      imgHeight = pdfHeight;
      imgWidth = pdfHeight * imgAspect;
    }

    pdf.addImage(pdfPages[pageIndex], 'PNG', 0, 0, imgWidth, imgHeight);

    const pageAccessPoints = accessPoints.filter(ap => ap.page === pageIndex);
    pageAccessPoints.forEach((ap) => {
      const type = accessPointTypes.find(t => t.id === ap.type);
      if (type) {
        const scaleX = imgWidth / (mapImageRef.current?.naturalWidth || 1);
        const scaleY = imgHeight / (mapImageRef.current?.naturalHeight || 1);

        const pdfX = ap.x * scaleX;
        const pdfY = ap.y * scaleY;

        pdf.setFillColor(type.color);
        pdf.circle(pdfX, pdfY, 3, 'F');

        pdf.setDrawColor(255, 255, 255);
        pdf.setLineWidth(0.5);
        pdf.circle(pdfX, pdfY, 3, 'S');

        pdf.setTextColor(0, 0, 0);
        pdf.setFontSize(8);
        pdf.text(ap.name, pdfX + 5, pdfY - 2);
      }
    });
  }

  // Add legend page
  pdf.addPage();
  pdf.setFontSize(16);
  pdf.setTextColor(0, 0, 0);
  pdf.text('Access Points Legend', 20, 20);

  let yPos = 40;
  accessPointTypes.forEach((type) => {
    const typePoints = accessPoints.filter(ap => ap.type === type.id);
    if (typePoints.length > 0) {
      pdf.setFillColor(type.color);
      pdf.circle(25, yPos - 2, 2, 'F');
      pdf.setFontSize(12);
      pdf.text(`${type.name} (${typePoints.length} points)`, 35, yPos);
      yPos += 10;

      typePoints.forEach((point) => {
        pdf.setFontSize(10);
        pdf.setTextColor(100, 100, 100);
        pdf.text(`  • ${point.name} - Page ${point.page + 1}`, 40, yPos);
        yPos += 6;
      });
      yPos += 5;
    }
  });

  pdf.save(`map-with-access-points-${Date.now()}.pdf`);
};

export const calculateVisibleRegion = (viewport, imageDimensions, containerWidth, containerHeight) => {
  const { zoomLevel, panOffset } = viewport;

  // Calculate the visible area in original image coordinates
  const visibleWidth = containerWidth / zoomLevel;
  const visibleHeight = containerHeight / zoomLevel;

  // Calculate top-left corner of visible area
  const visibleX = Math.max(0, -panOffset.x / zoomLevel);
  const visibleY = Math.max(0, -panOffset.y / zoomLevel);

  // Clamp to image bounds
  const clampedX = Math.max(0, Math.min(imageDimensions.width - visibleWidth, visibleX));
  const clampedY = Math.max(0, Math.min(imageDimensions.height - visibleHeight, visibleY));
  const clampedWidth = Math.min(visibleWidth, imageDimensions.width - clampedX);
  const clampedHeight = Math.min(visibleHeight, imageDimensions.height - clampedY);

  return {
    x: clampedX,
    y: clampedY,
    width: clampedWidth,
    height: clampedHeight
  };
};

export const cropImageToViewport = async (imageRef, viewport, imageDimensions, containerRef, dispatch, setPdfFile) => {
  if (!imageRef.current || !containerRef.current) return null;

  const containerRect = containerRef.current.getBoundingClientRect();
  const containerWidth = containerRect.width;
  const containerHeight = containerRect.height;

  // Create canvas for cropping
  const canvas = document.createElement('canvas');
  const ctx = canvas.getContext('2d');

  // Calculate the visible region in original image coordinates
  const visibleRegion = calculateVisibleRegion(viewport, imageDimensions, containerWidth, containerHeight);

  // Set canvas size to the visible region size
  canvas.width = visibleRegion.width;
  canvas.height = visibleRegion.height;

  // Create a temporary image to work with
  const tempImage = new Image();
  tempImage.crossOrigin = 'anonymous';

  return new Promise((resolve) => {
    tempImage.onload = async () => {
      // Apply rotation if needed
      if (viewport.rotation !== 0) {
        // Save context state
        ctx.save();

        // Translate to center and rotate
        ctx.translate(canvas.width / 2, canvas.height / 2);
        ctx.rotate((viewport.rotation * Math.PI) / 180);

        // Draw the cropped portion
        ctx.drawImage(
            tempImage,
            visibleRegion.x - canvas.width / 2,
            visibleRegion.y - canvas.height / 2,
            visibleRegion.width,
            visibleRegion.height,
            -canvas.width / 2,
            -canvas.height / 2,
            canvas.width,
            canvas.height
        );

        // Restore context
        ctx.restore();
      } else {
        // Simple crop without rotation
        ctx.drawImage(
            tempImage,
            visibleRegion.x,
            visibleRegion.y,
            visibleRegion.width,
            visibleRegion.height,
            0,
            0,
            canvas.width,
            canvas.height
        );
      }

      const mmWidth = canvas.width * 0.264583; // pixels to mm
      const mmHeight = canvas.height * 0.264583;

      const orientation = canvas.width > canvas.height ? 'landscape' : 'portrait';
      const {jsPDF} = window.jspdf;
      const pdf = new jsPDF('orientation', 'mm', [ mmWidth, mmHeight,]);

      let quality = 0.95;
      let croppedDataUrl;
      let pdfBlob;
      do {
        croppedDataUrl = canvas.toDataURL('image/jpeg', quality);
        pdf.addImage(croppedDataUrl, 'JPEG', 0, 0, mmWidth, mmHeight);

        pdfBlob = pdf.output('blob');
        const sizeMB = pdfBlob.size / (1024 * 1024);

        if (sizeMB > 3 && quality > 0.3) {
          // Too large, reduce quality and try again
          pdf.deletePage(1);
          quality -= 0.1;
        } else {
          break;
        }
      } while (true);

      const filename = `precision-map-crop-${Date.now()}.pdf`;
      const pdfFileObj = {
        file: new File([pdfBlob], filename, { type: 'application/pdf' }),
        uploadedPath: ''
      };
      setPdfFile(pdfFileObj);
      const uploadedPath = await uploadPdfToBackend(pdfFileObj.file);
      setPdfFile(prev => ({
        ...prev,
        uploadedPath
      }));
      resolve({
        dataUrl: croppedDataUrl,
        cropRegion: visibleRegion,
        newDimensions: {width: canvas.width, height: canvas.height}
      });
    };

    tempImage.src = imageRef.current.src;
  });
};

export const transformAccessPointsAfterCrop = (accessPoints, cropRegion, newDimensions) => {
  return accessPoints.map(ap => {
    // Check if access point is within the cropped region
    const isVisible =
        ap.x >= cropRegion.x &&
        ap.x <= cropRegion.x + cropRegion.width &&
        ap.y >= cropRegion.y &&
        ap.y <= cropRegion.y + cropRegion.height;

    if (!isVisible) {
      // Access point is outside the crop, remove it or move to edge
      return null; // Will be filtered out
    }

    // Transform coordinates to new coordinate system
    const newX = ((ap.x - cropRegion.x) / cropRegion.width) * newDimensions.width;
    const newY = ((ap.y - cropRegion.y) / cropRegion.height) * newDimensions.height;

    return {
      ...ap,
      x: newX,
      y: newY
    };
  }).filter(Boolean); // Remove null entries
};
