/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

import { useEuiI18n } from '../../i18n';
var MS_INTERVALS = {
  s: 1000,
  get m() {
    return 60 * this.s;
  },
  get h() {
    return 60 * this.m;
  },
  get d() {
    return 24 * this.h;
  }
};
/**
 * Pretty interval i18n strings
 *
 * Units should not be simply concatenated because different languages
 * will have different grammar/positions for time than English
 */
var usePrettyIntervalI18n = function usePrettyIntervalI18n(interval) {
  return {
    s: useEuiI18n('euiPrettyInterval.seconds', function (_ref) {
      var interval = _ref.interval;
      return "".concat(interval, " second").concat(interval > 1 ? 's' : '');
    }, {
      interval: interval
    }),
    m: useEuiI18n('euiPrettyInterval.minutes', function (_ref2) {
      var interval = _ref2.interval;
      return "".concat(interval, " minute").concat(interval > 1 ? 's' : '');
    }, {
      interval: interval
    }),
    h: useEuiI18n('euiPrettyInterval.hours', function (_ref3) {
      var interval = _ref3.interval;
      return "".concat(interval, " hour").concat(interval > 1 ? 's' : '');
    }, {
      interval: interval
    }),
    d: useEuiI18n('euiPrettyInterval.days', function (_ref4) {
      var interval = _ref4.interval;
      return "".concat(interval, " day").concat(interval > 1 ? 's' : '');
    }, {
      interval: interval
    }),
    shorthand: {
      s: useEuiI18n('euiPrettyInterval.secondsShorthand', '{interval} s', {
        interval: interval
      }),
      m: useEuiI18n('euiPrettyInterval.minutesShorthand', '{interval} m', {
        interval: interval
      }),
      h: useEuiI18n('euiPrettyInterval.hoursShorthand', '{interval} h', {
        interval: interval
      }),
      d: useEuiI18n('euiPrettyInterval.daysShorthand', '{interval} d', {
        interval: interval
      })
    }
  };
};
export var usePrettyInterval = function usePrettyInterval(isPaused, intervalInMs, options) {
  var _ref5 = options || {},
    _ref5$shortHand = _ref5.shortHand,
    shortHand = _ref5$shortHand === void 0 ? false : _ref5$shortHand,
    unit = _ref5.unit;
  var prettyInterval = '';
  var interval;
  var unitId;
  if (unit) {
    unitId = unit;
    interval = Math.round(intervalInMs / MS_INTERVALS[unit]);
  } else {
    if (intervalInMs < MS_INTERVALS.m) {
      interval = Math.round(intervalInMs / MS_INTERVALS.s);
      unitId = 's';
    } else if (intervalInMs < MS_INTERVALS.h) {
      interval = Math.round(intervalInMs / MS_INTERVALS.m);
      unitId = 'm';
    } else if (intervalInMs < MS_INTERVALS.d) {
      interval = Math.round(intervalInMs / MS_INTERVALS.h);
      unitId = 'h';
    } else {
      interval = Math.round(intervalInMs / MS_INTERVALS.d);
      unitId = 'd';
    }
  }
  var prettyIntervalI18n = usePrettyIntervalI18n(interval);
  prettyInterval = shortHand ? prettyIntervalI18n.shorthand[unitId] : prettyIntervalI18n[unitId];
  var off = useEuiI18n('euiPrettyInterval.off', 'Off');
  if (isPaused || intervalInMs === 0) {
    prettyInterval = off;
  }
  return prettyInterval;
};