/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

import React, { useCallback } from 'react';
import { EuiButtonEmpty, EuiButtonIcon } from '../button';
import { EuiToolTip } from '../tool_tip';
import { useGeneratedHtmlId } from '../../services/accessibility';
import { EuiScreenReaderOnly } from '../accessibility';
import { callWithItemIfFunction } from './action_types';
import { jsx as ___EmotionJSX } from "@emotion/react";
export var DefaultItemAction = function DefaultItemAction(_ref) {
  var action = _ref.action,
    enabled = _ref.enabled,
    item = _ref.item,
    className = _ref.className;
  if (!action.onClick && !action.href) {
    throw new Error("Cannot render item action [".concat(action.name, "]. Missing required 'onClick' callback\n      or 'href' string. If you want to provide a custom action control, make sure to define the 'render' callback"));
  }
  var onClick = useCallback(function (event) {
    if (!action.onClick) return;
    event.persist(); // TODO: Remove once React 16 support is dropped
    action.onClick(item, event);
  }, [action.onClick, item]);
  var color = action.color ? callWithItemIfFunction(item)(action.color) : 'primary';
  var icon = action.icon ? callWithItemIfFunction(item)(action.icon) : undefined;
  var actionContent = callWithItemIfFunction(item)(action.name);
  var tooltipContent = callWithItemIfFunction(item)(action.description);
  var href = callWithItemIfFunction(item)(action.href);
  var dataTestSubj = callWithItemIfFunction(item)(action['data-test-subj']);
  var ariaLabelId = useGeneratedHtmlId();
  var ariaLabelledBy;
  var button;
  if (action.type === 'icon') {
    if (!icon) {
      throw new Error("Cannot render item action [".concat(action.name, "]. It is configured to render as an icon but no\n      icon is provided. Make sure to set the 'icon' property of the action"));
    }
    button = ___EmotionJSX(EuiButtonIcon, {
      className: className,
      "aria-labelledby": ariaLabelId,
      isDisabled: !enabled,
      color: color,
      iconType: icon,
      onClick: onClick,
      href: href,
      target: action.target,
      "data-test-subj": dataTestSubj
      // If action is disabled, the normal tooltip can't show - attempt to
      // provide some amount of affordance with a browser title tooltip
      ,
      title: !enabled ? tooltipContent : undefined
    });
    // actionContent (action.name) is a ReactNode and must be rendered
    // to an element and referenced by ID for screen readers
    ariaLabelledBy = ___EmotionJSX(EuiScreenReaderOnly, null, ___EmotionJSX("span", {
      id: ariaLabelId
    }, actionContent));
  } else {
    button = ___EmotionJSX(EuiButtonEmpty, {
      className: className,
      size: "s",
      isDisabled: !enabled,
      color: color,
      iconType: icon,
      onClick: onClick,
      href: href,
      target: action.target,
      "data-test-subj": dataTestSubj,
      flush: "right"
    }, actionContent);
  }
  return enabled ? ___EmotionJSX(React.Fragment, null, ___EmotionJSX(EuiToolTip, {
    content: tooltipContent,
    delay: "long"
  }, button), ariaLabelledBy) : ___EmotionJSX(React.Fragment, null, button, ariaLabelledBy);
};