import _toArray from "@babel/runtime/helpers/toArray";
import _slicedToArray from "@babel/runtime/helpers/slicedToArray";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

import React, { useEffect, useState, useMemo } from 'react';
import { tabbable } from 'tabbable';
import classNames from 'classnames';
import { keys, useGeneratedHtmlId } from '../../../../services';
import { isDOMNode } from '../../../../utils';
import { EuiFocusTrap } from '../../../focus_trap';
import { EuiI18n } from '../../../i18n';

/**
 * This internal utility component is used by all cells, both header and body/footer cells.
 * It always handles:
 *   1. Removing any interactive children from keyboard tab order on cell mount
 *   2. Listening for focus on any interactive children and updating the cell focus context
 *
 * It should *only* render focus traps for:
 *   1. Header cells that are `actions: false` but still have interactive children
 *   2. Body cells that are `isExpandable: false` but still have interactive children
 */
import { jsx as ___EmotionJSX } from "@emotion/react";
export var HandleInteractiveChildren = function HandleInteractiveChildren(_ref) {
  var cellEl = _ref.cellEl,
    children = _ref.children,
    updateCellFocusContext = _ref.updateCellFocusContext,
    renderFocusTrap = _ref.renderFocusTrap,
    onInteractiveChildrenFound = _ref.onInteractiveChildrenFound;
  var _useState = useState(false),
    _useState2 = _slicedToArray(_useState, 2),
    hasInteractiveChildren = _useState2[0],
    setHasInteractiveChildren = _useState2[1];

  // On mount, disable all interactive children
  useEffect(function () {
    if (cellEl) {
      var interactives = disableInteractives(cellEl);
      onInteractiveChildrenFound === null || onInteractiveChildrenFound === void 0 || onInteractiveChildrenFound(interactives);
      setHasInteractiveChildren(interactives.length > 0);
    }
  }, [cellEl, onInteractiveChildrenFound]);

  // Ensure that any interactive children that are clicked update the latest cell focus context
  useEffect(function () {
    if (cellEl) {
      var onFocus = function onFocus() {
        return updateCellFocusContext();
      };
      cellEl.addEventListener('focus', onFocus, true); // useCapture listens for focus on children
      return function () {
        cellEl.removeEventListener('focus', onFocus, true);
      };
    }
  }, [cellEl, updateCellFocusContext]);
  var _children = useMemo(function () {
    return ___EmotionJSX(React.Fragment, null, children);
  }, [children]);
  if (!cellEl) return _children; // Do nothing if cell has yet to mount or is unmounting
  if (!renderFocusTrap) return _children; // Cells with default actions / expansion popovers do not need to trap
  if (!hasInteractiveChildren) return _children; // No need to focus trap if no children are interactive

  return ___EmotionJSX(FocusTrappedChildren, {
    cellEl: cellEl
  }, children);
};

/**
 * Cells with interactive children but no cell popover expansion should render a
 * focus trap that can be entered with the Enter key, which cycles keyboard tabs
 * through the cell contents only, and exited with the Escape key
 */
export var FocusTrappedChildren = function FocusTrappedChildren(_ref2) {
  var cellEl = _ref2.cellEl,
    children = _ref2.children;
  var _useState3 = useState(false),
    _useState4 = _slicedToArray(_useState3, 2),
    isCellEntered = _useState4[0],
    setIsCellEntered = _useState4[1];
  var _useState5 = useState(false),
    _useState6 = _slicedToArray(_useState5, 2),
    isExited = _useState6[0],
    setExited = _useState6[1];
  var ariaDescribedById = useGeneratedHtmlId({
    suffix: 'focusTrapHint'
  });

  // direct DOM manipulation as workaround to attach required hints
  useEffect(function () {
    var _currentAriaDescribed;
    var currentAriaDescribedbyId = cellEl.getAttribute('aria-describedby');
    // A11y: splitting ids to be able to append the first hint (sorting)
    // while other hints should follow the keyboard navigation hints
    var _ref3 = (_currentAriaDescribed = currentAriaDescribedbyId === null || currentAriaDescribedbyId === void 0 ? void 0 : currentAriaDescribedbyId.split(' ')) !== null && _currentAriaDescribed !== void 0 ? _currentAriaDescribed : [],
      _ref4 = _toArray(_ref3),
      sortingId = _ref4[0],
      rest = _ref4.slice(1);
    var remainingIds = rest.join(' ');
    cellEl.setAttribute('aria-describedby', classNames(sortingId, ariaDescribedById, !isCellEntered && remainingIds));
    return function () {
      if (currentAriaDescribedbyId) {
        cellEl.setAttribute('aria-describedby', currentAriaDescribedbyId);
      } else {
        cellEl.removeAttribute('aria-describedby');
      }
    };
  }, [cellEl, ariaDescribedById, isCellEntered]);
  useEffect(function () {
    if (isCellEntered) {
      enableAndFocusInteractives(cellEl);
    } else {
      disableInteractives(cellEl);
    }
  }, [isCellEntered, cellEl]);
  useEffect(function () {
    var onKeyUp = function onKeyUp(event) {
      switch (event.key) {
        case keys.ENTER:
        case keys.F2:
          event.preventDefault();
          setIsCellEntered(true);
          break;
        case keys.ESCAPE:
          event.preventDefault();
          setIsCellEntered(function (isCellEntered) {
            if (isCellEntered === true) {
              setExited(true);
              requestAnimationFrame(function () {
                return cellEl.focus();
              }); // move focus to cell
              return false;
            } else if (
            // when opened content is closed, we don't want Escape to return to the cell
            // immediately but instead return focus to a trigger as expected
            isCellEntered === false && isDOMNode(event.target) && isDOMNode(event.currentTarget) && event.currentTarget !== event.target && event.currentTarget.contains(event.target)) {
              return true;
            }
            return isCellEntered;
          });
          break;
      }
    };

    // ensures the SR text is reset when navigating to a different cell
    var onBlur = function onBlur() {
      return setExited(false);
    };
    cellEl.addEventListener('keyup', onKeyUp);
    cellEl.addEventListener('blur', onBlur);
    return function () {
      cellEl.removeEventListener('keyup', onKeyUp);
      cellEl.removeEventListener('blur', onBlur);
    };
  }, [cellEl]);
  return ___EmotionJSX(EuiFocusTrap, {
    disabled: !isCellEntered,
    clickOutsideDisables: true,
    onDeactivation: function onDeactivation() {
      return setIsCellEntered(false);
    }
  }, children, ___EmotionJSX("p", {
    id: ariaDescribedById,
    hidden: true
  }, isExited && ___EmotionJSX(EuiI18n
  // eslint-disable-next-line local/i18n
  , {
    token: "euiDataGridCell.focusTrapExitPrompt",
    default: "Exited cell content."
  }), !isCellEntered && ___EmotionJSX(EuiI18n
  // eslint-disable-next-line local/i18n
  , {
    token: "euiDataGridCell.focusTrapEnterPrompt",
    default: "Press the Enter key to interact with this cell's contents."
  })));
};

/**
 * Utility fns for managing child interactive tabIndex state
 */

var disableInteractives = function disableInteractives(cell) {
  var interactives = tabbable(cell);
  interactives.forEach(function (element) {
    element.setAttribute('data-euigrid-tab-managed', 'true');
    element.setAttribute('tabIndex', '-1');
  });
  return interactives;
};
var enableAndFocusInteractives = function enableAndFocusInteractives(cell) {
  var interactives = cell.querySelectorAll('[data-euigrid-tab-managed]');
  interactives.forEach(function (element, i) {
    element.setAttribute('tabIndex', '0');
    // focus the first element only if we're on the cell and not inside of it
    if (i === 0 && !cell.contains(document.activeElement)) {
      element.focus();
    }
  });
  return interactives;
};