/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

import React, { useMemo } from 'react';
import { useGeneratedHtmlId } from '../../../../services';
// Keep the i18n scope the same as EuiDataGridHeaderCell
/* eslint-disable local/i18n */
import { EuiI18n } from '../../../i18n';
import { EuiIcon } from '../../../icon';
import { jsx as ___EmotionJSX } from "@emotion/react";
/**
 * Column sorting utility helpers
 */
export var useColumnSorting = function useColumnSorting(_ref) {
  var _sorting$columns;
  var sorting = _ref.sorting,
    id = _ref.id,
    hasColumnActions = _ref.hasColumnActions;
  var sortedColumn = useMemo(function () {
    return sorting === null || sorting === void 0 ? void 0 : sorting.columns.find(function (col) {
      return col.id === id;
    });
  }, [sorting, id]);
  var isColumnSorted = !!sortedColumn;
  var hasOnlyOneSort = (sorting === null || sorting === void 0 || (_sorting$columns = sorting.columns) === null || _sorting$columns === void 0 ? void 0 : _sorting$columns.length) === 1;

  /**
   * Arrow icon
   */
  var sortingArrow = useMemo(function () {
    return isColumnSorted ? ___EmotionJSX(EuiIcon, {
      type: sortedColumn.direction === 'asc' ? 'sortUp' : 'sortDown',
      color: "text",
      className: "euiDataGridHeaderCell__sortingArrow",
      "data-test-subj": "dataGridHeaderCellSortingIcon-".concat(id)
    }) : null;
  }, [id, isColumnSorted, sortedColumn]);

  /**
   * aria-sort attribute - should only be used when a single column is being sorted
   * @see https://developer.mozilla.org/en-US/docs/Web/Accessibility/ARIA/Attributes/aria-sort
   * @see https://www.w3.org/WAI/ARIA/apg/example-index/table/sortable-table.html
   * @see https://github.com/w3c/aria/issues/283 for potential future multi-column usage
   */
  var ariaSort = isColumnSorted && hasOnlyOneSort ? sorting.columns[0].direction === 'asc' ? 'ascending' : 'descending' : undefined;

  // aria-describedby ID for when aria-sort isn't sufficient
  var sortingAriaId = useGeneratedHtmlId({
    prefix: 'euiDataGridCellHeader',
    suffix: 'sorting'
  });

  /**
   * Sorting status - screen reader text
   */
  var sortingScreenReaderText = useMemo(function () {
    var _sorting$columns2;
    if (!isColumnSorted) return null;
    if (!hasColumnActions && hasOnlyOneSort) return null; // in this scenario, the `aria-sort` attribute will be used by screen readers
    return ___EmotionJSX("p", {
      id: sortingAriaId,
      hidden: true
    }, sorting === null || sorting === void 0 || (_sorting$columns2 = sorting.columns) === null || _sorting$columns2 === void 0 ? void 0 : _sorting$columns2.map(function (_ref2, index) {
      var columnId = _ref2.id,
        direction = _ref2.direction;
      if (hasOnlyOneSort) {
        if (direction === 'asc') {
          return ___EmotionJSX(EuiI18n, {
            token: "euiDataGridHeaderCell.sortedByAscendingSingle",
            default: "Sorted ascending",
            key: index
          });
        } else {
          return ___EmotionJSX(EuiI18n, {
            token: "euiDataGridHeaderCell.sortedByDescendingSingle",
            default: "Sorted descending",
            key: index
          });
        }
      } else if (index === 0) {
        if (direction === 'asc') {
          return ___EmotionJSX(EuiI18n, {
            token: "euiDataGridHeaderCell.sortedByAscendingFirst",
            default: "Sorted by {columnId}, ascending",
            values: {
              columnId: columnId
            },
            key: index
          });
        } else {
          return ___EmotionJSX(EuiI18n, {
            token: "euiDataGridHeaderCell.sortedByDescendingFirst",
            default: "Sorted by {columnId}, descending",
            values: {
              columnId: columnId
            },
            key: index
          });
        }
      } else {
        if (direction === 'asc') {
          return ___EmotionJSX(EuiI18n, {
            token: "euiDataGridHeaderCell.sortedByAscendingMultiple",
            default: ", then sorted by {columnId}, ascending",
            values: {
              columnId: columnId
            },
            key: index
          });
        } else {
          return ___EmotionJSX(EuiI18n, {
            token: "euiDataGridHeaderCell.sortedByDescendingMultiple",
            default: ", then sorted by {columnId}, descending",
            values: {
              columnId: columnId
            },
            key: index
          });
        }
      }
    }), ".");
  }, [isColumnSorted, hasColumnActions, hasOnlyOneSort, sorting, sortingAriaId]);
  return {
    sortingArrow: sortingArrow,
    ariaSort: ariaSort,
    sortingAriaId: sortingAriaId,
    sortingScreenReaderText: sortingScreenReaderText
  };
};