import _slicedToArray from "@babel/runtime/helpers/slicedToArray";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

import { Children, cloneElement, useState, useEffect, useCallback } from 'react';
function isMutableRef(ref) {
  return ref != null && ref.hasOwnProperty('current');
}

/**
 * The `EuiValidatableControl` component should be used in scenarios where
 * we can render the validated `<input>` as its direct child.
 */

export var EuiValidatableControl = function EuiValidatableControl(_ref) {
  var isInvalid = _ref.isInvalid,
    children = _ref.children;
  // Note that this must be state and not a ref to cause a rerender/set invalid state on initial mount
  var _useState = useState(null),
    _useState2 = _slicedToArray(_useState, 2),
    control = _useState2[0],
    setControl = _useState2[1];
  var child = Children.only(children);
  var childRef = child.ref;
  var replacedRef = useCallback(function (element) {
    setControl(element);

    // Call the original ref, if any
    if (typeof childRef === 'function') {
      childRef(element);
    } else if (isMutableRef(childRef)) {
      childRef.current = element;
    }
  }, [childRef]);
  useSetControlValidity({
    controlEl: control,
    isInvalid: isInvalid
  });
  return /*#__PURE__*/cloneElement(child, {
    ref: replacedRef,
    'aria-invalid': isInvalid || child.props['aria-invalid']
  });
};

/**
 * The `UseEuiValidatableControl` hook should be used in scenarios where
 * we *cannot* control where the validated `<input>` is rendered (e.g., ReactDatePicker)
 * and instead need to access the input via a ref and pass the element in directly
 */

export var useEuiValidatableControl = function useEuiValidatableControl(_ref2) {
  var isInvalid = _ref2.isInvalid,
    controlEl = _ref2.controlEl;
  useSetControlValidity({
    controlEl: controlEl,
    isInvalid: isInvalid
  });
  useEffect(function () {
    if (!controlEl) return;
    if (typeof isInvalid === 'boolean') {
      controlEl.setAttribute('aria-invalid', String(isInvalid));
    } else {
      controlEl.removeAttribute('aria-invalid');
    }
  }, [isInvalid, controlEl]);
};

/**
 * Internal `setCustomValidity` helper
 */
var useSetControlValidity = function useSetControlValidity(_ref3) {
  var controlEl = _ref3.controlEl,
    isInvalid = _ref3.isInvalid;
  useEffect(function () {
    if (controlEl == null || typeof controlEl.setCustomValidity !== 'function') {
      return;
    }
    if (isInvalid) {
      controlEl.setCustomValidity('Invalid');
    } else {
      controlEl.setCustomValidity('');
    }
  }, [isInvalid, controlEl]);
};