/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

import { css } from '@emotion/react';
import { transparentize } from '../../services';
import { logicalCSS, euiFontSize, mathWithUnits } from '../../global_styling';

/**
 * Mixins
 */
// Internal utility for text scales/sizes
var euiScaleMarkdownFormatText = function euiScaleMarkdownFormatText(euiTheme, options) {
  var _euiFontSize = euiFontSize(euiTheme, 'm', options),
    fontSize = _euiFontSize.fontSize;

  // Custom scales
  var tablePaddingVertical = mathWithUnits(fontSize, function (x) {
    return x / 4;
  });
  var tablePaddingHorizontal = mathWithUnits(fontSize, function (x) {
    return x / 2;
  });
  return "\n    .euiMarkdownFormat__codeblockWrapper {\n      ".concat(logicalCSS('margin-bottom', fontSize), "\n    }\n\n    .euiMarkdownFormat__table {\n      ").concat(logicalCSS('margin-bottom', fontSize), "\n    }\n\n    .euiMarkdownFormat__table th,\n    .euiMarkdownFormat__table td {\n      ").concat(logicalCSS('padding-vertical', tablePaddingVertical), "\n      ").concat(logicalCSS('padding-horizontal', tablePaddingHorizontal), "\n    }\n  ");
};

// Internal utility for generating border colors based on EuiText colors
var euiMarkdownAdjustBorderColors = function euiMarkdownAdjustBorderColors(_ref, color) {
  var euiTheme = _ref.euiTheme;
  var border = "".concat(euiTheme.border.width.thin, " solid ").concat(color);
  return "\n    .euiMarkdownFormat__blockquote {\n      ".concat(logicalCSS('border-left-color', color), "\n    }\n\n    .euiHorizontalRule {\n      background-color: ").concat(color, ";\n      color: ").concat(color, "; /* ensure that firefox gets the currentColor */\n    }\n\n    /* Tables */\n\n    .euiMarkdownFormat__table {\n      display: block;\n      ").concat(logicalCSS('width', '100%'), "\n      overflow: auto;\n      border-spacing: 0;\n      border-collapse: collapse;\n      ").concat(logicalCSS('border-left', border), "\n    }\n\n    .euiMarkdownFormat__table th,\n    .euiMarkdownFormat__table td {\n      ").concat(logicalCSS('border-vertical', border), "\n\n      &:last-child {\n        ").concat(logicalCSS('border-right', border), "\n      }\n    }\n\n    .euiMarkdownFormat__table tr {\n      ").concat(logicalCSS('border-top', border), "\n    }\n  ");
};

/**
 * Styles
 */
export var euiMarkdownFormatStyles = function euiMarkdownFormatStyles(euiTheme) {
  return {
    euiMarkdownFormat: /*#__PURE__*/css(";label:euiMarkdownFormat;"),
    // Text sizes
    m: /*#__PURE__*/css(euiScaleMarkdownFormatText(euiTheme, {
      customScale: 'm'
    }), ";label:m;"),
    s: /*#__PURE__*/css(euiScaleMarkdownFormatText(euiTheme, {
      customScale: 's'
    }), ";label:s;"),
    xs: /*#__PURE__*/css(euiScaleMarkdownFormatText(euiTheme, {
      customScale: 'xs'
    }), ";label:xs;"),
    relative: /*#__PURE__*/css(euiScaleMarkdownFormatText(euiTheme, {
      unit: 'em'
    }), ";label:relative;"),
    colors: {
      default: /*#__PURE__*/css(euiMarkdownAdjustBorderColors(euiTheme, transparentize(euiTheme.euiTheme.colors.fullShade, 0.15)), ";label:default;"),
      subdued: /*#__PURE__*/css(euiMarkdownAdjustBorderColors(euiTheme, euiTheme.euiTheme.colors.subduedText), ";label:subdued;"),
      success: /*#__PURE__*/css(euiMarkdownAdjustBorderColors(euiTheme, euiTheme.euiTheme.colors.success), ";label:success;"),
      accent: /*#__PURE__*/css(euiMarkdownAdjustBorderColors(euiTheme, euiTheme.euiTheme.colors.accent), ";label:accent;"),
      warning: /*#__PURE__*/css(euiMarkdownAdjustBorderColors(euiTheme, euiTheme.euiTheme.colors.warning), ";label:warning;"),
      danger: /*#__PURE__*/css(euiMarkdownAdjustBorderColors(euiTheme, euiTheme.euiTheme.colors.danger), ";label:danger;"),
      ghost: /*#__PURE__*/css(euiMarkdownAdjustBorderColors(euiTheme, euiTheme.euiTheme.colors.ghost), ";label:ghost;"),
      inherit: /*#__PURE__*/css(euiMarkdownAdjustBorderColors(euiTheme, 'currentColor'), ";label:inherit;"),
      custom: /*#__PURE__*/css(euiMarkdownAdjustBorderColors(euiTheme, 'currentColor'), ";label:custom;")
    }
  };
};