import _extends from "@babel/runtime/helpers/extends";
import _slicedToArray from "@babel/runtime/helpers/slicedToArray";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

import React, { createContext, useContext, useState, useMemo, useCallback, forwardRef } from 'react';
import { useUpdateEffect } from '../hooks';
import { useEuiTheme } from './hooks';
import { emitEuiProviderWarning } from './warning';

// Typically an object of serialized css`` styles, but can have any amount of nesting, so it's not worth it to try and strictly type this
import { jsx as ___EmotionJSX } from "@emotion/react";
export var EuiThemeMemoizedStylesContext = /*#__PURE__*/createContext(new WeakMap());
export var EuiThemeMemoizedStylesProvider = function EuiThemeMemoizedStylesProvider(_ref) {
  var children = _ref.children;
  // Note: we *should not* use `useMemo` instead of `useState` here, as useMemo is not guaranteed
  // and its cache can get thrown away by React (https://react.dev/reference/react/useMemo#caveats)
  var _useState = useState(new WeakMap()),
    _useState2 = _slicedToArray(_useState, 2),
    componentStyles = _useState2[0],
    rerenderStyles = _useState2[1];

  // On theme update, wipe the map, which causes the below hook to recompute all styles
  var _useEuiTheme = useEuiTheme(),
    euiTheme = _useEuiTheme.euiTheme;
  useUpdateEffect(function () {
    rerenderStyles(new WeakMap());
  }, [euiTheme]);
  return ___EmotionJSX(EuiThemeMemoizedStylesContext.Provider, {
    value: componentStyles
  }, children);
};

/**
 * Internal util primarily responsible for getting the memoized styles (if they exist)
 * and if not, generating and setting the styles. DRYed out to facilitate usage
 * between both hook/function components and HOC/class components
 */
var getMemoizedStyles = function getMemoizedStyles(stylesGenerator, stylesMap, euiThemeContext) {
  if (!stylesGenerator.name) {
    emitEuiProviderWarning('Styles are memoized per function. Your style functions must be statically defined in order to not create a new map entry every rerender.');
  }
  var existingStyles = stylesMap.get(stylesGenerator);
  if (existingStyles) {
    return existingStyles;
  } else {
    var generatedStyles = stylesGenerator(euiThemeContext);
    stylesMap.set(stylesGenerator, generatedStyles);
    return generatedStyles;
  }
};

/**
 * Hook that memoizes the returned values of components style fns/generators
 * per-theme
 */
export var useEuiMemoizedStyles = function useEuiMemoizedStyles(stylesGenerator) {
  var memoizedStyles = useContext(EuiThemeMemoizedStylesContext);
  var euiThemeContext = useEuiTheme();
  var memoizedComponentStyles = useMemo(function () {
    return getMemoizedStyles(stylesGenerator, memoizedStyles, euiThemeContext);
  }, [stylesGenerator, memoizedStyles, euiThemeContext]);
  return memoizedComponentStyles;
};

/**
 * HOC for class components
 * Syntax is mostly copied from withEuiTheme HOC
 */

export var withEuiStylesMemoizer = function withEuiStylesMemoizer(Component) {
  var componentName = Component.displayName || Component.name || 'ComponentWithStylesMemoizer';
  var Render = function Render(props, ref) {
    var memoizedStyles = useContext(EuiThemeMemoizedStylesContext);
    var euiThemeContext = useEuiTheme();
    var stylesMemoizer = useCallback(function (stylesGenerator) {
      return getMemoizedStyles(stylesGenerator, memoizedStyles, euiThemeContext);
    }, [memoizedStyles, euiThemeContext]);
    return ___EmotionJSX(Component, _extends({
      stylesMemoizer: stylesMemoizer,
      ref: ref
    }, props));
  };
  var WithEuiStylesMemoizer = /*#__PURE__*/forwardRef(Render);
  WithEuiStylesMemoizer.displayName = componentName;
  return WithEuiStylesMemoizer;
};

/**
 * Render prop alternative for complex class components
 * Most useful for scenarios where a HOC may interfere with typing
 */
export var RenderWithEuiStylesMemoizer = function RenderWithEuiStylesMemoizer(_ref2) {
  var children = _ref2.children;
  var memoizedStyles = useContext(EuiThemeMemoizedStylesContext);
  var euiThemeContext = useEuiTheme();
  var stylesMemoizer = useCallback(function (stylesGenerator) {
    return getMemoizedStyles(stylesGenerator, memoizedStyles, euiThemeContext);
  }, [memoizedStyles, euiThemeContext]);
  return children(stylesMemoizer);
};