import _taggedTemplateLiteral from "@babel/runtime/helpers/taggedTemplateLiteral";
import _toConsumableArray from "@babel/runtime/helpers/toConsumableArray";
var _templateObject;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

import { css, keyframes } from '@emotion/react';
import { euiBackgroundColor, euiCanAnimate } from '../../../../global_styling';
import { hexToRgb, isColorDark, makeHighContrastColor, shade, tint, transparentize, useEuiMemoizedStyles } from '../../../../services';
export var BUTTON_COLORS = ['text', 'accent', 'primary', 'success', 'warning', 'danger'];
export var BUTTON_DISPLAYS = ['base', 'fill', 'empty'];
/**
 * Creates the `base` version of button styles with proper text contrast.
 * @param euiThemeContext
 * @param color One of the named button colors or 'disabled'
 * @returns Style object `{ backgroundColor, color }`
 */
export var euiButtonColor = function euiButtonColor(euiThemeContext, color) {
  var euiTheme = euiThemeContext.euiTheme,
    colorMode = euiThemeContext.colorMode;
  function tintOrShade(color) {
    return colorMode === 'DARK' ? shade(color, 0.7) : tint(color, 0.8);
  }
  var foreground;
  var background;
  switch (color) {
    case 'disabled':
      return {
        color: euiTheme.colors.disabledText,
        backgroundColor: transparentize(euiTheme.colors.lightShade, 0.15)
      };
    case 'text':
      foreground = euiTheme.colors[color];
      background = colorMode === 'DARK' ? shade(euiTheme.colors.lightShade, 0.2) : tint(euiTheme.colors.lightShade, 0.5);
      break;
    default:
      foreground = euiTheme.colors["".concat(color, "Text")];
      background = tintOrShade(euiTheme.colors[color]);
      break;
  }
  return {
    color: makeHighContrastColor(foreground)(background),
    backgroundColor: background
  };
};

/**
 * Creates the `fill` version of buttons styles with proper text contrast.
 * @param euiThemeContext
 * @param color One of the named button colors or 'disabled'
 * @returns Style object `{ backgroundColor, color }`
 */
export var euiButtonFillColor = function euiButtonFillColor(euiThemeContext, color) {
  var euiTheme = euiThemeContext.euiTheme,
    colorMode = euiThemeContext.colorMode;
  var getForegroundColor = function getForegroundColor(background) {
    return isColorDark.apply(void 0, _toConsumableArray(hexToRgb(background))) ? euiTheme.colors.ghost : euiTheme.colors.ink;
  };
  var background;
  var foreground;
  switch (color) {
    case 'disabled':
      background = euiButtonColor(euiThemeContext, color).backgroundColor;
      foreground = euiButtonColor(euiThemeContext, color).color;
      break;
    case 'text':
      background = colorMode === 'DARK' ? euiTheme.colors.text : euiTheme.colors.darkShade;
      foreground = getForegroundColor(background);
      break;
    case 'success':
    case 'accent':
      // Success / accent fills are hard to read on light mode even though they pass color contrast ratios
      // TODO: If WCAG 3 gets adopted (which would calculates luminosity & would allow us to use white text instead),
      // we can get rid of this case (https://blog.datawrapper.de/color-contrast-check-data-vis-wcag-apca/)
      background = colorMode === 'LIGHT' ? tint(euiTheme.colors[color], 0.3) : euiTheme.colors[color];
      foreground = getForegroundColor(background);
      break;
    default:
      background = euiTheme.colors[color];
      foreground = getForegroundColor(background);
      break;
  }
  return {
    color: foreground,
    backgroundColor: background
  };
};

/**
 * Creates the `empty` version of button styles using the text-variant and adding interactive styles.
 * @param euiThemeContext
 * @param color One of the named button colors or 'disabled'
 * @returns Style object `{ backgroundColor, color }` where `background` is typically used for interactive states
 */
export var euiButtonEmptyColor = function euiButtonEmptyColor(euiThemeContext, color) {
  var foreground;
  var background;
  switch (color) {
    case 'disabled':
      foreground = euiButtonColor(euiThemeContext, color).color;
      background = 'transparent';
      break;
    case 'text':
      foreground = euiButtonColor(euiThemeContext, color).color;
      background = euiBackgroundColor(euiThemeContext, 'subdued', {
        method: 'transparent'
      });
      break;
    default:
      foreground = euiButtonColor(euiThemeContext, color).color;
      background = euiBackgroundColor(euiThemeContext, color, {
        method: 'transparent'
      });
      break;
  }
  return {
    color: foreground,
    backgroundColor: background
  };
};

/**
 * Given the button display type, returns the Emotion based color keys.
 * @param options Button display type
 * @returns An object of `_EuiButtonColor` keys including `disabled`
 */
export var useEuiButtonColorCSS = function useEuiButtonColorCSS() {
  var options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
  var _options$display = options.display,
    display = _options$display === void 0 ? 'base' : _options$display;
  var colorsDisplaysMap = useEuiMemoizedStyles(euiButtonDisplaysColors);
  return colorsDisplaysMap[display];
};
var euiButtonDisplaysColors = function euiButtonDisplaysColors(euiThemeContext) {
  var COLORS = [].concat(BUTTON_COLORS, ['disabled']);
  var displaysColorsMap = {};
  BUTTON_DISPLAYS.forEach(function (display) {
    displaysColorsMap[display] = {};
    COLORS.forEach(function (color) {
      switch (display) {
        case 'base':
          displaysColorsMap[display][color] = /*#__PURE__*/css(euiButtonColor(euiThemeContext, color), ";;label:displaysColorsMap-display-color;");
          break;
        case 'fill':
          displaysColorsMap[display][color] = /*#__PURE__*/css(euiButtonFillColor(euiThemeContext, color), "outline-color:", euiThemeContext.colorMode === 'DARK' && color === 'text' ? 'currentColor' : euiThemeContext.euiTheme.colors.fullShade, ";;label:displaysColorsMap-display-color;");
          break;
        case 'empty':
          displaysColorsMap[display][color] = /*#__PURE__*/css("color:", euiButtonEmptyColor(euiThemeContext, color).color, ";&:focus,&:active{background-color:", euiButtonEmptyColor(euiThemeContext, color).backgroundColor, ";};label:displaysColorsMap-display-color;");
          break;
      }

      // Tweak auto-generated Emotion label/className output
      var emotionOutput = displaysColorsMap[display][color];
      emotionOutput.styles = emotionOutput.styles.replace('label:displaysColorsMap-display-color;', "label:".concat(display, "-").concat(color, ";"));
    });
  });
  return displaysColorsMap;
};

/**
 * Creates the translate animation when button is in focus.
 * @returns string
 */
export var useEuiButtonFocusCSS = function useEuiButtonFocusCSS() {
  return useEuiMemoizedStyles(euiButtonFocusCSS);
};
var euiButtonFocusAnimation = keyframes(_templateObject || (_templateObject = _taggedTemplateLiteral(["\n  50% {\n    transform: translateY(1px);\n  }\n"])));
var euiButtonFocusCSS = function euiButtonFocusCSS(_ref) {
  var euiTheme = _ref.euiTheme;
  var focusCSS = /*#__PURE__*/css(euiCanAnimate, "{transition:transform ", euiTheme.animation.normal, " ease-in-out,background-color ", euiTheme.animation.normal, " ease-in-out;&:hover:not(:disabled){transform:translateY(-1px);}&:focus{animation:", euiButtonFocusAnimation, " ", euiTheme.animation.normal, " ", euiTheme.animation.bounce, ";}&:active:not(:disabled){transform:translateY(1px);}};label:focusCSS;");
  // Remove the auto-generated label.
  // We could typically avoid a label by using a plain string `` instead of css``,
  // but we need css`` for keyframes`` to work for the focus animation
  focusCSS.styles = focusCSS.styles.replace('label:focusCSS;', '');
  return focusCSS;
};

/**
 * Map of `size` props to various sizings/scales
 * that should remain consistent across all buttons
 */
export var euiButtonSizeMap = function euiButtonSizeMap(_ref2) {
  var euiTheme = _ref2.euiTheme;
  return {
    xs: {
      height: euiTheme.size.l,
      radius: euiTheme.border.radius.small,
      fontScale: 'xs'
    },
    s: {
      height: euiTheme.size.xl,
      radius: euiTheme.border.radius.small,
      fontScale: 's'
    },
    m: {
      height: euiTheme.size.xxl,
      radius: euiTheme.border.radius.medium,
      fontScale: 's'
    }
  };
};