"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transparentize = exports.tintOrShade = exports.tint = exports.shadeOrTint = exports.shade = exports.saturate = exports.lightness = exports.desaturate = exports.darken = exports.brighten = void 0;
var _chromaJs = _interopRequireDefault(require("chroma-js"));
var _is_valid_hex = require("./is_valid_hex");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

var inOriginalFormat = function inOriginalFormat(originalColor, newColor) {
  return (0, _is_valid_hex.isValidHex)(originalColor) ? newColor.hex() : newColor.css();
};

/**
 * Makes a color more transparent.
 * @param color - Color to manipulate
 * @param alpha - alpha channel value. From 0-1.
 */
var transparentize = exports.transparentize = function transparentize(color, alpha) {
  return (0, _chromaJs.default)(color).alpha(alpha).css();
};

/**
 * Mixes a provided color with white.
 * @param color - Color to mix with white
 * @param ratio - Mix weight. From 0-1. Larger value indicates more white.
 */
var tint = exports.tint = function tint(color, ratio) {
  var tint = _chromaJs.default.mix(color, '#fff', ratio, 'rgb');
  return inOriginalFormat(color, tint);
};

/**
 * Mixes a provided color with black.
 * @param color - Color to mix with black
 * @param ratio - Mix weight. From 0-1. Larger value indicates more black.
 */
var shade = exports.shade = function shade(color, ratio) {
  var shade = _chromaJs.default.mix(color, '#000', ratio, 'rgb');
  return inOriginalFormat(color, shade);
};

/**
 * Returns the tinted color for light mode and shaded color for dark mode
 * @param color - Color to mix with white
 * @param ratio - Mix weight. From 0-1. Larger value indicates more white.
 * @param colorMode - Light or dark only
 */
var tintOrShade = exports.tintOrShade = function tintOrShade(color, ratio, colorMode) {
  return colorMode === 'DARK' ? shade(color, ratio) : tint(color, ratio);
};

/**
 * Returns the shaded color for light mode and tinted color for dark mode
 * @param color - Color to mix with white
 * @param ratio - Mix weight. From 0-1. Larger value indicates more white.
 * @param colorMode - Light or dark only
 */
var shadeOrTint = exports.shadeOrTint = function shadeOrTint(color, ratio, colorMode) {
  return colorMode === 'DARK' ? tint(color, ratio) : shade(color, ratio);
};

/**
 * Increases the saturation of a color by manipulating the hsl saturation.
 * @param color - Color to manipulate
 * @param amount - Amount to change in absolute terms. 0-1.
 */
var saturate = exports.saturate = function saturate(color, amount) {
  var saturate = (0, _chromaJs.default)(color).set('hsl.s', "+".concat(amount));
  return inOriginalFormat(color, saturate);
};

/**
 * Decreases the saturation of a color by manipulating the hsl saturation.
 * @param color - Color to manipulate
 * @param amount - Amount to change in absolute terms. 0-1.
 */
var desaturate = exports.desaturate = function desaturate(color, amount) {
  var desaturate = (0, _chromaJs.default)(color).set('hsl.s', "-".concat(amount));
  return inOriginalFormat(color, desaturate);
};

/**
 * Returns the lightness value of a color. 0-100
 * @param color
 */
var lightness = exports.lightness = function lightness(color) {
  return (0, _chromaJs.default)(color).get('hsl.l') * 100;
};

/**
 * Returns the darken value of a color. 0-100
 * @param color - Color to manipulate
 * @param amount - Amount to change in absolute terms. 0-1.
 */
var darken = exports.darken = function darken(color, amount) {
  return (0, _chromaJs.default)(color).darken(amount).hex();
};

/**
 * Returns the brighten value of a color. 0-100
 * @param color - Color to manipulate
 * @param amount - Amount to change in absolute terms. 0-1.
 */
var brighten = exports.brighten = function brighten(color, amount) {
  return (0, _chromaJs.default)(color).brighten(amount).hex();
};