import { HeadersAdapter } from '../web/spec-extension/adapters/headers';
import { workAsyncStorage } from '../app-render/work-async-storage.external';
import { getExpectedRequestStore } from '../app-render/work-unit-async-storage.external';
import { workUnitAsyncStorage } from '../app-render/work-unit-async-storage.external';
import { postponeWithTracking, abortAndThrowOnSynchronousRequestDataAccess, throwToInterruptStaticGeneration, trackDynamicDataInDynamicRender } from '../app-render/dynamic-rendering';
import { StaticGenBailoutError } from '../../client/components/static-generation-bailout';
import { makeResolvedReactPromise } from './utils';
import { makeHangingPromise } from '../dynamic-rendering-utils';
import { createDedupedByCallsiteServerErrorLoggerDev } from '../create-deduped-by-callsite-server-error-loger';
/**
 * This function allows you to read the HTTP incoming request headers in
 * [Server Components](https://nextjs.org/docs/app/building-your-application/rendering/server-components),
 * [Server Actions](https://nextjs.org/docs/app/building-your-application/data-fetching/server-actions-and-mutations),
 * [Route Handlers](https://nextjs.org/docs/app/building-your-application/routing/route-handlers) and
 * [Middleware](https://nextjs.org/docs/app/building-your-application/routing/middleware).
 *
 * Read more: [Next.js Docs: `headers`](https://nextjs.org/docs/app/api-reference/functions/headers)
 */ export function headers() {
    const workStore = workAsyncStorage.getStore();
    const workUnitStore = workUnitAsyncStorage.getStore();
    if (workStore) {
        if (workStore.forceStatic) {
            // When using forceStatic we override all other logic and always just return an empty
            // headers object without tracking
            const underlyingHeaders = HeadersAdapter.seal(new Headers({}));
            return makeUntrackedExoticHeaders(underlyingHeaders);
        }
        if (workUnitStore) {
            if (workUnitStore.type === 'cache') {
                throw new Error(`Route ${workStore.route} used "headers" inside "use cache". Accessing Dynamic data sources inside a cache scope is not supported. If you need this data inside a cached function use "headers" outside of the cached function and pass the required dynamic data in as an argument. See more info here: https://nextjs.org/docs/messages/next-request-in-use-cache`);
            } else if (workUnitStore.type === 'unstable-cache') {
                throw new Error(`Route ${workStore.route} used "headers" inside a function cached with "unstable_cache(...)". Accessing Dynamic data sources inside a cache scope is not supported. If you need this data inside a cached function use "headers" outside of the cached function and pass the required dynamic data in as an argument. See more info here: https://nextjs.org/docs/app/api-reference/functions/unstable_cache`);
            } else if (workUnitStore.phase === 'after') {
                throw new Error(`Route ${workStore.route} used "headers" inside "unstable_after(...)". This is not supported. If you need this data inside an "unstable_after" callback, use "headers" outside of the callback. See more info here: https://nextjs.org/docs/canary/app/api-reference/functions/unstable_after`);
            }
        }
        if (workStore.dynamicShouldError) {
            throw new StaticGenBailoutError(`Route ${workStore.route} with \`dynamic = "error"\` couldn't be rendered statically because it used \`headers\`. See more info here: https://nextjs.org/docs/app/building-your-application/rendering/static-and-dynamic#dynamic-rendering`);
        }
        if (workUnitStore) {
            if (workUnitStore.type === 'prerender') {
                // dynamicIO Prerender
                // We don't track dynamic access here because access will be tracked when you access
                // one of the properties of the headers object.
                return makeDynamicallyTrackedExoticHeaders(workStore.route, workUnitStore);
            } else if (workUnitStore.type === 'prerender-ppr') {
                // PPR Prerender (no dynamicIO)
                // We are prerendering with PPR. We need track dynamic access here eagerly
                // to keep continuity with how headers has worked in PPR without dynamicIO.
                // TODO consider switching the semantic to throw on property access instead
                postponeWithTracking(workStore.route, 'headers', workUnitStore.dynamicTracking);
            } else if (workUnitStore.type === 'prerender-legacy') {
                // Legacy Prerender
                // We are in a legacy static generation mode while prerendering
                // We track dynamic access here so we don't need to wrap the headers in
                // individual property access tracking.
                throwToInterruptStaticGeneration('headers', workStore, workUnitStore);
            }
        }
        // We fall through to the dynamic context below but we still track dynamic access
        // because in dev we can still error for things like using headers inside a cache context
        trackDynamicDataInDynamicRender(workStore, workUnitStore);
    }
    const requestStore = getExpectedRequestStore('headers');
    if (process.env.NODE_ENV === 'development' && !(workStore == null ? void 0 : workStore.isPrefetchRequest)) {
        return makeUntrackedExoticHeadersWithDevWarnings(requestStore.headers, workStore == null ? void 0 : workStore.route);
    } else {
        return makeUntrackedExoticHeaders(requestStore.headers);
    }
}
const CachedHeaders = new WeakMap();
function makeDynamicallyTrackedExoticHeaders(route, prerenderStore) {
    const cachedHeaders = CachedHeaders.get(prerenderStore);
    if (cachedHeaders) {
        return cachedHeaders;
    }
    const promise = makeHangingPromise(prerenderStore.renderSignal, '`headers()`');
    CachedHeaders.set(prerenderStore, promise);
    Object.defineProperties(promise, {
        append: {
            value: function append() {
                const expression = `headers().append(${describeNameArg(arguments[0])}, ...)`;
                const error = createSyncHeadersError(route, expression);
                abortAndThrowOnSynchronousRequestDataAccess(route, expression, error, prerenderStore);
            }
        },
        delete: {
            value: function _delete() {
                const expression = `headers().delete(${describeNameArg(arguments[0])})`;
                const error = createSyncHeadersError(route, expression);
                abortAndThrowOnSynchronousRequestDataAccess(route, expression, error, prerenderStore);
            }
        },
        get: {
            value: function get() {
                const expression = `headers().get(${describeNameArg(arguments[0])})`;
                const error = createSyncHeadersError(route, expression);
                abortAndThrowOnSynchronousRequestDataAccess(route, expression, error, prerenderStore);
            }
        },
        has: {
            value: function has() {
                const expression = `headers().has(${describeNameArg(arguments[0])})`;
                const error = createSyncHeadersError(route, expression);
                abortAndThrowOnSynchronousRequestDataAccess(route, expression, error, prerenderStore);
            }
        },
        set: {
            value: function set() {
                const expression = `headers().set(${describeNameArg(arguments[0])}, ...)`;
                const error = createSyncHeadersError(route, expression);
                abortAndThrowOnSynchronousRequestDataAccess(route, expression, error, prerenderStore);
            }
        },
        getSetCookie: {
            value: function getSetCookie() {
                const expression = `headers().getSetCookie()`;
                const error = createSyncHeadersError(route, expression);
                abortAndThrowOnSynchronousRequestDataAccess(route, expression, error, prerenderStore);
            }
        },
        forEach: {
            value: function forEach() {
                const expression = `headers().forEach(...)`;
                const error = createSyncHeadersError(route, expression);
                abortAndThrowOnSynchronousRequestDataAccess(route, expression, error, prerenderStore);
            }
        },
        keys: {
            value: function keys() {
                const expression = `headers().keys()`;
                const error = createSyncHeadersError(route, expression);
                abortAndThrowOnSynchronousRequestDataAccess(route, expression, error, prerenderStore);
            }
        },
        values: {
            value: function values() {
                const expression = `headers().values()`;
                const error = createSyncHeadersError(route, expression);
                abortAndThrowOnSynchronousRequestDataAccess(route, expression, error, prerenderStore);
            }
        },
        entries: {
            value: function entries() {
                const expression = `headers().entries()`;
                const error = createSyncHeadersError(route, expression);
                abortAndThrowOnSynchronousRequestDataAccess(route, expression, error, prerenderStore);
            }
        },
        [Symbol.iterator]: {
            value: function() {
                const expression = 'headers()[Symbol.iterator]()';
                const error = createSyncHeadersError(route, expression);
                abortAndThrowOnSynchronousRequestDataAccess(route, expression, error, prerenderStore);
            }
        }
    });
    return promise;
}
function makeUntrackedExoticHeaders(underlyingHeaders) {
    const cachedHeaders = CachedHeaders.get(underlyingHeaders);
    if (cachedHeaders) {
        return cachedHeaders;
    }
    const promise = makeResolvedReactPromise(underlyingHeaders);
    CachedHeaders.set(underlyingHeaders, promise);
    Object.defineProperties(promise, {
        append: {
            value: underlyingHeaders.append.bind(underlyingHeaders)
        },
        delete: {
            value: underlyingHeaders.delete.bind(underlyingHeaders)
        },
        get: {
            value: underlyingHeaders.get.bind(underlyingHeaders)
        },
        has: {
            value: underlyingHeaders.has.bind(underlyingHeaders)
        },
        set: {
            value: underlyingHeaders.set.bind(underlyingHeaders)
        },
        getSetCookie: {
            value: underlyingHeaders.getSetCookie.bind(underlyingHeaders)
        },
        forEach: {
            value: underlyingHeaders.forEach.bind(underlyingHeaders)
        },
        keys: {
            value: underlyingHeaders.keys.bind(underlyingHeaders)
        },
        values: {
            value: underlyingHeaders.values.bind(underlyingHeaders)
        },
        entries: {
            value: underlyingHeaders.entries.bind(underlyingHeaders)
        },
        [Symbol.iterator]: {
            value: underlyingHeaders[Symbol.iterator].bind(underlyingHeaders)
        }
    });
    return promise;
}
function makeUntrackedExoticHeadersWithDevWarnings(underlyingHeaders, route) {
    const cachedHeaders = CachedHeaders.get(underlyingHeaders);
    if (cachedHeaders) {
        return cachedHeaders;
    }
    const promise = makeResolvedReactPromise(underlyingHeaders);
    CachedHeaders.set(underlyingHeaders, promise);
    Object.defineProperties(promise, {
        append: {
            value: function append() {
                const expression = `headers().append(${describeNameArg(arguments[0])}, ...)`;
                warnForSyncAccess(route, expression);
                return underlyingHeaders.append.apply(underlyingHeaders, arguments);
            }
        },
        delete: {
            value: function _delete() {
                const expression = `headers().delete(${describeNameArg(arguments[0])})`;
                warnForSyncAccess(route, expression);
                return underlyingHeaders.delete.apply(underlyingHeaders, arguments);
            }
        },
        get: {
            value: function get() {
                const expression = `headers().get(${describeNameArg(arguments[0])})`;
                warnForSyncAccess(route, expression);
                return underlyingHeaders.get.apply(underlyingHeaders, arguments);
            }
        },
        has: {
            value: function has() {
                const expression = `headers().has(${describeNameArg(arguments[0])})`;
                warnForSyncAccess(route, expression);
                return underlyingHeaders.has.apply(underlyingHeaders, arguments);
            }
        },
        set: {
            value: function set() {
                const expression = `headers().set(${describeNameArg(arguments[0])}, ...)`;
                warnForSyncAccess(route, expression);
                return underlyingHeaders.set.apply(underlyingHeaders, arguments);
            }
        },
        getSetCookie: {
            value: function getSetCookie() {
                const expression = `headers().getSetCookie()`;
                warnForSyncAccess(route, expression);
                return underlyingHeaders.getSetCookie.apply(underlyingHeaders, arguments);
            }
        },
        forEach: {
            value: function forEach() {
                const expression = `headers().forEach(...)`;
                warnForSyncAccess(route, expression);
                return underlyingHeaders.forEach.apply(underlyingHeaders, arguments);
            }
        },
        keys: {
            value: function keys() {
                const expression = `headers().keys()`;
                warnForSyncAccess(route, expression);
                return underlyingHeaders.keys.apply(underlyingHeaders, arguments);
            }
        },
        values: {
            value: function values() {
                const expression = `headers().values()`;
                warnForSyncAccess(route, expression);
                return underlyingHeaders.values.apply(underlyingHeaders, arguments);
            }
        },
        entries: {
            value: function entries() {
                const expression = `headers().entries()`;
                warnForSyncAccess(route, expression);
                return underlyingHeaders.entries.apply(underlyingHeaders, arguments);
            }
        },
        [Symbol.iterator]: {
            value: function() {
                warnForSyncIteration(route);
                return underlyingHeaders[Symbol.iterator].apply(underlyingHeaders, arguments);
            }
        }
    });
    return promise;
}
function describeNameArg(arg) {
    return typeof arg === 'string' ? `'${arg}'` : '...';
}
const noop = ()=>{};
const warnForSyncIteration = process.env.__NEXT_DISABLE_SYNC_DYNAMIC_API_WARNINGS ? noop : createDedupedByCallsiteServerErrorLoggerDev(function getSyncIterationMessage(route) {
    const prefix = route ? ` In route ${route} ` : '';
    return new Error(`${prefix}headers were iterated over. ` + `\`headers()\` should be awaited before using its value. ` + `Learn more: https://nextjs.org/docs/messages/sync-dynamic-apis`);
});
const warnForSyncAccess = process.env.__NEXT_DISABLE_SYNC_DYNAMIC_API_WARNINGS ? noop : createDedupedByCallsiteServerErrorLoggerDev(function getSyncAccessMessage(route, expression) {
    const prefix = route ? ` In route ${route} a ` : 'A ';
    return new Error(`${prefix}header property was accessed directly with \`${expression}\`. ` + `\`headers()\` should be awaited before using its value. ` + `Learn more: https://nextjs.org/docs/messages/sync-dynamic-apis`);
});
function createSyncHeadersError(route, expression) {
    return new Error(`Route "${route}" used ${expression}. \`headers()\` now returns a Promise and should be \`awaited\` before using it's value. See more info here: https://nextjs.org/docs/messages/next-prerender-sync-headers`);
}

//# sourceMappingURL=headers.js.map