import React, { useState, useEffect } from 'react';
import { AnimatePresence, motion } from 'framer-motion';
import {
    useProjectLoader,
    useScreenManager
} from './hooks';
import {
    EuiPanel,
    EuiTitle,
    EuiText,
    EuiSpacer,
    EuiButton,
    EuiProgress,
    EuiLoadingSpinner
} from './components/ui';
import {
    FileUploadComponent,
    ProjectTypeCard
} from './components/forms';
import { ModuleCard } from './components/cards';
import { Background, Container } from './components/layout';
import { WelcomeScreen, LoadingScreen } from './components/screens';
import {
    SCREEN_TYPES,
    animations,
    isSmallScreen,
    isMediumScreen
} from '../setupApp/utils';
import { LoadingAP } from './components/screens/LoadingAP';
import AP from '@/app/types/AP';
import getDict from '@/lib/dict';
import { NetworkSetupScreen } from './components/screens/NetworkSetupScreen';
import { useNetworkManager } from './hooks/useNetworkManager';
import { useLoader } from '@/context/LoaderContext';
import MultiStepDemo from './components/ui/StepProgress';
import { useProjectManager } from '@/hooks/useProjectManager';
const dict = getDict("de");
export default function Test({ setCurrentScreen, status, savedProjects }) {
    const { navigateTo } = useScreenManager();
    const { loading, progress, setProgress, projects, completed, loadProjects } = useProjectLoader();
    const [uploadedFile, setUploadedFile] = useState(null);
    const [projectName, setProjectName] = useState('');
    const [projectType, setProjectType] = useState('');
    const [loadingProgress, setLoadingProgress] = useState(0);
    const [aps, setAPs] = React.useState<AP[]>([]);
    const { networkName, setNetworkName, networkType, setNetworkType, saveNetwork, saved, apiLoading } = useNetworkManager();
    const { currentScreen, setLoading, completeCurrentStep, setCurrentStep } = useLoader();

    useEffect(() => {
        if (completed && currentScreen === SCREEN_TYPES.LOADING_NETWORK) {
            setTimeout(() => {
                navigateTo(SCREEN_TYPES.NETWORK_SETUP);
            }, 500);
        }
        if (completed && currentScreen === SCREEN_TYPES.LOADING_AP) {
            setTimeout(() => {
                navigateTo(SCREEN_TYPES.SUCCESS);
            }, 2000);
        }
        if (completed && currentScreen === SCREEN_TYPES.SUCCESS) {
            setTimeout(() => {
                completeCurrentStep();
                setCurrentStep(prev => prev + 1);
                navigateTo(SCREEN_TYPES.UPLOAD);
            }, 4000);
        }
    }, [completed, currentScreen, navigateTo]);
    useEffect(() => {
        if (saved && currentScreen === SCREEN_TYPES.NETWORK_SETUP) {
            setLoading(true);
            setTimeout(() => {
                setCurrentStep(prev => prev + 1);
                completeCurrentStep();
                navigateTo(SCREEN_TYPES.LOADING_AP);
                setLoading(false);
            }, 2500);
        } else if (!saved && currentScreen === SCREEN_TYPES.NETWORK_SETUP) {
            setTimeout(() => {
                navigateTo(SCREEN_TYPES.FAIL);
            }, 2500);
        }

    },[apiLoading]);

    const handleGetStarted = () => {
        setCurrentStep(prev => prev + 1);
        completeCurrentStep();
        loadProjects();
    };

    const gridStyle = {
        display: 'grid',
        gridTemplateColumns: isSmallScreen() ? '1fr' : 'repeat(2, 1fr)',
        gap: isSmallScreen() ? '16px' : '20px'
    };
    useEffect(() => {
        setCurrentScreen(currentScreen);
    },[currentScreen]);
    const handleSaveNetwork = async () => {
        setLoading(true);
        setProgress(0);
        const networkData = {
            name: networkName.trim(),
            type: 'business',
        };

        try {
            await saveNetwork(networkData);
        } catch (error) {
            console.log(`Error saving project: ${error.message}`);
        } finally {
            setLoading(false);
        }
    };
    return (
        <>
            {currentScreen === SCREEN_TYPES.WELCOME && (
                <WelcomeScreen onNext={handleGetStarted} status={status} savedProjects={savedProjects} />
            )}
            {currentScreen === SCREEN_TYPES.LOADING_NETWORK && (
                <LoadingScreen text={dict.networkLoad} />
            )}
            {currentScreen === SCREEN_TYPES.LOADING_AP && (
                <LoadingScreen text={dict.apLoad} />
            )}

            {currentScreen === SCREEN_TYPES.NETWORK_SETUP && (
                <NetworkSetupScreen
                    networkName={networkName}
                    setNetworkName={setNetworkName}
                    onContinue={async () => {
                        await handleSaveNetwork();
                    }}
                    onSkip={() => {
                        setTimeout(() => {
                            setCurrentStep(prev => prev + 1);
                            completeCurrentStep();
                            navigateTo(SCREEN_TYPES.LOADING_AP);
                            setLoading(false);
                        }, 2500);
                    }}
                />
            )}
            {currentScreen === SCREEN_TYPES.SUCCESS && (
                <motion.div
                    key="success"
                    variants={animations.pageVariants}
                    initial="initial"
                    animate="in"
                    exit="out"
                >
                    <motion.div
                        variants={animations.containerVariants}
                        initial="hidden"
                        animate="visible"
                    >
                        <EuiPanel paddingSize="l" style={{ margin: '0 12px', textAlign: 'center' }}>
                            <motion.div
                                variants={animations.itemVariants}
                                initial={{ scale: 0 }}
                                animate={{ scale: 1 }}
                                transition={{
                                    type: "spring",
                                    stiffness: 200,
                                    damping: 15,
                                    delay: 0.2
                                }}
                                style={{
                                    fontSize: isSmallScreen() ? '48px' : '64px',
                                    marginBottom: isSmallScreen() ? '20px' : '24px',
                                    filter: 'drop-shadow(0 4px 12px rgba(220, 38, 38, 0.3))'
                                }}
                            >
                                ✅
                            </motion.div>

                            <motion.div variants={animations.itemVariants}>
                                <EuiTitle size="l">{dict.accessPoint} {dict.found}</EuiTitle>
                            </motion.div>

                            <EuiSpacer size="m" />

                            <LoadingAP aps={aps} setAPs={setAPs} />

                            <EuiSpacer size="l" />

                            <motion.div
                                initial={{ width: 0 }}
                                animate={{ width: '100%' }}
                                transition={{ duration: 2, ease: "easeInOut" }}
                                style={{
                                    height: '4px',
                                    backgroundColor: '#dc2626',
                                    borderRadius: '2px',
                                    margin: '0 auto',
                                    maxWidth: '200px'
                                }}
                            />
                        </EuiPanel>
                    </motion.div>
                </motion.div>
            )}
        </>
    );
}
