import React from 'react';
import { AnimatePresence, motion } from 'framer-motion';
import AP from '@/app/types/AP';
import { useToast } from '@/app/components/ToastProvider';
import { useLoader } from '@/context/LoaderContext';
import { getAccessPoints } from '@/services/accessPoints';
import {ApiError, getErrorCode } from '@/lib/errorsCodes';
import getDict from '@/lib/dict';
import { checkResponseShape } from '@/lib/checkResponseShape';
import {
    EuiSpacer,
    EuiText,
} from '../ui';
import { animations } from "@/setupApp/utils";
const dict = getDict("de");

export const LoadingAP = ({ aps, setAPs }) => {
    const { addToast } = useToast();
    const { setLoading } = useLoader();

    async function fetchAPs() {
        try {
            setLoading(true);
            const response = await getAccessPoints();
            if (!response.ok) {
                const errorCode = getErrorCode('ACCESS_POINT_FETCH', response.status);
                addToast({
                    title: errorCode,
                    color: "danger",
                    text: dict.apFetchToastFail,
                });
                setLoading(false);
                return;
            }
            const data = await response.json();
            // Dont rely on backend, double check response schema
            if (checkResponseShape(data)) {
                setAPs(data);
            } else {
                addToast({
                    title: 'Fetch Error',
                    color: "danger",
                    text: dict.apFetchToastFail,
                });
            }
            setLoading(false);
        } catch (err) {
            const error = err as ApiError;
            const errorCode = getErrorCode('ACCESS_POINT_EXCEPTION', error.status);
            setLoading(false);
            addToast({
                title: errorCode,
                color: "danger",
                text: dict.apFetchToastFail,
            });
        }
    }
    React.useEffect(() => {
        fetchAPs();
    },[])
    return (<>
        <motion.div variants={animations.itemVariants}>
            <EuiText style={{ fontSize: '18px', fontWeight: '600', color: '#dc2626' }}>
                {aps.length} {dict.detected}
            </EuiText>
        </motion.div>

        <EuiSpacer size="s" />

        <motion.div variants={animations.itemVariants}>
            <EuiText color="subdued">
                {dict.autoRedirect}
            </EuiText>
        </motion.div>
    </>)
}
