import React from 'react';
import { motion, animations } from 'framer-motion';
import { EuiPanel, EuiTitle, EuiText,EuiFlexItem, EuiSpacer, EuiFormRow, EuiFieldText,
    EuiProgress, EuiLoadingSpinner, EuiButton, EuiFlexGroup
} from '@elastic/eui';
import { ProjectTypeCard } from '../forms/ProjectTypeCard';
import getDict from '@/lib/dict';
import Network from '@/app/types/Network';
import { useLoader } from '@/context/LoaderContext';
import {ApiError, getErrorCode } from '@/lib/errorsCodes';
import { useToast } from '@/app/components/ToastProvider';
import { getNetworks } from '@/services/network';
import { isSmallScreen } from '@/setupApp/utils/helpers';
const dict = getDict("de");
export const NetworkSetupScreen = ({ networkName, setNetworkName, onContinue, onSkip }) => {
    const [networks, setNetworks] = React.useState<Network[]>([]);
    const { setLoading, loading } = useLoader();
    const { addToast } = useToast();

    async function fetchNetworks() {
        try {
            setLoading(true);
            const response = await getNetworks();
            if (!response.ok) {
                const errorCode = getErrorCode('NETWORK_FETCH', response.status);
                addToast({
                    title: errorCode,
                    color: "danger",
                    text: `${dict.networkFetchToastFail}`,
                });
                setLoading(false);
                return;
            }
            const data = await response.json();
            setNetworks(data);
            setLoading(false);
        } catch (err) {
            const error = err as ApiError;
            setLoading(false);
            const errorCode = getErrorCode('NETWORK_EXCEPTION', error.status);
            addToast({
                title: errorCode,
                color: "danger",
                text: `${dict.networkFetchToastFail}`,
            });
        }
    }
    React.useEffect(() => {
        fetchNetworks();
    } ,[])
    const isSmall = React.useMemo(() => isSmallScreen(), []);

    return (
        <motion.div
            key="project-modal"
            variants={animations.pageVariants}
            initial="initial"
            animate="in"
            exit="out"
        >
            <EuiPanel paddingSize="l" style={{margin: '0 12px'}}>
                {/* Header */}
                <motion.div
                    initial={{ opacity: 0, y: 20 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ delay: 0.1, duration: 0.4 }}
                    style={{textAlign: 'center', marginBottom: '28px'}}
                >
                    <EuiTitle size="m">
                        <h2>{networks.length === 0 ? dict.noNetwork : `${dict.network} ${dict.create}`}</h2>
                    </EuiTitle>

                    <EuiSpacer size="s"/>

                    <EuiText color="subdued">
                        {networks.length > 0 && `${networks.length} ${dict.network}${networks.length > 1 ? '(en)' : ''} ${dict.found}`}
                    </EuiText>

                    <EuiText color="subdued">
                        {networks.length === 0
                            ? dict.networkInfo
                            : `${dict.networkInfo} oder ${dict.skip}`
                        }
                    </EuiText>
                </motion.div>

                {/* Input Field */}
                <motion.div
                    initial={{ opacity: 0, y: 20 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ delay: 0.2, duration: 0.4 }}
                    style={{marginBottom: '24px'}}
                >
                    <EuiFormRow
                        label={dict.name}
                        fullWidth
                    >
                        <EuiFieldText
                            value={networkName}
                            onChange={(e) => setNetworkName(e.target.value)}
                            placeholder={dict.network}
                            fullWidth
                            disabled={loading}
                            autoFocus
                        />
                    </EuiFormRow>
                </motion.div>

                {/* Action Buttons */}
                <motion.div
                    initial={{ opacity: 0, y: 20 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ delay: 0.3, duration: 0.4 }}
                >
                    <EuiFlexGroup
                        justifyContent="center"
                        gutterSize="s"
                        responsive={false}
                        direction={isSmall ? "column" : "row"}
                    >
                        {loading ? (
                            <EuiFlexItem grow={false}>
                                <EuiLoadingSpinner size="l"/>
                            </EuiFlexItem>
                        ) : (
                            <>
                                <EuiFlexItem grow={false}>
                                    <EuiButton
                                        fill
                                        onClick={onContinue}
                                        disabled={!networkName.trim()}
                                        fullWidth={isSmall}
                                    >
                                        {dict.create.charAt(0).toUpperCase() + dict.create.slice(1)}
                                    </EuiButton>
                                </EuiFlexItem>

                                <EuiFlexItem grow={false}>
                                    <EuiButton
                                        onClick={() => {
                                            setLoading(true);
                                            return onSkip();
                                        }}
                                        disabled={networks.length === 0 && !networkName.trim()}
                                        fullWidth={isSmall}
                                    >
                                        {dict.skip}
                                    </EuiButton>
                                </EuiFlexItem>
                            </>
                        )}
                    </EuiFlexGroup>
                </motion.div>
            </EuiPanel>
        </motion.div>
    );
}
