import React from 'react';
import Image from "next/image";
import { motion } from 'framer-motion';
import { EuiFlexItem, EuiFlexGroup } from '@elastic/eui';
import { EuiPanel, EuiTitle, EuiText, EuiSpacer, EuiButton } from '@/setupApp/components/ui';
import { animations } from '@/setupApp/utils/animations';
import { isSmallScreen } from '@/setupApp/utils/helpers';
import branding from "@/lib/branding";

import getDict from '@/lib/dict';
import { EuiLoadingSpinner } from '@elastic/eui';
import { useProjectManager } from '@/hooks/useProjectManager';
import { useLoader } from '@/context/LoaderContext';
import { useProjectLoader } from '@/setupApp/hooks/useProjectLoader';
import { usePDFLoader } from '@/hooks/usePDFLoader';
import { useToast } from '@/app/components/ToastProvider';
import { SCREEN_TYPES } from '@/setupApp/utils/constants';
import { useScreenManager } from '@/setupApp/hooks/useScreenManager';

const dict = getDict("de");

export const WelcomeScreen = ({onNext, status, savedProjects}) => {
    const { globalAps, loading, completeCurrentStep } = useLoader();
    const { navigateTo } = useScreenManager();
    const [isLoading, setIsLoading] = React.useState(true);
    const { isClient, pdfJsLoaded } = usePDFLoader();
    const { addToast } = useToast();
    const isSmall = React.useMemo(() => isSmallScreen(), []);
    React.useEffect(() => {
        if (savedProjects.length > 0) {
            const timer = setTimeout(() => {
                completeCurrentStep();
                navigateTo(SCREEN_TYPES.PROJECT_SETUP);
            }, 4000);

            return () => clearTimeout(timer);
        }
    },[savedProjects])

    React.useEffect(() => {
        const timer = setTimeout(() => {
            setIsLoading(false);
        }, 2000);

        return () => clearTimeout(timer);
    }, []);
    return (
        <motion.div
            key="welcome"
            variants={animations.pageVariants}
            initial="initial"
            animate="in"
            exit="out"
        >
            <EuiPanel paddingSize="l" style={{margin: '0 12px', textAlign: 'center'}}>
                {/* Logo */}
                <motion.div
                    initial={{ opacity: 0, scale: 0.8 }}
                    animate={{ opacity: 1, scale: 1 }}
                    transition={{ duration: 0.5 }}
                    style={{
                        fontSize: isSmall ? '56px' : '72px',
                        marginBottom: isSmall ? '20px' : '28px',
                    }}
                >
                    <Image
                        src="/logo.svg"
                        alt="Logo"
                        width={80}
                        height={80}
                        priority
                        style={{
                            filter: 'drop-shadow(0 4px 8px rgba(0,0,0,0.1))'
                        }}
                    />

                </motion.div>

                {/* Title */}
                <motion.div
                    initial={{ opacity: 0, y: 20 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ delay: 0.2, duration: 0.5 }}
                >
                    <EuiTitle size="l">{dict.welcome}{branding.productName}</EuiTitle>
                </motion.div>

                    <EuiSpacer size="m"/>

                {/* Loading State */}
                {isLoading && (
                    <motion.div
                        initial={{ opacity: 0 }}
                        animate={{ opacity: 1 }}
                        exit={{ opacity: 0 }}
                        transition={{ duration: 0.3 }}
                    >
                        <EuiText color="subdued" style={{padding: '0 12px', maxWidth: '400px', margin: '0 auto'}}>
                            {dict.mapLoad}
                        </EuiText>
                        <EuiLoadingSpinner size="m"/>
                        <EuiSpacer size="m"/>

                    </motion.div>
                )}

                {/* Results State */}
                {!isLoading && (
                    <motion.div
                        initial={{ opacity: 0, y: 20 }}
                        animate={{ opacity: 1, y: 0 }}
                        transition={{ duration: 0.4 }}
                    >
                        <EuiText color="subdued" style={{padding: '0 12px', maxWidth: '400px', margin: '0 auto'}}>
                            {savedProjects.length === 0 ? dict.noProject : dict.projectFound}
                        </EuiText>

                        {savedProjects.length === 0 && (
                            <>
                                <EuiSpacer size="m"/>
                                <EuiText color="subdued" style={{padding: '0 12px', maxWidth: '400px', margin: '0 auto'}}>
                                    {dict.setup}
                                </EuiText>

                                <EuiSpacer size="m"/>

                                <EuiFlexGroup justifyContent="center" gutterSize="s">
                                    <EuiFlexItem grow={false}>
                                        <EuiButton
                                            fill
                                            onClick={onNext}
                                            color="primary"
                                            fullWidth={isSmall}
                                        >
                                            {dict.getStarted}
                                        </EuiButton>
                                    </EuiFlexItem>
                                </EuiFlexGroup>
                            </>
                        )}
                    </motion.div>
                )}
            </EuiPanel>
        </motion.div>
    )
};
