import React, { useState } from 'react';
import { motion } from 'framer-motion';
import { useLoader } from '@/context/LoaderContext';
import { useProjectManager } from '@/hooks/useProjectManager';

const MultiStepProgress = ({
                               steps = [],
                               currentStep = 0,
                               completedSteps = [], // Array of completed step indices [0, 1, 2] or boolean array [true, true, false, false]
                               animation = 'smooth',
                               color = '#dc2626',
                               backgroundColor = '#f3f4f6',
                               completedColor = '#10b981',
                               disabledColor = '#e5e7eb',
                               showLabels = true,
                               showNumbers = true,
                               showPercentage = true,
                               orientation = 'horizontal', // 'horizontal' | 'vertical'
                               size = 'md', // 'sm' | 'md' | 'lg'
                               clickable = false,
                               allowSkipping = false, // Allow clicking on future steps
                               strictMode = false, // Only allow next step if current is completed
                               style = {},
                               className = '',
                               currentScreen,
                               onStepClick,
                               onStepComplete, // Callback when a step is marked as complete
                               ...props
                           }) => {
    // Handle both array of step objects and simple number
    const stepCount = Array.isArray(steps) ? steps.length : steps;
    const stepLabels = Array.isArray(steps) ? steps : Array.from({ length: steps }, (_, i) => ({ label: `Step ${i + 1}` }));

    // Handle completed steps - can be array of indices [0, 1, 2] or boolean array [true, true, false]
    const getCompletedStatus = (stepIndex) => {
        if (Array.isArray(completedSteps)) {
            if (typeof completedSteps[0] === 'boolean') {
                return completedSteps[stepIndex] || false;
            } else {
                return completedSteps.includes(stepIndex);
            }
        }
        return false;
    };

    // Determine if a step is accessible (clickable)
    const isStepAccessible = (stepIndex) => {
        if (!clickable) return false;
        if (allowSkipping) return true;

        if (strictMode) {
            // In strict mode, can only access next step if current is completed
            if (stepIndex === currentStep + 1) {
                return getCompletedStatus(currentStep);
            }
            // Can access current step or any completed step
            return stepIndex <= currentStep || getCompletedStatus(stepIndex);
        } else {
            // Default: can access current step and any previous steps
            return stepIndex <= currentStep;
        }
    };

    // Get the highest accessible step
    const getMaxAccessibleStep = () => {
        if (allowSkipping) return stepCount - 1;

        if (strictMode) {
            // Find the furthest step that can be accessed
            let maxStep = 0;
            for (let i = 0; i < stepCount; i++) {
                if (getCompletedStatus(i)) {
                    maxStep = Math.min(i + 1, stepCount - 1);
                }
            }
            return Math.max(maxStep, currentStep);
        }

        return currentStep;
    };

    // Size configurations
    const sizeConfig = {
        sm: {
            stepSize: '24px',
            fontSize: '10px',
            progressHeight: '4px',
            gap: '8px',
            labelSize: '12px'
        },
        md: {
            stepSize: '32px',
            fontSize: '12px',
            progressHeight: '6px',
            gap: '12px',
            labelSize: '14px'
        },
        lg: {
            stepSize: '40px',
            fontSize: '14px',
            progressHeight: '8px',
            gap: '16px',
            labelSize: '16px'
        }
    };

    const config = sizeConfig[size];

    // Animation configs
    const animationConfig = {
        smooth: { duration: 0.4, ease: [0.4, 0, 0.2, 1] },
        bouncy: { type: "spring", stiffness: 100, damping: 15, duration: 0.6 },
        snap: { duration: 0.15, ease: "easeInOut" },
        buildup: { duration: 1.2, ease: "easeOut" }
    };

    const percentage = strictMode
        ? Math.round((completedSteps.length / stepCount) * 100)
        : Math.round(((currentStep + 1) / stepCount) * 100);
    const isVertical = orientation === 'vertical';
    const isClickable = clickable || (onStepClick && typeof onStepClick === 'function');

    const containerStyle = {
        display: 'flex',
        flexDirection: isVertical ? 'row' : 'column',
        width: '100%',
        ...style
    };

    const progressContainerStyle = {
        display: 'flex',
        flexDirection: isVertical ? 'column' : 'row',
        alignItems: 'center',
        width: '100%',
        gap: config.gap
    };

    return (
        <div className={className} style={containerStyle} {...props}>
            {/* Header with step info */}
            {(showLabels || showPercentage) && !isVertical && (
                <div style={{
                    display: 'flex',
                    justifyContent: 'space-between',
                    alignItems: 'center',
                    marginBottom: config.gap,
                    fontSize: config.labelSize,
                    fontWeight: '500'
                }}>
                    {showLabels && (
                        <span style={{ color: '#374151' }}>
            </span>
                    )}
                    {showPercentage && (
                        <span style={{ color: '#6b7280' }}>
              {percentage}%
            </span>
                    )}
                </div>
            )}

            <div style={progressContainerStyle}>
                {/* Step indicators */}
                {stepLabels.map((step, index) => {
                    const isCompleted = getCompletedStatus(index);
                    const isCurrent = index === currentStep;
                    const isActive = isCompleted || isCurrent;
                    const isAccessible = isStepAccessible(index);
                    const isDisabled = !isAccessible && clickable;

                    // Determine step color and appearance
                    const getStepColor = () => {
                        if (isCompleted) return completedColor;
                        if (isCurrent) return color;
                        if (isDisabled) return disabledColor;
                        return '#e5e7eb';
                    };

                    const stepColor = getStepColor();

                    return (
                        <React.Fragment key={index}>
                            {/* Step Circle and Label */}
                            <motion.div
                                initial={{ scale: 0.8, opacity: 0.5 }}
                                animate={{
                                    scale: isCurrent ? 1.1 : 1,
                                    opacity: isActive ? 1 : (isDisabled ? 0.4 : 0.5)
                                }}
                                transition={{
                                    delay: index * 0.05,
                                    ...animationConfig[animation]
                                }}
                                onClick={isAccessible ? () => onStepClick?.(index) : undefined}
                                style={{
                                    display: 'flex',
                                    flexDirection: isVertical ? 'column' : 'column',
                                    alignItems: 'center',
                                    gap: '8px',
                                    cursor: isAccessible ? 'pointer' : (clickable ? 'not-allowed' : 'default'),
                                    minWidth: isVertical ? 'auto' : '80px',
                                    textAlign: 'center',
                                    opacity: isDisabled ? 0.5 : 1
                                }}
                            >
                                {/* Step circle */}
                                <motion.div
                                    style={{
                                        width: config.stepSize,
                                        height: config.stepSize,
                                        borderRadius: '50%',
                                        backgroundColor: stepColor,
                                        display: 'flex',
                                        alignItems: 'center',
                                        justifyContent: 'center',
                                        color: isCompleted || isCurrent ? 'white' : '#9ca3af',
                                        fontSize: config.fontSize,
                                        fontWeight: '600',
                                        boxShadow: isActive && !isDisabled ? `0 2px 8px ${stepColor}40` : 'none',
                                        transition: 'all 0.2s ease',
                                        border: isCurrent && !isCompleted ? `2px solid ${color}` : 'none',
                                        position: 'relative'
                                    }}
                                    whileHover={isAccessible ? { scale: 1.05 } : {}}
                                    whileTap={isAccessible ? { scale: 0.95 } : {}}
                                >
                                    {showNumbers ? (
                                        isCompleted ? '✓' : index + 1
                                    ) : (
                                        isCompleted ? '✓' : '•'
                                    )}

                                    {/* Disabled overlay */}
                                    {isDisabled && (
                                        <div style={{
                                            position: 'absolute',
                                            top: '-2px',
                                            right: '-2px',
                                            width: '12px',
                                            height: '12px',
                                            backgroundColor: '#ef4444',
                                            borderRadius: '50%',
                                            display: 'flex',
                                            alignItems: 'center',
                                            justifyContent: 'center',
                                            fontSize: '8px',
                                            color: 'white'
                                        }}>
                                            🔒
                                        </div>
                                    )}
                                </motion.div>

                                {/* Step label */}
                                {step.label && (
                                    <div style={{
                                        fontSize: config.labelSize,
                                        fontWeight: isActive ? '600' : '400',
                                        color: isActive && !isDisabled ? '#374151' : '#6b7280',
                                        lineHeight: 1.2,
                                        maxWidth: isVertical ? '120px' : '80px'
                                    }}>
                                        {step.label}
                                    </div>
                                )}

                                {/* Step description */}
                                {step.description && (
                                    <div style={{
                                        fontSize: config.fontSize,
                                        color: isDisabled ? '#d1d5db' : '#9ca3af',
                                        lineHeight: 1.2,
                                        maxWidth: isVertical ? '120px' : '100px'
                                    }}>
                                        {step.description}
                                    </div>
                                )}
                            </motion.div>

                            {/* Progress Line between steps */}
                            {index < stepCount - 1 && (
                                <div style={{
                                    flex: 1,
                                    height: isVertical ? config.progressHeight : '2px',
                                    width: isVertical ? '2px' : '100%',
                                    backgroundColor: backgroundColor,
                                    borderRadius: config.progressHeight,
                                    overflow: 'hidden',
                                    position: 'relative',
                                    margin: isVertical ? `${config.gap} 0` : `0 ${config.gap}`
                                }}>
                                    <motion.div
                                        style={{
                                            height: '100%',
                                            width: '100%',
                                            background: `linear-gradient(${isVertical ? '180deg' : '90deg'}, ${completedColor} 0%, ${completedColor}dd 100%)`,
                                            borderRadius: config.progressHeight,
                                            boxShadow: `0 2px 4px ${completedColor}30`,
                                            transformOrigin: isVertical ? 'top' : 'left'
                                        }}
                                        initial={{
                                            [isVertical ? 'scaleY' : 'scaleX']: 0
                                        }}
                                        animate={{
                                            [isVertical ? 'scaleY' : 'scaleX']: getCompletedStatus(index) ? 1 : 0
                                        }}
                                        transition={{
                                            delay: index * 0.1,
                                            ...animationConfig[animation]
                                        }}
                                    />
                                </div>
                            )}
                        </React.Fragment>
                    );
                })}
            </div>

            {/* Progress bar (alternative visualization) */}
            {showPercentage && isVertical && (
                <div style={{
                    marginTop: config.gap,
                    fontSize: config.labelSize,
                    fontWeight: '600',
                    color: '#374151',
                    textAlign: 'center'
                }}>
                    {percentage}%
                </div>
            )}
        </div>
    );
};
export const stepConfigs = {
    numbered: [
        { label: 'Step 1' },
        { label: 'Step 2' },
        { label: 'Step 3' },
        { label: 'Step 4' },
        { label: 'Step 5' },
    ]
};
// Demo Component
const MultiStepDemo = () => {
    const { currentStep, setCurrentStep, completedSteps, setCompletedSteps, completeCurrentStep, currentScreen } = useLoader();
    const [selectedConfig, setSelectedConfig] = useState('numbered');
    const [orientation, setOrientation] = useState('horizontal');
    const [size, setSize] = useState('md');
    const [animation, setAnimation] = useState('smooth');
    const [strictMode, setStrictMode] = useState(true);
    const [allowSkipping, setAllowSkipping] = useState(false);
    const { savedProjects } = useProjectManager();
    const handleStepClick = (stepIndex) => {
        const maxSteps = stepConfigs[selectedConfig].length;

        if (strictMode && !allowSkipping) {
            // In strict mode, can only go to next step if current is completed
            if (stepIndex === currentStep + 1 && !completedSteps.includes(currentStep)) {
                return; // Can't proceed without completing current step
            }
        }

        if (stepIndex < maxSteps) {
            setCurrentStep(stepIndex);
        }
    };

    const nextStep = () => {
        const maxSteps = stepConfigs[selectedConfig].length;

        if (strictMode) {
            // In strict mode, complete current step before moving to next
            if (!completedSteps.includes(currentStep)) {
                completeCurrentStep();
            }
            if (currentStep < maxSteps - 1) {
                setCurrentStep(prev => prev + 1);
            }
        }
    };


    return (
        <>
            <div style={{
                    backgroundColor: 'white',
                    borderRadius: '16px',
                    padding: '40px',
                    marginBottom: '32px',
                    boxShadow: '0 10px 25px -5px rgba(0, 0, 0, 0.1)',
                    minHeight: orientation === 'vertical' ? '400px' : 'auto'
                }}>
                <MultiStepProgress
                    steps={stepConfigs[selectedConfig]}
                    currentStep={currentStep}
                    completedSteps={completedSteps}
                    animation={animation}
                    orientation={orientation}
                    size={size}
                    clickable={true}
                    strictMode={strictMode}
                    allowSkipping={allowSkipping}
                    onStepClick={undefined}
                    currentScreen={currentScreen}
                    style={{
                        justifyContent: orientation === 'vertical' ? 'center' : 'flex-start',
                        alignItems: orientation === 'vertical' ? 'center' : 'stretch'
                    }} onStepComplete={undefined}                />
            </div>
        </>
    );
};

export default MultiStepDemo;
