package handlers

import (
	"crypto/rand"
	"crypto/sha256"
	"encoding/hex"
	"encoding/json"
	"io/ioutil"
	"net/http"
	"regexp"
	"strings"
	"time"
	announcement "uniberg/uconfigd-controller/services"
	"uniberg/uconfigd-controller/services/config"
	"uniberg/uconfigd-controller/services/database"
	"uniberg/uconfigd-controller/services/devices"
	"uniberg/uconfigd-controller/services/statistics"

	"github.com/gin-gonic/gin"
)

type DeviceSubscribeRequestDeviceInfo struct {
	Compatible string `json:"compatible"`
	Model      string `json:"model"`
	Role       string `json:"role"`
}

type DeviceSubscribeRequest struct {
	Name      string `json:"name"`
	Mac       string `json:"mac"`
	Serial    string `json:"serial"`
	IpAddress string `json:"ip_address"`

	Radios []struct {
		Band       int `json:"band"`
	} `json:"radios"`

	Setup struct {
		SSID string `json:"ssid"`
		Key  string `json:"key"`
	} `json:"setup"`

	Device DeviceSubscribeRequestDeviceInfo `json:"device"`
}

type ClientBindRequest struct {
	MacAddress       string `json:"mac_address"`
	PasswordHashType string `json:"password_hash_type"`
	PasswordHash     string `json:"password_hash"`
	SSID             string `json:"ssid"`
}

func GetNetworkConfig() config.NetworkConfig {
	netconf, err := config.GetConfig()
	if err != nil {
		panic(err)
	}
	return netconf
}

func NetworkConfigGet(c *gin.Context) {
	networkConfig := GetNetworkConfig()
	c.JSON(http.StatusOK, networkConfig)
}

func NetworkConfigSet(c *gin.Context) {
	var networkConfig config.NetworkConfig
	if err := c.ShouldBindJSON(&networkConfig); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{"error": err.Error()})
		return
	}

	/* Valdiate Network Config */
	// ToDo: Check all network-UUIDS exist and are unique
	// ToDo: Check WiFI passwords are unique and are >= 8 characters and <= 63 characters

	config.SaveToFile(networkConfig)

	announcement.UpdateAnnouncementData()
	c.JSON(http.StatusOK, gin.H{"message": "Configuration Saved"})
}

func DeviceList(c *gin.Context) {
	var devices []devices.AccessPoint = devices.GetDevices()
	c.JSON(http.StatusOK, devices)
}

func DeviceDeploy(c *gin.Context) {
	c.JSON(http.StatusOK, gin.H{"message": "Deploying Device deprecated"})
}

func DeviceConfig(c *gin.Context) {
	mac_address := c.Param("node_id")
	var dev database.Device
	database.DB.Where("mac_address = ?", mac_address).Preload("Radios").First(&dev)
	netconf, err := config.GetConfig()
	if err != nil {
		c.JSON(http.StatusBadRequest, gin.H{"error": err.Error()})
		return
	}
	deviceConfig := config.GenerateDeviceConfig(netconf, dev)
	c.JSON(http.StatusOK, deviceConfig)
}

func DeviceMetrics(c *gin.Context) {
	mac_address := c.Param("node_id")
	statisticsData, err := statistics.GetStatisticData(mac_address)
	if err != nil {
		/* Check if Error is of type DeviceNotFoundError */
		if _, ok := err.(*statistics.DeviceNotFoundError); ok {
			c.JSON(http.StatusBadRequest, gin.H{"error": "Device not found"})
			return
		}

		c.JSON(http.StatusNotFound, gin.H{"error": "No Statistic Data found"})
		return
	}
	c.Data(http.StatusOK, "application/json", []byte(statisticsData))
}

func DeviceMetricsSave(c *gin.Context) {
	/* Check if device exists */
	nodeId := c.Param("node_id")
	var dev database.Device
	database.DB.Where("mac_address = ?", nodeId).Preload("Radios").First(&dev)
	if dev.ID == 0 {
		c.JSON(http.StatusBadRequest, gin.H{"error": "Device not found"})
		return
	}

	/* Read Request body to string */
	bodyBytes, err := ioutil.ReadAll(c.Request.Body)
	if err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{"error": "Failed to read request body"})
		return
	}
	bodyString := string(bodyBytes)

	/* Check if metrics are valid JSON */
	var js json.RawMessage
	if json.Unmarshal([]byte(bodyString), &js) != nil {
		c.JSON(http.StatusBadRequest, gin.H{"error": "Invalid JSON"})
		return
	}

	/* Add "lastUpdated" field to JSON */
	var data map[string]interface{}
	if err := json.Unmarshal([]byte(bodyString), &data); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{"error": "Invalid JSON"})
		return
	}
	data["lastUpdated"] = time.Now().Unix()

	/* Convert JSON back to string */
	bodyBytes, err = json.Marshal(data)
	if err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{"error": "Failed to convert JSON"})
		return
	}
	bodyString = string(bodyBytes)

	statistics.SaveStatisticData(nodeId, bodyString)
}

func BindClient(c *gin.Context) {
	var request ClientBindRequest

	/* Get JSON Body */
	if err := c.ShouldBindJSON(&request); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{"error": err.Error()})
		return
	}

	/* Get Network Config */
	netconf := GetNetworkConfig()

	/* Find Network with SSID */
	var network config.WirelessNetwork
	for _, n := range netconf.Wireless {
		if n.SSID == request.SSID {
			network = n
			break
		}
	}

	if network.SSID == "" {
		c.JSON(http.StatusNotFound, gin.H{"error": "SSID not found", "accepted": false})
		return
	}

	/* Get all Passwords of network */
	var passwords []database.Password
	database.DB.Where("network_id = ?", network.NetworkUUID).Find(&passwords)
	if len(passwords) == 0 {
		c.JSON(http.StatusNotFound, gin.H{"error": "No Passwords found", "accepted": false})
		return
	}

	/* Check if Password Hash matches */
	for _, password := range passwords {
		var hashedPasskey string
		cleartextPasskey := password.Password
		macAddress := password.MacAddress

		if request.PasswordHashType == "sha256" {
			/* Hash passkey from list */
			h := sha256.New()
			h.Write([]byte(cleartextPasskey))
			hashedPasskey = hex.EncodeToString(h.Sum(nil))
			if !strings.EqualFold(hashedPasskey, request.PasswordHash) {
				continue
			}
		} else {
			c.JSON(http.StatusBadRequest, gin.H{"error": "Invalid Password Hash Type", "accepted": false})
			return
		}

		/* Password Equal. Check if MAC matches or is assignable (ff:ff:ff:ff:ff:ff) */
		if strings.EqualFold(macAddress, request.MacAddress) {
			c.JSON(http.StatusOK, gin.H{"message": "Client already bound", "accepted": true})
			return
		}

		/* Check binding case */
		if strings.ToLower(macAddress) == "ff:ff:ff:ff:ff:ff" {
			/* Assign MAC Address */
			password.MacAddress = request.MacAddress
			database.DB.Save(&password)
			c.JSON(http.StatusOK, gin.H{"message": "Client bound", "accepted": true})
			IncreaseNetconfVersion()
			return
		}

		/* Check unbindable case */
		if strings.ToLower(macAddress) == "00:00:00:00:00:00" || strings.ToLower(macAddress) == "" {
			c.JSON(http.StatusOK, gin.H{"message": "Passkey not bindable", "accepted": true})
			return
		}
	}

	/* No Passkey found */
	c.JSON(http.StatusNotFound, gin.H{"error": "No Passkey found", "accepted": false})
}

func DeployDevices(c *gin.Context) {
	c.JSON(http.StatusOK, gin.H{"message": "Deploying Devices deprecated"})
}

func DeviceSubscribe(c *gin.Context) {
	var request DeviceSubscribeRequest
	if err := c.ShouldBindJSON(&request); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{"error": err.Error()})
		return
	}

	/* Check if Device with MAC Address exists */
	devices := database.DB.Where("mac_address = ?", request.Mac).Find(&database.Device{})
	if devices.RowsAffected == 0 {
		/* Check if Valid MAC Address */
		macRegex := `^([0-9A-Fa-f]{2}[:-]){5}([0-9A-Fa-f]{2})$`
		matched, err := regexp.MatchString(macRegex, request.Mac)
		if err != nil || !matched {
			c.JSON(http.StatusBadRequest, gin.H{"error": "Invalid MAC address"})
			return
		}

		/* Generate AES256 Key as hex */
		key := make([]byte, 32)
		_, err = rand.Read(key)
		if err != nil {
			c.JSON(http.StatusInternalServerError, gin.H{"error": "Failed to generate key"})
			return
		}
		keyHex := hex.EncodeToString(key)

		/* Create Radios */
		radios := []database.WirelessRadio{}
		for _, radio := range request.Radios {
			/* Check if Band matches 2.4GHz or 5GHz */
			if radio.Band != 2 && radio.Band != 5 {
				c.JSON(http.StatusBadRequest, gin.H{"error": "Invalid Band"})
				return
			}

			radio := database.WirelessRadio{
				Band:       radio.Band,
			}
			radios = append(radios, radio)
		}

		/* Get Setup WiFi Information */
		setupSSID := "UB-EDGE-Setup"
		setupKey := "9830097827" /* MD5 UNIBERG */
		if request.Setup.SSID != "" && request.Setup.Key != "" {
			setupSSID = request.Setup.SSID
			setupKey = request.Setup.Key
		}

		/* Create Device */
		device := database.Device{
			DeviceName:    request.Name,
			MacAddress:    request.Mac,
			IpAddress:     request.IpAddress,
			ModelType:     request.Device.Compatible,
			ModelName:     request.Device.Model,
			Role:          request.Device.Role,
			Radios:        radios,
			EncryptionKey: keyHex,

			SetupWiFiSSID: setupSSID,
			SetupWiFiKey:  setupKey,
		}

		database.DB.Create(&device)
		c.JSON(http.StatusOK, gin.H{"message": "Subscribed", "key": keyHex})
		return
	}

	c.JSON(http.StatusOK, gin.H{"message": "Subscribed", "key": nil})
}

func DeviceTest(c *gin.Context) {
	c.JSON(http.StatusOK, gin.H{"message": "OK"})
}

func SystemMetrics(c *gin.Context) {
	/* Get System Metrics */
	metrics := statistics.GetSystemMetrics()
	c.JSON(http.StatusOK, metrics)
}

type PSKRequestBody struct {
	Id         string `json:"id"`
	MacAddress string `json:"mac_address"`
	Password   string `json:"password"`
}

func NetworkPasswordList(c *gin.Context) {
	var passwords []database.Password
	var network_id = c.Param("network_id")
	database.DB.Where("network_id = ?", network_id).Find(&passwords)
	c.JSON(http.StatusOK, passwords)
}

func IncreaseNetconfVersion() {
	// Increase config version
	netconf := GetNetworkConfig()
	netconf.UUID = netconf.UUID + 1
	config.SaveToFile(netconf)
}

func NetworkPasswordUpdate(c *gin.Context) {
	var network_id = c.Param("network_id")

	var requestPasswords []PSKRequestBody
	if err := c.ShouldBindJSON(&requestPasswords); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{"error": err.Error})
		return
	}

	for _, password := range requestPasswords {
		if len(password.Password) < 8 || len(password.Password) > 63 {
			c.JSON(http.StatusBadRequest, gin.H{"error": "Password length must be between 8 and 63 characters"})
			return
		}
	}

	// Get all known passwords
	var dbPasswords []database.Password
	database.DB.Where("network_id = ?", network_id).Find(&dbPasswords)

	// Check which passwords to create, which to delete
	var passwordDelete []database.Password
	var passwordCreate []database.Password

	for _, dbPassword := range dbPasswords {
		var found = false
		for _, requestPassword := range requestPasswords {
			if dbPassword.Password == requestPassword.Password {
				found = true
			}
		}
		if found {
			continue
		}

		passwordDelete = append(passwordDelete, dbPassword)
	}

	for _, requestPassword := range requestPasswords {
		var found = false
		for _, dbPassword := range dbPasswords {
			if dbPassword.Password == requestPassword.Password {
				found = true
			}
		}

		if found {
			continue
		}
		passwordCreate = append(passwordCreate, database.Password{
			MacAddress: requestPassword.MacAddress,
			NetworkId:  network_id,
			Password:   requestPassword.Password,
		})
	}

	// Delete passwords
	for _, password := range passwordDelete {
		err := database.DB.Delete(&password)
		if err.Error != nil {
			c.JSON(http.StatusInternalServerError, gin.H{"error": "Failed to delete passwords"})
			return
		}
	}

	// Create passwords
	for _, password := range passwordCreate {
		err := database.DB.Create(&password)
		if err.Error != nil {
			c.JSON(http.StatusInternalServerError, gin.H{"error": "Failed to create passwords"})
			return
		}
	}

	c.JSON(http.StatusOK, gin.H{"message": "Passwords updated"})

	if len(passwordCreate) > 0 || len(passwordDelete) > 0 {
		// Increase config version
		IncreaseNetconfVersion()
	}
}
