package main

import (
	"context"
	"log"
	"net"
	"net/http"
	"os"
	"os/signal"
	"uniberg/uconfigd-controller/handlers"
	announcement "uniberg/uconfigd-controller/services"
	"uniberg/uconfigd-controller/services/config"
	"uniberg/uconfigd-controller/services/database"
	"uniberg/uconfigd-controller/services/statistics"

	"github.com/gin-gonic/gin"
)

func main() {
	/* First Argument is the Port, second the data directory */
	if len(os.Args) != 5 {
		panic("Usage: uconfigd-controller <port> <data-dir> <statistics-dir> <broadcast-if>")
	}

	listenPort := os.Args[1]
	dataDir := os.Args[2]
	statisticsDir := os.Args[3]
	broadcastIf := os.Args[4]

	/* Create Directories */
	os.MkdirAll(dataDir, os.ModePerm)
	os.MkdirAll(statisticsDir, os.ModePerm)

	statistics.SetStatisticDataDirectory(statisticsDir)
	config.SetConfigPath(dataDir + "/config.json")

	/* Open Database Connection */
	database.GetDB(dataDir + "/uconfigd.db")
	defer database.CloseDB()

	/* Update Announcement Data */
	announcement.AD.BroadcastIf = broadcastIf
	//announcement.UpdateAnnouncementData()

	/* Migrate Database */
	database.DB.AutoMigrate(&database.Device{})
	database.DB.AutoMigrate(&database.WirelessRadio{})
	database.DB.AutoMigrate(&database.Password{})

	gin.SetMode(gin.ReleaseMode)

	r := gin.Default()

	/* Config */
	r.GET("/config", handlers.NetworkConfigGet)
	r.POST("/config", handlers.NetworkConfigSet)

	/* Nodes */
	r.GET("/nodes", handlers.DeviceList)
	r.GET("/nodes/deploy", handlers.DeployDevices)
	r.GET("/nodes/:node_id/config", handlers.DeviceConfig)
	r.GET("/nodes/:node_id/metrics", handlers.DeviceMetrics)
	r.GET("/nodes/:node_id/deploy", handlers.DeviceDeploy)

	/* Passwords */
	r.GET("/network/:network_id/passwords", handlers.NetworkPasswordList)
	r.POST("/network/:network_id/password", handlers.NetworkPasswordUpdate)

	/* Clients */
	r.POST("/wireless/client/bind", handlers.BindClient)

	/* Node */
	r.POST("/node/subscribe", handlers.DeviceSubscribe)
	r.GET("/node/test", handlers.DeviceTest)
	r.GET("/node/:node_id/config", handlers.DeviceConfig)
	r.POST("/node/:node_id/metrics", handlers.DeviceMetricsSave)

	/* System */
	r.GET("/system/metrics", handlers.SystemMetrics)

	// Create an HTTP server
	server := &http.Server{
		Addr:    "[::]:" + listenPort,
		Handler: r,
	}

	ln, err := net.Listen("tcp", ":"+listenPort)
	if err != nil {
		panic(err)
	}

	// Asynchronously call AnnounceStatistics every 10 seconds
	// go announcement.AnnouncementFunction()

	// Handle graceful shutdown
	quit := make(chan os.Signal, 1)
	signal.Notify(quit, os.Interrupt)
	go func() {
		<-quit
		log.Println("Shutting down server...")

		// Shutdown the Gin server
		if err := server.Shutdown(context.Background()); err != nil {
			log.Fatal("Server forced to shutdown:", err)
		}
	}()

	// Start the HTTP server
	log.Printf("Starting server on port %s", listenPort)
	if err := server.Serve(ln); err != nil && err != http.ErrServerClosed {
		log.Fatalf("Server failed to start: %v", err)
	}
}
