package config

import (
	"bufio"
	"crypto/md5"
	"encoding/hex"
	"encoding/json"
	"io/ioutil"
	"os"
	"strconv"
	"strings"
	"uniberg/uconfigd-controller/services/database"
)

type PPSKPasskey struct {
	MacAddress  string `json:"mac_address"`
	Passkey     string `json:"passkey"`
	NetworkUUID string `json:"network_uuid"`
}

type Encryption struct {
	Method     string        `json:"method"`
	Passkeys   []PPSKPasskey `json:"passkeys"`
	Passphrase string        `json:"passphrase"`
}

type Network struct {
	Name   string `json:"name"`
	UUID   string `json:"uuid"`
	VlanId int    `json:"vlan_id"`
}

type WirelessNetwork struct {
	SSID        string     `json:"ssid"`
	NetworkUUID string     `json:"network_uuid"`
	Encryption  Encryption `json:"encryption"`
}

type NetworkConfig struct {
	UUID     int64             `json:"uuid"`
	Networks []Network         `json:"networks"`
	Wireless []WirelessNetwork `json:"wireless_networks"`
}

var configPath string
var networkConfig NetworkConfig

func ResolveNetworkUUIDToVlanId(networkConfig NetworkConfig, networkUUID string) int {
	for _, network := range networkConfig.Networks {
		if network.UUID == networkUUID {
			return network.VlanId
		}
	}
	return 0
}

func GetMacAddressFromPrefixAndMac(prefix string, mac string) string {
	prefix = "fd42:dead:c0de:1337::"
	macWithoutColons := strings.Replace(mac, ":", "", -1)
	address := prefix + macWithoutColons[:4] + ":" + macWithoutColons[4:8] + ":" + macWithoutColons[8:12]
	return address + "/64"
}

func GetNetworkShortId(networkUUID string) string {
	md5hash := md5.New()
	md5hash.Write([]byte(networkUUID))
	networkUUID = hex.EncodeToString(md5hash.Sum(nil))
	return networkUUID[:8]
}

func GenerateDeviceConfig(networkConfig NetworkConfig, device database.Device) DeviceConfig {
	ssh := []string{
		"ssh-ed25519 AAAAC3NzaC1lZDI1NTE5AAAAICfnCPGWR8aocwv8TwjLrrR81W+deqeNEuSlvR4xbcgr ub-edge-provisioning@uniberg.com",
		"ssh-ed25519 AAAAC3NzaC1lZDI1NTE5AAAAIIRBEQfvJQlt1ufW9BTtDI3Am+TplOQo1GCtU7CJyro1 david.bauer@uniberg.com",
		"ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAABgQDhUUBHJNYw2iuLHJXveX1R+TiZD0XxlXFxnbeSXsF56YV+SlhHnT1/0h5fQua3p1aTLHYO9kNErnRTHxnFzUnH9hOl8raBGTUjmO6SQ2fdkPUkHarFXWVOWApZQ8DW4rkOU6nGiEkBuKQ9H6N23jY+06j4FU7/muw2wCA+BZ9XgSP3J4NWTEY5yAOv7dMVlDAeeeviveuzfrc8rzv2yG9OBK+LWyJ3eagJzOs59RPd6ICTuO7nu+45eLcAN/JhIYt/HodSBAn/btM3ZJ519ur8zZn4aUX8yMdRQHXq1+vq7EKDdwqW0bGQmpIBWpd91sgN2VyaIsw7uOkZ6z3bQIu2bCugRQUFjXdbScMK9JpOyiwDKIYL4RMV7YDpJfj+9oHNffzXHA6EJRyA3c2HC4Lxw4MdbIAX7Pk7nlqLo8mc4qBZ0gQfwgLO7NsPWYc6ja+4UPGMKBiEe223tUYL6nvWu1GG+MWskg3rgngf3/g/+Dir29qYlF2kX7wjQJKs0Js= igdpukontaktmann69",
		"ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAABgQCX3T7y+eCA20iKQvVPSmE+EXpos2CvY8zI1zJUiGi8LMG/W+DjUpwVDE0VTPBh1IQdUW8sTMv7Oo+ZlrIyjY7I/ajC56Ww8aP1qkvmaS98ULtffWnxsij23MpjifoqePruALzknZNIkKwHThDeJBmpRV5odyD+BGVg0GL+9KnfPJST6ckMOCE7qiQaBaC4MMKi58xWAVzf+/QCsAds1FMq99xFOoL8TmUPlP+JbmxLsfhIdBcGzA1EtCcGEHMkLijjhBGynV2HxB3PezHsZ6mgbbarnMqXKWROTO2VF8g4grh/E7oVgc1lni6tbx/495/EZBqeCvnTrnNnXsPexvJ1in4eIGdUdrKZ8W1avlKnI5CvLYXcL4vvMwq8/0cfWC+qdXdSWTNDN3WuNPLpNRXCNB5I2CchpaWC4hQxUp4zDAr0Yqv82RlIEJrUbw8gW9akIbD2VMaMOgyf0uhC6xwkrhyn8R/8QC4KNNZ92j+IKvoLyCj+ejqAjZQ9UU6QS1U= jannis-der-knochenbrecher",
		"ssh-ed25519 AAAAC3NzaC1lZDI1NTE5AAAAIEY5mgC5SN7isZcFfgwsXr1ZBzjNdotJYTLA2TZyAdTH jannis@mbp",
		"ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAABAQCmqeV0CNq/3H95UejKAmwOdxCXOXlgQLXbcwaqFZXBImIsiC7vm28DZHTiOVEV5h0hokTugfU4Zr3Hu+u5Uffjsa9z1r8qehWrqhNDHln8V5Vm77fIXBMywNTulGympGZBPcwy0BmbuKORKhpLadfjJCq0mPbd/oyeA2rW23OwJxeENgY74TX1x8DLoWWgvyWPFgX3dh6Uk3TUWNowkCZ6T0QHbUSwEYz8E5iKtrXvGL0VciloTTTQ2GyANvsiIyp4wq486GRmvUH23OCtEeB4cTKJVZEZdbotuk9un1zrZ+oAj9tIm4c5BAtpkr0aGqAuiWAiH3s/33C2J5FQv6q/ bassemchagra@Bassems-MacBook-Pro.local",
		"ssh-ed25519 AAAAC3NzaC1lZDI1NTE5AAAAIIxXmg/wnOPFTTqBlbUsSTDBSOidFpJLyof7+rgHCpXo lukas.kirner@uniberg.com",
	}

	if f, err := os.Open("/etc/uniberg/configd-controller/data/authorized_keys"); err == nil {
		defer f.Close()
		scan := bufio.NewScanner(f)
		ssh = []string{}
		for scan.Scan() {
			ssh = append(ssh, scan.Text())
		}
	}

	deviceConfig := DeviceConfig{
		UUID:        networkConfig.UUID,
		Unit:        DeviceConfigUnit{Name: device.DeviceName, Location: "Uniberg", Timezone: "Europe/Berlin"},
		CountryCode: "DE",
		Password:    true,
		Interfaces:  make(map[string]DeviceConfigInterface),
		Radios:      make(map[string]DeviceConfigRadio),
		Services: DeviceConfigServices{SSH: DeviceConfigSSH{
			AuthorizedKeys: ssh,
			PasswordAuth:   false,
		}},
	}

	// Add Wireless Radios (Common to both AP and Controller)
	for _, radio := range device.Radios {
		channel := 1
		band := "2G"

		if radio.Band != 2 {
			channel = 36
			band = "5G"
		}

		deviceConfig.Radios[band] = DeviceConfigRadio{
			Channel:      channel,
		}
	}

	// LAN Interface (Common SSIDs, different configurations based on role)
	lanInterface := DeviceConfigInterface{
		/* Role is specific on AP / Controller */
		/* Ports are specific on AP / Controller */
		/* IPv6 config is specific on AP / Controller */
		Tunnel:   &DeviceConfigTunnel{Protocol: "mesh-batman", Wired: true},
		SSIDs:    make(map[string]DeviceConfigSSID),
		Services: []string{"ssh"},
		Vlan:     &DeviceConfigVlan{Id: 3},
		Mtu:      1440,
	}

	lanInterface.SSIDs["mesh"] = DeviceConfigSSID{
		SSID:       "uniberg-mesh",
		WiFiRadios: []string{"2G", "5G"},
		BSSMode:    "mesh",
		Encryption: DeviceConfigSSIDEncryption{Proto: "sae", Key: "uniberg-mesh"},
	}

	i := 0

	/* Add Networks */
	for _, network := range networkConfig.Networks {
		shortId := GetNetworkShortId(network.UUID)
		shortIdDec, err := strconv.ParseInt(shortId[:2], 16, 32)

		// todo: check david's ID generation
		if err != nil {
			shortIdDec = int64(0 + i)
		}

		/* Case for controller */
		net := DeviceConfigInterface{
			Role:   "downstream",
			Ports:  map[string]string{},
			IPv4:   &DeviceConfigIPv4{	Addressing: "static", Subnet: "192.168." + strconv.Itoa(int(shortIdDec)) + ".1/24",
							DHCP: map[string]interface{}{"lease-first": 50, "lease-count": 150, "lease-time": "2d"},
						},
			IPv6:   &map[string]string{"addressing": "static", "subnet": "fd42:9830:0978:" + shortId[:4] + "::1/64"},
			Tunnel: &DeviceConfigTunnel{Protocol: "mesh-batman", Wired: true},
			Vlan:   &DeviceConfigVlan{Id: network.VlanId},
			SSIDs:  make(map[string]DeviceConfigSSID),
			Mtu:    1440,
		}

		/* Case for AP */
		if device.Role == "ap" {
			net.Role = "upstream"
			net.Ports = map[string]string{}
			net.IPv4 = nil
			net.IPv6 = nil
		}

		for _, wireless := range networkConfig.Wireless {
			if wireless.NetworkUUID != network.UUID {
				continue
			}

			wirelessInterfaceConfig := DeviceConfigSSID{
				SSID:       wireless.SSID,
				WiFiRadios: []string{"2G", "5G"},
				BSSMode:    "ap",
			}
			if wireless.Encryption.Method == "psk2" {
				wirelessInterfaceConfig.Encryption.Proto = "psk2"
				wirelessInterfaceConfig.MultiPSK = append(wirelessInterfaceConfig.MultiPSK, DeviceConfigSSIDMultiPSK{
					Mac: "00:00:00:00:00:00",
					Key: wireless.Encryption.Passphrase,
				})
			} else if wireless.Encryption.Method == "ppsk2" {
				wirelessInterfaceConfig.Encryption.Proto = "psk2"

				// Get Passwords from DB
				var passkeys []database.Password
				database.DB.Where("network_id = ?", wireless.NetworkUUID).Find(&passkeys)
				for _, passkey := range passkeys {
					macaddress := passkey.MacAddress
					if macaddress == "" {
						macaddress = "00:00:00:00:00:00"
					}
					wirelessInterfaceConfig.MultiPSK = append(wirelessInterfaceConfig.MultiPSK, DeviceConfigSSIDMultiPSK{
						Mac: macaddress,
						Key: passkey.Password,
					})
				}
			}

			net.SSIDs[wireless.SSID] = wirelessInterfaceConfig
		}

		deviceConfig.Interfaces[shortId] = net
		i = i + 1
	}

	if device.Role == "controller" {
		/* LAN interface */
		lanInterface.Role = "downstream"
		lanInterface.Ports = map[string]string{"lan*": "auto"}
		lanInterface.IPv6 = &map[string]string{"addressing": "static", "subnet": "fd42:dead:c0de:1337::1/64"}
		lanInterface.Services = []string{"ssh", "controller"}

		lanInterface.IPv4 = &DeviceConfigIPv4{ 
			Addressing: "static",
			Subnet:     "192.168.133.1/24",
			DHCP: map[string]interface{}{
				"lease-first": 10,
				"lease-count": 100,
				"lease-time":  "5d",
			},
		}

		/* Add Setup-WiFi */
		lanInterface.SSIDs["setup"] = DeviceConfigSSID{
			SSID:       device.SetupWiFiSSID,
			WiFiRadios: []string{"2G", "5G"},
			BSSMode:    "ap",
			Encryption: DeviceConfigSSIDEncryption{Proto: "psk2", Key: device.SetupWiFiKey},
		}

		// WAN interface config
		wanInterface := DeviceConfigInterface{
			Role:  "upstream",
			Mtu:   1440,
			Ports: map[string]string{"wan*": "auto"},          // Add ports as necessary
			IPv4:  &DeviceConfigIPv4{Addressing: "dynamic"},    // Configure addressing
			IPv6:  &map[string]string{"addressing": "dynamic"}, // Configure addressing
			Services: []string{"ssh"},
		}
		deviceConfig.Interfaces["wan"] = wanInterface

	} else if device.Role == "ap" {
		/* LAN interface */
		lanInterface.Role = "upstream"
		lanInterface.Ports = map[string]string{"lan*": "auto", "wan*": "auto"}
		lanInterface.IPv6 = &map[string]string{"addressing": "static", "subnet": GetMacAddressFromPrefixAndMac("fd42:dead:c0de:1337::", device.MacAddress)}

		lanInterface.IPv4 = &DeviceConfigIPv4{Addressing: "dynamic"} // Set only addressing for AP
	}

	deviceConfig.Interfaces["lan"] = lanInterface
	return deviceConfig
}

func ParseJson(jsonData []byte) (NetworkConfig, error) {
	var networkConfig NetworkConfig
	err := json.Unmarshal(jsonData, &networkConfig)

	return networkConfig, err
}

func SetConfigPath(path string) {
	configPath = path
}

func GetConfig() (NetworkConfig, error) {
	if networkConfig.UUID != 0 {
		return networkConfig, nil
	}
	return LoadFromFile()
}

func LoadFromFile() (NetworkConfig, error) {
	// Read JSON from File
	jsonFile, err := os.Open(configPath)
	if err != nil {
		return NetworkConfig{}, err
	}
	defer jsonFile.Close()

	// Read File as String
	fileContent, _ := ioutil.ReadAll(jsonFile)
	networkConfig, _ = ParseJson(fileContent)

	return networkConfig, nil
}

func SaveToFile(netConf NetworkConfig) {
	// Convert to JSON
	jsonData, _ := json.MarshalIndent(netConf, "", "    ")

	networkConfig = netConf
	// Write JSON to File
	err := os.WriteFile(configPath, jsonData, 0644)
	if err != nil {
		panic(err)
	}
}
