package statistics

import (
	"bufio"
	"fmt"
	"os"
	"path/filepath"
	"strconv"
	"strings"
	"uniberg/uconfigd-controller/services/database"
)

var statisticDataDirectory string

type DeviceNotFoundError struct {
	DeviceMac string
}

type MemoryMetrics struct {
	MemTotal     int `json:"total"`
	MemFree      int `json:"free"`
	MemAvailable int `json:"available"`
}

type UplinkDataMetrics struct {
	BytesSent       int `json:"bytes_sent"`
	BytesReceived   int `json:"bytes_received"`
	PacketsSent     int `json:"packets_sent"`
	PacketsReceived int `json:"packets_received"`
}

type CellularMetrics struct {
	SignalStrength int               `json:"signal_strength"`
	CarrierName    string            `json:"carrier_name"`
	DataMetrics    UplinkDataMetrics `json:"data"`
}

type SystemMetrics struct {
	MemoryMetrics MemoryMetrics `json:"memory"`
	Uptime        int           `json:"uptime"`
	LoadAverage   float64       `json:"load_average"`
	OnlineStatus  bool          `json:"online_status"`

	CellularMetrics []CellularMetrics `json:"cellular"`
}

func (r *DeviceNotFoundError) Error() string {
	return fmt.Sprintf("Device %s not found", r.DeviceMac)
}

func SetStatisticDataDirectory(directory string) {
	// Set the directory to store statistic data
	statisticDataDirectory = directory
}

func SaveStatisticData(macAddress string, data string) {
	// Save the statistic data to the directory

	fileName := fmt.Sprintf("%s.json", macAddress)
	filePath := filepath.Join(statisticDataDirectory, fileName)

	file, err := os.Create(filePath)
	if err != nil {
		fmt.Printf("Error creating file: %v\n", err)
		return
	}
	defer file.Close()

	_, err = file.WriteString(data)
	if err != nil {
		fmt.Printf("Error writing to file: %v\n", err)
	}
}

func GetStatisticData(macAddress string) (string, error) {
	/* Check if device in inventory */
	var dev database.Device
	database.DB.Where("mac_address = ?", macAddress).Preload("Radios").First(&dev)
	if dev.ID == 0 {
		return "", &DeviceNotFoundError{DeviceMac: macAddress}
	}

	// Get the statistic data from the directory
	fileName := fmt.Sprintf("%s.json", macAddress)
	filePath := filepath.Join(statisticDataDirectory, fileName)

	file, err := os.Open(filePath)
	if err != nil {
		fmt.Printf("Error opening file: %v\n", err)
		return "", err
	}
	defer file.Close()

	statisticData, err := os.ReadFile(filePath)
	if err != nil {
		fmt.Printf("Error reading file: %v\n", err)
		return "", err
	}

	return string(statisticData), nil
}

func MemToInt(raw string) int {
	if raw == "" {
		return 0
	}
	res, err := strconv.Atoi(raw)
	if err != nil {
		panic(err)
	}
	return res
}

func ParseMemoryStatsLine(raw string) (key string, value int) {
	fmt.Println(raw)
	text := strings.ReplaceAll(raw[:len(raw)-2], " ", "")
	keyValue := strings.Split(text, ":")
	return keyValue[0], MemToInt(keyValue[1])
}

func ReadMemoryStats() MemoryMetrics {
	file, err := os.Open("/proc/meminfo")
	if err != nil {
		panic(err)
	}
	defer file.Close()
	bufio.NewScanner(file)
	scanner := bufio.NewScanner(file)
	res := MemoryMetrics{}
	for scanner.Scan() {
		key, value := ParseMemoryStatsLine(scanner.Text())
		switch key {
		case "MemTotal":
			res.MemTotal = value
		case "MemFree":
			res.MemFree = value
		case "MemAvailable":
			res.MemAvailable = value
		}
	}
	return res
}

func ReadSystemUptime() int {
	file, err := os.Open("/proc/uptime")
	if err != nil {
		panic(err)
	}
	defer file.Close()
	scanner := bufio.NewScanner(file)
	scanner.Scan()

	duration := scanner.Text()
	uptime, err := strconv.ParseFloat(strings.Split(duration, " ")[0], 64)
	if err != nil {
		panic(err)
	}
	return int(uptime)
}

func GetInterfaceMetric(interfaceName string, metricName string) int {
	file, err := os.Open(fmt.Sprintf("/sys/class/net/%s/statistics/%s", interfaceName, metricName))
	if err != nil {
		panic(err)
	}
	defer file.Close()

	scanner := bufio.NewScanner(file)
	scanner.Scan()
	value, err := strconv.Atoi(scanner.Text())
	if err != nil {
		panic(err)
	}
	return value
}

func GetUplinkDataMetricsForInterface(interfaceName string) UplinkDataMetrics {
	metrics := UplinkDataMetrics{}

	metrics.BytesSent = GetInterfaceMetric(interfaceName, "tx_bytes")
	metrics.BytesReceived = GetInterfaceMetric(interfaceName, "rx_bytes")
	metrics.PacketsSent = GetInterfaceMetric(interfaceName, "tx_packets")
	metrics.PacketsReceived = GetInterfaceMetric(interfaceName, "rx_packets")
	return metrics
}

func GetSystemMetrics() SystemMetrics {
	var systemMetrics SystemMetrics
	systemMetrics.MemoryMetrics = ReadMemoryStats()
	systemMetrics.Uptime = ReadSystemUptime()
	systemMetrics.LoadAverage = 0.0
	systemMetrics.OnlineStatus = true
	systemMetrics.CellularMetrics = []CellularMetrics{
		{
			SignalStrength: 0,
			CarrierName:    "T-Mobile",
			DataMetrics:    GetUplinkDataMetricsForInterface("wwan0"),
		},
	}
	return systemMetrics
}
